// vim: ts=2 sw=2 expandtab cindent
//
// BEGIN FLOCK GPL
//
// Copyright Flock Inc. 2005-2008
// http://flock.com
//
// This file may be used under the terms of the
// GNU General Public License Version 2 or later (the "GPL"),
// http://www.gnu.org/licenses/gpl.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// END FLOCK GPL

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cr = Components.results;
const CU = Components.utils;
CU.import("resource:///modules/FlockXMLUtils.jsm");

const SERVICES_LOCALE_PREFIX = "chrome://flock/locale/services/";
const SERVICES_LOCALE_POSTFIX = ".properties";
const SERVICES_DEFAULT_PROPERTIES_FILE = "chrome://flock/locale/services/services.properties";

function componentToolkit()
{
  this._logger = Cc["@flock.com/logger;1"].createInstance(Ci.flockILogger);
  this._logger.init("componentToolkit");
  this._logger.info("component toolkit object created");
}

componentToolkit.prototype.test =
function componentToolkit_test()
{
  this._logger.info("test!");
}

componentToolkit.prototype.addAllInterfaces =
function componentToolkit_addAllInterfaces(aComponent)
{
  this._logger.info(".addAllInterfaces('"+aComponent._ctk.shortName+"')");
  for each (var iface in aComponent._ctk.interfaces) {
    this.addInterface(aComponent, iface);
  }
}

componentToolkit.prototype.getACUtils =
function componentToolkit_getACUtils(aComponent)
{
  if (!aComponent.acUtils) {
    aComponent.acUtils = Cc["@flock.com/account-utils;1"]
      .getService(Ci.flockIAccountUtils);
  }
  return aComponent.acUtils;
}

componentToolkit.prototype.getWD =
function componentToolkit_getWD(aComponent)
{
  if (!aComponent.webDetective) {
    aComponent.webDetective =
      this.getACUtils(aComponent)
          .useWebDetective(aComponent._ctk.shortName.toLowerCase()+".xml");
  }
  return aComponent.webDetective;
}

componentToolkit.prototype.newResults =
function componentToolkit_newResults()
{
  return Cc["@mozilla.org/hash-property-bag;1"]
    .createInstance(Ci.nsIWritablePropertyBag2);
}

componentToolkit.prototype.getCoop =
function componentToolkit_getCoop(aComponent)
{
  if (!aComponent._coop) {
    aComponent._coop = Cc["@flock.com/singleton;1"]
                       .getService(Ci.flockISingleton)
                       .getSingleton("chrome://flock/content/common/load-faves-coop.js")
                       .wrappedJSObject;
  }
  return aComponent._coop;
}

componentToolkit.prototype.addInterface =
function componentToolkit_addInterface(aComponent, aInterface)
{
  if (!aInterface) return;
  // Add aInterface to aComponent's interface list, if it isn't already there
  if (!aComponent._ctk) { aComponent._ctk = {}; }
  var _ctk = aComponent._ctk;
  if (!_ctk.interfaces) { _ctk.interfaces = []; }
  if (_ctk.interfaces.indexOf(aInterface) < 0) _ctk.interfaces.push(aInterface);

  var proto = aComponent.__proto__;
  var ctk = this;

  switch (aInterface) {

    case "nsISupports":
    {
      if (!proto.QueryInterface) {
        proto.QueryInterface =
        function QueryInterface(aIID)
        {
          for each(var iface in this._ctk.interfaces) {
            if (aIID.equals(Ci[iface])) {
              return this;
            }
          }
          throw Cr.NS_ERROR_NO_INTERFACE;
        };
      }
    }; break;

    case "nsISupportsCString":
    {
      if (!proto.toString) {
        proto.toString =
        function toString()
        {
          return this._ctk.shortName;
        };
      }
    }; break;

    case "nsIClassInfo":
    {
      if (!proto.flags) {
        proto.flags = Ci.nsIClassInfo.SINGLETON;
      }
      if (!proto.implementationLanguage) {
        proto.implementationLanguage = Ci.nsIProgrammingLanguage.JAVASCRIPT;
      }
      if (!proto.classID) {
        proto.classID = aComponent._ctk.CID;
      }
      if (!proto.classDescription) {
        proto.classDescription = aComponent._ctk.description;
      }
      if (!proto.contractId) {
        proto.contractId = aComponent._ctk.contractID;
      }
      if (!proto.getInterfaces) {
        proto.getInterfaces =
        function getInterfaces(aCount)
        {
          var iids = [];
          for each(var iface in this._ctk.interfaces) {
            if (iface) {
              iids.push(Ci[iface]);
            }
          }
          aCount.value = iids.length;
          return iids;
        };
      }
      if (!proto.getHelperForLanguage) {
        proto.getHelperForLanguage =
        function getHelperForLanguage(aLanguage)
        {
          return null;
        };
      }
    }; break;

    case "nsIObserver":
    {
      if (!proto.observe) {
        proto.observe = function observe(aSubject, aTopic, aData) {};
      }
    }; break;

    case "flockIWebService":
    {
      if (!proto.urn) {
        proto.urn = "urn:"+aComponent._ctk.shortName.toLowerCase()+":service";
      }
      if (!proto.shortName) {
        proto.shortName = aComponent._ctk.shortName;
      }
      if (!proto.title) {
        proto.title = aComponent._ctk.fullName;
      }

      if (!proto.getStringBundle) {
        proto.getStringBundle =
        function getStringBundle() {
          if (!aComponent._stringBundle) {
            var sbs = Cc["@mozilla.org/intl/stringbundle;1"]
                      .getService(Ci.nsIStringBundleService);

            try {

              var bundlePath = aComponent._stringBundlePath;
              
              if (!bundlePath) {
                // A default has not been specified so assemble the standard 
                // bundle path for services (this will satisfy Flock created
                // services)
                bundlePath = SERVICES_LOCALE_PREFIX
                           + aComponent._ctk.shortName
                           + SERVICES_LOCALE_POSTFIX;
              }

              // Create the string bundle
              aComponent._stringBundle = sbs.createBundle(bundlePath);

            } catch (e) {
              // Could not create the bundle so use the default bundle for 
              // services in Flock
              aComponent._stringBundle = 
                sbs.createBundle(SERVICES_DEFAULT_PROPERTIES_FILE);
            }
          }

          return aComponent._stringBundle;
        };
      }

      if (!proto.icon) {
        proto.icon = aComponent._ctk.favicon;
      }
      if (!proto.isHidden) {
        proto.isHidden = function isHidden() {
          var pref = "flock.service." + this.shortName + ".hidden";
          var prefService = Cc["@mozilla.org/preferences-service;1"]
                            .getService(Ci.nsIPrefBranch);
          if (prefService.getPrefType(pref)) {
            var isHidden = prefService.getBoolPref(pref);
            return isHidden;
          }

          return false;
        };
      }
    }; break;

    case "flockIWebServiceAccount":
    {
      if (!proto.getService) {
        proto.getService =
        function getService()
        {
          var c_acct = ctk.getCoop(this).get(this.urn);
          return Cc[c_acct.serviceId].getService(Ci.flockIWebService);
        };
      }
      if (!proto.login) {
        proto.login =
        function login(aFlockListener)
        {
          if (aFlockListener) {
            aFlockListener.onError(null, "login");
          }
        };
      }
      if (!proto.logout) {
        proto.logout =
        function logout(aFlockListener)
        {
          var c_acct = this.coopObj;
          if (c_acct.isAuthenticated) {
            c_acct.isAuthenticated = false;
            Cc[c_acct.serviceId].getService(Ci.flockIWebService).logout();
          }
          if (aFlockListener) {
            aFlockListener.onSuccess(this, "logout");
          }
        };
      }
      if (!proto.keep) {
        proto.keep =
        function keep()
        {
          var c_acct = this.coopObj;
          c_acct.isTransient = false;
          var svc = Cc[c_acct.serviceId].getService(Ci.flockIWebService);
          ctk.getACUtils(this).makeTempPasswordPermanent(svc.urn+":"+c_acct.accountId);
        };
      }
      if (!proto.isAuthenticated) {
        proto.isAuthenticated =
        function isAuthenticated()
        {
          return this.coopObj.isAuthenticated;
        };
      }
      function _returnType(aValue, aType)
      {
        var ret = null;
        switch (aType) {
          case "boolean":
            ret = (aValue == "true") ? true : false;
            break;

          case "number":
            ret = parseInt(aValue);
            break;

          default:
            ret = aValue;
            break;
        }
        return ret;
      }
      function _reportError(aMethodName, aMsg) {
        ctk._logger.error("error." + aMethodName + ":" + aMsg);
        throw Components.Exception(aMsg);
      }

      if (!proto.getCustomParam) {
        proto.getCustomParam =
        function getCustomParam(aParamName) {
          ctk._logger.info(".getCustomParam('" + aParamName + "'");
          var paramList = this.coopObj.accountParamList;
          if (!paramList) {
            var msg = "paramList does not exist for account " + this.urn;
            _reportError("getCustomParam", msg);
          }
          param = paramList.getParam(aParamName);
          if (param) {
            return _returnType(param.value, param.type);
          } else {
            var msg = aParamName + " not found";
            _reportError("getCustomParam", msg);
            return null;
          }
        }
      }
      if (!proto.setCustomParam) {
        proto.setCustomParam =
        function setCustomParam(aParamName, aValue)
        {
          ctk._logger.info(".setCustomParam('" + aParamName + "','"
                                               + aValue + "'");
          var param = null;
          var paramList = this.coopObj.accountParamList;
          var val = null;
          if (!paramList) {
            var msg = "paramList does not exist for account " + this.urn;
            _reportError("setCustomParam", msg);
          }
          param = paramList.getParam(aParamName);
          if (param) {
            param.value = aValue;
          } else {
            // bug 12025 cannot make ctk.getCoop(this).AccountParam causes a
            // serious account coop object crash
            ctk.getCoop(this);
            var param = new this._coop.AccountParam(
              {
                key: aParamName,
                value: aValue,
                type: typeof(aValue)
              });
            paramList.children.add(param);
          }
        }
      }
      if (!proto.setParam) {
        proto.setParam = function setParam(aParamName, aValue) {
          try {
            this.coopObj.set(aParamName, aValue);
          } catch(e) {
            _reportError("setParam", aParamName + " does not exist!");
          }
        }
      }
      if (!proto.getParam) {
        proto.getParam = function getParam(aParamName) {
          try {
            return this.coopObj.get(aParamName);
          } catch (e) {
            _reportError("getParam", aParamName + " does not exist!");
            return null;
          }
        }
      }
      if (!proto.getAllCustomParams) {
        proto.getAllCustomParams =
        function getAllCustomParams()
        {
          var paramsList = Cc["@mozilla.org/hash-property-bag;1"]
                           .createInstance(Ci.nsIWritablePropertyBag);
          var paramList = this.coopObj.accountParamList;
          if (!paramList) {
            var msg = "paramList does not exist for account " + this.urn;
            _reportError("getAllCustomParams", msg);
          }
          var list = paramList.children.enumerate();

          while (list.hasMoreElements()) {
            var param = list.getNext();
            var paramValue = _returnType(param.value, param.type);
            paramsList.setProperty(param.key, paramValue);
          }
          return paramsList;
        }
      }
            
    }; break;

    case "flockILoginWebService":
    {
      if (proto.needPassword == undefined) {
        if (aComponent._ctk.needPassword == undefined) {
          proto.needPassword = true;
        } else {
          proto.needPassword = aComponent._ctk.needPassword;
        }
      }
      if (!proto.getAccount) {
        proto.getAccount =
        function getAccount(aURN) {
          var c_acct = ctk.getCoop(this).get(aURN);
          if (!c_acct) {
            return null;
          }
          var acct = new aComponent._ctk.accountClass();
          acct.urn = c_acct.id();
          acct.username = c_acct.name;
          acct.coopObj = c_acct;
          return acct;
        };
      }
      if (!proto.getAuthenticatedAccount) {
        proto.getAuthenticatedAccount =
        function getAuthenticatedAccount() {
          var accounts = ctk.getCoop(this).Account.find(
          {
            serviceId: this._ctk.contractID,
            isAuthenticated: true
          });
          if (accounts.length) {
            var acct = new aComponent._ctk.accountClass();
            acct.coopObj = accounts[0];
            acct.urn = acct.coopObj.id();
            acct.username = acct.coopObj.name;
            return acct;
          }
          return null;
        }
      }
      if (!proto.getAccounts) {
        proto.getAccounts =
        function getAccounts() {
          return ctk.getACUtils(this)
                    .getAccountsForService(aComponent._ctk.contractID);
        };
      }
      if (!proto.removeAccount) {
        proto.removeAccount =
        function removeAccount(aURN) {
          ctk.getACUtils(this).removeAccount(aURN);
        };
      }
      if (!proto.logout) {
        proto.logout =
        function logout() {
          var cookies = ctk.getWD(this)
                           .getSessionCookies(aComponent._ctk.shortName);
          if (cookies) {
            ctk.getACUtils(this).removeCookies(cookies);
          }
        };
      }
      if (!proto.ownsLoginForm) {
        proto.ownsLoginForm =
        function ownsLoginForm(aForm)
        {
          aForm.QueryInterface(Ci.nsIDOMHTMLFormElement);
          var wd = ctk.getWD(this);
          var shortName = aComponent._ctk.shortName.toLowerCase();
          if (wd.detectForm(shortName, "login", aForm, ctk.newResults())) {
            return true;
          }
          if (wd.detectForm(shortName, "signup", aForm, ctk.newResults())) {
            return true;
          }
          if (wd.detectForm(shortName, "changepassword", aForm, ctk.newResults())) {
            return true;
          }
          return false;
        };
      }
      if (!proto.getCredentialsFromForm) {
        proto.getCredentialsFromForm =
        function getCredentialsFromForm(aForm)
        {
          aForm.QueryInterface(Ci.nsIDOMHTMLFormElement);
          var wd = ctk.getWD(this);
          var shortName = aComponent._ctk.shortName.toLowerCase();
          var formType = "login";
          var results = ctk.newResults();
          if (!wd.detectForm(shortName, formType, aForm, results)) {
            formType = "signup"
            results = ctk.newResults();
            if (!wd.detectForm(shortName, formType, aForm, results)) {
              formType = "changepassword"
              results = ctk.newResults();
              if (!wd.detectForm(shortName, formType, aForm, results)) {
                results = null;
              }
            }
          }
          if (results) {
            var pw = {
              QueryInterface: function proto_gcff_pw_QueryInterface(aIID) {
                if (!aIID.equals(CI.nsISupports) &&
                    !aIID.equals(CI.nsILoginInfo) &&
                    !aIID.equals(CI.flockILoginInfo))
                {
                  throw CI.NS_ERROR_NO_INTERFACE;
                }
                return this;
              },
              username: results.getPropertyAsAString("username"),
              password: results.getPropertyAsAString("password"),
              formType: formType
            };
            return pw;
          }
          return null;
        };
      }
      if (!proto.ownsDocument) {
        // Note that ownsDocument() only gets called for documents hosted in a
        // domain for this service... so if no further tests are necessary,
        // it's ok to just return 'true' here.
        proto.ownsDocument =
        function ownsDocument(aDocument)
        {
          return true;
        };
      }
      if (!proto.docRepresentsSuccessfulLogin) {
        proto.docRepresentsSuccessfulLogin =
        function docRepresentsSuccessfulLogin(aDocument)
        {
          return ctk.getWD(this).detect( aComponent._ctk.shortName.toLowerCase(),
                                         "loggedin", aDocument, null );
        };
      }
      if (!proto.getAccountIDFromDocument) {
        proto.getAccountIDFromDocument =
        function getAccountIDFromDocument(aDocument)
        {
          var results = ctk.newResults();
          if (ctk.getWD(this).detect( aComponent._ctk.shortName.toLowerCase(),
                                      "accountinfo", aDocument, results ))
          {
            return results.getPropertyAsAString("accountid");
          }
          return null;
        };
      }
      if (!proto.getSessionValue) {
        proto.getSessionValue =
        function getSessionValue() {
          var svcName = aComponent._ctk.shortName.toLowerCase();
          var cookies = ctk.getWD(this).getSessionCookies(svcName);
          var ckStr, ckUrl = "", val = "";
          while (cookies && cookies.hasMoreElements()) {
            var cookie = cookies.getNext().QueryInterface(CI.nsICookie);
            var url = cookie.host + cookie.path;
            var lengthDiff = ckUrl.length - url.length;
            // no need to re-get the cookie if it's a subdomain of the domain
            if (lengthDiff < 0 || ckUrl.indexOf(url, lengthDiff) == -1) {
              ckUrl = url;
              ckStr = ctk.getACUtils(this).getCookie("http://" + ckUrl, null);
            }
            var index = ckStr ? ckStr.indexOf(cookie.name + "=") : -1;
            if (index != -1) {
              index += cookie.name.length + 1;
              var end = ckStr.indexOf(";", index);  
              val += unescape(ckStr.substring(index, 
                                              end == -1 ? ckStr.length : end));
            }
          }
          return val;
        }
      }
    }; break;

    case "flockISocialWebService":
    {
      if (!proto.markAllMediaSeen) {
        proto.markAllMediaSeen = function markAllMediaSeen(aPersonURN) {
          var identity = ctk.getCoop(this).get(aPersonURN);
          if (!identity) {
            ctk._logger.error("flockISocialWebService.markAllMediaSeen: "
                              + "identity \""
                              + aPersonURN
                              + "\" doesn't exist.");
            return;
          }
          identity.unseenMedia = 0;
        };
      }
    }; break;

    case "flockISocialAccount":
    {
      // AString getFormattedFriendUpdateType(aLastUpdateType, aStatus)
      if (!proto.getFormattedFriendUpdateType) {
        proto.getFormattedFriendUpdateType =
        function defaultImpl_mgetFormattedFriendUpdateType(aLastUpdateType,
                                                           aStatus)
        {
          var pplProperties = "chrome://flock/locale/people/people.properties";
          var sb = CC["@mozilla.org/intl/stringbundle;1"]
                   .getService(CI.nsIStringBundleService)
                   .createBundle(pplProperties);
          switch (aLastUpdateType) {
            case "profile":
              return sb.GetStringFromName("flock.people.mecard"
                                          + ".lastUpdateTypePretty.profile");
              break;
              
            case "status":
              if (aStatus == "") {
                return sb.GetStringFromName("flock.people.mecard"
                                            + ".lastUpdateTypePretty"
                                            + ".statusCleared");
              }
              return sb.GetStringFromName("flock.people.mecard"
                                          + ".lastUpdateTypePretty"
                                          + ".statusUpdated");
              break;

            case "media":
              return sb.GetStringFromName("flock.people.mecard"
                                          + ".lastUpdateTypePretty"
                                          + ".mediaUpdated");
              break;
          }
          return null;
        };
      }
      if (!proto.formatFriendActivityForDisplay) {
        proto.formatFriendActivityForDisplay =
        function formatFriendActivityForDisplay(aFriendActivityUrn) {

          var friendActivity = ctk.getCoop(this).get(aFriendActivityUrn);
          var svc = Cc[friendActivity.identity.serviceId]
                    .getService(Ci.flockIWebService);
          var bundle = svc.getStringBundle();

          if (!bundle) {
            return "";
          }

          var result = "";

          switch (friendActivity.updateType) {
            case "profile":
              result = 
                bundle.GetStringFromName("flock.friendFeed.lastUpdateTypePretty.profile");
              break;

            case "status":
              if (friendActivity.updateValue == "") {
                result = 
                  bundle.GetStringFromName("flock.friendFeed.lastUpdateTypePretty.statusCleared");
              } else {
                result = friendActivity.updateValue;
              }
              break;

            case "media":
              result = 
                bundle.GetStringFromName("flock.friendFeed.lastUpdateTypePretty.mediaUpdated");
              break;
          }

          return(flockXMLDecode(result));
        }
      }

      if (!proto.getInviteFriendsURL) {
        // AString getInviteFriendsURL();
        proto.getInviteFriendsURL =
        function getInviteFriendsURL() {
          var service = aComponent.getService()
                                  .QueryInterface(CI.flockIWebService);
          return ctk.getWD(this).getString(service.shortName.toLowerCase(),
                                           "inviteFriendsURL",
                                           null);
        };
      }

      if (!proto.getFriendCount) {
        proto.getFriendCount = function getFriendCount() {
          var count = 0;
          var c_friendslist = ctk.getCoop(this).get(this.urn).friendsList;
          if (c_friendslist) {
            var friends = c_friendslist.children.enumerate();
            while (friends.hasMoreElements()) {
              count++;
              friends.getNext();
            }
          }
          return count;
        }
      }
      if (!proto.enumerateFriends) {
        proto.enumerateFriends =
        function enumerateFriends() {
          function FriendsEnum(aCoopAccount) {
            this._enum = aCoopAccount.friendsList.children.enumerate();
          }
          FriendsEnum.prototype = {
            hasMoreElements: function FriendsEnum_hasMoreElements() {
              return this._enum.hasMoreElements();
            },
            getNext: function FriendsEnum_getNext() {
              var coopPerson = this._enum.getNext();
              var person = Cc["@flock.com/person;1"]
                           .createInstance(Ci.flockIPerson);
              person.accountId = coopPerson.accountId;
              person.lastUpdate = coopPerson.lastUpdate;
              person.lastUpdateType = coopPerson.lastUpdateType;
              person.name = coopPerson.name;
              person.avatar = coopPerson.avatar;
              person.statusMessage = coopPerson.statusMessage;
              person.unseenMedia = coopPerson.unseenMedia;
              return person;
            }
          }
          var coopAccount = ctk.getCoop(this).get(this.urn);
          return new FriendsEnum(coopAccount);
        };
      }
      if (!proto.markAllMeNotificationsSeen) {
        proto.markAllMeNotificationsSeen =
        function markAllMeNotificationsSeen(aType) { };
      }
    }; break;

    case "flockIMediaWebService":
    {
      if (!proto.decorateForMedia) {
        proto.decorateForMedia = function decorateForMedia(aDocument) { };
      }
      if (!proto.getChannel) {
        proto.getChannel = function getChannel(aChannelId) {
          if (!(aChannelId in this._channels)) {
            return null;
          }

          var mc = Cc["@flock.com/media-channel;1"]
                   .createInstance(Ci.flockIMediaChannel);
          mc.id = aChannelId;
          mc.title = this._channels[aChannelId].title
          mc.supportsSearch = this._channels[aChannelId].supportsSearch;
          return mc;
        }
      }
      if (!proto.enumerateChannels) {
        proto.enumerateChannels = function enumerateChannel() {
          var enumerateChannelsRecursively =
          function enumerateChannels_Recursively(aChannelObj) {
            var ar = [];
            for (var id in aChannelObj) {
              var mc = Cc["@flock.com/media-channel;1"]
                       .createInstance(Ci.flockIMediaChannel);
              mc.id = id;
              mc.title = aChannelObj[id].title;
              mc.supportsSearch = aChannelObj[id].supportsSearch;
              var subChannels = null; 
              if (aChannelObj[id].subChannels) {
                // Recursively append subChannels
                subChannels =
                  enumerateChannelsRecursively(aChannelObj[id].subChannels);
              }
              mc.subChannels = subChannels;
              ar.push(mc);
            };
            var rval = {
              getNext: function rval_getNext() {
                var rval = ar.shift();
                return rval;
              },
              hasMoreElements: function rval_hasMoreElements() {
                return (ar.length > 0);
              }
            };      
            return rval;
          };
          return enumerateChannelsRecursively(this._channels);
        }
      }
      if (!proto.getMediaItemFormatter) {
        proto.getMediaItemFormatter = function getMediaItemFormatter() {
          // Use the default formatting functions defined in flockPhoto.js
          return null;
        }
      }
      if (!proto.getIconForQuery) {
        proto.getIconForQuery = function getIconForQuery(aQuery) {
          return this.icon;
        }
      }
    }; break;

    case "flockIMediaUploadWebService":
    {
      if (!proto.getAlbumsForUpload) {
        proto.getAlbumsForUpload =
        function getAlbumsForUpload(aFlockListener, aUsername) {
          this.getAlbums(aFlockListener, aUsername);
        }
      }
    }; break;

    case "flockIMediaEmbedWebService":
    {
      if (!proto.checkIsStreamUrl) {
        proto.checkIsStreamUrl = function checkIsStreamUrl(aUrl) {
          return false;
        }
      }
      if (!proto.getMediaQueryFromURL) {
        proto.getMediaQueryFromURL = function getMediaQueryFromURL(aURL, aFlockListener) {
          aFlockListener.onError(null, "Media service is not supported.");
        }
      }
    }; break;

    case "flockIRichContentDropHandler":
    {
      if (!proto._handleTextareaDrop) {
        // Helper function to handle a DnD to a supported service TEXTAREA
        proto._handleTextareaDrop = function _handleTextareaDrop(aServicename,
                                                                 aDomains,
                                                                 aTextarea,
                                                                 aCallback)
        {
          const WD_XPATH = 0;
          const WD_FLAVOUR = 1;

          if (aTextarea instanceof Ci.nsIDOMHTMLTextAreaElement) {
            var doc = aTextarea.ownerDocument;
            if (doc instanceof Ci.nsIDOMHTMLDocument) {
              var wd = ctk.getWD(this);
              var domainsArray = aDomains.split(",");
              for each (var domain in domainsArray) {
                if (wd.testDomain(doc.URL, domain)) {
                  // Retrieve the specific fields from Web Detective we cannot
                  // DnD to
                  var fields = wd.getString(aServicename,
                                            "avoidDnDXPathFields", null);
                  if (fields) {
                    fields = fields.split(";");
                    for each (var avoidDnD in fields) {
                      var xPath = wd.getString(aServicename, avoidDnD, null);
                      if (xPath) {
                        var results = doc.evaluate(xPath, doc, null,
                                                   Ci.nsIDOMXPathResult.ANY_TYPE,
                                                   null);
                        if (results && results.iterateNext() == aTextarea) {
                          // The matching field does not accept rich content,
                          // bail
                          return true;
                        }
                      }
                    }
                  }

                  // Retrieve the specific fields from Web Detective we can
                  // DnD to
                  var pairs = [];
                  fields = wd.getString(aServicename, "dndXPathFields", null);
                  if (fields) {
                    fields = fields.split(";");
                    for each (var xpfFields in fields) {
                      pairs.push(xpfFields);
                    }
                  }

                  // Go through the list of DnD fields to find a match
                  for each (var xpfPair in pairs) {
                    var xpf = xpfPair.split(",");
                    var xPath = wd.getString(aServicename, xpf[WD_XPATH], null);
                    if (xPath) {
                      var results = doc.evaluate(xPath, doc, null,
                                                 Ci.nsIDOMXPathResult.ANY_TYPE,
                                                 null);
                      if (results && results.iterateNext() == aTextarea) {
                        // Let the service perform the drop via the callback
                        aCallback(xpf[WD_FLAVOUR]);
                        return true;
                      }
                    }
                  }
                }
              }
            }
          }
          return false;
        }
      }
    }; break;

    default:
      break;
  }
}
