// BEGIN FLOCK GPL
//
// Copyright Flock Inc. 2005-2008
// http://flock.com
//
// This file may be used under the terms of the
// GNU General Public License Version 2 or later (the "GPL"),
// http://www.gnu.org/licenses/gpl.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// END FLOCK GPL

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cu = Components.utils;
  
Cu.import("resource:///modules/FlockStringBundleHelpers.jsm");
Cu.import("resource:///modules/FlockSvcUtils.jsm");

const BLOGGER_CID = Components.ID('{dec6797b-2155-4a79-9872-30f142074f0d}');
const BLOGGER_CONTRACTID = '@flock.com/people/blogger;1';
const BLOGGER_FAVICON = "chrome://flock/content/services/blogger/favicon.png";
const SERVICE_ENABLED_PREF          = "flock.service.blogger.enabled";
const CATEGORY_COMPONENT_NAME       = "Blogger JS Component"
const CATEGORY_ENTRY_NAME           = "blogger"


var gCompTK;
function getCompTK() {
  if (!gCompTK) {
    gCompTK = Components.classes["@flock.com/singleton;1"]
                        .getService(Components.interfaces.flockISingleton)
                        .getSingleton("chrome://flock/content/services/common/load-compTK.js")
                        .wrappedJSObject;
  }
  return gCompTK;
}

function flockBLService() {
  var loader = Cc['@mozilla.org/moz/jssubscript-loader;1']
               .getService(Ci.mozIJSSubScriptLoader);
  loader.loadSubScript('chrome://browser/content/utilityOverlay.js');
  loader.loadSubScript("chrome://flock/content/blog/atom.js");
  loader.loadSubScript("chrome://flock/content/blog/blogBackendLib.js");

  var obs = Cc["@mozilla.org/observer-service;1"]
                      .getService(Ci.nsIObserverService);
  obs.addObserver(this, 'xpcom-shutdown', false);

  this.acUtils = Cc["@flock.com/account-utils;1"].getService(Ci.flockIAccountUtils);

  this.supportsPostReplace = true;
  this.mIsInitialized = false;

  this._ctk = {
    interfaces: [
      "nsISupports",
      "nsIClassInfo",
      "nsIObserver",
      "nsISupportsCString",
      "flockIWebService",
      "flockIAuthenticateNewAccount",
      "flockILoginWebService",
      "flockIBlogWebService",
      "flockIMigratable"
    ],
    shortName: "blogger",
    fullName: "Blogger",
    description: "Google's Blogger.com Web Service",
    favicon: BLOGGER_FAVICON,
    CID: BLOGGER_CID,
    contractID: BLOGGER_CONTRACTID,
    accountClass: flockBLAccount
  };
  this._profiler = Cc["@flock.com/profiler;1"].getService(Ci.flockIProfiler);

  FlockSvcUtils.flockIWebService.addDefaultMethod(this, "url");
  FlockSvcUtils.flockILoginWebService.addDefaultMethod(this, "loginURL");

  this.init();
}

// Helpers

// For Google Auth (ClientLogin)
flockBLService.prototype._parseError =
function flockBLService__parseError(aErrorString) {
  var result = {};
  var lines = aErrorString.split('\n');
  for (i in lines) {
    if (lines[i].length > 0) {
      entry = lines[i].split('=');
      result[entry[0]] = entry[1];
    }
  }

  return result;
};

// Migrate coop objects
flockBLService.prototype._migrateFromCoop =
function flockBLService__migrateFromCoop(aContext) {
  this._logger.debug("_migrateFromCoop(...)");
  if (aContext.majorVersion < 2) {
    // 2.0 changes
    var accounts = this.getAccounts();
    while (accounts.hasMoreElements()) {
      var account = accounts.getNext();
      if (account instanceof Ci.flockIWebServiceAccount) {
        // Migrate old coop urn (USER@gmail.com or user@googlemail.com) to
        // new urn (USER)
        var urn = account.urn.match(/(.+:(.+))@g(?:|oogle)mail\.com/);
        if (urn) {
          // Elements 0, 1 and 2 are valid if urn is non-null
          // Format:
          //   urn[0]: urn:flock:<service>:account:<accountId>@gmail.com
          //   urn[1]: urn:flock:<service>:account:<accountId>
          //   urn[2]: <accountId>
          this._logger.debug("  urn[0]: " + urn[0] + "\n  urn[1]: " + urn[1]
                             + "\n  urn[2]: " + urn[2]);
          var coopAccount = this.faves_coop.get(urn[0]);
          coopAccount.accountId = urn[2];
          coopAccount.changeId(urn[1]);

          // Update the Blog child nodes so we don't create RDF orphans
          var childrenEnum = coopAccount.children.enumerate();
          while (childrenEnum.hasMoreElements()) {
            var child = childrenEnum.getNext();
            if (child.isInstanceOf(this.faves_coop.Blog)) {
              var oldId = child.id();
              var newId = oldId.replace(/@g(?:|oogle)mail\.com/, "");
              if (oldId != newId) {
                this._logger.debug("Migrating '" + oldId
                                   + "' to '" + newId + "'\n");
                this.faves_coop.get(oldId).changeId(newId);
              }
            }
          }

          // Update the urn entry in the Password Manager
          var oldAcctId = urn[0].replace(/^urn:flock:blogger:account:/, "");
          var pw = this.acUtils.removePassword(this.urn + ":" + oldAcctId);
          if (pw) {
            this.acUtils.setPassword(this.urn + ":" + urn[2],
                                     pw.username,
                                     pw.password);
          }
        }
      }
    }
  }
  // Makes this a generator-iterator
  yield true;
};

// nsIObserver
flockBLService.prototype.observe =
function flockBLService_observe(subject, topic, state) {
  switch (topic) {
    case 'xpcom-shutdown':
      var obs = Cc["@mozilla.org/observer-service;1"]
        .getService(Ci.nsIObserverService);
      obs.removeObserver(this, 'xpcom-shutdown');
      return;
  }
};

// flockIWebService implementation
flockBLService.prototype.addAccount =
function flockBLService_addAccount(aAccountID, aIsTransient, aFlockListener) {
  this._logger.debug("addAccount('" + aAccountID + "', "
                     + aIsTransient + ", aFlockListener)");

  var pw = this.acUtils.getPassword(this.urn + ":" + aAccountID);
  var user = pw ? pw.username : aAccountID;
  var acctId = aAccountID.replace(/@g(?:|oogle)mail\.com$/, "");
  var accountURN =
    this.acUtils.createAccount(this, acctId, user, null, aIsTransient);
  var account = this.faves_coop.get(accountURN);
  this.USER = accountURN;

  var acct;

  // Add the blog account
  blsvc = this;
  var blogListener = {
    onResult: function bl_onResult(aSimpleEnum) {
      var theBlog;
      while (aSimpleEnum.hasMoreElements()) {
        theBlog = aSimpleEnum.getNext();
        theBlog.QueryInterface(Ci.nsIPropertyBag2);
        var title = theBlog.getPropertyAsAString("title");
        var URL = theBlog.getPropertyAsAString("URL");
        var APILink = theBlog.getPropertyAsAString("APILink");
        var authToken = theBlog.getPropertyAsAString("authToken");
        var blogURN = accountURN + ":" + title;
        theCoopBlog = new blsvc.faves_coop.Blog(blogURN, {
          name: title,
          title: title,
          blogid: "",
          URL: URL,
          apiLink: APILink,
          authtoken: authToken
        });
        account.children.addOnce(theCoopBlog);
      }
      if (aFlockListener) {
        aFlockListener.onSuccess(acct, "addAccount");
      }
    },
    onFault: function bl_onFault(aError) {
      blsvc.faves_coop.accounts_root.children.remove(account);
      account.destroy();
      if (aFlockListener) {
        var error = Components.classes['@flock.com/error;1'].createInstance(Components.interfaces.flockIError);
        error.errorCode = Ci.flockIError.BLOG_UNKNOWN_ERROR;
        error.serviceErrorString = aError;
        aFlockListener.onError(error, "FAULT");
      }
    },
    onError: function bl_onError(aFlockError) {
      blsvc.faves_coop.accounts_root.children.remove(account);
      account.destroy();
      if(aFlockListener) {
        aFlockListener.onError(aFlockError, "ERROR");
      }
    }
  }

  var apiLink = "http://www.blogger.com/feeds/default/blogs";
  this.getUsersBlogs(blogListener, apiLink, acctId, pw.password);

  acct = this.getAccount(accountURN);
  return acct;
};

flockBLService.prototype.init =
function flockBLService_init() {
  FlockSvcUtils.getLogger(this);
  this._logger.init("blogger");
  this._logger.debug(".init()");

  // Prevent re-entry
  if (this.mIsInitialized) return;
  this.mIsInitialized = true;

  var evtID = this._profiler.profileEventStart("blogger-init");

  this.prefService = Components.classes["@mozilla.org/preferences-service;1"]
                               .getService(Components.interfaces.nsIPrefBranch);
  if ( this.prefService.getPrefType(SERVICE_ENABLED_PREF) &&
       !this.prefService.getBoolPref(SERVICE_ENABLED_PREF) )
  {
    this._logger.info("Pref " + SERVICE_ENABLED_PREF
                              + " set to FALSE... not initializing.");
    var catMgr = Cc["@mozilla.org/categorymanager;1"]
                 .getService(Ci.nsICategoryManager);
    catMgr.deleteCategoryEntry("wsm-startup", CATEGORY_COMPONENT_NAME, true);
    catMgr.deleteCategoryEntry("flockMigratable", CATEGORY_ENTRY_NAME, true);
    catMgr.deleteCategoryEntry("flockWebService", CATEGORY_ENTRY_NAME, true);
    return;
  }

  this.acUtils = Cc["@flock.com/account-utils;1"]
                 .getService(Ci.flockIAccountUtils);

  this.faves_coop = Cc["@flock.com/singleton;1"]
                    .getService(Ci.flockISingleton)
                    .getSingleton("chrome://flock/content/common/load-faves-coop.js")
                    .wrappedJSObject;

  this.blService = new this.faves_coop.Service('urn:blogger:service');
  this.blService.name = 'blogger';
  this.blService.desc = 'The Blogger.com Service';
  this.blService.logoutOption = false;
  this.blService.domains = "google.com,blogger.com";
  this.blService.serviceId = BLOGGER_CONTRACTID;

  this.urn = this.blService.id();
  this.webDetective = this.acUtils.useWebDetective("blogger.xml");
  
  // Initialize member that specifies path for localized string bundle
  this._stringBundlePath = "";

  // Load auth manager
  this._authMgr = Cc["@flock.com/authmgr;1"]
                  .getService(Ci.flockIAuthenticationManager);
  this._authMgr.joinNetwork(BLOGGER_CONTRACTID, "google");

  this._profiler.profileEventEnd(evtID, "");
};

flockBLService.prototype.refresh =
function flockBLService_refresh(aURN, aPollingListener) {
  throw Components.results.NS_ERROR_ABORT;
};


// BEGIN flockIBlogWebService interface

function atomBlogListener(aPublishListener) {
  FlockSvcUtils.getLogger(this).init("atomBlogListener");
  this.publishListener = aPublishListener;
}

atomBlogListener.prototype = {
  onResult: function atomBlogListener_onResult(aResult) {
    this.publishListener.onResult(aResult.atomid);
  },
  onError: function atomBlogListener_onError(aError) {
    this._logger.error("<<<<<<<<<< Google API: SERVER TO FLOCK");
    this._logger.error("ERROR " + aError);
    this.publishListener.onError(aError);
  },
  onFault: function atomBlogListener_onFault(aError) {
    this._logger.error("<<<<<<<<<< Google API: SERVER TO FLOCK");
    this._logger.error("FAULT " + aError);
    this.publishListener.onFault(aError);
  }
};

flockBLService.prototype.doAuthRequest =
function flockBLService_doAuthRequest(aBlogListener,
                                      method,
                                      url,
                                      body,
                                      processor)
{
  var inst = this;
  this._req = Cc['@mozilla.org/xmlextras/xmlhttprequest;1']
              .createInstance(Ci.nsIXMLHttpRequest);
  this._req.onreadystatechange = function doAuthRequest_ORSC(aEvt) {
    inst._logger.info("<<<<<<<<<< Google API: SERVER TO FLOCK");
    inst._logger.info("Request readyState: " + inst._req.readyState);
    if (inst._req.readyState == 4) {
      var status = inst._req.status;
      inst._logger.info("Request status: " + inst._req.status);
      inst._logger.debug("\nRESPONSE\n" + inst._req.responseText);
      if (status == 200 || status == 201 || status == 205) {
        try {
          processor(aBlogListener, inst);
        } catch (e) {
          inst._logger.error(e + " " + e.lineNumber);
        }
      } else {
        inst._logger.error("Error: " + inst._req.responseText);
        var answer = inst._parseError(inst._req.responseText);
        var error = Cc['@flock.com/error;1'].createInstance(Ci.flockIError);
        error.serviceErrorString = answer["Error"];
        inst._logger.error("***" + error.serviceErrorString + "***");
        switch (error.serviceErrorString) {
          case "BadAuthentication": // Invalid login/pass
            error.errorCode = Ci.flockIError.BLOG_INVALID_AUTH;
            break;
          case "CaptchaRequired":
            error.errorCode = Ci.flockIError.BLOG_CAPTCHA_REQUIRED;
            break;
           case "NotVerified":
            error.errorCode = Ci.flockIError.BLOG_NOT_VERIFIED;
            break;           
          default: // Unknown error code
            error.errorCode = Ci.flockIError.BLOG_UNKNOWN_ERROR;
        }
        aBlogListener.onError(error);
      }
    }
  };
  rval = this._req.open(method, url, true);
  this._req.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
  this._logger.info(">>>>>>>>>> Google API: FLOCK TO SERVER");
  this._logger.info("\nSENDING\n" + body);
  this._logger.info("\nTO\n" + method + " @ " + url);
  rval = this._req.send(body);
};

flockBLService.prototype.doRequest =
function flockBLService_doRequest(aBlogListener,
                                  method,
                                  url,
                                  body,
                                  authToken,
                                  processor) 
{
  var inst = this;
  this._req = Cc['@mozilla.org/xmlextras/xmlhttprequest;1']
              .createInstance(Ci.nsIXMLHttpRequest);
  this._req.onreadystatechange = function doRequest_ORSC(aEvt) {
    inst._logger.info("<<<<<<<<<< Google API: SERVER TO FLOCK");
    inst._logger.info("Request readyState: " + inst._req.readyState);
    if (inst._req.readyState == 4) {
      inst._logger.info("\nRESPONSE\n" + inst._req.responseText);
      var status = inst._req.status;
      if (status == 200 || status == 201 || status == 205) {
        try {
          processor(aBlogListener, inst, authToken);
        } catch(e) {
          inst._logger.error(e + " " + e.lineNumber);
        }
      } else {
        var faultString = inst._req.responseText;
        inst._logger.error(faultString);
        aBlogListener.onFault(faultString);
      }
    }
  };
  rval = this._req.open(method, url, true);
  this._req.setRequestHeader("Authorization", "GoogleLogin auth="+authToken);
  this._logger.info(">>>>>>>>>> Google API: FLOCK TO SERVER");
  this._logger.info("\nSENDING\n" + body);
  this._logger.info("\nTO\n" + method + " @ " + url);
  rval = this._req.send(body);
};

flockBLService.prototype.parseUsersBlogs =
function flockBLService_parseUsersBlogs(aBlogListener, inst, authToken) {
  try {
    var result = new Array();
    var dom = inst._req.responseXML;

    var domEntries = dom.getElementsByTagName("entry");
    for (i=0; i<domEntries.length; i++) {
      domEntry = domEntries[i];
      title = domEntry.getElementsByTagName("title")[0].textContent;
      var blog = Cc["@mozilla.org/hash-property-bag;1"]
                 .createInstance(Ci.nsIWritablePropertyBag2);
      blog.setPropertyAsAString("title", title);
      blog.setPropertyAsAString("api", this.shortName);
      blog.setPropertyAsAString("authToken", authToken);
      var links = domEntry.getElementsByTagName("link");
      for (j=0; j<links.length; j++) {
        var link = links[j]
        switch (link.getAttribute("rel")) {
          case "alternate":
            blog.setPropertyAsAString("URL", link.getAttribute("href"));
            break;
          case "http://schemas.google.com/g/2005#post":
            blog.setPropertyAsAString("APILink", link.getAttribute("href"));
            break;
        }
      }
      result.push(blog);
    }
    debug("Found "+ result.length +" blogs\n");
    aBlogListener.onResult(new simpleEnumerator(result));
  }
  catch(e) {
    var logger = Cc['@flock.com/logger;1']
       .createInstance(Ci.flockILogger);
    logger.init("blog");
    logger.error(e + " " + e.lineNumber);
    aBlogListener.onError(e + " " + e.lineNumber);
  }
};

flockBLService.prototype.parseRecentPosts =
function flockBLService_parseRecentPosts(aBlogListener, inst) {
  var getNamedChild = function(node, name) {
    for(var i=0;i<node.childNodes.length;++i) {
      if(node.childNodes[i].nodeName==name)
        return node.childNodes[i];
    }
    return null;
  };

  var result = new Array();

  debug(inst._req.responseText);

  var dom = inst._req.responseXML;
  var entries = dom.getElementsByTagName("entry");
  for(var i=0;i<entries.length;++i) {
    try {
      var entry_n = entries[i];
      var post = new BlogPost();
      post.title = getNamedChild(entry_n, "title").firstChild.nodeValue;
      post.issued = getNamedChild(entry_n, "published").firstChild.nodeValue;

      var atomid_n = entry_n.getElementsByTagName("id")[0];
      post.postid = "";
      if(atomid_n) post.postid = atomid_n.firstChild.nodeValue; //.split('/').pop();

      var link_n = null;
      for(var j=0;j<entry_n.childNodes.length;++j) {
        if(entry_n.childNodes[j].nodeName=="link") {
          var tmp = entry_n.childNodes[j];
          if(tmp.getAttribute("rel").match(/edit/)) {
            link_n = tmp;
          }
          if(tmp.getAttribute("rel").match(/alternate/)) {
            permalink_n = tmp;
          }
        }
      }

      var permaLink = permalink_n.getAttribute("href");
      var href = link_n.getAttribute("href");
      // href.match(/.+\/(.+)/);
      post.editURI = href; // RegExp.$1;
      debug("post.editURI: "+post.editURI+"\n");

      result.push(post);
    }
    catch(e) {
      var logger = Cc['@flock.com/logger;1']
        .createInstance(Ci.flockILogger);
      logger.error(e + " " + e.lineNumber + " " + e.fileName);
    }
  }
  aBlogListener.onResult(new simpleEnumerator(result));
};

flockBLService.prototype.newPost =
function flockBLService_newPost(aPublishListener,
                                aBlogId,
                                aPost,
                                aPublish,
                                aNotifications)
{
  var svc = this;
  var gBlogService = Cc['@flock.com/flock-blog;1']
                     .getService(Ci['flockIBlogService']);
  var account = gBlogService.getAccount(aBlogId).wrappedJSObject;

  var blogListener = {
    onResult: function newPost_onResult(aResult) {
      aPublishListener.onResult(aResult.atomid);
    },
    onError: function newPost_onError(aError) {
      svc._logger.error("<<<<<<<<<< Google API: SERVER TO FLOCK");
      svc._logger.error("ERROR " + aError.serviceErrorString);
      aPublishListener.onError(aError);
    },
    onFault: function newPost_onFault(aError) {
      svc._logger.error("<<<<<<<<<< Google API: SERVER TO FLOCK");
      svc._logger.error("FAULT " + error.serviceErrorString);
      if (error.serviceErrorCode == 401) {
        // The token is bad or missing, let's get a brand new one
        var parseAuthToken = function (aBlogListener, inst) {
          var response = inst._req.responseText;
          response.match(/Auth=(.+)/);
          var token = RegExp.$1;
          var coopBlog = svc.faves_coop.get(aBlogId);
          coopBlog.authtoken = token;
          svc.newPost(aPublishListener, aBlogId, aPost, aPublish, aNotifications);
        };
        var body = "Email="
                 + encodeURIComponent(account.username)
                 + "&Passwd="
                 + encodeURIComponent(account.password)
                 + "&service=blogger&source=FlockInc-Flock-0.8";
        var url = "https://www.google.com/accounts/ClientLogin";
        svc.doAuthRequest(blogListener, "POST", url, body, parseAuthToken);
      }
      else
        aListener.onFault(error);
    }
  }

  var atomEntry = {
    title: aPost.title,
    content: aPost.description
  };
  var labels = new Array();
  if (aPost.tags)
    while (aPost.tags.hasMore()) {
      var label = aPost.tags.getNext();
      if (label.length > 0)
        labels.push(label);
    }
  if (labels.length > 0)
    atomEntry.categories = labels;
  // Hack because Blogger announce www but takes beta
  var url = account.apiLink.replace('www', 'beta');
  flockAtomPost(blogListener, url, atomEntry, account.authtoken);
};

flockBLService.prototype.editPost =
function flockBLService_editPost(aPublishListener,
                                 aBlogId,
                                 aPost,
                                 aPublish,
                                 aNotifications)
{
  var gBlogService = Cc['@flock.com/flock-blog;1']
                     .getService(Ci['flockIBlogService']);
  var account = gBlogService.getAccount(aBlogId).wrappedJSObject;

  if (account.username) { // Blogger beta
    var blogListener = new atomBlogListener(aPublishListener);
    var account = gBlogService.getAccount(aBlogId).wrappedJSObject;

    var atomEntry = {
      id: aPost.postid,
      title: aPost.title,
      content: aPost.description,
      issued: aPost.issued
    };
    var labels = new Array();
    while (aPost.tags.hasMore())
      labels.push(aPost.tags.getNext());
    if (labels.length > 0)
      atomEntry.categories = labels;
    // Hack because Blogger announce www but takes beta
    var url = aPost.editURI.replace('www', 'beta');
    flockAtomEdit(blogListener, url, atomEntry, account.authtoken);
  }
  else {
    var atomAPI = Cc['@flock.com/blog/service/atom;1']
                  .getService(Ci.flockIBlogWebService);
    atomAPI.editPost(aPublishListener, aBlogId, aPost, aPublish, aNotifications);
  }
};

flockBLService.prototype.deletePost =
function flockBLService_deletePost(aBlogListener, aBlogId, aPostid) {
  var gBlogService = Cc['@flock.com/flock-blog;1']
                     .getService(Ci['flockIBlogService']);
  var account = gBlogService.getAccount(aBlogId).wrappedJSObject;

  if (account.username) { // Blogger beta
    var handleDelete = function(listener, inst) {
      listener.onResult(1);
    }

    var gBlogService = Cc['@flock.com/flock-blog;1']
                       .getService(Ci['flockIBlogService']);
    var account = gBlogService.getAccount(aBlogId).wrappedJSObject;
    var url = account.apiLink;
    url += "/" + aPostid;
    // if(aEditURI) url = aEditURI;
    this.doRequest(aBlogListener, "DELETE", url, null,
                   account.authtoken, handleDelete);
  }
  else {
    var atomAPI = Cc['@flock.com/blog/service/atom;1']
                  .getService(Ci.flockIBlogWebService);
    atomAPI.deletePost(aBlogListener, aBlogId, aPostid);
  }
};

flockBLService.prototype.getUsersBlogs =
function flockBLService_getUsersBlogs(aBlogListener,
                                      aAPILink,
                                      aUsername,
                                      aPassword)
{
  var inst = this;

  var parseAuthToken = function (listener, inst){
    var response = inst._req.responseText;
    response.match(/Auth=(.+)/);
    var token = RegExp.$1;
    debug("Found the token: "+token+"\n");
    inst.doRequest(aBlogListener, "GET", aAPILink, null,
                   token, inst.parseUsersBlogs);
  };

  var body = "Email="
           + encodeURIComponent(aUsername)
           + "&Passwd="
           + encodeURIComponent(aPassword)
           + "&service=blogger&source=FlockInc-Flock-1.2";
  var url = "https://www.google.com/accounts/ClientLogin";
  this.doAuthRequest(aBlogListener, "POST", url, body, parseAuthToken);
};

flockBLService.prototype.getRecentPosts =
function flockBLService_getRecentPosts(aBlogListener, aBlogId, aNumber) {
  var gBlogService = Cc['@flock.com/flock-blog;1']
                     .getService(Ci['flockIBlogService']);
  var account = gBlogService.getAccount(aBlogId).wrappedJSObject;

  var url = account.apiLink;
  url.match(/(.+\/)(.+)/);
  if (RegExp.$2 == "post") {
    url = RegExp.$1 + "feed";
  }
  
  var blogListener = {
    onResult: function getRecentPosts_onResult(aResult) {
      aBlogListener.onResult(aResult);
    },
    onError: function getRecentPosts_onError(aError) {
      svc._logger.error("<<<<<<<<<< Google API: SERVER TO FLOCK");
      svc._logger.error("ERROR " + aError.serviceErrorString);
      aBlogListener.onError(aError);
    },
    onFault: function getRecentPosts_onFault(aError) {
      svc._logger.error("<<<<<<<<<< Google API: SERVER TO FLOCK");
      svc._logger.error("FAULT " + aError.serviceErrorString);
      if (error.serviceErrorCode == 401) {
        // The token is bad or missing, let's get a brand new one
        var parseAuthToken = function (listener, inst){
          var response = inst._req.responseText;
          response.match(/Auth=(.+)/);
          var token = RegExp.$1;
          var coopBlog = svc.faves_coop.get(aBlogId);
          coopBlog.authtoken = token;
          inst.doRequest(blogListener, "GET", url, null,
                         token, inst.parseRecentPosts);
        };
        var body = "Email="
                 + encodeURIComponent(account.username)
                 + "&Passwd="
                 + encodeURIComponent(account.password)
                 + "&service=blogger&source=FlockInc-Flock-0.8";
        var url = "https://www.google.com/accounts/ClientLogin";
        inst.doAuthRequest(blogListener, "POST", url, body, parseAuthToken);
      } else {
        aBlogListener.onFault(error);
      }
    }
  }
  
  this.doRequest(blogListener, "GET", url, null,
                 account.authtoken, this.parseRecentPosts);
};

flockBLService.prototype.getCategoryList =
function flockBLService_getCategoryList(aBlogListener, aBlogId) {
  aBlogListener.onResult(null);
};

// END flockIBlogWebService interface


// BEGIN flockILoginWebService interface

flockBLService.prototype.docRepresentsSuccessfulLogin =
function flockBLService_docRepresentsSuccessfulLogin(aDocument) {
  this._logger.debug(".docRepresentsSuccessfulLogin(...)");
  aDocument.QueryInterface(Components.interfaces.nsIDOMHTMLDocument);
  return this.webDetective.detect("blogger", "loggedin", aDocument, null);
};

flockBLService.prototype.ownsDocument =
function flockBLService_ownsDocument(aDocument) {
  this._logger.debug(".ownsDocument(...)");
  aDocument.QueryInterface(Components.interfaces.nsIDOMHTMLDocument);
  return this.webDetective.detect("blogger", "ownsDocument", aDocument, null);
};

flockBLService.prototype.updateAccountStatusFromDocument =
function flockBLService_updateAccountStatusFromDocument(aDocument,
                                                        aAcctURN,
                                                        aFlockListener)
{
  this._logger.debug("updateAccountStatusFromDocument('" + aAcctURN + "')");
  if (aAcctURN) {
    // We're logged in to this account, but we still need to grab the user's
    // avatar
    this._authMgr.reportCompleteLoginToNetwork(aAcctURN);
    this.acUtils.ensureOnlyAuthenticatedAccount(aAcctURN);
    var results = Cc["@mozilla.org/hash-property-bag;1"]
                  .createInstance(Ci.nsIWritablePropertyBag2);
    if (this.webDetective
            .detect("blogger", "accountinfo", aDocument, results))
    {
      var avatarURL = null;
      try {
        avatarURL = results.getPropertyAsAString("avatarURL");
      } catch(e) {
        this._logger.debug("No avatar found");
      }
      if (avatarURL) {
        var c_acct = this.faves_coop.get(aAcctURN);
        c_acct.avatar = avatarURL;
      }
    }
  } else if (this.webDetective
                 .detect("blogger", "loggedout", aDocument, null))
  {
    // We logged out (of all accounts)
    this.acUtils.markAllAccountsAsLoggedOut(BLOGGER_CONTRACTID);
    this._authMgr.reportCompleteLogoutFromNetwork(this.contractId);
  }
};

flockBLService.prototype.logout =
function flockBLService_logout() {
  this._logger.debug(".logout()");
  var cookieManager = Components.classes["@mozilla.org/cookiemanager;1"]
                                .getService(Components.interfaces.nsICookieManager);
  cookieManager.remove(".blogger.com", "blogger_SID", "/", false);
  cookieManager.remove(".blogger.com", "B2I", "/", false);
  cookieManager.remove(".blogger.com", "ServerID", "/", false);
  cookieManager.remove(".blogger.com", "S", "/", false);
  cookieManager.remove(".blogger.com", "__utma", "/", false);
  cookieManager.remove(".blogger.com", "__utmb", "/", false);
  cookieManager.remove(".blogger.com", "__utmc", "/", false);
  cookieManager.remove(".blogger.com", "__utmz", "/", false);
  cookieManager.remove(".blogger.com", "NSC_cmphhfs-fyu", "/", false);
  cookieManager.remove("blogger.com", "NSC_cmphhfs-fyu", "/", false);
  cookieManager.remove("www.blogger.com", "NSC_cmphhfs-fyu", "/", false);
  cookieManager.remove("www.blogger.com", "JSESSIONID", "/", false);
  cookieManager.remove(".www2.blogger.com", "__utma", "/", false);
  cookieManager.remove(".www2.blogger.com", "__utmb", "/", false);
  cookieManager.remove(".www2.blogger.com", "__utmc", "/", false);
  cookieManager.remove(".www2.blogger.com", "__utmz", "/", false);
  cookieManager.remove("www2.blogger.com", "S", "/", false);
  cookieManager.remove("www.google.com", "LSID", "/accounts", false);
};

// END flockILoginWebService interface


// BEGIN flockIMigratable interface

flockBLService.prototype.__defineGetter__("migrationName",
function flockBLService_getter_migrationName() {
  this._logger.debug("_getter_migrationName()");
  return flockGetString("common/migrationNames", "migration.name.blog");
});

// boolean needsMigration(in string aOldVersion);
flockBLService.prototype.needsMigration =
function flockBLService_needsMigration(aOldVersion) {
  this._logger.debug("needsMigration(" + aOldVersion + ")");
  // Version: major.minor.revision
  var major = parseInt(aOldVersion[0], 10);
  var minor = parseInt(aOldVersion[2], 10);
  if (major == 1 && (minor == 1 || minor == 2)) {
    // Grackle and Grouse: This service was added for Grackle and changed for
    // Ibis.
    return true;
  }
  return false;
};

// nsISupports startMigration(in string aOldVersion,
//                            in flockIMigrationProgressListener aListener);
flockBLService.prototype.startMigration =
function flockBLService_startMigration(aOldVersion,
                                       aFlockMigrationProgressListener)
{
  this._logger.debug("startMigration(" + aOldVersion
                     + ", aFlockMigrationProgressListener)");
  var ctxt = {
    majorVersion: parseInt(aOldVersion[0], 10),
    minorVersion: parseInt(aOldVersion[2], 10),
    listener: aFlockMigrationProgressListener
  };
  return { wrappedJSObject: ctxt };
};

// boolean doMigrationWork(in nsISupports aMigrationContext);
flockBLService.prototype.doMigrationWork =
function flockBLService_doMigrationWork(aMigrationContext) {
  this._logger.debug("doMigrationWork(aMigrationContext)");
  var context = aMigrationContext.wrappedJSObject;
  if (!context.bloggerMigrator) {
    context.bloggerMigrator = this._migrateFromCoop(context);
  }
  if (context.bloggerMigrator.next()) {
    context.bloggerMigrator = null;
  }
  return (context.bloggerMigrator != null);
};

// void finishMigration(in nsISupports aMigrationContext);
flockBLService.prototype.finishMigration =
function flockBLService_finishMigration(aMigrationContext) {
  this._logger.debug("finishMigration(aMigrationContext)");
};

// END flockIMigratable interface


// ================================================
// ========== BEGIN XPCOM Module support ==========
// ================================================

function createModule(aParams) {
  var Cc = Components.classes;
  var Ci = Components.interfaces;
  var Cr = Components.results;
  return {
    registerSelf: function (aCompMgr, aFileSpec, aLocation, aType) {
      aCompMgr.QueryInterface(Ci.nsIComponentRegistrar);
      aCompMgr.registerFactoryLocation( aParams.CID, aParams.componentName,
                                        aParams.contractID, aFileSpec,
                                        aLocation, aType );
      var catMgr = Cc["@mozilla.org/categorymanager;1"]
        .getService(Ci.nsICategoryManager);
      if (!aParams.categories) { aParams.categories = []; }
      for (var i = 0; i < aParams.categories.length; i++) {
        var cat = aParams.categories[i];
        catMgr.addCategoryEntry( cat.category, cat.entry,
                                 cat.value, true, true );
      }
    },
    getClassObject: function (aCompMgr, aCID, aIID) {
      if (!aCID.equals(aParams.CID)) { throw Cr.NS_ERROR_NO_INTERFACE; }
      if (!aIID.equals(Ci.nsIFactory)) { throw Cr.NS_ERROR_NOT_IMPLEMENTED; }
      return { // Factory
        createInstance: function (aOuter, aIID) {
          if (aOuter != null) { throw Cr.NS_ERROR_NO_AGGREGATION; }
          var comp = new aParams.componentClass();
          if (aParams.implementationFunc) { aParams.implementationFunc(comp); }
          return comp.QueryInterface(aIID);
        }
      };
    },
    canUnload: function (aCompMgr) { return true; }
  };
}

// NS Module entrypoint
function NSGetModule(aCompMgr, aFileSpec) {
  return createModule({
    componentClass: flockBLService,
    CID: BLOGGER_CID,
    contractID: BLOGGER_CONTRACTID,
    componentName: CATEGORY_COMPONENT_NAME,
    implementationFunc: function (aComp) { getCompTK().addAllInterfaces(aComp); },
    categories: [
      { category: "wsm-startup", entry: CATEGORY_COMPONENT_NAME, value: BLOGGER_CONTRACTID },
      { category: "flockMigratable", entry: CATEGORY_ENTRY_NAME, value: BLOGGER_CONTRACTID },
      { category: "flockWebService", entry: CATEGORY_ENTRY_NAME, value: BLOGGER_CONTRACTID }
    ]
  });
}

// ========== END XPCOM Module support ==========


/* ********** Account Class ********** */

function flockBLAccount() {
  FlockSvcUtils.getLogger(this).init(CATEGORY_ENTRY_NAME + ":Account");

  this.faves_coop = Cc["@flock.com/singleton;1"]
                    .getService(Ci.flockISingleton)
                    .getSingleton("chrome://flock/content/common/load-faves-coop.js")
                    .wrappedJSObject;

  this.acUtils = Cc["@flock.com/account-utils;1"].getService(Ci.flockIAccountUtils);

  // XXX: I should be able to use FlockSvcUtils.getWD() here, but it can
  //      only be called by the service.
  this._WebDetective = Cc["@flock.com/web-detective;1"]
                       .getService(Ci.flockIWebDetective);

  this.service = Cc[BLOGGER_CONTRACTID].getService(Ci.flockIBlogWebService)

  this._ctk = {
    interfaces: [
      "nsISupports",
      "flockIWebServiceAccount",
      "flockIBlogAccount"
    ]
  };
  getCompTK().addAllInterfaces(this);
}

// nsISupports implementation
flockBLAccount.prototype.QueryInterface =
function flockBLAccount_QueryInterface(iid) {
  if (!iid.equals(Ci.nsISupports) &&
    !iid.equals(Ci.flockIWebServiceAccount) &&
    !iid.equals(Ci.flockIBlogAccount))
  {
    throw Components.results.NS_ERROR_NO_INTERFACE;
  }
  return this;
};

// flockIWebServiceAccount implementation
flockBLAccount.prototype.login =
function flockBLAccount_login(listener) {
  this._logger.info("login(listener)");
  var c_acct = this.faves_coop.get(this.urn);
  if (!c_acct.isAuthenticated) {
    c_acct.isAuthenticated = true;
  }
  if (listener) {
    listener.onSuccess(this, "login");
  }
};

flockBLAccount.prototype.logout =
function flockBLAccount_logout(listener) {
  this._logger.info("logout(listener)");
  var c_acct = this.faves_coop.get(this.urn);
  if (c_acct.isAuthenticated) {
    c_acct.isAuthenticated = false;
    var cookieManager = Components.classes["@mozilla.org/cookiemanager;1"]
                                  .getService(Components.interfaces.nsICookieManager);
    cookieManager.remove(".blogger.com", "B2I", "/", false);
    cookieManager.remove(".blogger.com", "blogger_SID", "/", false);
    cookieManager.remove(".blogger.com", "ServerID", "/", false);
    cookieManager.remove(".blogger.com", "__utma", "/", false);
    cookieManager.remove(".blogger.com", "__utmb", "/", false);
    cookieManager.remove(".blogger.com", "__utmc", "/", false);
    cookieManager.remove(".blogger.com", "__utmz", "/", false);
    cookieManager.remove(".blogger.com", "NSC_cmphhfs-fyu", "/", false);
    cookieManager.remove("blogger.com", "NSC_cmphhfs-fyu", "/", false);
    cookieManager.remove("www.blogger.com", "NSC_cmphhfs-fyu", "/", false);
    cookieManager.remove("www.blogger.com", "JSESSIONID", "/", false);
    cookieManager.remove(".www2.blogger.com", "__utma", "/", false);
    cookieManager.remove(".www2.blogger.com", "__utmb", "/", false);
    cookieManager.remove(".www2.blogger.com", "__utmc", "/", false);
    cookieManager.remove(".www2.blogger.com", "__utmz", "/", false);
    cookieManager.remove("www2.blogger.com", "S", "/", false);
    cookieManager.remove("www.google.com", "LSID", "/accounts", false);
  }
  if (listener) {
    listener.onSuccess(this, "logout");
  }
};

flockBLAccount.prototype.keep =
function flockBLAccount_keep() {
  var c_acct = this.faves_coop.get(this.urn);
  c_acct.isTransient = false;
  this.acUtils.makeTempPasswordPermanent(this.service.urn+':'+c_acct.accountId);
};

// flockIBlogAccount implementation
flockBLAccount.prototype.getBlogs =
function flockBLAccount_getBlogs() {
  this._logger.info("getBlogs()");
  var blogsEnum = {
    QueryInterface : function(iid) {
      if (!iid.equals(Ci.nsISupports) &&
          !iid.equals(Ci.nsISimpleEnumerator))
      {
        throw Components.results.NS_ERROR_NO_INTERFACE;
      }
      return this;
    },
    hasMoreElements : function() {
      return false;
    },
    getNext : function() {
    }
  };
  return blogsEnum;
};
