// vim: ts=2 sw=2 expandtab cindent
//
// BEGIN FLOCK GPL
//
// Copyright Flock Inc. 2005-2008
// http://flock.com
//
// This file may be used under the terms of the
// GNU General Public License Version 2 or later (the "GPL"),
// http://www.gnu.org/licenses/gpl.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// END FLOCK GPL

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cr = Components.results;

Components.utils.import("resource:///modules/FlockCryptoHash.jsm");
Components.utils.import("resource:///modules/FlockScheduler.jsm");
Components.utils.import("resource:///modules/FlockStringBundleHelpers.jsm");
Components.utils.import("resource:///modules/FlockSvcUtils.jsm");

const FLICKR_TITLE                  = "Flickr Web Service";
const FLICKR_FAVICON                = "chrome://flock/content/services/flickr/favicon.png";
const FLICKR_CID                    = Components.ID("{db720a5c-6315-49bf-a39f-b4d4aa5ed142}");
const FLICKR_CONTRACTID             = "@flock.com/?photo-api-flickr;1";
const SERVICE_ENABLED_PREF          = "flock.service.flickr.enabled";
const CATEGORY_COMPONENT_NAME       = "Flickr JS Component"
const CATEGORY_ENTRY_NAME           = "flickr"

const flockIError                   = Components.interfaces.flockIError;
const flockIPhotoAlbum              = Components.interfaces.flockIPhotoAlbum;

const FLOCK_PHOTO_CONTRACTID        = "@flock.com/photo;1";
const FLOCK_PHOTOPERSON_CONTRACTID  = "@flock.com/photo-person;1";
const FLOCK_PHOTO_ALBUM_CONTRACTID  = "@flock.com/photo-album;1";
const FLOCK_ERROR_CONTRACTID        = "@flock.com/error;1";
const FLOCK_RDNDS_CONTRACTID        = "@flock.com/rich-dnd-service;1";

const PEOPLE_PROPERTIES = "chrome://flock/locale/people/people.properties";
const FLICKR_IDENTITY_URN_PREFIX = "urn:flock:identity:flickr:";
const XMLHTTPREQUEST_READYSTATE_COMPLETED = 4;

// The delay between two refreshes when the sidebar is closed (in seconds)
const FLICKR_REFRESH_INTERVAL = 3600; // 60 minutes
// The delay between two refreshes when the sidebar is open (in seconds)
const FLICKR_SHORT_INTERVAL = 3600; // 60 minutes
// The first time, only get photos not older than one week
const MEDIA_INITIAL_FETCH = 7 * 24 * 3600;
const MILLISEC = 1000;

var gCompTK;
function getCompTK() {
  if (!gCompTK) {
    gCompTK = Cc["@flock.com/singleton;1"]
              .getService(Ci.flockISingleton)
              .getSingleton("chrome://flock/content/services/common/load-compTK.js")
              .wrappedJSObject;
  }
  return gCompTK;
}

var gFlickrAPI;

function Namespace(ns) { return function (arg) {
  return Cc["@mozilla.org/rdf/rdf-service;1"]
         .getService(Ci.nsIRDFService)
         .GetResource (ns+arg); } }
const FLRDF = Namespace("http://flock.com/rdf#");

const FLICKR_STRING_BUNDLE = "chrome://flock/locale/services/flickr.properties";
const SERVICES_STRING_BUNDLE = "services/services";
const SERVICES_PHOTO_FEED_COMMENT_STRING = "flock.photo.feeds.comment.title";

// String defaults... may be updated later through Web Detective
var gStrings = {
  "domains": "flickr.com,yahoo.com",
  "userprofile": "http://www.flickr.com/people/%accountid%/",
  "photopage": "http://www.flickr.com/photos/%accountid%/%photoid%/",
  "commentsreceivedRSS": "http://api.flickr.com/services/feeds/activity.gne?id=%accountid%&format=rss_200",
  "staticimage": "http://static.flickr.com/%server%/%photoid%_%secret%%size%.jpg",
  "buddyicon": "http://www.flickr.com/images/buddyicon.jpg",
  "staticbuddyicon": "http://static.flickr.com/%iconserver%/buddyicons/%owner%.jpg",
  "nofriendavataricon": "http://static.flickr.com/0/buddyicons/%owner%.jpg",
  "nomeavataricon": "http://www.flickr.com/images/buddyicon.jpg?%owner%",
  "apikey": "92c2a562f0e9c2ed8dfe78f42a7734c7",
  "endpoint": "http://www.flickr.com/services/rest/",
  "uploadendpoint": "http://www.flickr.com/services/upload/",
  "authendpoint": "http://www.flickr.com/services/auth/"
};


// ====================================================
// ========== BEGIN General helper functions ==========
// ====================================================

function loadSubScript(spec)
{
  var loader = Components.classes["@mozilla.org/moz/jssubscript-loader;1"]
              .getService(Components.interfaces.mozIJSSubScriptLoader);
  var context = {};
  loader.loadSubScript(spec, context);
  return context;
}

function loadLibraryFromSpec(aSpec)
{
  var loader = Components.classes["@mozilla.org/moz/jssubscript-loader;1"]
                         .getService(Components.interfaces.mozIJSSubScriptLoader);
  loader.loadSubScript(aSpec);
}

function flock_flickrBuildPhotoUrl(server, id, secret, size)
{
  var convertedSize = "";
  switch (size) {
    case "square": convertedSize = "_s"; break;
    case "thumbnail": convertedSize = "_t"; break;
    case "small": convertedSize = "_m"; break;
    case "medium": convertedSize = "_d"; break;
    case "large": convertedSize = "_b"; break;
    case "original": convertedSize = "_o"; break;
    default: convertedSize = "";
  }
  return gStrings["staticimage"].replace("%server%", server)
                                .replace("%photoid%", id)
                                .replace("%secret%", secret)
                                .replace("%size%", convertedSize);
}

function _getIdentityUrn(aAccountId, aUid) {
  var result = FLICKR_IDENTITY_URN_PREFIX
             + aAccountId + ":"
             + aUid;
  return result;
}

function flock_flickrBuildPageUrl(aUserID, aPhotoID) {
  return gStrings["photopage"].replace("%accountid%", aUserID).replace("%photoid%", aPhotoID);
}

// refresh the mediabar if it is open and contains private media
function flock_refreshMediabarIfHasPrivate() {
  var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"]
                     .getService(Components.interfaces.nsIWindowMediator);
  var win = wm.getMostRecentWindow("navigator:browser");
  // if the mediabar is open
  if (win &&
      win.document.getElementById("mediabar") &&
      win.document.getElementById("mediabar").getAttribute("hidden") != "true" &&
      win.FlockTopbar.JXCtx().gPhotoDrawer &&
      win.FlockTopbar.JXCtx().gPhotoDrawer.mHasPrivateMedia)
  {
      win.FlockTopbar.JXCtx().gPhotoDrawer.refreshView();
  }
}

// ===============================================
// ========== BEGIN flickrService class ==========
// ===============================================

function flickrService()
{
  loadSubScript("chrome://browser/content/utilityOverlay.js");
  loadLibraryFromSpec("chrome://flock/content/photo/photoAPI.js");
  this.obs = Components.classes["@mozilla.org/observer-service;1"]
                       .getService(Components.interfaces.nsIObserverService);

  this.acUtils = Components.classes["@flock.com/account-utils;1"]
                           .getService(Components.interfaces.flockIAccountUtils);
  this._ppSvc = Cc["@flock.com/people-service;1"]
                .getService(Ci.flockIPeopleService);
  this.mIsInitialized = false;
  this._ctk = {
    interfaces: [
      "nsISupports",
      "nsIClassInfo",
      "nsISupportsCString",
      "nsIObserver",
      "flockIPollingService",
      "flockIWebService",
      "flockIAuthenticateNewAccount",
      "flockILoginWebService",
      "flockIMediaWebService",
      "flockIMediaUploadWebService",
      "flockIMediaEmbedWebService",
      "flockISocialWebService",
      "flockIRichContentDropHandler"
    ],
    shortName: "flickr",
    fullName: "Flickr",
    description: FLICKR_TITLE,
    favicon: FLICKR_FAVICON,
    CID: FLICKR_CID,
    contractID: FLICKR_CONTRACTID,
    accountClass: flickrAccount,
    needPassword: false
  };
  FlockSvcUtils.flockIMediaEmbedWebService
               .addDefaultMethod(this, "getSharingContent");
  this._logger = Cc['@flock.com/logger;1'].createInstance(Ci.flockILogger);
  this._logger.init('flickr');
  this._profiler = Cc["@flock.com/profiler;1"].getService(Ci.flockIProfiler);

  var sbs = Cc["@mozilla.org/intl/stringbundle;1"]
          .getService(Ci.nsIStringBundleService);
  var bundle = sbs.createBundle(FLICKR_STRING_BUNDLE);

  this._channels = {
    "special:recent": {
      title: bundle.GetStringFromName("flock.flickr.title.recent"),
      supportsSearch: false
    },
    "special:interestingness": {
      title: bundle.GetStringFromName("flock.flickr.title.interestingness"),
      supportsSearch: true
    }
  };
  this.init();

  FlockSvcUtils.flockIWebService.addDefaultMethod(this, "url");
  FlockSvcUtils.flockIWebService.addDefaultMethod(this, "getStringBundle");
  FlockSvcUtils.flockILoginWebService.addDefaultMethod(this, "loginURL");
}

flickrService.prototype.serviceName = "Flickr";
flickrService.prototype.shortName = "flickr";

flickrService.prototype.handlePhotosResult =
function (aXML, aUserid)
{
  var rval = [];
  var photoList = aXML.getElementsByTagName("photo");
  for (var i = 0; i < photoList.length; i++) {
    var photo = photoList[i];

    var id = photo.getAttribute("id");
    var server = photo.getAttribute("server");
    var secret = photo.getAttribute("secret");
    var title = photo.getAttribute("title");
    var owner = photo.getAttribute("owner");
    if (!owner) owner = aUserid;
    var owner_name = photo.getAttribute("ownername");
    var date_upload = photo.getAttribute("dateupload");
    var date_added = photo.getAttribute("dateadded");
    
    var icon_server = photo.getAttribute("iconserver");
    var square_url = flock_flickrBuildPhotoUrl(server,id,secret,'square');
    var small_url = flock_flickrBuildPhotoUrl(server,id,secret,'small');
    var med_url = flock_flickrBuildPhotoUrl(server,id,secret,'medium');
    var page_url = flock_flickrBuildPageUrl(owner,id);
    var media = photo.getAttribute("media");
    var newMediaItem = Cc["@flock.com/photo;1"]
                       .createInstance(Ci.flockIMediaItem);
    newMediaItem.init(this.shortName, null);
    newMediaItem.webPageUrl = page_url;
    newMediaItem.thumbnail = square_url;
    newMediaItem.midSizePhoto = small_url;
    newMediaItem.largeSizePhoto = med_url;
    newMediaItem.username = owner_name;
    newMediaItem.userid = owner;
    newMediaItem.title = title;
    newMediaItem.id = id;
    if (icon_server == '1') {
      newMediaItem.icon = gStrings["buddyicon"];
    }
    else {
      newMediaItem.icon = gStrings["staticbuddyicon"]
                      .replace("%iconserver%", icon_server)
                      .replace("%owner%", owner);
    }
    newMediaItem.uploadDate = parseInt(date_added ? date_added : date_upload)
                            * 1000;

    // if the photo result set doesn't return back the ispublic
    // attribute, we assume that the server is returning only
    // public photos back in the response
    if (photo.hasAttribute("ispublic")) {
      var ispublic = photo.getAttribute("ispublic");
      if (ispublic =="1") ispublic = "true";
      else ispublic = "false";
    } else {
      ispublic = "true";
    }

    newMediaItem.is_public = ispublic;

    if (media == "video") {
      newMediaItem.is_video = true;
      newMediaItem.mediaType = Ci.flockIMediaItem.MEDIA_TYPE_VIDEO;
    } else {
      newMediaItem.mediaType = Ci.flockIMediaItem.MEDIA_TYPE_IMAGE;
    }

    rval.push(newMediaItem);
  }
  return rval;
}


flickrService.prototype.handleContactsResult =
function (aXML)
{
  var rval = [];
  var contactList = aXML.getElementsByTagName("contact");
  for (var i = 0; i < contactList.length; i++) {
    var contact = contactList[i];
    var nsid = contact.getAttribute("nsid");
    var username = contact.getAttribute("username");
    var fullname = contact.getAttribute("fullname");
    var family = contact.getAttribute("family");
    var friend = contact.getAttribute("friend");
    var icon_server = contact.getAttribute("iconserver");

    if (true || family == "1" || friend == "1") {
      this._logger.debug(".handleContactsResult(): id=" + nsid
                        + " username=" + username
                        + " family=" + family
                        + " friend=" + friend);

      var newContact = {};
      newContact.id = nsid;
      newContact.username = username;
      newContact.fullname = username;
      
      // If friend has no avatar, set it to null here and people sidebar
      // code will fill it with the standard Flock no avatar image.
      var noAvatarUrl = gStrings["nofriendavataricon"].replace("%owner%", nsid);
      var avatarUrl = gStrings["staticbuddyicon"]
                             .replace("%iconserver%", icon_server)
                             .replace("%owner%", nsid);
      if (noAvatarUrl == avatarUrl) {
        newContact.avatarUrl = null;
      } else {
        newContact.avatarUrl = avatarUrl;
      }

      rval[nsid] = newContact;
    }
  }
  return rval;
}

flickrService.prototype.handleAlbumsResult =
function (aXML)
{
  var rval = [];
  var photosetList = aXML.getElementsByTagName("photoset");
  var titleList = aXML.getElementsByTagName("title");
  for (var i = 0; i < photosetList.length; i++) {
    var id = photosetList[i].getAttribute("id");
    var title = titleList[i].firstChild.nodeValue;

    var newAlbum = Components.classes[FLOCK_PHOTO_ALBUM_CONTRACTID]
                             .createInstance(flockIPhotoAlbum);
    newAlbum.id = id;
    newAlbum.title = title;

    rval.push(newAlbum);
  }
  return rval;
}

flickrService.prototype._getContacts =
function flickrService__getContacts(aListener)
{
  this._logger.info("._getContacts(...)");
  var inst = this;
  var myListener = {
    onResult: function (aXML) {
      var peeps = inst.handleContactsResult(aXML);
      var result = {
        wrappedJS: peeps
      }
      aListener.onGetContactsResult(result);
    },
    onError: aListener.onError
  }
  this.api.authenticatedCall(myListener, "flickr.contacts.getList");
}


flickrService.prototype.createAlbum =
function (aFlockListener, aAlbumName)
{
  // trim white space from front and end of string
  aAlbumName = aAlbumName.replace(/^\s+|\s+$/g, "");
  if (aAlbumName) {
    var newAlbum = Components.classes[FLOCK_PHOTO_ALBUM_CONTRACTID]
                             .createInstance(flockIPhotoAlbum);
    newAlbum.title = aAlbumName;
    var date = new Date();  //hopefully this won't collide with an actual id!
    newAlbum.id = date.getTime();
    this.api.fakeAlbums[newAlbum.id] = newAlbum;
    aFlockListener.onSuccess(newAlbum, "");
  } else {
    var error = Cc[FLOCK_ERROR_CONTRACTID].createInstance(Ci.flockIError);
    error.errorCode = error.PHOTOSERVICE_EMPTY_ALBUMNAME;
    aFlockListener.onError(error, null);
  }
}

flickrService.prototype.getAlbums =
function flickrService_getAlbums(aFlockListener, aUsername)
{
  var inst = this;
  var myListener = {
    onResult: function (aXML) {
      try {
        var rval = inst.handleAlbumsResult(aXML);
        var albumsEnum = {
          hasMoreElements: function albumsEnum_hasMoreElements() {
            return (rval.length > 0);
          },
          getNext: function albumsEnum_getNext() {
            return rval.shift();
          },
          QueryInterface: function albumsEnum_QI(aIID) {
            if (aIID.equals(Ci.nsISimpleEnumerator)) {
              return this;
            }
            throw Cr.NS_ERROR_NO_INTERFACE;
          }
        };
        aFlockListener.onSuccess(albumsEnum, "");
      } catch(e) {
        aFlockListener.onError(null, null);
      }
    },
    onError: function (aFlockError) {
      aFlockListener.onError(aFlockError, null);
    }
  }
  var params = {};
  if (aUsername) {
    params.user_id  = aUsername;
  }
  // Force authentication if the account is authenticated (for upload)
  var authOptional = (this.api.authAccount == null);
  this.api.authenticatedCall(myListener,
                             "flickr.photosets.getList", params, authOptional);
}

flickrService.prototype.getAccountStatus =
function flickrService_getAccountStatus(aFlockListener)
{
  var inst = this;
  var myListener = {
    onResult:function (aXML) {
      try {
        inst._logger.info('we got a result for account status...');
        var result = Cc["@mozilla.org/hash-property-bag;1"]
                     .createInstance(Ci.nsIWritablePropertyBag2);
        var username = aXML.getElementsByTagName('username')[0].firstChild.nodeValue;
        var bandwidth = aXML.getElementsByTagName('bandwidth')[0];
        var maxFileSize = aXML.getElementsByTagName('filesize')[0];
        var isPro = aXML.getElementsByTagName('user')[0].getAttribute('ispro');
        var maxSpace = bandwidth.getAttribute("max");
        var usedSpace = bandwidth.getAttribute("used");
        result.setPropertyAsAString("username", username);
        result.setPropertyAsAString("maxSpace", maxSpace);
        result.setPropertyAsAString("usedSpace", usedSpace);
        result.setPropertyAsAString("maxFileSize", maxFileSize);
        result.setPropertyAsAString("usageUnits", "bytes");
        result.setPropertyAsBool("isPremium", (isPro == "1"));
        aFlockListener.onSuccess(result, "");
      } catch (ex) {
        inst._logger.info('getaccountstatus error >>>>>>>>>>>>>>>>>>>>' + ex);
      }
    },
    onError: function (aError) {
      aFlockListener.onError(aError, "");
    }
  }
  this.api.authenticatedCall(myListener, "flickr.people.getUploadStatus");
}

flickrService.prototype.findByUsername =
function flickrService_findByUsername(aFlockListener, aUsername)
{
  var inst = this;
  var userListener =
  function fls_findByUsername_ul(aFlockListener) {
    this.mFlockListener = aFlockListener;
  }
  userListener.prototype.onGetInfo =
  function fls_findByUsername_ul_onGetInfo(aPerson) {
    this.mFlockListener.onSuccess(aPerson, "");
  }
  userListener.prototype.onError =
  function fls_findByUsername_ul_onError(aError) {
    inst._logger.info(aError);
    this.mFlockListener.onError(null, null);
  }
  var myListener = {
    onResult: function fls_findByUsername_ml_onResult(aXML) {
      var user = aXML.getElementsByTagName("user")[0];
      var id = user.getAttribute("id");
      inst.people_getInfo(new userListener(aFlockListener), id);
    },
    onError: function fls_findByUsername_ml_onError(aFlockError) {
      aFlockListener.onError(aFlockError, null);
    }
  }
  var params = {};
  params.username = aUsername;
  this.call(myListener, "flickr.people.findByUsername", params);
}


flickrService.prototype.lookupUser =
function (aListener, aURL)
{
  var inst = this;
  var myListener = {
    onResult: function (aXML) {
      var newUserObj = Components.classes[FLOCK_PHOTOPERSON_CONTRACTID]
                                 .createInstance(flockIPhotoPerson);
      var user = aXML.getElementsByTagName("user")[0];
      newUserObj.service = inst;
      newUserObj.id = user.getAttribute("id");
      newUserObj.username = user.getElementsByTagName("username")[0].childNodes[0].nodeValue;
      newUserObj.fullname = newUserObj.username;
      aListener.onLookupUser(newUserObj);
    },
    onError: function (aXML) {
      aListener.onError(aXML);
    }
  }
  var params = {};
  params.url = aURL;
  this.call(myListener, "flickr.urls.lookupUser", params);
}

flickrService.prototype.people_getInfo =
function (aListener, aUserId)
{
  var inst = this;
  var myListener = {
    onResult: function (aXML) {
      var newUserObj = Cc["@flock.com/person;1"]
                       .createInstance(Ci.flockIPerson);
      var person = aXML.getElementsByTagName("person")[0];
      var icon_server = person.getAttribute("iconserver");
      var count = person.getElementsByTagName("photos")[0]
                        .getElementsByTagName("count")[0]
                        .firstChild
                        .nodeValue;
      newUserObj.serviceId = inst.contractId;
      newUserObj.accountId = person.getAttribute("id");
      newUserObj.name = person.getElementsByTagName('username')[0]
                              .firstChild
                              .nodeValue;
      newUserObj.avatar = gStrings["staticbuddyicon"]
                          .replace("%iconserver%", icon_server)
                          .replace("%owner%", newUserObj.accountId);
      newUserObj.totalMedia = parseInt(count);
      aListener.onGetInfo(newUserObj);
    },
    onError: function (aXML) {
      aListener.onError(aXML);
    }
  }
  var params = {};
  params.user_id = aUserId;
  this.call(myListener, "flickr.people.getInfo", params);
}

flickrService.prototype.queryChannel =
function flickrService_queryChannel(aFlockListener,
                                    aQueryString,
                                    aCount,
                                    aPage,
                                    aRequestId)
{
  var aQuery = new queryHelper(aQueryString);
  var inst = this;
  var myListener = {
    onResult: function (aXML) {
      var rval = inst.handlePhotosResult(aXML);
      var resultsEnum = {
        hasMoreElements: function resultsEnum_hasMoreElements() {
          return (rval.length > 0);
        },
        getNext: function resultsEnum_getNext() {
          return rval.shift();
        },
        QueryInterface: function resultsEnum_QueryInterface(aIID) {
          if (aIID.equals(Ci.nsISimpleEnumerator)) {
            return this;
          }
          throw Cr.NS_ERROR_NO_INTERFACE;
        }
      };
      aFlockListener.onSuccess(resultsEnum, aRequestId);
    },
    onError: function (aError) {
      aFlockListener.onError(aError, aRequestId);
    }
  };

  var params = {
    per_page: aCount,
    page: aPage,
    extras: "owner_name,icon_server,date_upload,license,media"
  };

  if (aQuery.getKey("special") == "recent") {
    if (aQuery.hasKey("search")) {
      params.text = aQuery.getKey("search");
      this.call(myListener, "flickr.photos.search", params);
    } else {
      this.call(myListener, "flickr.photos.getRecent", params);
    }
  } else if (aQuery.getKey("special") == "interestingness") {
    if (aQuery.hasKey("search")) {
      params.text = aQuery.getKey("search");
      params.sort = "interestingness-desc";
      this.call(myListener, "flickr.photos.search", params);
    } else {
      this.call(myListener, "flickr.interestingness.getList", params);
    }
  } else if (aQuery.hasKey("search")) {
    aText = aQuery.getKey("search");
    // If search string only contains whitespace, consider it blank
    if (aText.replace(/\s/g, "") == "") {
      aText = "";
    }
    if (aText && aText.length) {
      if (aQuery.sort) {
        params.sort = aQuery.sort;
      }
      params.text = aText;
      this.call(myListener, "flickr.photos.search", params);
    } else {
      // No search string so just return empty result
      var rval = [];
      var emptyEnum = {
        hasMoreElements: function emptyEnum_hasMoreElements() {
          return (rval.length>0);
        },
        getNext: function emptyEnum_getNext() {
          return rval.shift();
        },
        QueryInterface: function emptyEnum_QueryInterface(aIID) {
          if (aIID.equals(Ci.nsISimpleEnumerator)) {
            return this;
          }
          throw Cr.NS_ERROR_NO_INTERFACE;
        }
      };
      aFlockListener.onSuccess(emptyEnum, aRequestId);
    }
  }
};

flickrService.prototype.poolSearch =
function flickrService_poolSearch(aFlockListener,
                                  aQueryString,
                                  aCount,
                                  aPage,
                                  aRequestId)
{
  var aQuery = new queryHelper(aQueryString);
  var inst = this;
  var myListener = {
    onResult: function (aXML) {
      var rval = inst.handlePhotosResult(aXML);
      var resultsEnum = {
        hasMoreElements: function resultsEnum_hasMoreElements() {
          return (rval.length>0);
        },
        getNext: function resultsEnum_getNext() {
          return rval.shift();
        },
        QueryInterface: function resultsEnum_QueryInterface(aIID) {
          if (aIID.equals(Ci.nsISimpleEnumerator)) {
            return this;
          }
          throw Cr.NS_ERROR_NO_INTERFACE;
        }
      }
      aFlockListener.onSuccess(resultsEnum, aRequestId);
    },
    onError: function (aError) {
      aFlockListener.onError(aError, aRequestId);
    }
  }

/*
  var aText;

  if (aQuery.split('?').length > 1) {
    aText = aQuery.split('?')[1];
    aQuery = aQuery.split('?')[0];
  }

  aQuery = aQuery.split(':')[1]
  */

  var params = {
    group_id: aQuery.pool,
    per_page: aCount,
    page: aPage,
    extras: "owner_name,icon_server,date_upload,license,media"
  };

  if (aQuery.search) {
    params.tags = aQuery.search;
    this.call(myListener, "flickr.groups.pools.getPhotos", params);
  } else {
    this.call(myListener, "flickr.groups.pools.getPhotos", params);
  }
}

flickrService.prototype.search =
function flickrService_search(aFlockListener,
                              aQueryString,
                              aCount,
                              aPage,
                              aRequestId)
{
  this._logger.info(".search(" + aQueryString + ")");

  var aQuery = new queryHelper(aQueryString);
  if (aQuery.pool) {
    this.poolSearch(aFlockListener, aQueryString, aCount, aPage, aRequestId);
    return;
  }
  if (!aQuery.user) {
    this.queryChannel(aFlockListener, aQueryString, aCount, aPage, aRequestId);
    return;
  }
  var aUserid = aQuery.user;
  var inst = this;
  var myListener = {
    onResult: function (aXML) {
      var rval = inst.handlePhotosResult(aXML, aUserid);
      var resultsEnum = {
        hasMoreElements: function resultsEnum_hasMoreElements() {
          return (rval.length>0);
        },
        getNext: function resultsEnum_getNext() {
          return rval.shift();
        },
        QueryInterface: function resultsEnum_QueryInterface(aIID) {
          if (aIID.equals(Ci.nsISimpleEnumerator)) {
            return this;
          }
          throw Cr.NS_ERROR_NO_INTERFACE;
        }
      }

      aFlockListener.onSuccess(resultsEnum, aRequestId);
    },
    onError: function (aError) {
      aFlockListener.onError(aError, aRequestId);
    }
  }
  // Flickr return the photo in the reverse side when calling flickr.photosets.getPhotos!
  // So we need to enumerate backward :-/
  var myPhotosetListener = {
    onResult: function (aXML) {
      var rval = inst.handlePhotosResult(aXML, aUserid);
      var resultsEnum = {
        hasMoreElements: function resultsEnum_hasMoreElements() {
          return (rval.length>0);
        },
        getNext: function resultsEnum_getNext() {
          return rval.pop();
        },
        QueryInterface: function resultsEnum_QueryInterface(aIID) {
          if (aIID.equals(Ci.nsISimpleEnumerator)) {
            return this;
          }
          throw Cr.NS_ERROR_NO_INTERFACE;
        }
      }
      aFlockListener.onSuccess(resultsEnum, aRequestId);
    },
    onError: function (aError) {
      aFlockListener.onError(aError, aRequestId);
    }
  }

  var params = {};
  if (aUserid && aUserid.length) params.user_id = aUserid;
  params.per_page = aCount;
  params.page = aPage;

  if (aQuery.search) params.text = aQuery.search;

  params.tag_mode = "all";
  params.extras = "owner_name,license,date_upload,icon_server,media";
  if (aQuery.album) {
    params.photoset_id = aQuery.album;
  }

  if (params.photoset_id) {
    this.api.authenticatedCall(myPhotosetListener,
                               "flickr.photosets.getPhotos", params, true);
  } else if (params.user_id || params.text) {
    this.api.authenticatedCall(myListener,
                               "flickr.photos.search", params, true);
  }
}

flickrService.prototype.supportsSearch =
function flickrService_supportsSearch( aQueryString ) {
  var aQuery = new queryHelper(aQueryString);

  if (aQuery.special) {
    var channel = this._channels["special:" + aQuery.special];
    if (channel) {
      return channel.supportsSearch;
    }
  }

  if (aQuery.album) {
    return false;
  }
  if (aQuery.user) {
    return true;
  }
  if (aQuery.search) {
    return false;
  }
  return false;
}

flickrService.prototype.mSupportsTitle = true;
flickrService.prototype.mSupportsTags = true;
flickrService.prototype.mSupportsContacts = true;
flickrService.prototype.mSupportsPrivacy = true;

flickrService.prototype.supportsFeature =
function flickrService_supportsFeature(aFeature)
{
  var supports = {};
  supports.tags = true;
  supports.title = true;
  supports.fileName = false;
  supports.contacts = true;
  supports.description = true;
  supports.privacy = true;
  supports.albumCreation = true;
  return (supports[aFeature] == true);
}

flickrService.prototype.init =
function flickrService_init()
{
  // Prevent re-entry
  if (this.mIsInitialized) return;
  this.mIsInitialized = true;

  this._logger.info(".init()");
  var evtID = this._profiler.profileEventStart("flickr-init");

  // Determine whether this service has been disabled, and unregister if so.
  this.prefService = Cc["@mozilla.org/preferences-service;1"].getService(Ci.nsIPrefBranch);
  if ( this.prefService.getPrefType(SERVICE_ENABLED_PREF) &&
       !this.prefService.getBoolPref(SERVICE_ENABLED_PREF) )
  {
    this._logger.info("Pref "+SERVICE_ENABLED_PREF+" set to FALSE... not initializing.");
    var catMgr = Cc["@mozilla.org/categorymanager;1"].getService(Ci.nsICategoryManager);
    catMgr.deleteCategoryEntry("wsm-startup", CATEGORY_COMPONENT_NAME, true);
    catMgr.deleteCategoryEntry("flockWebService", CATEGORY_ENTRY_NAME, true);
    catMgr.deleteCategoryEntry("flockMediaProvider", CATEGORY_ENTRY_NAME, true);
    return;
  }

  if (!gFlickrAPI) {
    gFlickrAPI = new FlickrAPI();
  }
  this.api = gFlickrAPI;
  var inst = this;

  this.faves_coop = Components.classes["@flock.com/singleton;1"]
                              .getService(Components.interfaces.flockISingleton)
                              .getSingleton("chrome://flock/content/common/load-faves-coop.js")
                              .wrappedJSObject;

  this.urn = "urn:flickr:service";
  this.svcCoopObj = new this.faves_coop.Service(
    this.urn,
    {
      name: "flickr",
      desc: "The Flickr Service"
    }
  );
  this.svcCoopObj.serviceId = FLICKR_CONTRACTID;

  // Initialize member that specifies path for localized string bundle
  this._stringBundlePath = FLICKR_STRING_BUNDLE;

  // Load Web Detective file
  this.webDetective = this.acUtils.useWebDetective("flickr.xml");
  for (var s in gStrings) {
    gStrings[s] = this.webDetective.getString("flickr", s, gStrings[s]);
  }
  this.svcCoopObj.domains = gStrings["domains"];

  // Load auth manager
  this._authMgr = Cc["@flock.com/authmgr;1"]
                  .getService(Ci.flockIAuthenticationManager);
  this._authMgr.joinNetwork(FLICKR_CONTRACTID, "yahoo");

  // Update auth states
  try {
    if (this.webDetective.detectCookies("flickr", "loggedout", null)) {
      this.acUtils.markAllAccountsAsLoggedOut(FLICKR_CONTRACTID);
      this.api.logout();
    }
  } catch (ex) {
    this._logger.error("ERROR updating auth states for Flickr: "+ex);
  }

  // On browser restart if a Flickr coop account is still marked as
  // authenticated we will also need to reauth the api. We can't use
  // getAuthenticatedAccount because this service is not yet registered.
  var query = {serviceId: FLICKR_CONTRACTID, isAuthenticated: true};
  var authenticatedAccounts = this.faves_coop.Account.find(query);

  if (authenticatedAccounts.length != 0) {
    this.api.setAuthAccount(authenticatedAccounts[0].id());
  }

  var timerFunc = {
    inProgress: false,
    count: 0,
    observe: function(subject, topic, state) {
      //if (topic == 'nsPref:changed' && state == FLICKR_TOKEN_PREF_ID) {
      //  var oldToken = flock_getCharPref(FLICKR_TOKEN_PREF_ID);
      //  if (oldToken && oldToken.length && !inst.api.isLoggedIn()) {
      //    this.inProgress = true;
      //    inst.api.checkToken(this, oldToken);
      //  }
      //}
    },
    notify: function(aTimer) {
      if (this.inProgress) return;
      inst.api.processPendingUploadTickets();
      inst.api.processPendingAlbumAdditions();
    },
    onResult: function(aXML) {
      this.inProgress = false;
    },
    onSuccess: function(aXML) {
      this.inProgress = false;
    },
    onError: function(aError) {
      inst.logout();
      this.inProgress = false;
    }
  }
  timer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
  timer.initWithCallback(timerFunc, 5 * 1 * 1000, 1);  //re-check token

  this._profiler.profileEventEnd(evtID, "");
}

flickrService.prototype.__defineGetter__('supportsTitle', function () { return this.mSupportsTitle; })
flickrService.prototype.__defineGetter__('supportsTags', function () { return this.mSupportsTags; })
flickrService.prototype.__defineGetter__('supportsContacts', function () { return this.mSupportsContacts; })
flickrService.prototype.__defineGetter__('supportsPrivacy', function () { return this.mSupportsPrivacy; })

flickrService.prototype.getAuthState =
function flickrService_getAuthState()
{
  return this.state;
}

flickrService.prototype.__defineGetter__('authState', function () { return this.state; })

flickrService.prototype.logout =
function flickrService_logout()
{
  this.api.logout();
  this.acUtils.removeCookies(this.webDetective.getSessionCookies("flickr"));
}

flickrService.prototype.__defineGetter__('isUploading', function () { return this.running; })

/** XXX: this doesn't appear to be used?!? */
flickrService.prototype.dictionary2Params =
function flickrService_dictionary2Params(aDictionary) {
  var params = {};
  var obj = {};
  var count = {};
  var keys = aDictionary.getKeys(count, obj);
  for (var i = 0; i < keys.length;++i) {
    var supports = aDictionary.getValue(keys[i]);
    var supportsString = supports.QueryInterface(Ci.nsISupportsString);
    var val = supportsString.toString();
    params[keys[i]] = val;
  }
  return params;
}

flickrService.prototype.call =
function flickrService_call(aListener, aMethod, aParams)
{
  this.api.call(aListener, aMethod, aParams);
}

flickrService.prototype.upload =
function flickrService_upload(aUploadListener, aUpload, aFile)
{
  var params = {};
  params.title = aUpload.title;
  params.description = aUpload.description;
  var prefService = Cc["@mozilla.org/preferences-service;1"]
                    .getService(Ci.nsIPrefService);
  var prefBranch = prefService.getBranch("flock.photo.uploader.");
  if (prefBranch.getPrefType("breadcrumb.enabled") &&
      prefBranch.getBoolPref("breadcrumb.enabled"))
  {
    var sbs = Cc["@mozilla.org/intl/stringbundle;1"]
              .getService(Ci.nsIStringBundleService);
    var sb = sbs.createBundle(FLICKR_STRING_BUNDLE);
    params.description += sb.GetStringFromName("flock.flickr.uploader.breadcrumb");
  }

  params.is_family = aUpload.is_family;
  params.is_friend = aUpload.is_friend;
  params.is_public = aUpload.is_public;
  params.async = "1";
  params.tags = aUpload.tags;
  this.api.upload(aUploadListener, aFile, params, aUpload);
}

flickrService.prototype.cancelUpload =
function flickrService_cancelUpload()
{
  try { this.api.uploader.req.abort(); }
  catch(e){};
}

flickrService.prototype.addCoopPerson =
function flickrService_addCoopPerson(aPhotoPerson, aCoopAccount) {
  var person = aPhotoPerson;
  var profileURL = gStrings["userprofile"].replace("%accountid%", person.id);
  // We include the current accountId in the identity urn to prevent friend
  // collisions if multiple Facebook accounts have the same friend.
  var identityUrn = _getIdentityUrn(aCoopAccount.accountId,
                                    person.id);
  var updating = this.faves_coop.Identity.exists(identityUrn);
  var identity = this.faves_coop.get(identityUrn);
  if (identity) {
    if (this._personUpdateRequired (aCoopAccount,person)) {
      // update data of the identity coop obj here
      identity.name = person.username;
      identity.URL = profileURL;
    }
  } else {
    identity = new this.faves_coop.Identity(
      identityUrn,
      {
        name: person.username,
        serviceId: FLICKR_CONTRACTID,
        accountId: person.id,
        statusMessage: '',
        lastUpdateType: "media",
        URL: profileURL
      }
    );
    aCoopAccount.friendsList.children.add(identity);
    identity.avatar = person.avatarUrl;
    identity.lastUpdate = 0; // triggers the RDF observers
  }

  if (person.media) {
    this._incrementMedia(person.id, person.media.count, person.media.latest);
  }
}

flickrService.prototype._incrementMedia =
function flickrService__incrementMedia(aUid, aCount, aLatest) {
  this._logger.info("._incrementMedia('" + aUid + "')");
  var currAcct = this.getAuthenticatedAccount();
  var currAcctId = currAcct.getParam("accountId");

  // Update the count on the identity...
  var identityUrn = _getIdentityUrn(currAcctId, aUid);
  identity = this.faves_coop.get(identityUrn);

  if (!identity.lastUpdate) {
    identity.lastUpdate = aLatest;
    var lastweek = parseInt(Date.now() / 1000) - MEDIA_INITIAL_FETCH;
    // Friend uploaded pictures in the last week!
    if (aLatest > lastweek) {
      identity.unseenMedia += aCount;
    }
  } else if (aLatest > identity.lastUpdate) {
    identity.unseenMedia += aCount;
    identity.lastUpdate = aLatest;
  }
}

flickrService.prototype._personUpdateRequired =
function flickrService__personUpdateRequired(aCoopPerson, aPerson) {
  return (aCoopPerson.name != aPerson.fullname)
      || (aCoopPerson.avatar != aPerson.pic_square);
}

flickrService.prototype._updateUserCommentCount =
function flickrService__updateUserCommentCount(aCoopAccount)
{
  this._logger.info("._updateUserCommentCount('" + aCoopAccount.id() + "')");
  var timespanMS = 0;
  var lastClearedDateSec = 0;
  var account = this.getAccount(aCoopAccount.id());
  if (!aCoopAccount.lastUpdateDate) {
    // First time refreshing this account, so notify on all photo comments for
    // the past week.
    aCoopAccount.lastUpdateDate = new Date();
    var updateTimeMS =  aCoopAccount.lastUpdateDate.getTime();
    var lastweek = updateTimeMS - (MEDIA_INITIAL_FETCH * MILLISEC);
    timespanMS = updateTimeMS - lastweek;
    account.setCustomParam("flickr_comments", 0);
    account.setCustomParam("flickr_comments_timespan", 0);
    lastClearedDateSec = parseInt(lastweek / MILLISEC);
  } else {
    var today = new Date();
    var updateTimeMS =  aCoopAccount.lastUpdateDate.getTime();
    timespanMS = today.getTime() - updateTimeMS;
    lastClearedDateSec = parseInt(updateTimeMS / MILLISEC);
  }
  // This controls the time span (in hours) that photo comments will be shown
  // for when the user chooses to view comment noties.
  var timespanHrs = timespanMS / (60 * 60 * MILLISEC);
  var acc_timespanHrs = account.getCustomParam("flickr_comments");
  var timespan = acc_timespanHrs + timespanHrs;
  account.setCustomParam("flickr_comments_timespan", timespan);
  
  var api_key = this.api_key;
  var params = {
    api_key: api_key,
    timeframe: timespanHrs + "h",
  };
  var inst = this;
  var commentListener = {
    onResult: function commentListener_onResult(aXML) {
      var items = aXML.getElementsByTagName("item");
      var lightUp = false;
      for (var i = 0; i< items.length; i++) {
        var events = items[i].getElementsByTagName("event");
        for (var j = 0; j< events.length; j++) {
          var dateadded = events[j].getAttribute("dateadded");
          // Dateadded greater than last checked time,
          // we have new comment to see! Light up!
          dateadded = parseInt(dateadded);
          if (lastClearedDateSec < dateadded) {
            var newdate = new Date();
            newdate.setTime(dateadded * MILLISEC);
            aCoopAccount.lastUpdateDate = newdate;
            var comments = account.getCustomParam("flickr_comments");
            account.setCustomParam("flickr_comments", ++comments);
            lightUp = true;
          }
        }
      }
      if (lightUp) {
        inst._ppSvc.togglePeopleIcon(true);
      }
    },
    onError: function commentListener_onError(aFlockError) {
    }
  };
  this.api.authenticatedCall(commentListener,
                             "flickr.activity.userPhotos", params);
}

flickrService.prototype._updateUserAccount =
function flickrService__updateUserAccount(aCoopAccount)
{
  this._logger.info("._updateUserAccount('" + aCoopAccount.id() + "')");
  var inst = this;
  var hr = Cc["@mozilla.org/xmlextras/xmlhttprequest;1"]
           .createInstance(Ci.nsIXMLHttpRequest);
  hr.mozBackgroundRequest = true;
  hr.onreadystatechange = function updateAccount_onreadystatechange(aEvent) {
    if (hr.readyState == XMLHTTPREQUEST_READYSTATE_COMPLETED) {
      var results = Cc["@mozilla.org/hash-property-bag;1"]
                    .createInstance(Ci.nsIWritablePropertyBag2);
      if (inst.webDetective.detectNoDOM("flickr", "accountinfo", "",
                                        hr.responseText, results))
      {
        inst._updateUserStatusFromResults(aCoopAccount, results);
        inst._updateUserCommentCount(aCoopAccount);
        inst.obs.notifyObservers(aCoopAccount.resource(),
                                 "flock-acct-refresh",
                                 "user-info");
      }
    }
  }
  hr.open("GET", this.webDetective.getString("flickr", "scrapeAccountInfo",
                                             "http://www.flickr.com/"));
  hr.send(null);
}

flickrService.prototype._updateUserStatusFromResults =
function flickrService__updateUserStatusFromResults(aCoopAccount, aResults)
{
  this._logger.info("._updateUserStatusFromResults('" + aCoopAccount.id() + "')");
  var username = aResults.getPropertyAsAString("username");
  if (username && username.length) {
    aCoopAccount.name = username;
  }
  var avatarURL = aResults.getPropertyAsAString("avatarURL");
  if (avatarURL && avatarURL.length) {
    // If user account has no avatar, set it to null here and people sidebar
    // code will fill it with the standard Flock no avatar image.
    var accountid = aResults.getPropertyAsAString("accountid");
    var noAvatarUrl = gStrings["nomeavataricon"].replace("%owner%", accountid);
    if (noAvatarUrl == avatarURL) {
      aCoopAccount.avatar = null;
    } else {
      aCoopAccount.avatar = avatarURL;
    }
  }
  var messages = "0";
  try {
    messages = aResults.getPropertyAsAString("messages");
    if (messages && messages.length) {
      if (aCoopAccount.accountMessages < messages) {
        this._ppSvc.togglePeopleIcon(true);
      }
    }
    else {
      messages = "0";
    }
  } catch(ex) {}
  aCoopAccount.accountMessages = messages;
}

// BEGIN flockIPollingService
flickrService.prototype.refresh =
function flickrService_refresh(aURN, aPollingListener)
{
  this._logger.info("{flockIPollingService}.refresh('"+aURN+"', ...)");
  var refreshItem = this.faves_coop.get(aURN);
  var inst = this;
  if (refreshItem instanceof this.faves_coop.Account) {
    var peopleHash;

    if (!refreshItem.isAuthenticated) {
      this._logger.debug("account not logged in - skipping refresh");
      aPollingListener.onResult();
      return;
    }
    var recentPhotosListener = {
      onResult: function recentPhotosListener_onResult(aXML) {
        var photosList = aXML.getElementsByTagName("photo");
        for (var i = 0; i < photosList.length; i++) {
          var photo = photosList[i];
          var personId = photo.getAttribute("owner");
          var latest = photo.getAttribute("dateupload");

          // Add media info to people hash
          peopleHash[personId].media = {
            count: 1,
            latest: latest
          };
        }

        // Now that we have all we need, update the RDF
        function myWorker(aShouldYield) {
          if (refreshItem.isAuthenticated) {
            // REMOVE locally people removed on the server
            var localEnum = refreshItem.friendsList.children.enumerate();
            while (localEnum.hasMoreElements()) {
              var identity = localEnum.getNext();
              if (!peopleHash[identity.accountId]) {
                inst._logger.info("Friend " + identity.accountId
                                            + " has been deleted on the server");
                refreshItem.friendsList.children.remove(identity);
                identity.destroy();
              }
            }

            // ADD or update existing people
            for (var uid in peopleHash) {
              inst.addCoopPerson(peopleHash[uid], refreshItem);
              if (aShouldYield()) {
                yield;
                if (!refreshItem.isAuthenticated) {
                  // Account has just been deleted or logged out
                  break;
                }
              }
            }
          }

          if (inst.acUtils.isPeopleSidebarOpen() &&
              refreshItem.isAuthenticated)
          {
            refreshItem.nextRefresh = new Date(Date.now()
                                               + FLICKR_SHORT_INTERVAL * 1000);
          }
          aPollingListener.onResult();
        } // end myWorker()
        FlockScheduler.schedule(null, 0.05, 10, myWorker);
      },
      onError: function recentPhotosListener_onError(aFlockError) {
        aPollingListener.onError(aFlockError);
        inst._logger.error("recentPhotosListener error");
      }
    };

    this._getContacts({
      onGetContactsResult: function getContacts_onGetContactsResult(aResult) {
        inst._updateUserAccount(refreshItem);
        peopleHash = aResult.wrappedJS;

        var api_key = inst.api_key;
        var params = {
          api_key: api_key,
          single_photo: 1,
          count: 50,
          extras: "date_upload"
        };

        inst.api.authenticatedCall(recentPhotosListener,
                                   "flickr.photos.getContactsPhotos", params);
      },
      onError: function getContacts_onError(aFlockError) {
        aPollingListener.onError(aFlockError);
      }
    });

  } else {
    throw CR.NS_ERROR_ABORT;
    this._logger.error("can't refresh " + aURN + " (unsupported type)");
    aPollingListener.onError(null);
  }
}
// END flockIPollingService


flickrService.prototype.migrateAccount =
function flickrService_migrateAccount( aId, aUsername ) {
  this.init();

  var token = '';
  /*
  try {
    token = flock_getCharPref('flock.photo.flickr.token');
  } catch (e) { }
  */

  this.addAccount( aId, false, null, aUsername, token);
}

// BEGIN flockIWebService interface
flickrService.prototype.addAccount =
function flickrService_addAccount(aAccountID, aIsTransient,
                                  aFlockListener, aUsername, aToken)
{
  this._logger.info("{flockIWebService}.addAccount('"+aAccountID+"', "+aIsTransient+")");

  if (!aUsername) {
    // Get the password associated with this account
    var pw = this.acUtils.getPassword(this.urn+':'+aAccountID);
    // FIX ME - name shouldn't be email address (pw.username) - ja
    var name = (pw) ? pw.username : aAccountID;
    var token = '';
    var pollable = false;
    var auth = false;
  } else {
    var name = aUsername;
    var token = aToken;
    var pollable = true;
    var auth = true;
  }
  // Account
  var url = gStrings["userprofile"].replace("%accountid%", aAccountID);
  var accountUrn = this.acUtils.createAccount(this,
                                              aAccountID,
                                              name,
                                              url,
                                              aIsTransient);
  var coopAccount = this.faves_coop.get(accountUrn);
  coopAccount.refreshInterval = FLICKR_REFRESH_INTERVAL;
  var list = Cc["@mozilla.org/hash-property-bag;1"]
             .createInstance(Ci.nsIWritablePropertyBag2);
  list.setPropertyAsAString("flickr_username", "");
  list.setPropertyAsAString("flickr_fullname", "");
  list.setPropertyAsAString("flickr_nsid", "");
  list.setPropertyAsInt32("flickr_comments", 0);
  list.setPropertyAsInt32("flickr_comments_timespan", 0);
  this.acUtils.addCustomParamsToAccount(list, coopAccount.id());
  // Instanciate account component
  var acct = this.getAccount(coopAccount.id());
  if (aFlockListener) {
    aFlockListener.onSuccess(acct, "addAccount");
  }
  return acct;
}
// END flockIWebService interface


// BEGIN flockIAuthenticateNewAccount interface
flickrService.prototype.authenticateNewAccount =
function flickrService_authenticateNewAccount() {
  this._logger.info("authenticateNewAccount()");
  var inst = this;
  var frobListener = {
    onResult: function (aXML) {
      inst._logger.info(".authenticateNewAccount(): frobListener: onResult()");
      var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"]
                         .getService(Components.interfaces.nsIWindowMediator);
      var ww = Components.classes["@mozilla.org/embedcomp/window-watcher;1"]
                         .getService(Components.interfaces.nsIWindowWatcher);
      var topWin = wm.getMostRecentWindow(null);
      var frob = aXML.getElementsByTagName("frob")[0].firstChild.nodeValue;
      var authURL = inst.api.getAuthUrl(frob, "write");
      var url = "chrome://flock/content/photo/photoLoginWindow.xul?"
              + "url="+escape(authURL)+"&finalString=logout.gne";
      var chrome = "chrome,close,titlebar,resizable=yes,toolbar,dialog=no,"
                 + "scrollbars=yes,modal,centerscreen";
      topWin.open(url, "_blank", chrome);

      var account = inst.getAuthenticatedAccount();
      if (!account) {
        inst._logger.error(".authenticateNewAccount(): frobListener: "
                           + "onResult(): ERROR: account was not created");
      }
    },
    onError: function (aError) {
      inst._logger.info(".authenticateNewAccount(): frobListener: onError()");
    }
  };
  this.api.call(frobListener, "flickr.auth.getFrob", null);
}
// END flockIAuthenticateNewAccount


// BEGIN flockILoginWebService interface
flickrService.prototype.updateAccountStatusFromDocument =
function flickrService_updateAccountStatusFromDocument(aDocument, 
                                                       aAcctURN, 
                                                       aFlockListener)
{
  this._logger.info("updateAccountStatusFromDocument('" + aAcctURN + "')");
  if (aAcctURN) {
    // We're logged in to this account
    var account = this.getAccount(aAcctURN);
    if (!account.isAuthenticated()) {
      var inst = this;
      var flockListener = {
        onSuccess: function UA_onSuccess(aSubject, aTopic) {
          inst._logger.info(".updateAccountStatusFromDocument(): loginListener: onSuccess()");
          inst._authMgr.reportCompleteLoginToNetwork(aAcctURN);
          if (aFlockListener) {
            aFlockListener.onSuccess(account, aTopic);
          }
        },
        onError: function UA_onError(aFlockError, aTopic) {
          inst._logger.info(".updateAccountStatusFromDocument(): loginListener: onError()");
          inst._logger.info(aFlockError ? (aFlockError.errorString) : "No details");
        }
      };
      account.login(flockListener);
    }
  } else {
    // No login was detected.  So let's see if we can detect a logged out
    // condition.
    var flickrLogout =
      this.webDetective.detect("flickr", "loggedout", aDocument, null);
    var yahooNetworkLogout =
      this.webDetective.detect("flickr", "completeLogout", aDocument, null);
    if (flickrLogout || yahooNetworkLogout) {
      // Yes, we're logged out.  We do slightly different things depending on
      // whether we've just logged out of Flickr, or the whole Yahoo network.
      var loggedoutURN = null;
      if (flickrLogout) {
        var account = this.getAuthenticatedAccount();
        if (account) {
          loggedoutURN = account.urn;
        }
      }
      this.acUtils.markAllAccountsAsLoggedOut(FLICKR_CONTRACTID);
      this.api.logout();
      flock_refreshMediabarIfHasPrivate();
      if (flickrLogout && loggedoutURN) {
        this._authMgr.reportAccountAuthState(loggedoutURN, false);
      }
      if (yahooNetworkLogout) {
        this._authMgr.reportCompleteLogoutFromNetwork(FLICKR_CONTRACTID);
      }
    }
  }
}

flickrService.prototype.ownsDocument =
function flickrService_ownsDocument(aDocument)
{
  this._logger.debug("domain: " + aDocument.domain);

  // We need to override the default ownsDocument in this case because
  // multiple services share the same login page (yahoo.com in this case)
  return this.webDetective.detect("flickr", "ownsDocument", aDocument, null);
}
// END flockILoginWebService interface


// BEGIN flockISocialWebService interface
flickrService.prototype.maxStatusLength = 0;
// END flockISocialWebService interface


// BEGIN flockIMediaWebService interface
// readonly attribute boolean supportsUsers;
flickrService.prototype.supportsUsers = true;

flickrService.prototype.decorateForMedia =
function flickrService_decorateForMedia(aDocument)
{
  this._logger.info("{flockIMediaWebService}.decorateForMedia(aDocument)");
  var results = Components.classes["@mozilla.org/hash-property-bag;1"]
                          .createInstance(Components.interfaces.nsIWritablePropertyBag2);
  var mediaArr = [];
  if (this.webDetective.detect("flickr", "person", aDocument, results)) {
    var media = {
      name: results.getPropertyAsAString("username"),
      query: "user:" + results.getPropertyAsAString("userid") + "|username:" + results.getPropertyAsAString("username"),
      label: results.getPropertyAsAString("username"),
      favicon: this.icon,
      service: this.shortName
    }
    mediaArr.push(media);
  }
  
  if (this.webDetective.detect("flickr", "pool", aDocument, results)) {
    var media = {
      name: results.getPropertyAsAString("groupname"),
      query: "pool:" + results.getPropertyAsAString("groupid") + "|albumlabel:" + results.getPropertyAsAString("groupname"),
      label: results.getPropertyAsAString("groupname"),
      favicon: this.icon,
      service: this.shortName
    }
    mediaArr.push(media);
  }
  
  if (mediaArr.length > 0) {
    if (!aDocument._flock_decorations) {
      aDocument._flock_decorations = {};
    }

    if (aDocument._flock_decorations.mediaArr) {
      aDocument._flock_decorations.mediaArr =
        aDocument._flock_decorations.mediaArr.concat(mediaArr);
    } else {
      aDocument._flock_decorations.mediaArr = mediaArr;
    }

    this.obs.notifyObservers(aDocument, "media", "media:update");
  }
}

flickrService.prototype.checkIsStreamUrl =
function flickrService_checkIsStreamUrl(aUrl)
{
  if (this.webDetective.detectNoDOM("flickr", "isStreamUrl", "", aUrl, null)) {
    this._logger.debug("Checking if url is flickr stream: YES: " + aUrl);
    return true;
  }
  this._logger.debug("Checking if url is flickr stream: NO: " + aUrl);
  return false;
}

flickrService.prototype.getMediaQueryFromURL =
function flickrService_getMediaQueryFromURL(aUrl, aFlockListener)
{
  var myListener = {
    onResult:function (aXML) {
      try {
        var results = Components.classes["@mozilla.org/hash-property-bag;1"]
                                .createInstance(Components.interfaces.nsIWritablePropertyBag2);
        var userID = aXML.getElementsByTagName('user')[0].getAttribute('id');
        var userName = aXML.getElementsByTagName('username')[0].firstChild.nodeValue;

        results.setPropertyAsAString("query", "user:" + userID + "|username:" + userName);
        results.setPropertyAsAString("title", userName);
        aFlockListener.onSuccess(results, "query");
      } catch (ex) {
        aFlockListener.onError(null, "Unable to get user.");
      }
    },
    onError: function (aError) {
      aFlockListener.onError(aError, null);
    }
  }

  this._logger.debug("Finding User ID from Url: " + aUrl);
  var params = {};
  params.url = aUrl;
  this.call(myListener, "flickr.urls.lookupUser", params);
}
// END flockIMediaWebService

// BEGIN flockIMediaEmbedWebService interface
// DEFAULT: boolean getSharingContent(in nsIDOMHTMLElement aSrc,
//                                    in nsIWritablePropertyBag2 aProp);
// END flockIMediaEmbedWebService interface

// Checks if the TEXTAREA is drag and droppable
flickrService.prototype._isDnDableTextarea =
function fs__isDnDableTextarea(aDocument, aXPath, aTextarea) {
  if (aDocument && aXPath && aTextarea) {
    var xpath = this.webDetective.getString("flickr", aXPath, "");
    var results = aDocument.evaluate(xpath, aDocument, null,
                                     Ci.nsIDOMXPathResult.ANY_TYPE, null);
    if (results && results.iterateNext() == aTextarea) {
      return true;
    }
  }
  return false;
}


// BEGIN flockIRichContentDropHandler

// boolean handleDrop(in nsITransferable aFlavours,
//                    in nsIDOMHTMLElement aTargetElement);
flickrService.prototype.handleDrop =
function flickrService_handleDrop(aFlavours, aTargetElement)
{
  this._logger.info(".handleDrop()");

  // Handle textarea drops
  if (aTargetElement instanceof Ci.nsIDOMHTMLTextAreaElement) {
    var dropCallback = function flickr_dropCallback(aFlav) {
      var data = {};
      var len = {};
      aFlavours.getTransferData(aFlav, data, len);
      var caretPos = aTargetElement.selectionEnd;
      var currentValue = aTargetElement.value;

      // Add a trailing space so that we don't mangle the url
      var nextChar = currentValue.charAt(caretPos);
      var trailingSpace = ((nextChar == "") ||
                           (nextChar == " ") || 
                           (nextChar == "\n"))
                        ? ""
                        : " ";

      // We'll only add a breadcrumb if there isn't one already present
      var breadcrumb = "";
      var richDnDSvc = Cc[FLOCK_RDNDS_CONTRACTID]
                       .getService(Ci.flockIRichDNDService);
      if (!richDnDSvc.hasBreadcrumb(aTargetElement)) {
        breadcrumb = richDnDSvc.getBreadcrumb("plain");
      }

      aTargetElement.value = currentValue.substring(0, caretPos)
                           + data.value.QueryInterface(Ci.nsISupportsString)
                                 .data.replace(/: /, ":\n")
                           + trailingSpace
                           + currentValue.substring(caretPos)
                           + breadcrumb;
    };

    return this._handleTextareaDrop(CATEGORY_ENTRY_NAME,
                                    this.svcCoopObj.domains,
                                    aTargetElement,
                                    dropCallback);
  }

  // Default handling otherwise
  return false;
};
// END flockIRichContentDropHandler


// ========== END flickrService class ==========



// ================================================
// ========== BEGIN XPCOM Module support ==========
// ================================================

function createModule(aParams) {
  return {
    registerSelf: function (aCompMgr, aFileSpec, aLocation, aType) {
      aCompMgr.QueryInterface(Ci.nsIComponentRegistrar);
      aCompMgr.registerFactoryLocation( aParams.CID, aParams.componentName,
                                        aParams.contractID, aFileSpec,
                                        aLocation, aType );
      var catMgr = Cc["@mozilla.org/categorymanager;1"]
        .getService(Ci.nsICategoryManager);
      if (!aParams.categories) { aParams.categories = []; }
      for (var i = 0; i < aParams.categories.length; i++) {
        var cat = aParams.categories[i];
        catMgr.addCategoryEntry( cat.category, cat.entry,
                                 cat.value, true, true );
      }
    },
    getClassObject: function (aCompMgr, aCID, aIID) {
      if (!aCID.equals(aParams.CID)) { throw Cr.NS_ERROR_NO_INTERFACE; }
      if (!aIID.equals(Ci.nsIFactory)) { throw Cr.NS_ERROR_NOT_IMPLEMENTED; }
      return { // Factory
        createInstance: function (aOuter, aIID) {
          if (aOuter != null) { throw Cr.NS_ERROR_NO_AGGREGATION; }
          var comp = new aParams.componentClass();
          if (aParams.implementationFunc) { aParams.implementationFunc(comp); }
          return comp.QueryInterface(aIID);
        }
      };
    },
    canUnload: function (aCompMgr) { return true; }
  };
}

// NS Module entrypoint
function NSGetModule(aCompMgr, aFileSpec) {
  return createModule({
    componentClass: flickrService,
    CID: FLICKR_CID,
    contractID: FLICKR_CONTRACTID,
    componentName: CATEGORY_COMPONENT_NAME,
    implementationFunc: function (aComp) { getCompTK().addAllInterfaces(aComp); },
    categories: [
      { category: "wsm-startup", entry: CATEGORY_COMPONENT_NAME, value: FLICKR_CONTRACTID },
      { category: "flockWebService", entry: CATEGORY_ENTRY_NAME, value: FLICKR_CONTRACTID },
      { category: "flockMediaProvider", entry: CATEGORY_ENTRY_NAME, value: FLICKR_CONTRACTID },
      { category: "flockRichContentHandler", entry: CATEGORY_ENTRY_NAME, value: FLICKR_CONTRACTID }
    ]
  });
}

// ========== END XPCOM Module support ==========


// ===========================================
// ========== BEGIN FlickrAPI class ==========
// ===========================================

function FlickrAPI() {
  this._logger = Components.classes['@flock.com/logger;1'].createInstance(Components.interfaces.flockILogger);
  this._logger.init('flickrAPI');
  this._logger.info('Created Flickr API Object');
  this.acUtils = Components.classes["@flock.com/account-utils;1"]
                           .getService(Components.interfaces.flockIAccountUtils);
  this.webDetective = this.acUtils.useWebDetective("flickr.xml");
  this.faves_coop = Components.classes["@flock.com/singleton;1"]
                              .getService(Components.interfaces.flockISingleton)
                              .getSingleton("chrome://flock/content/common/load-faves-coop.js")
                              .wrappedJSObject;

  this.authAccount = null;
  this.need2CreateAlbum = false;
  this.api_key = gStrings["apikey"];
  this.api_secret = "17b26c20558cf979";
  this.endpoint = gStrings["endpoint"];
  this.upload_endpoint = gStrings["uploadendpoint"];
  this.auth_endpoint = gStrings["authendpoint"];
  this.fakeAlbums = {};
  this.realishAlbums = {};

  this.req = null;
  this.frob = null;
  this.lastToken = new Date();
  this.authUser = null;
  this.filesToDelete = [];
  this.setEndpoint = function(aEndpoint) {
    this.endpoint = aEndpoint;
  };

  this.reset = function() {
    this.hasCreateAlbum = null;
  };

  this.getAuthUser = function() {
    return this.authUser;
  };

  this.setAuthAccount = function flickrAPI_setAuthAccount(authAccountUrn) {
    this.authAccount = authAccountUrn;
  };

  this._getAccount = function flickrAPI__getAccount(authAccountUrn) {
    var service = Cc[FLICKR_CONTRACTID].getService(Ci.flockIWebService);
    return service.getAccount(authAccountUrn);
  };

  this._getNewToken = function flickrAPI__getNewToken(aListener, aFrob) {
    this._logger.info("entering _getNewToken");
    var acctCoopObj = this.faves_coop.get(this.authAccount);
    var flickrUser = new FlickrUser();
    var inst = this;
    var tokenListener = {
        onResult: function (aXML) {
          flickrUser.token = aXML.getElementsByTagName("token")[0].firstChild.nodeValue;
          var user = aXML.getElementsByTagName("user")[0];
          flickrUser.nsid = user.getAttribute("nsid");
          flickrUser.id = user.getAttribute("nsid");
          flickrUser.username = user.getAttribute("username");
          flickrUser.fullname = user.getAttribute("fullname");
          inst.authUser = flickrUser;

          acctCoopObj.authToken = flickrUser.token;
          var account = inst._getAccount(acctCoopObj.id());
          account.setCustomParam("flickr_username", flickrUser.username);
          account.setCustomParam("flickr_fullname", flickrUser.fullname);
          account.setCustomParam("flickr_nsid", flickrUser.fullname);

          inst._logger.info("got token");
          this.lastToken = new Date();
          aListener.onSuccess();
        },
        onError: function (aError) {
          inst._logger.info("error getting token");
          inst.acUtils.markAllAccountsAsLoggedOut(FLICKR_CONTRACTID);
          this.frob = null;
          if (!aError.errorCode) aError.errorCode = 1002;
          aListener.onError(aError);
        }
      }

      var frob = aFrob;
      if (!frob) {
        frob = this.frob;
      }
      var params = {
        frob: frob
      };
      this.call(tokenListener, "flickr.auth.getToken", params);
  },
  this.logout = function flickrAPI_logout() {
    this.authUser = null;
    this.authAccount = null;
    this.frob = null;
    flock_refreshMediabarIfHasPrivate()
  };
  this.login = function flickrAPI_login(aAccountURN, aListener) {
    this._logger.info("api.login('" + aAccountURN + "')");
    this.authAccount = aAccountURN;
    this.frob = null;
    var api = this;

    // Get new frob
    this._logger.info("api.login('" + aAccountURN + "'): need new token");
    var frobListener = {
      onResult: function FL_onResult(aXML) {
        try {
          var frobNode = aXML.getElementsByTagName("frob")[0];
          var frob = frobNode.firstChild.nodeValue;
          var authUrl = api.getAuthUrl(frob, "write");
          api.frob = frob;

          var hr = Cc['@mozilla.org/xmlextras/xmlhttprequest;1']
                   .createInstance(Ci.nsIXMLHttpRequest);

          var onReadyStateFunc = function login_onReadyStateFunc(eEvt) {
            if (hr.readyState == XMLHTTPREQUEST_READYSTATE_COMPLETED) {
              var wd = api.webDetective;
              var rpt = hr.responseText;
              if (wd.detectNoDOM("flickr", "apiAuth2", "", rpt, null)) {
                api._logger.info("Successfully authorized2!");
                api._getNewToken(aListener, frob);
              } else {
                var results = Cc["@mozilla.org/hash-property-bag;1"]
                              .createInstance(Ci.nsIWritablePropertyBag2);
                if (wd.detectNoDOM("flickr", "apiAuth1", "", rpt, results)) {

                  var hr2 = Cc["@mozilla.org/xmlextras/xmlhttprequest;1"]
                            .createInstance(Ci.nsIXMLHttpRequest);
                  var postBody = "magic_cookie=" +
                                 results.getPropertyAsAString("magic_cookie");
                  postBody+= "&api_key=" + api.api_key + "&api_sig="
                             + results.getPropertyAsAString("api_sig")
                             + "&perms=write&frob=" + frob + "&done_auth=1";
                  hr2.onreadystatechange = function hr2_onreadystate(eEvt2) {
                    if (hr2.readyState == XMLHTTPREQUEST_READYSTATE_COMPLETED) {
                      if (wd.detectNoDOM("flickr", "apiAuth2", "",
                                         hr2.responseText, null))
                      {
                        api._logger.info("Successfully authorized!");
                        api._getNewToken(aListener, frob);
                      } else {
                        api._logger.info("NOT successfully authorized!");
                        api._logger.info(hr2.responseText);
                        aListener.onError(null);
                      }
                    }
                  }
                  hr2.mozBackgroundRequest = true;
                  hr2.overrideMimeType("text/txt");
                  hr2.open("POST", gStrings["authendpoint"]);
                  hr2.setRequestHeader("Content-Type",
                                       "application/x-www-form-urlencoded");
                  hr2.send(postBody);
                } else if (wd.detectNoDOM("flickr", "apiAuth3", "", rpt, null))
                {
                  api._logger.info("Successfully authorized with apiAuth3!");
                  api._getNewToken(aListener, frob);
                } else {
                  api._logger.debug("Error - did not detect apiAuth1 or 3");
                  aListener.onError(null);
                }
              }
            }
          };

          hr.onreadystatechange = onReadyStateFunc;
          hr.mozBackgroundRequest = true;
          hr.overrideMimeType("text/txt");
          hr.open("GET", authUrl,true);
          hr.send(null);
        } catch(e) {
          api._logger.info("caught this one in login: " + e);
        }
      },
      onError: function FL_onError(aError) {
        api._logger.debug("There was an error getting the frob.");
        aListener.onError(aError);
      }
    };
    this.call(frobListener, "flickr.auth.getFrob", null);
  };
  this.add2Album = function(aUploadListener, aUpload, aID) {
    var inst = this;
    var listener = {
      onResult: function(aXML) {
        inst.finalizePhoto(aUploadListener, aUpload, aID);
      },
      onError: function(aError) {
        inst.finalizePhoto(aUploadListener, aUpload, aID);
      }
    }
    var params = {};
    params.photoset_id = aUpload.album;
    params.photo_id = aID;
    this.authenticatedCall(listener, "flickr.photosets.addPhoto", params);
  };
  this.reallyCreateAlbum =
  function flickrAPI_reallyCreateAlbum(aListener, aTitle, aID) {
    var myListener = {
      onResult: function fAPIrCA_onResult(aXML) {
        var newAlbum = Cc[FLOCK_PHOTO_ALBUM_CONTRACTID]
                       .createInstance(Ci.flockIPhotoAlbum);
        var photoset = aXML.getElementsByTagName("photoset")[0];
        newAlbum.title = aTitle;
        newAlbum.id = photoset.getAttribute("id");;
        aListener.onCreateAlbumResult(newAlbum);
      },
      onError: function fAPIrCA_onError(aXML) {
        aListener.onError(aXML);
      }
    }
    var params = {
      title: aTitle,
      primary_photo_id: aID
    };
    this.authenticatedCall(myListener, "flickr.photosets.create", params);
  };

  this._handleInfoResult = function(aXML) {
    var photo = aXML.getElementsByTagName("photo")[0];
    var owner = aXML.getElementsByTagName("owner")[0];
    var visibility = aXML.getElementsByTagName("visibility")[0];
    var title = aXML.getElementsByTagName("title")[0].firstChild.nodeValue;
    var dates =  aXML.getElementsByTagName("dates")[0];
    var id = photo.getAttribute("id");
    var server = photo.getAttribute("server");
    var secret = photo.getAttribute("secret");
    var lastUpdate = dates.lastupdate;
    var uploadDate = dates.posted;
    var square_url = flock_flickrBuildPhotoUrl(server,id,secret,'square');
    var small_url = flock_flickrBuildPhotoUrl(server,id,secret,'small');
    var med_url = flock_flickrBuildPhotoUrl(server,id,secret,'medium');
    var page_url = flock_flickrBuildPageUrl(owner.getAttribute('nsid'),id);
    var icon_server = photo.getAttribute("iconserver");
    var newMediaItem = Cc["@flock.com/photo;1"]
                       .createInstance(Ci.flockIMediaItem);
    newMediaItem.init("flickr", null);
    newMediaItem.webPageUrl = page_url;
    newMediaItem.thumbnail = square_url;
    newMediaItem.midSizePhoto = small_url;
    newMediaItem.largeSizePhoto = med_url;
    newMediaItem.username = owner.getAttribute("username");
    newMediaItem.title = title;
    newMediaItem.id = id;
    newMediaItem.lastUpdate = lastUpdate;
    newMediaItem.uploadDate = parseInt(uploadDate) * 1000;
    if (icon_server == "1") {
      newMediaItem.icon = gStrings["buddyicon"];
    } else {
      newMediaItem.icon = gStrings["staticbuddyicon"]
                      .replace("%iconserver%", icon_server)
                      .replace("%owner%", owner);
    }
    var visibility_ispublic = visibility.getAttribute("ispublic");
    var ispublic = (visibility_ispublic == "1") ? "true" : "false";
    newMediaItem.is_public = ispublic;
    newMediaItem.is_video = false;
    return newMediaItem;
  }

  this._getPhoto =
  function (aListener, aID) {
    var inst = this;
    var myListener = {
      onResult: function (aXML) {
        var mediaItem = inst._handleInfoResult(aXML);
        aListener.onGetPhoto(mediaItem);
      },
      onError: function (aError) {
        aListener.onError(aError);
      }
    }
    var params = {};
    params.photo_id = aID;
    this.authenticatedCall(myListener, "flickr.photos.getInfo", params, true);
  }

  this.finalizePhoto =
  function flickrAPI_finalizePhoto(aUploadListener, aUpload, aID) {
    try {
      var getPhotoListener = {
        onGetPhoto: function(aMediaItem) {
          aUploadListener.onUploadFinalized(aUpload, aMediaItem);
        },
        onError: function(aError) {
          aUploadListener.onError(null);
        }
      }
      this._getPhoto(getPhotoListener, aID);
    } catch(e) {
      this._logger.error(e);
    }
  };
  this.mAddingAlbums = false;
  this.mPhotos2Album = [];
  this.processPendingAlbumAdditions = function() {
    if (this.mAddingAlbums) return;
    if (this.mPhotos2Album.length == 0) return;
    this.mAddingAlbums = true;

    var obj = this.mPhotos2Album[0];
    var photoid = obj.photoid;
    var albumid = obj.albumid;
    var inst = this;

    if (this.fakeAlbums[albumid] && !this.realishAlbums[albumid]) {
      var listener = {
        onCreateAlbumResult: function(aAlbum) {
          inst.realishAlbums[albumid] = aAlbum.id;
          inst.mAddingAlbums = false;
        },
        onError: function(aError) {
          inst.mAddingAlbums = false;
        }
      }
//      var params = {};
//      params.photoset_id = photoid;
//      params.photo_id = albumid;
      var fakeAlbum = this.fakeAlbums[albumid];
      this.reallyCreateAlbum(listener,  fakeAlbum.title, photoid);
    }
    else {
      if (inst.realishAlbums[albumid]) {
        albumid = inst.realishAlbums[albumid];
      }
      var listener = {
        onResult: function(aXML) {
          inst.finalizePhoto(obj.listener, obj.upload, photoid);
          inst.mPhotos2Album.shift();
          inst.mAddingAlbums = false;
          inst.processPendingAlbumAdditions();
          //re-enter to optimize
        },
        onError: function(aError) {
          inst.finalizePhoto(obj.listener, obj.upload, photoid);
          inst.mPhotos2Album.shift();
          inst.mAddingAlbums = false;
          //re-enter to optimize?? this is an error condition, por
          //supuesto
        }
      }
      var params = {};
      params.photoset_id = albumid;
      params.photo_id = photoid;
      this.authenticatedCall(listener, "flickr.photosets.addPhoto", params);
    }
  };
  this.mCheckTicketsInProcess = false;
  this.processPendingUploadTickets = function() {
    if (this.mCheckTicketsInProcess) return;

    var ticketList = "";

    for (var p in this.mPendingTickets) {
      var obj = this.mPendingTickets[p];
      if (!obj) continue;//really should be removeing these - make it an array?
      ticketList += obj.ticketid + ",";
    }
    if (ticketList.length == 0) return;

    this.mCheckTicketsInProcess = true;

    var params = {
      tickets: ticketList
    };

    var inst = this;

    var ticketListener = {
      onResult: function (aXML) {
        inst.mCheckTicketsInProcess = false;
        var ticketList = aXML.getElementsByTagName("ticket");
        for (var i = 0; i < ticketList.length; i++) {
          var ticket = ticketList[i];
          var id = ticket.getAttribute("id");
          var photoid = ticket.getAttribute("photoid");
          var complete = ticket.getAttribute("complete");
          var invalid = ticket.getAttribute("invalid");
          var obj = inst.mPendingTickets[id];

          if (complete =="0") {
            continue;
          }
          else if (invalid) {
            obj.listener.onError(null);
          }
          else if (complete =="1") {
            if (obj.upload.album && obj.upload.album.length>0) {
              obj.photoid = photoid;
              obj.albumid = obj.upload.album;
              obj.listener = obj.listener;
              inst.mPhotos2Album.push(obj);
            }
            else {
              inst.finalizePhoto(obj.listener, obj.upload, photoid);
            }
          }
          else if (complete =="2") {
            obj.listener.onError(null);
          }
          inst.mPendingTickets[id] = null;//that doesn't really remove it tho
        }
      },
      onError: function (aXML) {
        inst.mCheckTicketsInProcess = false;
      }
    }

    this.call(ticketListener, "flickr.photos.upload.checkTickets", params);
  };
  this.mPendingTickets = {};
  this.upload = function(aListener, aPhoto, aParams, aUpload) {
    var inst = this;
    this.uploader = new PhotoUploader();
    var myListener = {
      onResult: function FlickrAPI_upload_onResult(aResponseText) {
        var parser = Components.classes["@mozilla.org/xmlextras/domparser;1"]
                     .createInstance(Components.interfaces.nsIDOMParser);
        var xml = parser.parseFromString(aResponseText, "text/xml");
        var rsp = xml.getElementsByTagName("rsp")[0];
        var stat = rsp.getAttribute("stat");
        if (stat != "ok") {
          var err = xml.getElementsByTagName("err")[0];
          var error= inst.getError("SERVICE_ERROR", xml, null);
          aListener.onError(error);
        }
        else {
          var ticketid = xml.getElementsByTagName("ticketid")[0].firstChild.nodeValue;
          inst._logger.info(ticketid + "pre ticketid");
          inst.mPendingTickets[ticketid] = {
            ticketid: ticketid,
            upload: aUpload,
            listener: aListener
          }
          inst._logger.info(ticketid + "post ticketid");
          aListener.onUploadComplete(aUpload);
          /*
          if (aUpload.album && aUpload.album.length>0) {
            var ticketid = aXML.getElementsByTagName("ticketid")[0].firstChild.nodeValue;
            inst.mPendingTickets[ticketid] = {
              ticketid: ticketid,
              upload: aUpload,
              listener: aListener,
            }
          }
          else {
            aListener.onUploadFinalized(aUpload, null);
          }
          */
        }
      },
      onError: function(aErrorCode) {
        if (aErrorCode) {
          aListener.onError(inst.getError('HTTP_ERROR', null, aErrorCode));
        } else {
          aListener.onError(inst.getError('SERVICE_ERROR', null));
        }
      },
      onProgress: function(aCurrentProgress) {
        aListener.onProgress(aCurrentProgress);
      }
    }
    this.convertBools(aParams);
    this.convertTags(aParams);

    aParams.auth_token = this.authUser.token;
    aParams.api_key = this.api_key;
    aParams = this.appendSignature(aParams);

    this.uploader.setEndpoint(this.upload_endpoint);
    this.uploader.upload(myListener, aPhoto, aParams);
    return;
  };
  this.convertBools = function(aParams) {
    for (var p in aParams) {
      if (!p.match(/^is/)) continue;
      // I hope that this doesn't break anything
      if (aParams[p]=="true") aParams[p] = "1";
      if (aParams[p]=="false") aParams[p] = "0";
    }
  };
  this.convertTags = function(aParams) {
    for (var p in aParams) {
      if (p != "tags") continue;
      var tags = aParams[p].split(",");
      for (var i = 0; i < tags.length;++i) {
        tags[i] = '"' + tags[i] + '"';
        tags[i] = tags[i].replace(/\"+/g,'"');
      }
      aParams[p] = tags.join(",");
    }
  };
  this.authenticatedCall =
  function flickrAPI_authCall(aListener, aMethod, aParams, aAuthOptional) {
    if (!aParams) {
      aParams = {};
    }
    if (this.authUser) {
      aParams.auth_token = this.authUser.token;
    }

    if (this.authUser || (aAuthOptional && !this.authAccount)) {
      this.call(aListener, aMethod, aParams);
    } else {
      if (!this.authAccount) {
        throw "not logged in";
      }
      var inst = this;
      var authListener = {
        onSuccess: function authListener_onSuccess() {
          inst.authenticatedCall(aListener, aMethod, aParams);
        },
        onError: function authListener_onError(aFlockError) {
          inst.authAccount = false;
          inst.acUtils.markAllAccountsAsLoggedOut(FLICKR_CONTRACTID);
          if(!aAuthOptional) {
            aListener.onError(aFlockError);
          }
          inst.call(aListener, aMethod, aParams);
        }
      }
      this.login(this.authAccount, authListener);
    }
  };
  this.call = function(aListener, aMethod, aParams) {
    this.convertBools(aParams);
    this.convertTags(aParams);
    if (aParams == null) aParams = [];
    if (aMethod) aParams.method = aMethod;
    aParams.api_key = this.api_key;
    var paramString = this.getParamString(aParams, (aMethod == null));
    var url = this.endpoint + "?" + paramString;
    this._doCall(aListener, url, null);
  };
  this._doCall = function flApi__doCall(aListener, aUrl, aContent) {
    this._logger.info("Sending " + aUrl);
    this.req = Components.classes['@mozilla.org/xmlextras/xmlhttprequest;1']
                         .createInstance(Components.interfaces.nsIXMLHttpRequest);
    this.req.QueryInterface(Components.interfaces.nsIJSXMLHttpRequest);
    this.req.mozBackgroundRequest = true;
    this.req.open('GET', aUrl, true);
    var req = this.req;
    var inst = this;
    this.req.onreadystatechange =
    function flApi__doCall_onreadystatechange(aEvt) {
      if (req.readyState == XMLHTTPREQUEST_READYSTATE_COMPLETED) {
        if (Math.floor(req.status/100) == 2) {
           try {
            var responseXML = req.responseXML;
            var rsp = responseXML.getElementsByTagName("rsp")[0];
            var stat = rsp.getAttribute("stat");
            if (stat != "ok") {
              var err = responseXML.getElementsByTagName("err")[0];
              var error = inst.getError("SERVICE_ERROR", responseXML, null);
              aListener.onError(error);
            } else {
              aListener.onResult(responseXML);
            }
          } catch (ex) {
            // error parsing xml
            inst._logger.error(ex);
            aListener.onError(inst.getError("SERVICE_ERROR", null, null));
          }
        } else {
          // HTTP errors (0 for connection lost)
          aListener.onError(inst.getError("HTTP_ERROR", null, req.status));
        }
      }
    }
    this.req.send(null);
  };
  this.getError = function(aErrorType, aXML, aHTTPErrorCode) {
    var error = Components.classes[FLOCK_ERROR_CONTRACTID].createInstance(flockIError);
    if  (aErrorType == "HTTP_ERROR") {
      error.errorCode = aHTTPErrorCode;
    } else if (aErrorType == "SERVICE_ERROR") {
      var errorCode;
      var errorMessage;
      var serviceErrorMessage;
      try {
        errorCode = aXML.getElementsByTagName("err")[0].getAttribute('code');
        serviceErrorMessage = aXML.getElementsByTagName("err")[0].getAttribute('msg');
      } catch (ex) {
        errorCode = "999" // in case the error xml is invalid
      }

      switch (errorCode) {
        case "1":
          error.errorCode = error.PHOTOSERVICE_INVALID_USER;
        break;

        case "3":
          error.errorCode = error.PHOTOSERVICE_UPLOAD_ERROR;
        break;

        case "4":
          error.errorCode = error.PHOTOSERVICE_FILE_IS_OVER_SIZE_LIMIT;
        break;

        case "5":
          error.errorCode = error.PHOTOSERVICE_INVALID_UPLOAD_FILE;
        break;

        case "6":
          error.errorCode = error.PHOTOSERVICE_BANDWIDTH_REACHED;
        break;

        case "10":

        break;

        case "98":
          error.errorCode = error.PHOTOSERVICE_LOGIN_FAILED;
        break;

        case "99":
          error.errorCode = error.PHOTOSERVICE_USER_NOT_LOGGED_IN;
        break;

        case "100":
          error.errorCode = error.PHOTOSERVICE_INVALID_API_KEY;
          break;

        case "105":
          error.errorCode = error.PHOTOSERVICE_UNAVAILABLE;
        break;

        case "999":
          error.errorCode = error.PHOTOSERVICE_UNKNOWN_ERROR;
        break;

        default:
          error.errorCode = error.PHOTOSERVICE_UNKNOWN_ERROR;
          error.serviceErrorString = serviceErrorMessage;
        break;
      }
    }
    //inst._logger.info('<<<<<<<<<<<<<<'  + error.errorString + '\n');
    return error;
  };
  this.getAuthUrl = function(aFrob, aPerms) {
    var api_key = this.api_key;
    var params = {
      api_key: api_key,
      perms: aPerms,
      frob: aFrob
    }

    var paramString = this.getParamString(params, true);
    return this.auth_endpoint + "?" + paramString;
  };
  this.getParamString = function(aParams, aNoMethod) {
    aParams = this.appendSignature(aParams);
    var rval = "";
    if (!aNoMethod) rval += "method=" + aParams.method + "&";

    var count = 0;
    for (var p in aParams) {
      if (p.match(/method/)) continue;

      if (count++ != 0) rval += "&";
      rval += encodeURIComponent(p) + "=" + encodeURIComponent(aParams[p]);
    }
    return rval;
  };
  this.appendSignature = function(aParams) {
    var keys = [];
    for (var p in aParams) {
      keys.push(p);
    }
    keys.sort();
    var preHash = this.api_secret;
    for (var i = 0; i < keys.length; ++i) {
      preHash += keys[i] + aParams[keys[i]];
    }
    this._logger.info("preHash " + preHash);

    var converter =
      Components.classes["@mozilla.org/intl/scriptableunicodeconverter"]
                .createInstance(Components.interfaces.nsIScriptableUnicodeConverter);
    converter.charset = "UTF-8";

    var inputStream = converter.convertToInputStream(preHash);
    aParams.api_sig = FlockCryptoHash.md5Stream(inputStream);
    return aParams;
  };
}

FlickrAPI.prototype = {};

// ========== END FlickrAPI class ==========



// ============================================
// ========== BEGIN FlickrUser class ==========
// ============================================

function FlickrUser() {
  this.token = "";
  this.perms = "";
  this.nsid = "";
  this.id = "";
  this.username = "";
  this.fullname = "";
}

FlickrUser.prototype = {};

// ========== END FlickrUser class =========



// ===============================================
// ========== BEGIN flickrAccount class ==========
// ===============================================

function flickrAccount()
{
  this._logger = Components.classes['@flock.com/logger;1'].createInstance(Components.interfaces.flockILogger);
  this._logger.init('flickrAccount');
  this._logger.info('Created Flickr Account Object');

  this.acUtils = Components.classes["@flock.com/account-utils;1"]
                           .getService(Components.interfaces.flockIAccountUtils);
  this.svc = Components.classes[FLICKR_CONTRACTID]
                       .getService(Components.interfaces.flockIMediaWebService)
                       .QueryInterface(Components.interfaces.flockIWebService);
  if (!gFlickrAPI) {
    gFlickrAPI = new FlickrAPI();
  }
  this.api = gFlickrAPI;

  this._coop = Components.classes["@flock.com/singleton;1"]
                         .getService(Components.interfaces.flockISingleton)
                         .getSingleton("chrome://flock/content/common/load-faves-coop.js")
                         .wrappedJSObject;
  this.webDetective = Cc["@flock.com/web-detective;1"]
                      .getService(Ci.flockIWebDetective);
  this._ctk = {
    interfaces: [
      "nsISupports",
      "flockIWebServiceAccount",
      "flockISocialAccount",
      "flockIMediaAccount",
      "flockIMediaUploadAccount",
      "flockIFlickrAccount"
    ]
  };
  getCompTK().addAllInterfaces(this);
}


// BEGIN flockIWebServiceAccount interface
flickrAccount.prototype.urn = null;

flickrAccount.prototype.login =
function flickrAccount_login(aFlockListener)
{
  this._logger.info("{flockIWebServiceAccount}.login()");
  this.api.setAuthAccount(this.urn);
  this.acUtils.ensureOnlyAuthenticatedAccount(this.urn);
  // Force refresh on login
  var pollerSvc = Cc["@flock.com/poller-service;1"]
                  .getService(Ci.flockIPollerService);
  pollerSvc.forceRefresh(this.urn);
  if (aFlockListener) {
    aFlockListener.onSuccess(this, "login");
  }
}

flickrAccount.prototype.logout =
function flickrAccount_logout(aFlockListener)
{
  this._logger.info("{flockIWebServiceAccount}.logout()");
  var acctCoopObj = this._coop.get(this.urn);
  if (acctCoopObj.isAuthenticated) {
    acctCoopObj.isAuthenticated = false;
    this.svc.logout();
  }
}

flickrAccount.prototype.keep =
function flickrAccount_keep()
{
  this._logger.info(".keep(): "+this.urn);
  var c_acct = this._coop.get(this.urn);
  c_acct.isTransient = false;
  this.acUtils.makeTempPasswordPermanent(this.svc.urn + ":"
                                         + c_acct.accountId);

  // Subscribe to the recent activity RSS feed for this account
  var feedMgr = Cc["@flock.com/feed-manager;1"]
                .getService(Ci.flockIFeedManager);
  var inst = this;
  var feedListener = {
    onGetFeedComplete: function keep_feed_complete(aFeed) {
      feedMgr.getFeedContext("news").getRoot().subscribeFeed(aFeed);
      Cc["@flock.com/metrics-service;1"]
        .getService(Ci.flockIMetricsService)
        .report("FeedsSidebar-AddFeed",  "FlickrAccountKeep");

      var coop = FlockSvcUtils.getCoop(this);

      var title = flockGetString(SERVICES_STRING_BUNDLE,
                                 SERVICES_PHOTO_FEED_COMMENT_STRING);
      var username = coop.get(inst.urn).name;

      var feedTitle = title.replace("%s", username);

      coop.get(aFeed.id()).name = feedTitle;
    },
    onError: function keep_feed_error(aError) {
      inst._logger.debug("There was a problem subscribing to the recent "
                         + "activity feed for Flickr account "
                         + c_acct.accountId);
    }
  };
  try {
    var feedURI = Cc["@mozilla.org/network/standard-url;1"]
                  .createInstance(Ci.nsIURI);
    feedURI.spec = gStrings["commentsreceivedRSS"]
                   .replace("%accountid%", c_acct.accountId);
    feedMgr.getFeed(feedURI, feedListener);
  } catch (ex) {
    this._logger.debug("There was an error subscribing to the recent "
                       + "activity feed for Flickr account "
                       + c_acct.accountId);
  }
}
// END flockIWebServiceAccount interface


// BEGIN flockISocialAccount interface
flickrAccount.prototype.hasFriendActions = true;
flickrAccount.prototype.isMeStatusSupported = false;
flickrAccount.prototype.isFriendStatusSupported = false;
flickrAccount.prototype.isStatusEditable  = false;
flickrAccount.prototype.isPostLinkSupported = false;
flickrAccount.prototype.isMyMediaFavoritesSupported = false;

flickrAccount.prototype.setStatus =
function flickrAccount_setStatus(aStatusMessage, aFlockListener)
{
  this._logger.info(".setStatus('" + aStatusMessage + "', aFlockListener)");
}

flickrAccount.prototype.getEditableStatus =
function flickrAccount_getEditableStatus()
{
  this._logger.info("{flockISocialAccount}.getEditableStatus()");
  return "";
}

flickrAccount.prototype.formatStatusForDisplay =
function flickrAccount_formatStatusForDisplay(aStatusMessage)
{
  return "";
}

flickrAccount.prototype.markAllMeNotificationsSeen =
function flickrAccount_markAllMeNotificationsSeen(aType) {
  this._logger.debug(".markAllMeNotificationsSeen('" + aType + "')");
  var c_acct = this._coop.get(this.urn);
  switch (aType) {
    case "meMessages":
      c_acct.accountMessages = 0;
      break;
    case "meCommentActivity":
      var comments = this.getCustomParam("flickr_comments");
      if (comments > 0) {
        c_acct.lastUpdateDate = new Date();
        this.setCustomParam("flickr_comments", 0);
        this.setCustomParam("flickr_comments_timespan", 0);
      }
      break;
      
    default:
      break;
  }
}

flickrAccount.prototype.getMeNotifications =
function flickrAccount_getMeNotifications()
{
  this._logger.info(".getMeNotifications()");

  var sbs = Cc["@mozilla.org/intl/stringbundle;1"]
            .getService(Ci.nsIStringBundleService);
  var bundle = sbs.createBundle(FLICKR_STRING_BUNDLE);

  var noties = [];
  var inst = this;
  function _addNotie(aType, aCount, aUrl) {
    var stringName = "flock.flickr.noties."
                   + aType + "."
                   + ((parseInt(aCount) <= 0) ? "none" : "some");
    noties.push({
      class: aType,
      tooltip: bundle.GetStringFromName(stringName),
      metricsName: aType,
      count: aCount,
      URL: aUrl
    });
  }
  var c_acct = this._coop.get(this.urn);
  var url = this.webDetective.getString("flickr", "meMessages_URL", "");
  _addNotie("meMessages", c_acct.accountMessages, url);

  var timespan = this.getCustomParam("flickr_comments_timespan");
  timespan = (timespan) ? timespan : "";
  url = this.webDetective.getString("flickr", "meCommentActivity_URL", "")
            .replace("%timespan%", timespan);

  var comments = this.getCustomParam("flickr_comments");
  comments = (comments) ? comments : 0;
  _addNotie("meCommentActivity", comments, url);
  var nsJSON = Cc["@mozilla.org/dom/json;1"].createInstance(Ci.nsIJSON);
  return nsJSON.encode(noties);
}

flickrAccount.prototype.getFriendActions =
function flickrAccount_getFriendActions(aFriendURN)
{
  this._logger.info(".getFriendActions('" + aFriendURN + "')");

  var actionNames = ["friendMessage",
                     "friendViewProfile",
                     "friendShareFlock"];

  var sbs = Cc["@mozilla.org/intl/stringbundle;1"]
            .getService(Ci.nsIStringBundleService);
  var bundle = sbs.createBundle(FLICKR_STRING_BUNDLE);

  var actions = [];
  var c_friend = this._coop.get(aFriendURN);
  if (c_friend) {
    var c_acct = this._coop.get(this.urn);
    for each (var i in actionNames) {
      actions.push({
        label: bundle.GetStringFromName("flock.flickr.actions." + i),
        class: i,
        spec: this.webDetective.getString("flickr", i, "")
                  .replace("%accountid%", c_acct.accountId)
                  .replace("%friendid%", c_friend.accountId)
      });
    }
  }
  var nsJSON = Cc["@mozilla.org/dom/json;1"].createInstance(Ci.nsIJSON);
  return nsJSON.encode(actions);
}

flickrAccount.prototype.getSharingAction =
function flickrAccount_getSharingAction(aFriendURN, aTransferable)
{
  this._logger.info(".getSharingAction('" + aFriendURN + "')");

  var sharingAction = "";
  var c_friend = this._coop.get(aFriendURN);
  if (c_friend) {
    // Flavors we want to support, in order of preference
    var flavors = ["text/x-flock-media",
                   "text/x-moz-url",
                   "text/unicode",
                   "text/html"];

    var message = Cc[FLOCK_RDNDS_CONTRACTID]
                  .getService(Ci.flockIRichDNDService)
                  .getMessageFromTransferable(aTransferable,
                                              flavors.length,
                                              flavors);
    if (message.body) {
      sharingAction = this.webDetective.getString("flickr", "shareAction", "")
                          .replace("%friendid%", c_friend.accountId);
    }
  }
  this._logger.info("sharingAction = "+sharingAction);
  return sharingAction;
}

flickrAccount.prototype.getProfileURLForFriend = 
function flickrAccount_getProfileURLForFriend(aFriendURN)
{
  this._logger.info(".getProfileURLForFriend('" + aFriendURN + "')");

  var url = "";
  var c_friend = this._coop.get(aFriendURN);
  if (c_friend) {
    url = this.webDetective.getString("flickr", "friendprofile", "")
                           .replace("%accountid%", c_friend.accountId);
  }

  return url;
}

flickrAccount.prototype.getPostLinkAction =
function flickrAccount_getPostLinkAction(aTransferable)
{
  return "";
}
// END flockISocialAccount interface


// BEGIN flockIFlickrAccount interface
flickrAccount.prototype.shareFlock =
function flickrAccount_shareFlock(aFriendURN) 
{
  this._logger.info(".shareFlock('" + aFriendURN + "')");
  var sbs = Cc["@mozilla.org/intl/stringbundle;1"]
              .getService(Ci.nsIStringBundleService);
  var bundle = sbs.createBundle(FLICKR_STRING_BUNDLE);
  var body = bundle.GetStringFromName("flock.flickr.friendShareFlock.message");
  var subj = bundle.GetStringFromName("flock.flickr.friendShareFlock.subject");
  this._composeMessage(aFriendURN, subj, body, false);
}

flickrAccount.prototype.flickrMail =
function flickrAccount_flickrMail(aFriendURN, aTransferable)
{
  this._logger.info(".flickrMail('" + aFriendURN + "')");

  var flavors = ["text/x-flock-media",
                 "text/x-moz-url",
                 "text/unicode",
                 "text/html"];

  var message = Cc[FLOCK_RDNDS_CONTRACTID]
                  .getService(Ci.flockIRichDNDService)
                  .getMessageFromTransferable(aTransferable,
                                              flavors.length,
                                              flavors);
  if (message.body) {
    this._composeMessage(aFriendURN, message.subject, message.body, true);
  }
}

flickrAccount.prototype._composeMessage =
function flickrAccount__composeMessage(aFriendURN, aSubject, aBody, addBreadCrumb)
{
  var body = aBody;
  var subject = aSubject;
  var c_friend = this._coop.get(aFriendURN);
  var url = this.webDetective.getString("flickr", "flickrmail_URL", "")
                             .replace("%friendid%", c_friend.accountId);
  var wm = Cc["@mozilla.org/appshell/window-mediator;1"]
             .getService(Ci.nsIWindowMediator);
  var win = wm.getMostRecentWindow("navigator:browser");
  if (win) {
   var browser = win.getBrowser();
   var newTab = browser.loadOneTab(url, null, null, null, false, false);
   var obs = Cc["@mozilla.org/observer-service;1"]
             .getService(Ci.nsIObserverService);
   var inst = this;
   var observer = {
     observe: function openSendMessageTabForFill_observer(aContent,
                                                          aTopic,
                                                          aContextUrl)
     {
       var contentWindow = newTab.linkedBrowser.docShell
                                 .QueryInterface(Ci.nsIInterfaceRequestor)
                                 .getInterface(Ci.nsIDOMWindow);
       function insertContent(aWebDicString, aMessage) {
         var xpathquery = inst.webDetective.getString("flickr", aWebDicString, "");
         var doc = contentWindow.document;
         var formItems = doc.evaluate(xpathquery, doc, null,
                                      Ci.nsIDOMXPathResult.ANY_TYPE, null);
         if (formItems) {
           var formItem = formItems.iterateNext();
           if (formItem.hasAttribute("value")) {
             formItem.setAttribute("value", aMessage);
           } else {
             var textNode = doc.createTextNode(aMessage);
             formItem.appendChild(textNode);
             inst._logger.info("aMessage " + aMessage);
           }
         }
       }
       if (contentWindow == aContent) {
         obs.removeObserver(this, "EndDocumentLoad");
         insertContent("flickrmail_subjectXPath", subject);
         if(addBreadCrumb)
         {
            // Add breadcrumb to message body
            var breadcrumb = Cc[FLOCK_RDNDS_CONTRACTID]
                               .getService(Ci.flockIRichDNDService)
                               .getBreadcrumb("plain");
           if (breadcrumb) {
             body += breadcrumb;
           }
        }
         insertContent("flickrmail_bodyXPath", body);
       }
     }
   };
   obs.addObserver(observer, "EndDocumentLoad", false);
  }
}
// END flockIFlickrAccount interface

// ========== END flickrAccount class ==========
