// BEGIN FLOCK GPL
//
// Copyright Flock Inc. 2005-2008
// http://flock.com
//
// This file may be used under the terms of the
// GNU General Public License Version 2 or later (the "GPL"),
// http://www.gnu.org/licenses/gpl.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// END FLOCK GPL

// CONTENTS OF THIS FILE:
// This file contains the implementation of the LiveJournal service. This
// involves the following components:
//   flockLJService - the service class itself
//   flockLJController - the controller class
//   flockLJServiceFactory - factory object for flockLJService
//   flockLJModule - module object for XPCOM registration
//   flockLJAccount - the account class

const CC = Components.classes;
const CI = Components.interfaces;
const CR = Components.results;
const CU = Components.utils;

CU.import("resource:///modules/FlockXPCOMUtils.jsm");
FlockXPCOMUtils.debug = false;
CU.import("resource:///modules/FlockSvcUtils.jsm");

const LIVEJOURNAL_CID = Components.ID("{e642271b-1b0d-4d5d-ab51-34de23f75fc7}");
const LIVEJOURNAL_CONTRACTID = "@flock.com/people/livejournal;1";
const LIVEJOURNAL_TITLE = "LiveJournal Web Service";
const LIVEJOURNAL_FAVICON = "chrome://flock/content/services/livejournal/favicon.png";
const SERVICE_ENABLED_PREF = "flock.service.livejournal.enabled";
const CATEGORY_COMPONENT_NAME = "LiveJournal JS Component"
const CATEGORY_ENTRY_NAME = "livejournal"

const CLASS_SHORT_NAME = "livejournal";
const CLASS_NAME = "LiveJournal";
const MODULE_NAME = "livejournal";


function loadSubScript(spec) {
  var loader = CC["@mozilla.org/moz/jssubscript-loader;1"]
               .getService(CI.mozIJSSubScriptLoader);
  var context = {};
  loader.loadSubScript(spec, context);
  return context;
}

// String defaults... may be updated later through Web Detective
var gStrings = {
  "domains": "livejournal.com",
  "userprofile": "http://%accountid%.livejournal.com/profile",
  "editprofile": "http://www.livejournal.com/manage/profile/",
  "userblog": "http://%accountid%.livejournal.com/",
  "userfriends": "http://%accountid%.livejournal.com/friends",
  "userphotos": "http://pics.livejournal.com/%accountid%",
  "userpics": "http://www.livejournal.com/allpics.bml?user=%accountid%",
  "postcomment": "http://fixme.com",
  "userfoaf": "http://www.livejournal.com/users/%accountid%/data/foaf"
};

// Helper function for LJ URLs - needs to be updated periodically
function makeLiveJournalURL(aType, aFriendID) {
  switch (aType) {
    case "openProfile":
      return gStrings["userprofile"].replace("%accountid%", aFriendID);
    case "openBlog":
      return gStrings["userblog"].replace("%accountid%", aFriendID);
    case "viewFriends":
      return gStrings["userfriends"].replace("%accountid%", aFriendID);
    case "postComment":
      return gStrings["postcomment"].replace("%accountid%", aFriendID);
    case "openPhotos":
      return gStrings["userphotos"].replace("%accountid%", aFriendID);
    case "userpics":
      return gStrings["userpics"].replace("%accountid%", aFriendID);
    default:
      return gStrings[aType].replace("%accountid%", aFriendID);
  }
}

function userBlogsListener(aBlogListener) {
  this.blogListener = aBlogListener;
}

userBlogsListener.prototype = {
  // aSimpleEnum is an Array (simpleEnumerator) of struct objects
  onResult: function UBL_onResult(aSimpleEnum) {
    var result = new Array();
    for each (var entry in aSimpleEnum){
      var blog = CC["@mozilla.org/hash-property-bag;1"]
                 .createInstance(CI.nsIWritablePropertyBag2);
      blog.setPropertyAsAString("title", entry.blogName);
      blog.setPropertyAsAString("blogID", entry.blogid);
      blog.setPropertyAsAString("APILink", "");
      blog.setPropertyAsAString("URL", entry.url);
      result.push(blog);
    }
    this.blogListener.onResult(new simpleEnumerator(result), null);
  },
  onError: function UBL_onError(aErrorCode, aErrorMsg) {
    var error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
    error.serviceErrorCode = aErrorCode;
    error.serviceErrorString = aErrorMsg;
    switch (aErrorCode) {
      case 0:
        // Invalid login/pass
        error.errorCode = CI.flockIError.BLOG_INVALID_AUTH;
        break;
      default:
        // Unknown error code
        error.errorCode = CI.flockIError.BLOG_UNKNOWN_ERROR;
    }
    this.blogListener.onError(error);
  },
  onFault: function UBL_onFault(aErrorCode, aErrorMsg) {
    var error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
    error.serviceErrorCode = aErrorCode;
    error.serviceErrorString = aErrorMsg;
    switch (aErrorCode) {
      case 0:
        // Invalid login/pass
        error.errorCode = CI.flockIError.BLOG_INVALID_AUTH;
        break;
      default:
        // Unknown error code
        error.errorCode = CI.flockIError.BLOG_UNKNOWN_ERROR;
    }
    this.blogListener.onError(error);
  }
}



/*************************************************************************
 * Component: Flock Livejournal Service
/*************************************************************************/

function flockLJService() {
  var loader = CC["@mozilla.org/moz/jssubscript-loader;1"]
               .getService(CI.mozIJSSubScriptLoader);
  loader.loadSubScript("chrome://browser/content/utilityOverlay.js");
  loader.loadSubScript("chrome://flock/content/xmlrpc/xmlrpchelper.js");
  loader.loadSubScript("chrome://flock/content/blog/blogBackendLib.js");

  this.initialized = false;

  // flockIBlogWebService
  this.supportsCategories = 0;
  this.supportsPostReplace = true;
  this.metadataOverlay = "chrome://flock/content/services/livejournal/ljBlogOverlay.xul";

  this._profiler = CC["@flock.com/profiler;1"].getService(CI.flockIProfiler);

  this._accountClassCtor = flockLJAccount;

  this._init();

  var ws = FlockSvcUtils.flockIWebService;
  ws.addDefaultMethod(this, "url");
  ws.addDefaultMethod(this, "isHidden");
  ws.addDefaultMethod(this, "getStringBundle");

  var lws = FlockSvcUtils.flockILoginWebService;
  lws.addDefaultMethod(this, "loginURL");
  lws.addDefaultMethod(this, "getAccount");
  lws.addDefaultMethod(this, "getAuthenticatedAccount");
  lws.addDefaultMethod(this, "getAccounts");
  lws.addDefaultMethod(this, "removeAccount");
  lws.addDefaultMethod(this, "docRepresentsSuccessfulLogin");
  lws.addDefaultMethod(this, "ownsLoginForm");
  lws.addDefaultMethod(this, "getSessionValue");
  lws.addDefaultMethod(this, "getCredentialsFromForm");
  lws.addDefaultMethod(this, "getAccountIDFromDocument");

  ws.addDefaultMethod(this, "getStringBundle");
}

/*************************************************************************
 * flockLJService: XPCOM Component Creation
 *************************************************************************/

flockLJService.prototype = new FlockXPCOMUtils.genericComponent(
  LIVEJOURNAL_TITLE,
  LIVEJOURNAL_CID,
  LIVEJOURNAL_CONTRACTID,
  flockLJService,
  CI.nsIClassInfo.SINGLETON,
  [
    CI.flockIWebService,
    CI.flockIAuthenticateNewAccount,
    CI.flockILoginWebService,
    CI.flockIBlogWebService
  ]
);

// FlockXPCOMUtils.genericModule() categories
flockLJService.prototype._xpcom_categories = [
  { category: "wsm-startup" },
  { category: "flockWebService", entry: CLASS_SHORT_NAME }
];

/*************************************************************************
 * flockLJService: flockIWebService implementation
 *************************************************************************/

// readonly attribute AString contractId;
// ALMOST duplicated by nsIClassInfo::contractID
// with different case: contractId != contractID
flockLJService.prototype.contractId = LIVEJOURNAL_CONTRACTID;

// readonly attribute AString icon;
flockLJService.prototype.icon = LIVEJOURNAL_FAVICON;

// readonly attribute boolean needPassword;
flockLJService.prototype.needPassword = true;

// readonly attribute AString shortName;
flockLJService.prototype.shortName = CLASS_SHORT_NAME;

// readonly attribute AString title;
flockLJService.prototype.title = CLASS_NAME;

flockLJService.prototype.addAccount =
function flockLJService_addAccount(aAccountID, aIsTransient, aFlockListener) {
  this._logger.debug("addAccount('" + aAccountID + "', "
                     + aIsTransient + ", aFlockListener)");
  var ljAccountURN = "urn:flock:lj" + aAccountID;
  var ljAccount = new this._coop.Account(ljAccountURN, {
    name: aAccountID,
    serviceId: LIVEJOURNAL_CONTRACTID,
    service: this.ljService,
    accountId: aAccountID,
    favicon: LIVEJOURNAL_FAVICON,
    URL: "http://www.livejournal.com/portal",
    isPollable: true,
    isTransient: aIsTransient
  });
  this.account_root.children.addOnce(ljAccount);

  this.USER = ljAccountURN;
  // Add the blog(s)
  var ljsvc = this;
  var blogListener = {
    onResult: function(aSimpleEnum) {
      ljsvc._logger.info("addAccount: got the blog list from the server");
      var theBlog;
      while (aSimpleEnum.hasMoreElements()) {
        theBlog = aSimpleEnum.getNext();
        theBlog.QueryInterface(CI.nsIPropertyBag2);
        var blogID = theBlog.getPropertyAsAString("blogID");
        var title = theBlog.getPropertyAsAString("title");
        var URL = theBlog.getPropertyAsAString("URL");
        var blogURN = ljAccountURN + ":" + title;
        theCoopBlog = new ljsvc._coop.Blog(blogURN, {
          name: title,
          title: title,
          blogid: blogID,
          URL: URL,
          apiLink: "https://www.livejournal.com/interface/xmlrpc"
        });
        ljAccount.children.addOnce(theCoopBlog);
      }
      if (aFlockListener) {
        aFlockListener.onSuccess(acct, "addAccount");
      }
    },
    onError: function addAcc_listener_onError(aFlockError) {
      ljsvc._coop.accounts_root.children.remove(ljAccount);
      ljAccount.destroy();
      if (aFlockListener) {
        aFlockListener.onError(aFlockError, null);
      }
    }
  }

  var apiLink = "https://www.livejournal.com/interface/blogger";
  var username = this._coop.get(this.USER).name;
  var pw = this._acUtils.getPassword(this.urn + ":" + username);
  this.getUsersBlogs(blogListener, apiLink, username, pw.password);

  acct = this.getAccount(ljAccountURN);
  return acct;
}

flockLJService.prototype.logout =
function flockLJService_logout() {
  this._acUtils.removeCookies(this.webDetective
                                 .getSessionCookies("livejournal"));
  this.USER = null;
}


// BEGIN flockILoginWebService interface
flockLJService.prototype.ownsDocument =
function flockLJService_ownsDocument(aDocument) {
  this._logger.debug(".ownsDocument(...)");
  aDocument.QueryInterface(CI.nsIDOMHTMLDocument);
  var ios = CC["@mozilla.org/network/io-service;1"]
            .getService(CI.nsIIOService);
  var uri = ios.newURI(aDocument.URL, null, null);
  if ((uri.host.indexOf("livejournal.com") == 0) ||
      (uri.host.indexOf(".livejournal.com") > 0))
  {
    if (uri.host.indexOf("adserver.livejournal.com") >= 0) {
      return false;
    } else {
      return true;
    }
  }
  return false;
}

flockLJService.prototype.updateAccountStatusFromDocument =
function flockLJService_updateAccountStatusFromDocument(aDocument,
                                                        aAcctURN,
                                                        aFlockListener)
{
  this._logger.debug(".updateAccountStatusFromDocument(...)");
  if (aAcctURN) {
    // We're logged in to this account
    this._acUtils.ensureOnlyAuthenticatedAccount(aAcctURN);
    this.USER = aAcctURN;
  } else if (this.webDetective
                 .detect("livejournal", "loggedout", aDocument, null))
  {
    // We're logged out (of all accounts)
    this._acUtils.markAllAccountsAsLoggedOut(LIVEJOURNAL_CONTRACTID);
  }
}
// END flockILoginWebService interface


// BEGIN flockIBlogWebService interface
flockLJService.prototype.parseUsersBlogs =
function flockLJService_parseUsersBlogs(listener, inst) {
  try {
    this._logger.debug(".parseUsersBlogs(...)");
    var result = [];
    var dom = inst._req.responseXML;
    var domEntries = dom.getElementsByTagName("entry");

    if (domEntries.length > 0) {
      // Regular ATOM feed (ATOM)
      for (var i = 0; i < domEntries.length; i++) {
        domEntry = domEntries[i];
        title = domEntry.getElementsByTagName("title")[0].textContent;
        var newAccount = new BlogAccount(title);
        newAccount.api = this.shortName;
        var links = domEntry.getElementsByTagName("link");
        for (var j = 0; j < links.length; j++) {
          var link = links[j]
          switch (link.getAttribute("rel")) {
            case "alternate":
              newAccount.URL = link.getAttribute("href");
              break;
            case "http://schemas.google.com/g/2005#post":
              newAccount.apiLink = link.getAttribute("href");
              break;
          }
        }
        result.push(newAccount);
      }
    }
    else { // Just a list of links (Livejournal)
      var links = dom.getElementsByTagName("link");
      var result = [];
      for (var i = 0; i < links.length; i++) {
        var link = links[i];
        var title = link.getAttribute("title");
        var rel = link.getAttribute("rel");
        var href = link.getAttribute("href");
        switch (rel) {
          case "service.post":
          {
            href.match(/.+\/(.+)/);
            var newAccount = new BlogAccount(title);
            newAccount.api = this.shortName;
            newAccount.blogid = RegExp.$1;
            newAccount.apiLink = href;
            newAccount.editLink = href;
            result.push(newAccount);
          }; break;
          case "alternate":
          {
            for (j in result) {
              if (result[j].title == title) {
                result[j].URL = href;
              }
            }
          }; break;
        }
      }
    }
    this._logger.info("Found " + result.length + " blogs");
    listener.onResult(new simpleEnumerator(result));
  }
  catch(e) {
    var logger = CC["@flock.com/logger;1"]
                 .createInstance(CI.flockILogger);
    logger.init("livejournal");
    logger.error(e + " " + e.lineNumber);
    listener.onError(e + " " + e.lineNumber);
  }
}


flockLJService.prototype.newPost =
function flockLJService_newPost(aPublishListener,
                                aBlogId,
                                aPost,
                                aPublish,
                                aNotifications)
{
  var svc = this;

  var coopBlog = this._coop.get(aBlogId);
  var coopAccount = coopBlog.getParent();

  var pw = this._acUtils.getPassword(this.urn + ":" + coopAccount.accountId);

  var extra = [];
  if (aPost.extra) {
    while (aPost.extra.hasMore()) {
      extra.push(aPost.extra.getNext());
    }
  }

  var nocomments = (extra[2] == "0") ? false : true;
  var notifsArray = [];

  var date = new Date();
  var hours = date.getHours();
  var month = date.getMonth() + 1;
  var day = date.getDate();
  var year = date.getFullYear();
  var hour = date.getHours();
  var min = date.getMinutes();

  var labels = [];
  if (aPost.tags) {
    while (aPost.tags.hasMore()) {
      labels.push(aPost.tags.getNext());
    }
  }
  var content = aPost.description.replace("</lj>", "", "gi");

  var listener = {
    onResult: function(aResult) {
      aPublishListener.onResult(aResult);
    },
    onError: function(errorcode, errormsg) {
      svc._logger.error("<<<<<<<<<< Livejournal: SERVER TO FLOCK");
      svc._logger.error("FAULT " + errorcode + " " + errormsg);

      var error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
      error.serviceErrorCode = errorcode;
      error.serviceErrorString = errormsg;
      switch (errorcode) {
        case 0: // Invalid login/pass
          error.errorCode = CI.flockIError.BLOG_INVALID_AUTH;
          break;
        default: // Unknown error code
          error.errorCode = CI.flockIError.BLOG_UNKNOWN_ERROR;
      }
      aPublishListener.onError(error);
    },
    onFault: function(errorcode, errormsg) {
      svc._logger.error("<<<<<<<<<< Livejournal: SERVER TO FLOCK");
      svc._logger.error("FAULT " + errorcode + " " + errormsg);

      var error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
      error.serviceErrorCode = errorcode;
      error.serviceErrorString = errormsg;
      switch (errorcode) {
        case 0: // Invalid login/pass
          error.errorCode = CI.flockIError.BLOG_INVALID_AUTH;
          break;
        default: // Unknown error code
          error.errorCode = CI.flockIError.BLOG_UNKNOWN_ERROR;
      }
      aPublishListener.onFault(error);
    }
  };

  var xmlrpcServer = new flockXmlRpcServer(coopBlog.apiLink);
  var args = [{username: coopAccount.accountId, password: pw.password,
               subject: aPost.title, event: content,
               security: extra[0], allowmask: extra[1], ver: 1,
               year: year, mon: month, day: day, hour: hour, min: min,
               // props: [{opts_nocomments: nocomments}],
               props: {taglist: labels.join(","), opt_nocomments: nocomments}
               }];
  xmlrpcServer.call("LJ.XMLRPC.postevent", args, listener);
}

flockLJService.prototype.editPost =
function flockLJService_editPost(aPublishListener,
                                 aBlogId,
                                 aPost,
                                 aPublish,
                                 aNotifications)
{
  var svc = this;

  var coopBlog = this._coop.get(aBlogId);
  var coopAccount = coopBlog.getParent();

  var pw = this._acUtils.getPassword(this.urn + ":" + coopAccount.accountId);

  var labels = [];
  if (aPost.tags) {
    while (aPost.tags.hasMore()) {
      labels.push(aPost.tags.getNext());
    }
  }
  var content = aPost.description.replace("</lj>", "", "gi");
  var extra = [];
  if (aPost.extra) {
    while (aPost.extra.hasMore()) {
      extra.push(aPost.extra.getNext());
    }
  }
  var nocomments = (extra[2] == "0") ? false : true;

  var notifsArray = [];

  var listener = {
    onResult: function(aResult) {
      aPublishListener.onResult(aResult);
    },
    onError: function(error) {
      svc._logger.error("<<<<<<<<<< LiveJournal API: SERVER TO FLOCK");
      svc._logger.error("ERROR " + error);
      aPublishListener.onError(error);
    },
    onFault: function(error) {
      svc._logger.error("<<<<<<<<<< LiveJournal API: SERVER TO FLOCK");
      svc._logger.error("FAULT " + error);
      aPublishListener.onFault(error);
    }
  };

  var xmlrpcServer = new flockXmlRpcServer(coopBlog.apiLink);
  var args = [{username: coopAccount.accountId,
               password: pw.password,
               itemid: aPost.postid,
               subject: aPost.title,
               event: content,
               security: extra[0],
               allowmask: extra[1],
               ver: 1,
               props: {
                 taglist: labels.join(","),
                 opt_nocomments: nocomments
               }
              }];
  xmlrpcServer.call("LJ.XMLRPC.editevent", args, listener);
}

flockLJService.prototype.deletePost =
function flockLJService_deletePost(aListener, aBlogId, aPostid) {
  this._logger.debug(".deletePost: not supported!");
}

flockLJService.prototype.getUsersBlogs =
function flockLJService_getUsersBlogs(aBlogListener, aAPILink, aUsername, aPassword) {
  var rpcListener = new userBlogsListener(aBlogListener);
  var xmlrpcServer = new flockXmlRpcServer(aAPILink);
  var args = ["flockbrowser", aUsername, aPassword];
  xmlrpcServer.call("blogger.getUsersBlogs", args, rpcListener);
}

flockLJService.prototype.getRecentPosts =
function flockLJService_getRecentPosts(aBlogListener, aBlogId, aNumber) {
  var svc = this;

  var coopBlog = this._coop.get(aBlogId);
  var coopAccount = coopBlog.getParent();

  var pw = this._acUtils.getPassword(this.urn + ":" + coopAccount.accountId);

  var listener = {
    onResult: function(aResult) {
      var result = [];
      svc._logger.info("<<<<<<<<<< LiveJournal API: SERVER TO FLOCK");
      for (i=0; i<aResult.length; i++) {
        var post = new BlogPost();
        var content = aResult[i].content;
        if (content.match(/<title.*>(.+?)</)) {
          post.title = RegExp.$1;
        }
        post.issued = aResult[i].dateCreated.getTime();
        post.postid = aResult[i].postId.split(":")[1];
        result.push(post);
      }

      // and return the data source through the listener
      aBlogListener.onResult(new simpleEnumerator(result));
    },
    onError: function(error) {
      svc._logger.error("<<<<<<<<<< LiveJournal API: SERVER TO FLOCK");
      svc._logger.error("ERROR " + error);
      aBlogListener.onError(error);
    },
    onFault: function(code, error) {
      svc._logger.error("<<<<<<<<<< LiveJournal API: SERVER TO FLOCK");
      svc._logger.error("FAULT #" + code + ": " + error);
      var flerror = CC["@flock.com/error;1"].createInstance(CI.flockIError);
      flerror.serviceErrorCode = code;
      flerror.serviceErrorString = error;
      flerror.errorCode = CI.flockIError.BLOG_UNKNOWN_ERROR;

      aBlogListener.onFault(flerror);// TODO: bubble up the error to the UI
    }
  }

  var xmlrpcServer = new flockXmlRpcServer("http://www.livejournal.com/interface/blogger");
  // The username is used as the blogid for this call
  var args = [
    "flockblog",
    coopAccount.accountId,
    coopAccount.accountId,
    pw.password,
    aNumber
  ];
  xmlrpcServer.call("blogger.getRecentPosts", args, listener);
}

flockLJService.prototype.getCategoryList =
function flockLJService_getCategoryList(aBlogListener, aAccount) {
  aBlogListener.onResult(new simpleEnumerator([]));
}

flockLJService.prototype._init =
function flockLJService__init() {
  if (this.initialized) {
    return;
  }

  FlockSvcUtils.getLogger(this).debug(".init()");
  FlockSvcUtils.getACUtils(this);
  FlockSvcUtils.getCoop(this);

  var evtID = this._profiler.profileEventStart("lj-init");

  this.prefService = CC["@mozilla.org/preferences-service;1"]
                     .getService(CI.nsIPrefBranch);
  if (this.prefService.getPrefType(SERVICE_ENABLED_PREF) &&
      !this.prefService.getBoolPref(SERVICE_ENABLED_PREF))
  {
    this._logger.info("Pref " + SERVICE_ENABLED_PREF
                              + " set to FALSE... not initializing.");
    var catMgr = CC["@mozilla.org/categorymanager;1"]
                 .getService(CI.nsICategoryManager);
    catMgr.deleteCategoryEntry("wsm-startup", CATEGORY_COMPONENT_NAME, true);
    catMgr.deleteCategoryEntry("flockWebService", CATEGORY_ENTRY_NAME, true);
    return;
  }

  this.account_root = this._coop.accounts_root;

  this.ljService = new this._coop.Service(
  "urn:livejournal:service", {
    name: "livejournal",
    desc: "The LiveJournal Service"
  });
  this.ljService.serviceId = LIVEJOURNAL_CONTRACTID;

  // Load Web Detective file
  this.webDetective = this._acUtils.useWebDetective("livejournal.xml");
  for (var s in gStrings) {
    gStrings[s] = this.webDetective.getString("livejournal", s, gStrings[s]);
  }
  this.ljService.domains = gStrings["domains"];

  // Initialize member that specifies path for localized string bundle
  this._stringBundlePath = "";

  this.urn = this.ljService.id();

  // Update auth states
  if (this.webDetective.detectCookies("livejournal", "loggedout", null)) {
    this._acUtils.markAllAccountsAsLoggedOut(LIVEJOURNAL_CONTRACTID);
  }

  this.initialized = true;

  this._profiler.profileEventEnd(evtID, "");
}


/*************************************************************************
 * Component: flockLJAccount
 *************************************************************************/

function flockLJAccount() {
  FlockSvcUtils.getLogger(this).info(".init()");
  FlockSvcUtils.getACUtils(this);
  FlockSvcUtils.getCoop(this);

  var wsa = FlockSvcUtils.flockIWebServiceAccount;
  wsa.addDefaultMethod(this, "getService");
  wsa.addDefaultMethod(this, "login");
  wsa.addDefaultMethod(this, "logout");
  wsa.addDefaultMethod(this, "keep");
  wsa.addDefaultMethod(this, "getParam");
  wsa.addDefaultMethod(this, "setParam");
  wsa.addDefaultMethod(this, "getCustomParam");
  wsa.addDefaultMethod(this, "getAllCustomParams");
  wsa.addDefaultMethod(this, "setCustomParam");
  wsa.addDefaultMethod(this, "isAuthenticated");
}

/**************************************************************************
 * Flock LJ Account: XPCOM Component Creation
 **************************************************************************/

flockLJAccount.prototype = new FlockXPCOMUtils.genericComponent(
  CLASS_NAME + " Account",
  "",
  "",
  flockLJAccount,
  0,
  [
    CI.flockIWebServiceAccount,
    CI.flockIBlogWebServiceAccount
  ]
);

/**************************************************************************
 * Flock LJ Account: flockIBlogWebServiceAccount Implementation
 **************************************************************************/

flockLJAccount.prototype.getBlogs =
function flockLJAccount_getBlogs() {
  this._logger.debug(".getBlogs()");
  var blogsEnum = {
    QueryInterface: function(iid) {
      if (!iid.equals(CI.nsISupports) &&
          !iid.equals(CI.nsISimpleEnumerator))
      {
        throw CR.NS_ERROR_NO_INTERFACE;
      }
      return this;
    },
    hasMoreElements: function() {
      return false;
    },
    getNext: function() {
    }
  };
  return blogsEnum;
}
// END flockIBlogAccount interface

/*************************************************************************
 * XPCOM Support - Module Construction
 *************************************************************************/

// Create array of components.
var componentsArray = [flockLJService];

// Generate a module for XPCOM to find.
var NSGetModule = FlockXPCOMUtils.generateNSGetModule(MODULE_NAME,
                                                      componentsArray);
