// BEGIN FLOCK GPL
//
// Copyright Flock Inc. 2005-2008
// http://flock.com
//
// This file may be used under the terms of the
// GNU General Public License Version 2 or later (the "GPL"),
// http://www.gnu.org/licenses/gpl.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// END FLOCK GPL

const CC = Components.classes;
const CI = Components.interfaces;
const CR = Components.results;
const CU = Components.utils;

CU.import("resource:///modules/FlockXPCOMUtils.jsm");
FlockXPCOMUtils.debug = false;
CU.import("resource:///modules/FlockSvcUtils.jsm");
CU.import("resource:///modules/FlockXMLUtils.jsm");

const MODULE_NAME = "Pownce";
const CLASS_NAME = "Flock Pownce Service";
const CLASS_SHORT_NAME = "pownce";
const CLASS_TITLE = "Pownce";
const CLASS_ID = Components.ID("{7C8077D0-B7E0-4F76-9E69-58BBACEBB3C2}");
const CONTRACT_ID = "@flock.com/webservice/pownce;1";
const FLOCK_ERROR_CONTRACTID = "@flock.com/error;1";
// From nsIXMLHttpRequest.idl
const XMLHTTPREQUEST_READYSTATE_COMPLETED = 4;
const HTTP_CODE_OK = 200;
const HTTP_CODE_FOUND = 302;
const FLOCK_RICH_DND_CONTRACTID = "@flock.com/rich-dnd-service;1";

const SERVICE_ENABLED_PREF = "flock.service.pownce.enabled";

const FAVICON = "chrome://flock/content/services/pownce/favicon.png";

// The delay between two refreshes when the sidebar is closed (in seconds)
const REFRESH_INTERVAL = 1800; // seconds == 30 minutes

const POWNCE_API_URL = "http://api.pownce.com/2.1/";

const POWNCE_FRIENDS_PAGE_SIZE = 20;
const POWNCE_MESSAGES_PAGE_SIZE = 20;

const POWNCE_MAX_STATUS_LENGTH = 4000;

const POWNCE_PROPERTIES = "chrome://flock/locale/services/pownce.properties";

const POWNCE_FLOCK_APP_KEY = "0ngf12r8x98659z180m1o7frcq31d96h";
const POWNCE_FLOCK_APP_SECRET = "pu62n713j20t5m1e828vqx70o32o351a";

var gApi = null;
var gSvc = null;

// Appends the Flock breadcrumb to the given text field
function _addBreadcrumb(aField) {
  if (aField) {
    // We'll only add a breadcrumb if there isn't one already present
    var breadcrumb = "";
    aField.focus();
    var richDnDSvc = CC[FLOCK_RICH_DND_CONTRACTID]
                     .getService(CI.flockIRichDNDService);
    if (!richDnDSvc.hasBreadcrumb(aField)) {
      breadcrumb = richDnDSvc.getBreadcrumb("plain");
    }
    aField.value += breadcrumb;
  }
}

function _dispatchMouseEvent(aArea) {
  if (aArea) {
    var doc = aArea.ownerDocument;
    var docView = doc.QueryInterface(CI.nsIDOMDocumentView).defaultView;
    var e = doc.createEvent("mouseevents");
    e.initMouseEvent("click", true, true, docView, 0, 1, 1, 1, 1,
                     false, false, false, false, 0, null);
    aArea.dispatchEvent(e);
  } else {
    inst._logger.error("dispatchMouseEvent: aArea is undefined");
  }
}

/*************************************************************************
 * Component: PownceAPI
 *************************************************************************/
function PownceAPI() {
  FlockSvcUtils.getLogger(this);
  this._logger.init("PownceAPI");

  FlockSvcUtils.getACUtils(this);
  FlockSvcUtils.getCoop(this);

  this._logger.debug("constructor");
}


/*************************************************************************
 * PownceAPI: XPCOM Component Creation
 *************************************************************************/
// Use genericComponent() for the goodness it provides (QI, nsIClassInfo, etc),
// but do NOT add this component to the list of constructors passed to
// FlockXPCOMUtils.genericModule().
PownceAPI.prototype = new FlockXPCOMUtils.genericComponent(
  CLASS_NAME + " API",
  "",
  "",
  PownceAPI,
  0,
  []
  );


/*************************************************************************
 * PownceAPI: flockIWebServiceAPI Implementation
 *************************************************************************/

/**
 * void call(in AString aMethod,
 *           in nsISupports aParams,
 *           in nsISupports aPostVars, // not part of flockIWebServiceAPI
 *           in flockIListener aFlockListener);
 * @see flockIWebServiceAPI#call
 */
PownceAPI.prototype.call =
function PownceAPI_call(aMethod, aParams, aPostData, aFlockListener) {
  // If post data was passed in, assume this request is a POST.
  var isPost = (aPostData) ? true : false;

  var responseFormat = ".json";
  var url = POWNCE_API_URL + aMethod + responseFormat;
  var error = {};

  // All authenticated Pownce API requests must include the app_key
  var params = (aParams) ? aParams : {};
  params.app_key = POWNCE_FLOCK_APP_KEY;

  var idx = 0;
  for (var p in params) {
    // Only use "?" for the first param.  Use "&" after.
    url += (idx == 0) ? "?" : "&";
    url += p + "=" + params[p];
    idx++;
  }

  this._logger.debug(".call() url: " + url);

  var req = CC["@mozilla.org/xmlextras/xmlhttprequest;1"]
            .createInstance(CI.nsIXMLHttpRequest)
            .QueryInterface(CI.nsIJSXMLHttpRequest);

  // Don't pop nsIAuthPrompts if auth fails
  req.mozBackgroundRequest = true;

  req.open((isPost) ? "POST" : "GET", url, true);

  if (this.isAuthenticated) {
    var acct = gSvc.getAuthenticatedAccount();
    var passwordUrn = "urn:pownce:service:" + acct.getParam("accountId");
    var creds = this._acUtils.getPassword(passwordUrn);

    // Pownce wants the authn creds as base64(username:password)
    var credString = btoa(creds.username + ":" + creds.password);
    req.setRequestHeader("Authorization", "basic " + credString);
    this._logger.debug(".call() This is authenticated as user "
                       + creds.username);
  } else {
    this._logger.debug(".call() This is unauthenticated");
  }

  var inst = this;
  req.onreadystatechange = function api_call_orsc(aEvent) {
    inst._logger.debug(".call() ReadyState: " + req.readyState);

    if (req.readyState == XMLHTTPREQUEST_READYSTATE_COMPLETED) {
      inst._logger.debug(".call() Status: " + req.status);

      if (Math.floor(req.status/100) == 2) {
        inst._logger.debug(".call() response:\n" + req.responseText);
        var nsJSON = CC["@mozilla.org/dom/json;1"].createInstance(CI.nsIJSON);
        var result = nsJSON.decode(req.responseText);

        if (result) {
          // Pownce can return a 200 with an error described in the JSON.
          if (result.error) {
            error = inst.getError(result.error.status_code);
            error.serviceErrorCode = result.error.status_code;
            error.serviceErrorString = result.error.message;
            aFlockListener.onError(error, arguments.callee.name);
          } else {
            aFlockListener.onSuccess(result, arguments.callee.name);
          }
        } else {
          inst._logger.debug(".call() error: "
                             + "empty or poorly formatted result");
          error = inst.getError(null);
          aFlockListener.onError(error, arguments.callee.name);
        }
      } else {
        // HTTP errors (0 for connection lost)
        inst._logger.debug(".call() HTTP Error: " + req.status);
        error = inst.getHttpError(req.status);
        error.serviceErrorString = req.responseText;
        aFlockListener.onError(error, arguments.callee.name);
      }
    }
  };

  if (isPost) {
    req.setRequestHeader("Content-Type",
                         "application/x-www-form-urlencoded; charset=UTF-8");
    var postBody = "";
    for (var key in aPostData) {
      if (postBody.length) {
        postBody += "&";
      }
      postBody += key + "=" + encodeURIComponent(aPostData[key]);
    }
  }

  if (isPost && postBody && postBody.length) {
    this._logger.debug(".call() postBody: " + postBody);
    req.send(postBody);
  } else {
    req.send(null);
  }
};

/**
 * flockIError getError(in AString aErrorCode);
 * @see flockIWebServiceAPI#getError
 */
PownceAPI.prototype.getError =
function PownceAPI_getError(aErrorCode) {
  this._logger.debug(".getError('" + aErrorCode + "')");
  var error = CC[FLOCK_ERROR_CONTRACTID].createInstance(CI.flockIError);

  switch (aErrorCode) {
    default:
      error.errorCode = CI.flockIError.PHOTOSERVICE_UNKNOWN_ERROR;
  }
  return error;
};

/**
 * flockIError getHttpError(in AString aHttpErrorCode);
 * @see flockIWebServiceAPI#getHttpError
 */
PownceAPI.prototype.getHttpError =
function PownceAPI_getHttpError(aHttpErrorCode) {
  this._logger.debug(".getHttpError('" + aHttpErrorCode + "')");
  var error = CC[FLOCK_ERROR_CONTRACTID].createInstance(CI.flockIError);
  error.errorCode = aHttpErrorCode;
  error.serviceErrorCode = aHttpErrorCode;
  return error;
};


/*************************************************************************
 * PownceAPI: flockIAuthenticatedAPI Implementation
 *************************************************************************/

// readonly attribute boolean isAuthenticated;
PownceAPI.prototype.isAuthenticated = false;

/**
 * void authenticate(in nsILoginInfo aLoginInfo,
 *                   in flockIListener aFlockListener);
 * @see flockIAuthenticatedAPI#authenticate
 */
PownceAPI.prototype.authenticate =
function PownceAPI_authenticate(aLoginInfo, aFlockListener) {
  var logMessage = ".authenticate(";
  logMessage += (aLoginInfo) ? "'" + aLoginInfo.username + "', " : "";
  logMessage += "...)";
  this._logger.debug(logMessage);
  this.isAuthenticated = true;
  if (aFlockListener) {
    aFlockListener.onSuccess(this, arguments.callee.name);
  }
};

/**
 * void deauthenticate();
 * @see flockIAuthenticatedAPI#deauthenticate
 */
PownceAPI.prototype.deauthenticate =
function PownceAPI_deauthenticate() {
  this._logger.debug(".deauthenticate()");
  this.isAuthenticated = false;
};


/*************************************************************************
 * PownceAPI: Private data and functions
 *************************************************************************/
/**
 * Ensure the API call is authenticated, or don't perform it at all.
 * void _authnCall(in AString aMethod,
 *                 in nsISupports aParams,
 *                 in nsISupports aPostVars,
 *                 in flockIListener aFlockListener);
 * @see flockIWebServiceAPI#call
 */
PownceAPI.prototype._authnCall =
function PownceAPI__authnCall(aMethod, aParams, aPostVars, aFlockListener) {

  var inst = this;
  var authnFlockListener = {
    onSuccess: function authnFlockListener_onSuccess(aSubject, aTopic) {
      inst.call(aMethod, aParams, aPostVars, aFlockListener);
    },
    onError: function authnFlockListener_onError(aFlockError, aTopic) {
      // log error and just bail
      inst._logger.error(arguments.callee.name)
    }
  };

  if (!this.isAuthenticated) {
    this.authenticate(null, authnFlockListener);
  } else {
    authnFlockListener.onSuccess(null, arguments.callee.name);
  }
};

/**
 * Get profile info about a user.
 * @param  aUsername  Pownce username of the user to query.
 * @param  aFlockListener
 */
PownceAPI.prototype._userShow =
function PownceAPI__userShow(aUsername, aFlockListener) {
  this._logger.debug("._userShow(" + aUsername + ")");

  var url = "users/" + aUsername;
  gApi._authnCall(url, null, null, aFlockListener);
};


/**
 * Gets the latest note posted by a user that the "owner" of the
 * authenticated API call is authorized to see.
 * @param  aUsername  Pownce username of the user to query.
 * @param  aFlockListener
 */
PownceAPI.prototype._getLastNoteByUser =
function PownceAPI__getLastNoteByUser(aUsername, aFlockListener) {
  this._logger.debug("._getLastNoteByUser('" + aUsername + "', ...)");

  var url = "note_lists/" + aUsername
  var params = {
    type: "messages",
    filter: "sent",
    limit: "1"
  };
  gApi._authnCall(url, params, null, aFlockListener);
};

/**
 * Get a user's friends' updates.
 * @param  aUsername  Pownce username of the user whose friends to view.
 * @param  aFlockListener
 */
PownceAPI.prototype._getFriendsStatus =
function PownceAPI__getFriendsStatus(aUsername, aFlockListener) {
  this._logger.debug("._getFriendsStatus(" + aUsername + ", ...)");

  var peopleHash = {};
  var inst = this;

  var params = {};
  params.page = 0;
  params.with_notes = 1; // include latest message with each friend's details

  var url = "users/" + aUsername + "/friends";
  var friendsLoopFlockListener = {
    onSuccess: function friendsLoopFlockListener_onSuccess(aResult, aTopic) {
      if (!(aResult && aResult.friends)) {
        aFlockListener.onError(null, arguments.callee.name);
        return;
      }
      var friendsResult = aResult.friends;

      // If we get users back, lets check the next page.
      if (friendsResult.has_next_page) {
        params.page++;
        gApi._authnCall(url, params, null, friendsLoopFlockListener);
      }

      for each (var userObj in friendsResult.users) {
        var id = userObj.id;
        peopleHash[id] = userObj;
        inst._logger.debug("Found Pownce person: '"
                           + peopleHash[id].id + "' '"
                           + peopleHash[id].username + "'");
      }
      aFlockListener.onSuccess(peopleHash);
    },
    onError: function friendsLoopFlockListener_onError(aError, aTopic) {
      aFlockListener.onError(null, aTopic);
    }
  };

  gApi._authnCall(url, params, null, friendsLoopFlockListener);
};

/**
 * Set the user's status
 * @param  aStatusMessage  a string containing the message to set
 * @param  aFlockListener
 */
PownceAPI.prototype._setStatus =
function PownceAPI__setStatus(aStatusMessage, aFlockListener) {
  this._logger.debug("._setStatus('" + aStatusMessage + "', ...)");

// From Pownce API 2.0 documentation:
//
// Post a note that contains only a message.
//   url: http://api.pownce.com/2.0/send/message.{format}
//   request methods: POST
//   authentication: HTTP basic, OAuth
//   required url parameters:
//     {format}  The expected response format. Options are xml, or json.
//     {app_key} The application key assigned during registration.
//   required POST parameters:
//     {note_to} The recipient(s) of the note. Options are public, all,
//               friend_x, or set_x. Available options for the authenticated
//               user can be found from the Send To List endpoint.
//     {note_body} The main text body of the note.
//   notes: Returns the note that was posted in the requested response format.

  var url = "send/message";

  var postData = {
    note_to: "public",
    note_body: aStatusMessage
  };

  gApi._authnCall(url, null, postData, aFlockListener);
};


/*************************************************************************
 * Component: PownceService
 *************************************************************************/
function PownceService() {
  FlockSvcUtils.getLogger(this);
  this._logger.init("PownceService");

  // Determine whether this service has been disabled, and unregister if so.
  var prefService = CC["@mozilla.org/preferences-service;1"]
                    .getService(CI.nsIPrefBranch);
  if (prefService.getPrefType(SERVICE_ENABLED_PREF) &&
      !prefService.getBoolPref(SERVICE_ENABLED_PREF))
  {
    this._logger.info(SERVICE_ENABLED_PREF + " is FALSE! Not initializing.");
    this.deleteCategories();
    return;
  }

  var profiler = CC["@flock.com/profiler;1"].getService(CI.flockIProfiler);
  var evtID = profiler.profileEventStart("pownce-init");

  this._obs = CC["@mozilla.org/observer-service;1"]
              .getService(CI.nsIObserverService);
  this._obs.addObserver(this, "xpcom-shutdown", false);
  
  this._ppSvc = CC["@flock.com/people-service;1"]
                .getService(CI.flockIPeopleService);

  this._accountClassCtor = PownceAccount;

  // getWD() also adds the "_acUtils" property.
  FlockSvcUtils.getWD(this);
  FlockSvcUtils.getCoopService(this);

  this.ppUtils = CC["@flock.com/people-utils;1"]
                 .getService(CI.flockIPeopleUtils);

  // Initialize API
  gApi = new PownceAPI();

  // Update auth states
  try {
    if (this._WebDetective.detectCookies(CLASS_SHORT_NAME, "loggedout", null))
    {
      this._acUtils.markAllAccountsAsLoggedOut(CONTRACT_ID);
      gApi.deauthenticate();
    }
  } catch (ex) {
    this._logger.error("ERROR updating auth states for " + CLASS_SHORT_NAME
                       + ": " + ex);
  }

  profiler.profileEventEnd(evtID, "");
  gSvc = this;

  var ws = FlockSvcUtils.flockIWebService;
  ws.addDefaultMethod(this, "url");
  ws.addDefaultMethod(this, "getStringBundle");
  ws.addDefaultMethod(this, "isHidden");

  var lws = FlockSvcUtils.flockILoginWebService;
  lws.addDefaultMethod(this, "loginURL");
  lws.addDefaultMethod(this, "docRepresentsSuccessfulLogin");
  lws.addDefaultMethod(this, "getAccount");
  lws.addDefaultMethod(this, "getAccounts");
  lws.addDefaultMethod(this, "getAccountIDFromDocument");
  lws.addDefaultMethod(this, "getAuthenticatedAccount");
  lws.addDefaultMethod(this, "getCredentialsFromForm");
  lws.addDefaultMethod(this, "getSessionValue");
  lws.addDefaultMethod(this, "ownsDocument");
  lws.addDefaultMethod(this, "ownsLoginForm");
  lws.addDefaultMethod(this, "removeAccount");

  var rcdh = FlockSvcUtils.flockIRichContentDropHandler;
  rcdh.addDefaultMethod(this, "_handleTextareaDrop");
}


/*************************************************************************
 * PownceService: XPCOM Component Creation
 *************************************************************************/

PownceService.prototype = new FlockXPCOMUtils.genericComponent(
  CLASS_NAME,
  CLASS_ID,
  CONTRACT_ID,
  PownceService,
  CI.nsIClassInfo.SINGLETON,
  [
    CI.nsIObserver,
    CI.flockIWebService,
    CI.flockILoginWebService,
    CI.flockIPollingService,
    CI.flockISocialWebService,
    CI.flockIRichContentDropHandler
  ]
);

// FlockXPCOMUtils.genericModule() categories
PownceService.prototype._xpcom_categories = [
  { category: "wsm-startup" },
  { category: "flockWebService", entry: CLASS_SHORT_NAME },
  { category: "flockRichContentHandler", entry: CLASS_SHORT_NAME }
];


/*************************************************************************
 * PownceService: Private Data and Functions
 *************************************************************************/
PownceService.prototype._addPerson =
function PownceService__addPerson(aAccount, aPerson) {
/* Can't get last profile update time from Pownce
  // Make sure null update times aren't missed
  var lastProfileUpdate = 0;
  if (aPerson.profile_update_time) {
    lastProfileUpdate = parseInt(aPerson.profile_update_time, 10);
  }
*/
  // Make sure null status times aren't missed
  var statusTime = 0;
  var statusMessage = "";
  if (aPerson.note && aPerson.note.body && aPerson.note.timestamp) {
    statusTime = parseInt(aPerson.note.timestamp, 10);
    statusMessage = aPerson.note.body;
  }

  var lastUpdate = 0;
  var lastUpdateType;

  //if (statusTime > lastProfileUpdate) {
    lastUpdate = statusTime;
    lastUpdateType = "status";
  /*
  } else {
    lastUpdate = lastProfileUpdate;
    lastUpdateType = "profile";
  }
  */
  var avatarUrl = null;
  if (aPerson.profile_photo_urls &&
      aPerson.profile_photo_urls.medium_photo_url &&
      !this._hasDefaultAvatar(aPerson.profile_photo_urls.medium_photo_url))
  {
    avatarUrl = aPerson.profile_photo_urls.medium_photo_url;
  }

  var person = CC["@flock.com/person;1"]
               .createInstance(CI.flockIPerson);
  person.accountId = aPerson.id;
  person.screenName = aPerson.username;
  person.lastUpdate = lastUpdate;
  person.lastUpdateType = lastUpdateType;
  person.name = aPerson.short_name;
  person.avatar = avatarUrl;
  person.statusMessage = statusMessage;

  this._logger.debug("Adding person: '"
                     + person.accountId + "' '"
                     + person.screenName + "' '"
                     + person.statusMessage + "'");

  this.ppUtils.addPerson(aAccount, person);
};


/**
 * Helper function to determine if the user has customized their avatar based
 * on the passed in URL.
 * @param  aUrl  A string containing the contents of the Pownce user's
 *               "profile_photo_urls.medium_photo_url" property.
 * @return  true if the user is still using the default avatar, else false
 */
PownceService.prototype._hasDefaultAvatar =
function PownceService__hasDefaultAvatar(aUrl) {
  this._logger.debug("_hasDefaultAvatar(" + aUrl + ")");
  var defaultUrl = this._WebDetective
                       .getString(CLASS_SHORT_NAME, "noAvatar", "");
  return (aUrl.indexOf(defaultUrl) != -1);
};


/**************************************************************************
 * PownceService: nsIObserver Implementation
 **************************************************************************/

PownceService.prototype.observe =
function PownceService_observe(aSubject, aTopic, aState) {
  this._logger.info(".observe(..., '" + aTopic + "', '" + aState + "')");
  switch (aTopic) {
    case "xpcom-shutdown":
      this._obs.removeObserver(this, "xpcom-shutdown");
      break;
  }
};


/*************************************************************************
 * PownceService: flockIWebService Implementation
 *************************************************************************/

// readonly attribute AString urn;
PownceService.prototype.urn = "urn:" + CLASS_SHORT_NAME + ":service";

// readonly attribute AString title;
PownceService.prototype.title = CLASS_TITLE;

// readonly attribute AString icon;
PownceService.prototype.icon = FAVICON;

// readonly attribute AString contractId;
// ALMOST duplicated by nsIClassInfo::contractID
// with different case: contractId != contractID
PownceService.prototype.contractId = CONTRACT_ID;

// readonly attribute AString shortName;
PownceService.prototype.shortName = CLASS_SHORT_NAME;


/*************************************************************************
 * PownceService: flockILoginWebService Implementation
 *************************************************************************/

// readonly attribute AString domains;
PownceService.prototype.__defineGetter__("domains",
function PownceService_getdomains() {
  this._logger.debug("Getting property: domains");
  return this._WebDetective
             .getString(CLASS_SHORT_NAME, "domains",
                        "pownce.com");
});

// readonly attribute boolean needPassword;
PownceService.prototype.needPassword = true;

/**
 * @see flockILoginWebService#addAccount
 */
PownceService.prototype.addAccount =
function PownceService_addAccount(aAccountId, aIsTransient, aFlockListener) {
  this._logger.debug(".addAccount('" + aAccountId + "', "
                                     + aIsTransient + ", ...)");

  if (!aAccountId) {
    if (aFlockListener) {
      var error = CC[FLOCK_ERROR_CONTRACTID].createInstance(CI.flockIError);
      // XXX See bug 10749 - flockIError cleanup
      error.errorCode = 9999;
      aFlockListener.onError(error, arguments.callee.name);
    }
    return;
  }

  // XXX TODO: Scrape user's short name (ex: "Matthew W.") and use that for
  //           the coop.Account.name.
  var pw = this._acUtils.getPassword(this.urn + ":" + aAccountId);
  var name = (pw) ? pw.username : aAccountId;
  var url = this._WebDetective.getString(CLASS_SHORT_NAME, "userprofile", null)
                .replace("%accountid%", aAccountId);
  var accountUrn = this._acUtils.createAccount(this,
                                               aAccountId,
                                               name,
                                               url,
                                               aIsTransient);

  // Add custom parameters to be used
  var customParams = FlockSvcUtils.newResults();
  customParams.setPropertyAsInt32("friendRequests", 0);
  customParams.setPropertyAsInt32("maxUploadSize", 0);
  this._acUtils.addCustomParamsToAccount(customParams, accountUrn);

  // Instantiate account component
  var account = this.getAccount(accountUrn);
  // Is this service pollable by flockIPollerService?
  account.setParam("isPollable", true);
  account.setParam("refreshInterval", REFRESH_INTERVAL);
  if (aFlockListener) {
    aFlockListener.onSuccess(account, arguments.callee.name);
  }
  return account;
};

// DEFAULT: boolean docRepresentsSuccessfulLogin(in nsIDOMHTMLDocument aDocument);
// DEFAULT: flockIWebServiceAccount getAccount(in AString aAccountURN);
// DEFAULT: AString getAccountIDFromDocument(in nsIDOMHTMLDocument aDocument);
// DEFAULT: nsISimpleEnumerator getAccounts();
// DEFAULT: flockIWebServiceAccount getAuthenticatedAccount();
// DEFAULT: flockILoginInfo getCredentialsFromForm(in nsIDOMHTMLFormElement aForm);
// DEFAULT: AString getSessionValue();

/**
 * @see flockILoginWebService#logout
 */
PownceService.prototype.logout =
function PownceService_logout() {
  this._logger.debug(".logout()");
  var cookies = this._WebDetective.getSessionCookies(this.shortName);
  if (cookies) {
    this._acUtils.removeCookies(cookies);
  }

  gApi.deauthenticate();
};

// DEFAULT: boolean ownsDocument(in nsIDOMHTMLDocument aDocument);
// DEFAULT: boolean ownsLoginForm(in nsIDOMHTMLFormElement aForm);

/**
 * @see flockILoginWebService#removeAccount
 */
PownceService.prototype.removeAccount =
function PownceService_removeAccount(aUrn) {
  this._logger.debug(".removeAccount(" + aUrn + ")");
  this._acUtils.removeAccount(aUrn);
};

// DEFAULT: flockIWebServiceAccount getAccount(in AString aUrn);
// DEFAULT: nsISimpleEnumerator getAccounts();


/**
 * void updateAccountStatusFromDocument(in nsIDOMHTMLDocument aDocument,
 *                                      in AString aAcctUrn,
 *                                      in flockIListener aFlockListener);
 * @see flockILoginWebService#updateAccountStatusFromDocument
 */
PownceService.prototype.updateAccountStatusFromDocument =
function PownceService_updateAccountStatusFromDocument(aDocument,
                                                       aAcctUrn,
                                                       aFlockListener)
{
  this._logger.info(".updateAccountStatusFromDocument('" + aDocument.URL + "', '"
                                                         + aAcctUrn + "', ...)");
  if (aAcctUrn) {
    this._logger.debug("We're logged in!");
    var account = this.getAccount(aAcctUrn);
    var newlyAuthenticated = !account.isAuthenticated();
    this._acUtils.ensureOnlyAuthenticatedAccount(aAcctUrn);
    if (newlyAuthenticated) {
      gApi.authenticate(null, null);
      if (aFlockListener) {
        aFlockListener.onSuccess(account, arguments.callee.name);
      }
    }
  } else if (this._WebDetective
                 .detect(this.shortName, "loggedout", aDocument, null))
  {
    this._logger.debug("We're logged out!");
    this._acUtils.markAllAccountsAsLoggedOut(CONTRACT_ID);
    gApi.deauthenticate();
  } else {
    this._logger.debug("We don't match 'loggedout' or 'logged in'");
  }
};


/*************************************************************************
 * PownceService: flockIPollingService Implementation
 *************************************************************************/
/**
 * @see flockIPollingService#refresh
 */
PownceService.prototype.refresh =
function PownceService_refresh(aUrn, aPollingListener) {
  this._logger.debug(".refresh('" + aUrn + "', ...)");
  var inst = this;
  var account = this.getAccount(aUrn).QueryInterface(CI.flockISocialAccount);

  // If the user is not logged in, return a success without attempting to
  // refresh anything.
  if (!account.isAuthenticated()) {
    this._logger.debug(".refresh: Account is NOT authenticated. Short circuiting.");
    aPollingListener.onResult();
    return;
  }
  this._logger.debug(".refresh: Account is authenticated. Continuing.");

  var pplRefreshListener = CC["@flock.com/people-refresh-listener;1"]
                           .createInstance(CI.flockIPeopleRefreshListener);
  pplRefreshListener.init(3, aUrn, aPollingListener, 0);

  // Example JSON from Pownce API v2.0
  // {
  //   "username": "lilmatt",
  //   "permalink": "http:\/\/pownce.com\/lilmatt\/",
  //   "short_name": "Matthew W.",
  //   "country": "United States",
  //   "age": 33,
  //   "first_name": "Matthew",
  //   "blurb": "lilmatt == (VW && Mozilla && Apple)",
  //   "gender": "Male",
  //   "profile_photo_urls": {
  //     "smedium_photo_url": "http:\/\/pownce.com\/profile_photos\/l\/i\/l\/lilmatt\/141242_smedium.jpg",
  //     "small_photo_url": "http:\/\/pownce.com\/profile_photos\/l\/i\/l\/lilmatt\/141242_small.jpg",
  //     "tiny_photo_url": "http:\/\/pownce.com\/profile_photos\/l\/i\/l\/lilmatt\/141242_tiny.jpg",
  //     "medium_photo_url": "http:\/\/pownce.com\/profile_photos\/l\/i\/l\/lilmatt\/141242_medium.jpg",
  //     "large_photo_url": "http:\/\/pownce.com\/profile_photos\/l\/i\/l\/lilmatt\/141242_large.jpg"
  //   },
  //   "id": 141242,
  //   "is_pro": 0,
  //   "max_upload_mb": 0,
  //   "location": "Ithaca, NY"
  // }

  // This flockIListener handles getting the account owner's information.
  var userShowFlockListener = {
    onSuccess: function userShowFlockListener_onSuccess(aResult, aTopic) {
      if (aResult.short_name) {
        inst._logger.debug(arguments.callee.name
                           + ": name: " + aResult.short_name);
        account.setParam("name", aResult.short_name);
      }

      if (aResult.username) {
        inst._logger.debug(arguments.callee.name
                           + ": screen_name: " + aResult.username);
        account.setParam("screenName", aResult.username);
      }

      if (aResult.is_pro) {
        inst._logger.debug(arguments.callee.name
                           + ": is_pro: " + aResult.is_pro);
        account.setParam("isPro", aResult.is_pro);
      }

      if (aResult.max_upload_mb) {
        inst._logger.debug(arguments.callee.name
                           + ": max_upload_mb: " + aResult.max_upload_mb);
        account.setCustomParam("maxUploadSize", aResult.max_upload_mb);
      }

      if (aResult.permalink) {
        inst._logger.debug(arguments.callee.name
                           + ": URL: " + aResult.permalink);
        account.setParam("URL", aResult.permalink);
      }

      if (aResult.profile_photo_urls &&
          aResult.profile_photo_urls.medium_photo_url)
      {
        var photoUrl = aResult.profile_photo_urls.medium_photo_url;
        inst._logger.debug(arguments.callee.name + " avatar: " + photoUrl);

        // If avatar returned is the default image, set coop.Account.avatar
        // to null and let the people sidebar code set the Flock common image.
        if (inst._hasDefaultAvatar(photoUrl)) {
          inst._logger.debug(arguments.callee.name
                             + ": No avatar for account. Setting to null.");
          account.setParam("avatar", null);
        } else {
          account.setParam("avatar", photoUrl);
        }
      }

      if (aResult.friend_request_count) {
        var friendRequests = aResult.friend_request_count;
        inst._logger.debug(arguments.callee.name
                           + " friendRequests: " + friendRequests);

        // Light people icon if necessary
        var existingFriendRequests = account.getCustomParam("friendRequests");
        if (existingFriendRequests < friendRequests) {
          inst._ppSvc.togglePeopleIcon(true);
        }

        account.setCustomParam("friendRequests", friendRequests);
      }
      inst._obs.notifyObservers(account.coopObj.resource(),
                                "flock-acct-refresh",
                                "user-info");
      pplRefreshListener.onSuccess(null, arguments.callee.name);
    },
    onError: function userShowFlockListener_onError(aFlockError, aTopic) {
      inst._logger.error(arguments.callee.name);
      pplRefreshListener.onError(aFlockError, aTopic);
    }
  };

  // This handles getting the user's latest public note (a.k.a. "status").
  var getMyNoteFlockListener = {
    onSuccess: function getMyNoteFlockListener_onSuccess(aResult, aTopic) {
      inst._logger.debug(arguments.callee.name);

      var result = aResult.notes[0];
      if (result && result.body && result.timestamp) {
        inst._logger.debug(arguments.callee.name
                           + ": result.body: " + result.body);
        account.setParam("statusMessage", result.body);
        account.setParam("lastStatusMessageUpdateDate", result.timestamp);
      } else {
        inst._logger.debug(arguments.callee.name + ": No result to parse!");
        account.setParam("statusMessage", null);
        account.setParam("lastStatusMessageUpdateDate", 0);
      }
      inst._obs.notifyObservers(account.coopObj.resource(),
                                "flock-acct-refresh",
                                "user-info");
      pplRefreshListener.onSuccess(null, arguments.callee.name);
    },
    onError: function getMyNoteFlockListener_onError(aFlockError, aTopic) {
      inst._logger.error(arguments.callee.name);
      pplRefreshListener.onError(aFlockError, aTopic);
    }
  };

  // This listener handles getting the user's friends' information.
  var friendsFlockListener = {
    onSuccess: function friendsFlockListener_onSuccess(aResult) {
      inst._logger.debug(arguments.callee.name);

      // Now that we have all the results we need, update the RDF.

      // ADD or update existing people
      for (var friend in aResult) {
        inst._addPerson(account, aResult[friend]);
      }

      // REMOVE locally people removed on the server
      var localFriends = account.enumerateFriends();
      while (localFriends.hasMoreElements()) {
        var localFriend = localFriends.getNext();
        if (!aResult[localFriend.accountId]) {
          inst._logger.debug("Friend " + localFriend.accountId
                             + " has been deleted on the server");
          inst.ppUtils.removePerson(account, localFriend);
        }
      }
      pplRefreshListener.onSuccess(null, arguments.callee.name);
    },
    onError: function friendsFlockListener_onError(aFlockError, aTopic) {
      inst._logger.error(arguments.callee.name);
      pplRefreshListener.onError(aFlockError, aTopic);
    }
  };

  var uid = account.getParam("accountId");
  gApi._userShow(uid, userShowFlockListener);
  gApi._getLastNoteByUser(uid, getMyNoteFlockListener);
  gApi._getFriendsStatus(uid, friendsFlockListener);

  // Prevent JS strict warnings.  We used a return earlier to short-circuit
  // this in the case of being unauthenticated, so we need to return here also.
  return;
};


/*************************************************************************
 * PownceService: flockISocialWebService implementation
 *************************************************************************/

// readonly attribute long maxStatusLength;
PownceService.prototype.maxStatusLength = POWNCE_MAX_STATUS_LENGTH;

// void markAllMediaSeen(in AString aPersonUrn);
PownceService.prototype.markAllMediaSeen =
function PownceService_markAllMediaSeen(aPersonUrn) {
  this._logger.debug(".markAllMediaSeen('" + aPersonUrn + "')");

  // Pownce has no media, so this does nothing and shouldn't be called.
  throw (NS_ERROR_NOT_IMPLEMENTED);
};


/**************************************************************************
 * PownceService: flockIRichContentDropHandler Implementation
 **************************************************************************/

// boolean handleDrop(in nsITransferable aFlavours,
//                    in nsIDOMHTMLElement aTargetElement);
PownceService.prototype.handleDrop =
function PownceService_handleDrop(aFlavours, aTargetElement) {
  this._logger.debug(".handleDrop()");
  // Handle the rich content drop
  var inst = this;
  // Handle textarea drops
  var dropCallback = function pownce_dropCallback(aXPath, aFlav) {
    // Get the content from the transferable
    var dataObj = {};
    var len = {};
    aFlavours.getTransferData(aFlav, dataObj, len);
    var content = dataObj.value.QueryInterface(CI.nsISupportsString).data;
    inst._logger.debug("pownce_dropCallback: content = " + content);
    // Get the message data from the content
    var contentParts = content.split(": ");
    var message = {
      title: "",
      link: ""
    };
    if (contentParts.length > 1) {
      message.link = contentParts.pop();
      message.title = contentParts.join(": ");
    } else {
      message.link = content;
    }
    // Determine where we're dropping the content
    var doc = aTargetElement.ownerDocument;
    if (aXPath == "messageNoteXPath") {
      // Dropped on "Write" pane of Publisher: move the link to the
      // "Link" pane of the Publisher so that the link will be attached
      // to the message
      if (message.link) {
        var writeLinkListener = {
          onSuccess: function writeLinkListener_onSuccess(aLinkArea) {
            // Add User's post from their wall to the comment area
            aLinkArea.focus();
            aLinkArea.value += message.title;
            _addBreadcrumb(aLinkArea);
          },
          onError: function writeLinkListener_onError(aFlockError) {
            // No error should be returned
            inst._logger.error("writeLinkListener: Error occured while trying "
                               + "to set up link area");
          }
        };
        inst._setUpShareLink(message.link, doc, writeLinkListener, true);
        // Link will be dropped in listener, so we're done
        return;
      }
    } else if (aXPath == "messageLinkXPath") {
      // Dropped on "Link" pane of Publisher
      if (message.link) {
        var shareLinkListener = {
          onSuccess: function shareLinkListener_onSuccess(aLinkArea) {
            // Add breadcrumb to link comment
            aLinkArea.focus();
            aLinkArea.value += message.title;
            _addBreadcrumb(aLinkArea);
          },
          onError: function shareLinkListener_onError(aFlockError) {
            // No error should be returned
            inst._logger.error("shareLinkListener: Error occured while trying "
                               + "to set up link area");
          }
        };
        inst._setUpShareLink(message.link, doc, shareLinkListener, false);
        // Link will be dropped in listener, so we're done
        return;
      }
    }
    // Drop the content onto the element
    inst._dropMessage(aTargetElement, content);
  };
  return this._handleAreaDrop(aTargetElement, dropCallback);
};

PownceService.prototype._dropMessage =
function PownceService__dropMessage(aTargetElement, aMsg) {
  // Make sure the textarea has focus. This will remove any placeholders from
  // the text before we start manipulating it.
  aTargetElement.focus();

  var caretPos = aTargetElement.selectionEnd;
  var currentValue = aTargetElement.value;

  // Add a trailing space so that we don't mangle the url
  var nextChar = currentValue.charAt(caretPos);
  var trailingSpace = ((nextChar == "") ||
                       (nextChar == " ") ||
                       (nextChar == "\n"))
                    ? ""
                    : " ";
  aTargetElement.value = currentValue.substring(0, caretPos)
                       + aMsg.replace(/: /, ":\n")
                       + trailingSpace
                       + currentValue.substring(caretPos);
  _addBreadcrumb(aTargetElement);
};

PownceService.prototype._handleAreaDrop =
function PownceService__handleAreaDrop(aElement, aCallback) {
  const WD_XPATH = 0;
  const WD_FLAVOUR = 1;
  var doc = aElement.ownerDocument;
  if (doc instanceof CI.nsIDOMHTMLDocument) {
    var domainsArray = this.domains.split(",");
    for each (var domain in domainsArray) {
      if (this._WebDetective.testDomain(doc.URL, this.domains)) {
        // Retrieve the specific fields from Web Detective to which
        // we cannot DnD
        var fields = this._WebDetective.getString(this.shortName,
                                                  "avoidDnDXPathFields",
                                                  null);
        if (fields) {
          fields = fields.split(";");
          for each (var avoidDnD in fields) {
            var xPath = this._WebDetective.getString(this.shortName,
                                                     avoidDnD,
                                                     null);
            if (xPath) {
              var results = doc.evaluate(xPath, doc, null,
                                         CI.nsIDOMXPathResult.ANY_TYPE,
                                         null);
              if (results) {
                var node = results.iterateNext();
                while (node) {
                  if (node == aElement) {
                    // The matching field does not accept rich content, bail
                    return true;
                  }
                  node = results.iterateNext();
                }
              }
            }
          }
        }

        // Retrieve the specific fields from Web Detective to which
        // we can DnD
        var pairs = [];
        fields = this._WebDetective.getString(this.shortName,
                                              "dndXPathFields",
                                              null);
        if (fields) {
          fields = fields.split(";");
          for each (var xpfFields in fields) {
            pairs.push(xpfFields);
          }
        }

        // Go through the list of DnD fields to find a match
        for each (var xpfPair in pairs) {
          var xpf = xpfPair.split(",");
          var xPath = this._WebDetective.getString(this.shortName,
                                                   xpf[WD_XPATH],
                                                   null);
          if (xPath) {
            var results = doc.evaluate(xPath, doc, null,
                                       CI.nsIDOMXPathResult.ANY_TYPE,
                                       null);
            if (results) {
              var node = results.iterateNext();
              while (node) {
                if (node == aElement) {
                  // Let the service perform the drop via the callback
                  aCallback(xpf[WD_XPATH], xpf[WD_FLAVOUR]);
                  return true;
                }
                node = results.iterateNext();
              }
            }
          }
        }
      }
    }
  }
  return false;
};

// Put a link into the Share Link pane of the Publisher, hit the Preview button,
// and wait for the comment area to appear
PownceService.prototype._setUpShareLink =
function PownceService_setUpShareLink(aNewLink,
                                      aDoc,
                                      aFlockListener,
                                      aClickTab)
{
  // Do we need to go to the pane?
  if (aClickTab) {
    // Click on the Share Link tab
    var linkArea = this._getHtmlElement("messageLinkURLXPath", aDoc);
    linkArea.QueryInterface(CI.nsIDOMElement);
    _dispatchMouseEvent(linkArea);
  }
  // Put the new link into the input field
  var inputArea = this._getHtmlElement("messageLinkXPath", aDoc);
  inputArea.value = aNewLink;
  _dispatchMouseEvent(inputArea);
  var commentArea = this._getHtmlElement("messageNoteXPath", aDoc);
  _dispatchMouseEvent(commentArea);
  aFlockListener.onSuccess(commentArea);
};

PownceService.prototype._getHtmlElement =
function PownceService__getHtmlElement(aWebDicXPath, aDoc) {
  var textAreaXPATH = this._WebDetective.getString(this.shortName,
                                                   aWebDicXPath,
                                                   "");
  var area = aDoc.evaluate(textAreaXPATH, aDoc, null,
                           CI.nsIDOMXPathResult.ANY_TYPE, null);
  return area ? area.iterateNext() : null;
};

/*************************************************************************
 * Component: PownceAccount
 *************************************************************************/
function PownceAccount() {
  FlockSvcUtils.getLogger(this);
  this._logger.init("PownceAccount");

  FlockSvcUtils.getACUtils(this);
  FlockSvcUtils.getCoop(this);
  this.ppUtils = CC["@flock.com/people-utils;1"]
                  .getService(CI.flockIPeopleUtils);
  // XXX: I should be able to use FlockSvcUtils.getWD() here, but it can
  //      only be called by the service.
  this._WebDetective = CC["@flock.com/web-detective;1"]
                       .getService(CI.flockIWebDetective);

  var wsa = FlockSvcUtils.flockIWebServiceAccount;
  wsa.addDefaultMethod(this, "getAllCustomParams");
  wsa.addDefaultMethod(this, "getCustomParam");
  wsa.addDefaultMethod(this, "getParam");
  wsa.addDefaultMethod(this, "getService");
  wsa.addDefaultMethod(this, "isAuthenticated");
  wsa.addDefaultMethod(this, "login");
  wsa.addDefaultMethod(this, "logout");
  wsa.addDefaultMethod(this, "setCustomParam");
  wsa.addDefaultMethod(this, "setParam");

  var sa = FlockSvcUtils.flockISocialAccount;
  sa.addDefaultMethod(this, "enumerateFriends");
  sa.addDefaultMethod(this, "getInviteFriendsURL");
  //sa.addDefaultMethod(this, "formatFriendActivityForDisplay");
  sa.addDefaultMethod(this, "getFriendCount");
  sa.addDefaultMethod(this, "getFormattedFriendUpdateType");

  var sbs = CC["@mozilla.org/intl/stringbundle;1"]
            .getService(CI.nsIStringBundleService);
  this._bundle = sbs.createBundle(POWNCE_PROPERTIES);
}


/*************************************************************************
 * PownceAccount: XPCOM Component Creation
 *************************************************************************/

// Use genericComponent() for the goodness it provides (QI, nsIClassInfo, etc),
// but do NOT add this component to the list of constructors passed to
// FlockXPCOMUtils.genericModule().
PownceAccount.prototype = new FlockXPCOMUtils.genericComponent(
  CLASS_NAME + " Account",
  "",
  "",
  PownceAccount,
  0,
  [
    CI.flockIWebServiceAccount,
    CI.flockISocialAccount,
    CI.flockIPownceAccount
  ]
  );


/*************************************************************************
 * PownceAccount: flockIWebServiceAccount Implementation
 *************************************************************************/

// readonly attribute AString urn;
PownceAccount.prototype.urn = "";

// DEFAULT: nsIPropertyBag getAllCustomParams();
// DEFAULT: nsIVariant getCustomParam(in AString aParamName);
// DEFAULT: nsIVariant getParam(in AString aParamName);
// DEFAULT: flockILoginWebService getService();
// DEFAULT: boolean isAuthenticated();

/**
 * void keep();
 * @see flockILoginWebService#keep
 */
PownceAccount.prototype.keep =
function PownceAccount_keep() {
  this._logger.debug(".keep()");
  this.setParam("isTransient", false);
  var urn = this.urn.replace("account:", "service:").replace("flock:", "");
  this._acUtils.makeTempPasswordPermanent(urn);
};

// DEFAULT: void login(in flockIListener aFlockListener);
// DEFAULT: void logout(in flockIListener aFlockListener);
// DEFAULT: void setCustomParam(in AString aParamName, in nsIVariant aValue);
// DEFAULT: void setParam(in AString aParamName, in nsIVariant aValue);


/*************************************************************************
 * PownceAccount: flockISocialAccount Implementation
 *************************************************************************/

// readonly attribute boolean hasFriendActions;
PownceAccount.prototype.hasFriendActions = true;

// readonly attribute boolean isMyMediaFavoritesSupported;
PownceAccount.prototype.isMyMediaFavoritesSupported = false;

// readonly attribute boolean isPostLinkSupported;
PownceAccount.prototype.isPostLinkSupported = true;

// readonly attribute boolean isStatusEditable;
PownceAccount.prototype.isStatusEditable = true;

// readonly attribute boolean isMeStatusSupported;
PownceAccount.prototype.isMeStatusSupported = true;

// readonly attribute boolean isFriendStatusSupported;
PownceAccount.prototype.isFriendStatusSupported = true;

// DEFAULT: nsISimpleEnumerator enumerateFriends();

// AString formatFriendActivityForDisplay(in AString aFriendActivityUrn);
PownceAccount.prototype.formatFriendActivityForDisplay =
function PownceAccount_formatFriendActivityForDisplay(aFriendActivityUrn) {
  this._logger.debug(".formatFriendActivityForDisplay()");

  var friendActivity = this._coop.get(aFriendActivityUrn);
  var prefix = "flock." + CLASS_SHORT_NAME + ".";

  var sb = this._bundle;
  var out = "";

  switch (friendActivity.updateType) {
    case "profile":
      out = sb.GetStringFromName(prefix + "lastUpdateTypePretty.profile");
      break;

    case "status":
      if (friendActivity.updateValue == "") {
        out = sb.GetStringFromName(prefix + "lastUpdateTypePretty.statusCleared");
      } else {
        out = sb.formatStringFromName(prefix + "lastUpdateTypePretty.statusUpdated",
                                     [friendActivity.updateValue], 1);
      }
      break;
  }

  return (flockXMLDecode(out));
};

/**
 * AString formatStatusForDisplay(in AString aStatusMessage);
 * @see flockISocialAccount#formatStatusForDisplay
 */
PownceAccount.prototype.formatStatusForDisplay =
function PownceAccount_formatStatusForDisplay(aStatusMessage) {
  this._logger.debug(".formatStatusForDisplay('" + aStatusMessage + "')");

  var message = (aStatusMessage) ? aStatusMessage : "";

  // Pownce uses "\r\n" to represent <br/>
  message = message.replace(/\r\n/g, " ");
  return flockXMLDecode(message);
};

/**
 * void setStatus(in AString aStatusMessage, in flockIListener aFlockListener);
 * @see flockISocialAccount#setStatus
 */
PownceAccount.prototype.setStatus =
function PownceAccount_setStatus(aStatusMessage, aFlockListener) {
  this._logger.debug(".setStatus('" + aStatusMessage + "', ...)");

  var inst = this;
  var setStatusFlockListener = {
    onSuccess: function setStatusFlockListener_onSuccess(aResult, aTopic) {
      inst._logger.debug(arguments.callee.name);

      // If the API call succeeded, also set the account status.
      var message = (aResult && aResult.body) ? aResult.body : aStatusMessage;
      inst.setParam("statusMessage", message);
      if (aFlockListener) {
        aFlockListener.onSuccess(aResult, aTopic);
      }
    },
    onError: function setStatusFlockListener_onError(aFlockError, aTopic) {
      inst._logger.error(arguments.callee.name);
      if (aFlockListener) {
        aFlockListener.onError(aFlockError, aTopic);
      }
    }
  };
  gApi._setStatus(aStatusMessage, setStatusFlockListener);
};

/**
 * AString getEditableStatus();
 * @see flockISocialAccount#getEditableStatus
 */
PownceAccount.prototype.getEditableStatus =
function PownceAccount_getEditableStatus() {
  this._logger.debug(".getEditableStatus()");
  var message = this.getParam("statusMessage");
  message = (message) ? flockXMLDecode(message) : "";
  return this.formatStatusForDisplay(message);
};

/**
 * AString getFriendActions(in AString aFriendUrn);
 * @see flockISocialAccount#getFriendActions
 */
PownceAccount.prototype.getFriendActions =
function PownceAccount_getFriendActions(aFriendUrn) {
  this._logger.debug(".getFriendActions('" + aFriendUrn + "')");

  var actionNames = ["friendMessage",
                     "friendShareLink",
                     "friendViewProfile"];

  var actions = [];
  var friend = this.ppUtils.getPerson(aFriendUrn);
  if (friend) {
    var coopAccount = this._coop.get(this.urn);
    for each (var action in actionNames) {
      actions.push({
        label: this._bundle.GetStringFromName("flock."
                                              + CLASS_SHORT_NAME
                                              + ".actions." + action),
        class: action,
        spec: this._WebDetective.getString(CLASS_SHORT_NAME, action, "")
                  .replace("%accountid%", coopAccount.accountId)
                  .replace("%friendid%", friend.accountId)
                  .replace("%friendScreenName%", friend.screenName)
      });
    }
  }

  var nsJSON = CC["@mozilla.org/dom/json;1"].createInstance(CI.nsIJSON);
  return nsJSON.encode(actions);
};

// DEFAULT: long getFriendCount();

/**
 * AString getMeNotifications();
 * @see flockISocialAccount#getMeNotifications
 */
PownceAccount.prototype.getMeNotifications =
function PownceAccount_getMeNotifications() {
  this._logger.debug(".getMeNotifications()");

  var noties = [];
  var inst = this;
  function addNotie(aType, aCount) {
    var stringName = "flock." + CLASS_SHORT_NAME + ".noties." + aType + "."
                   + ((parseInt(aCount) <= 0) ? "none" : "some");

    inst._logger.debug("aType: " + aType
                       + " aCount: " + aCount
                       + " name: " + stringName);
    noties.push({
      class: aType,
      tooltip: inst._bundle.GetStringFromName(stringName),
      count: aCount,
      URL: inst._WebDetective.getString(CLASS_SHORT_NAME, aType + "_URL", "")
                             .replace("%accountid%",
                                      inst.getParam("accountId"))
    });
  }
  var coopAccount = this._coop.get(this.urn);
  //addNotie("meMessages", coopAccount.accountMessages);
  addNotie("meFriendRequests", this.getCustomParam("friendRequests"));

  var nsJSON = CC["@mozilla.org/dom/json;1"].createInstance(CI.nsIJSON);
  return nsJSON.encode(noties);
};

/**
 * AString getPostLinkAction(in nsITransferable aTransferable);
 * @see flockISocialAccount#getPostLinkAction
 */
PownceAccount.prototype.getPostLinkAction =
function PownceAccount_getPostLinkAction(aTransferable) {
  return this._WebDetective
             .getString(CLASS_SHORT_NAME, "shareAction_postLinkMethod", "");
};

/**
 * AString getProfileURLForFriend(in AString aFriendUrn);
 * @see flockISocialAccount#getProfileURLForFriend
 */
PownceAccount.prototype.getProfileURLForFriend =
function PownceAccount_getProfileURLForFriend(aFriendUrn) {
  this._logger.debug(".getProfileURLForFriend('" + aFriendUrn + "')");

  var url = "";
  var friend = this.ppUtils.getPerson(aFriendUrn);
  var screenName = friend.screenName;
  if (screenName) {
    this._logger.debug(".getProfileURLForFriend screenName: " + screenName);
    url = this._WebDetective.getString(CLASS_SHORT_NAME, "friendProfile", "")
                            .replace("%screenName%", screenName);
  }

  return url;
};

/**
 * AString getSharingAction(in AString aFriendUrn,
 *                          in nsITransferable aTransferable);
 * @see flockISocialAccount#getSharingAction
 */
PownceAccount.prototype.getSharingAction =
function PownceAccount_getSharingAction(aFriendUrn, aTransferable) {
  this._logger.debug(".getSharingAction('" + aFriendUrn + "')");
  return this._WebDetective
             .getString(CLASS_SHORT_NAME, "friendShareLink", "");
};

/**
 * AString postPownceLink(in nsITransferable aTransferable);
 * @see flockISocialAccount#postPownceLink
 */
PownceAccount.prototype.postPownceLink =
function PownceAccount_postPownceLink(aTransferable) {
  this._logger.debug(".postPownceLink('" + aTransferable +"')");
  var url;
  var message = null;
  if (aTransferable) {
    var flavors = ["text/x-flock-media",
                   "text/x-moz-url",
                   "text/unicode",
                   "text/html"];
    message = CC[FLOCK_RICH_DND_CONTRACTID]
              .getService(CI.flockIRichDNDService)
              .getMessageFromTransferable(aTransferable,
                                          flavors.length,
                                          flavors);
  
    if (message && message.body) {
      if (this._WebDetective.detectNoDOM(CLASS_SHORT_NAME,
                                         "isLink",
                                         "",
                                         message.body,
                                         null))
      {
        url = this._WebDetective
                  .getString(CLASS_SHORT_NAME, "shareAction_postLink", "")
                  .replace("%url%", encodeURIComponent(message.body));
      }
    }
  } else {
    // The "Post Link" button was clicked: get the current tab's URL
    message = this._buildMessageFromCurrentTab();
    if (!message.body) {
      // We don't have a link, do not proceed with posting link
      this._logger.error("postPownceLink() unable to get link");
      return;
    }
    url = this._WebDetective
              .getString(CLASS_SHORT_NAME, "shareAction_postLink", "")
              .replace("%url%", encodeURIComponent(message.body));
  }
  this._composeNote(url, message.subject);
};

PownceAccount.prototype._buildMessageFromCurrentTab =
function PownceAccount__buildMessageFromCurrentTab() {
  // Get the current tab's URL
  var win = CC["@mozilla.org/appshell/window-mediator;1"]
            .getService(CI.nsIWindowMediator)
            .getMostRecentWindow("navigator:browser");
  var msg = {};
  if (win) {
    msg.body = win.gBrowser.currentURI.spec;
    msg.subject = win.gBrowser.mCurrentTab.linkedBrowser.contentTitle;
  }
  return msg;
};

/**
 *  sendPownceNote(aFriendURN, aTransferable);
 *  creates a note to send to a friend defined by aFriendUrn
 *  if aTransferable exists. Set up pownce based on the type of content
 *  (link or text)
 *  @param aFriendURN urn of friend creating a note for
 *  @param aTransferable message being dropped (can be null)
 */
PownceAccount.prototype.sendPownceNote =
function PownceAccount_sendPownceNote(aFriendURN, aTransferable) {
  this._logger.debug(".sendPownceNote('" + aFriendURN + "')");
  var url = null;
  var c_friend = this._coop.get(aFriendURN);
  var message = null;
  if (aTransferable) {
    var flavors = ["text/x-flock-media",
                   "text/x-moz-url",
                   "text/unicode",
                   "text/html"];
    message = CC[FLOCK_RICH_DND_CONTRACTID]
              .getService(CI.flockIRichDNDService)
              .getMessageFromTransferable(aTransferable,
                                          flavors.length,
                                          flavors);

    if (message && message.body) {
      if (this._WebDetective.detectNoDOM(CLASS_SHORT_NAME,
                                         "isLink",
                                         "",
                                         message.body,
                                         null))
      {
        url = this._WebDetective
                  .getString(CLASS_SHORT_NAME,
                             "shareAction_friendShareLink",
                             "")
                  .replace("%friendid%", c_friend.accountId)
                  .replace("%currentURL%", encodeURIComponent(message.body));
        // message.body is a url, we don't need it anymore
        message.body = null;
      } else {
        url = this._WebDetective
                  .getString(CLASS_SHORT_NAME, "friendShareMessage", "")
                  .replace("%friendid%", c_friend.accountId);
      }
    }
  } else {
    // We are not dropping message so build message from current tab
    message = this._buildMessageFromCurrentTab();
    if (!message.body) {
      // we don't have a link, do not proceed
      this._logger.error("sendPownceNote() unable to get link");
      return;
    }
    url = this._WebDetective
              .getString(CLASS_SHORT_NAME, "shareAction_friendShareLink", "")
              .replace("%friendid%", c_friend.accountId)
              .replace("%currentURL%", encodeURIComponent(message.body));
    // message.body is a url, we don't need it anymore
    message.body = null;
  }
  this._composeNote(url, message.body ? message.body : message.subject);
};

/**
 *  _composeNote(aUrl, aBody);
 *  Loads aUrl in a new tab. Once loaded,drops message defined by
 *  aBody and breadcrumb
 *  @param aUrl url to load
 *  @param aBody message being dropped 
 */
PownceAccount.prototype._composeNote =
function PownceAccount__composeNote(aUrl, aBody) {
  this._logger.debug("._composeNote('"
                     + aUrl + "', '"
                     + aBody + "')");

  var wm = CC["@mozilla.org/appshell/window-mediator;1"]
           .getService(CI.nsIWindowMediator);
  var win = wm.getMostRecentWindow("navigator:browser");
  if (win) {
    var browser = win.getBrowser();
    var newTab = browser.loadOneTab(aUrl, null, null, null, false, false);
    var obs = CC["@mozilla.org/observer-service;1"]
              .getService(CI.nsIObserverService);
    var inst = this;
    var observer = {
      observe: function openSendMessageTabForFill_observer(aContent,
                                                           aTopic,
                                                           aContextUrl)
      {
        var contentWindow = newTab.linkedBrowser.docShell
                                  .QueryInterface(CI.nsIInterfaceRequestor)
                                  .getInterface(CI.nsIDOMWindow);
        var xpathquery = inst._WebDetective
                             .getString("pownce", "messageNoteXPath", "");
        var doc = contentWindow.document;
        var textarea = doc.evaluate(xpathquery, doc, null,
                                    CI.nsIDOMXPathResult.ANY_TYPE, null);
        if (textarea) {
          textarea = textarea.iterateNext();
          textarea.value = aBody;
          _addBreadcrumb(textarea);
        }
        obs.removeObserver(observer, "EndDocumentLoad", false);
      }
    };
    obs.addObserver(observer, "EndDocumentLoad", false);
  }
};

/**
 * void markAllMeNotificationsSeen(in AString aType);
 * @see flockISocialAccount#markAllMediaNotificationsSeen
 */
PownceAccount.prototype.markAllMeNotificationsSeen =
function PownceAccount_markAllMeNotificationsSeen(aType) {
  this._logger.debug(".markAllMeNotificationsSeen('" + aType + "')");
  switch (aType) {
    case "meMessages":
      this.setParam("accountMessages", 0);
      break;
    case "meFriendRequests":
      this.setCustomParam("friendRequests", 0);
      break;
    default:
      break;
  }
};


/*************************************************************************
 * XPCOM Support - Module Construction
 *************************************************************************/

// Create array of components.
var componentsArray = [PownceService];

// Generate a module for XPCOM to find.
var NSGetModule = FlockXPCOMUtils.generateNSGetModule(MODULE_NAME,
                                                      componentsArray);
