/*
 * BEGIN FLOCK GPL
 *
 * Copyright Flock Inc. 2005-2008
 * http://flock.com
 *
 * This file may be used under the terms of the
 * GNU General Public License Version 2 or later (the "GPL"),
 * http://www.gnu.org/licenses/gpl.html
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * END FLOCK GPL
 */

/**
 * Convenience wrappers around the Mozilla string bundle API,
 * tailored for Flock.
 *
 * Import this module through
 *
 * Components.utils.import("resource:///modules/FlockStringBundleHelpers.jsm");
 */

const CC = Components.classes;
const CI = Components.interfaces;
const CR = Components.results;

var EXPORTED_SYMBOLS = ["flockGetString", "brandGetString",
                        "getStringFromBundle", "flockGetStringBundle"];

/**
 * Gets the value of a string in a .properties file
 *
 * @param aBundleName  the name of the properties file.  The function
                       prepends "chrome://flock/locale/" to the name.
 * @param aStringName  the name of the string within the properties file.
 * @param aParams      optional array of parameters to format the string.
 */
function flockGetString(aBundleName, aStringName, aParams) {
  var bundle = flockGetStringBundle("flock", aBundleName);
  return getStringFromBundle(bundle, aStringName, aParams);
}

/**
 * Gets the value of a string in a .properties file in the branding
 * namespace. This is does the same thing as flockGetString except
 * "chrome://branding/locale/" is prepended to the name instead.
 *
 * @param aBundleName  the name of the properties file.  The function
                       prepends "chrome://branding/locale/" to the name.
 * @param aStringName  the name of the string within the properties file.
 * @param aParams      optional array of parameters to format the string.
 */
function brandGetString(aBundleName, aStringName, aParams) {
  var bundle = flockGetStringBundle("branding", aBundleName);
  return getStringFromBundle(bundle, aStringName, aParams);
}

/**
 * Gets a string from an nsIStringBundle.
 *
 * @param aBundle      the nsIStringBundle to use.
 * @param aStringName  the name of the string within the bundle.
 * @param aParams      optional array of parameters to format the string.
 */
function getStringFromBundle(aBundle, aStringName, aParams) {
  try {
    if (aParams && aParams.length) {
      return aBundle.formatStringFromName(aStringName,
                                          aParams, aParams.length);
    } else {
      return aBundle.GetStringFromName(aStringName);
    }
  }
  catch (ex) {
    throw ("L10N bug! Request for entity: '" + aStringName + "' "
           + "failed\n" + ex);
  }
}

/**
 * Gets an nsIStringBundle from a prefix and name. The URI is constructed
 * as follows: "chrome://prefix/local/name.properties"
 *
 * @param aBundlePrefix the prefix for the bundle.
 * @param aBundleName   the name of the properties file.
 */
function flockGetStringBundle(aBundlePrefix, aBundleName) {
  var sbs = CC["@mozilla.org/intl/stringbundle;1"]
            .getService(CI.nsIStringBundleService);
  var bundle = sbs.createBundle("chrome://" + aBundlePrefix + "/locale/"
                                + aBundleName + ".properties");
  return bundle;
}
