// BEGIN FLOCK GPL
//
// Copyright Flock Inc. 2005-2008
// http://flock.com
//
// This file may be used under the terms of the
// GNU General Public License Version 2 or later (the "GPL"),
// http://www.gnu.org/licenses/gpl.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// END FLOCK GPL

var _logger = Cc["@flock.com/logger;1"].createInstance(Ci.flockILogger);
_logger.init("myworld");

var _coop = Cc["@flock.com/singleton;1"]
            .getService(Ci.flockISingleton)
            .getSingleton("chrome://flock/content/common/load-faves-coop.js")
            .wrappedJSObject;

const BLOG_SVC = Cc["@flock.com/flock-blog;1"].getService(Ci.flockIBlogService);

const PREFS = Cc["@mozilla.org/preferences-service;1"].getService(Ci.nsIPrefBranch);
const PREF_BROWSER_STARTUP_HOMEPAGE = "browser.startup.homepage";
const PREF_MYWORLD_CURRENT_ENGINE   = "flock.myworld.currentEngine";
const PREF_MYWORLD_TITLE            = "flock.myworld.currentTitle";
const PREF_MYWORLD_WIDGETS_LAYOUT = "flock.myworld.widgets.layout";
const PREF_OPEN_SEARCH_IN = "flock.myworld.search.openLocation";
const XMLHTTPREQUEST_CONTRACTID     = "@mozilla.org/xmlextras/xmlhttprequest;1";
const nsIXMLHttpRequest             = Components.interfaces.nsIXMLHttpRequest;

const MYWORLD_TITLE_FONTSCALE_LC = 13;
const MYWORLD_TITLE_FONTSCALE_UC = 19;

const nsIPrefLocalizedString = Components.interfaces.nsIPrefLocalizedString;

var gMyworldTitleMinWidth;
var gMyworldWidgets = [];
var gMyworldWidgetsVisible = 0;

var currentEng;

var gObsrvr = null;

function onUnload() {
  this.gMetrics.report("MyWorld-Close", null);

  this.removeWidgetObservers();
}

function onLoad()
{
  var gTimer = null;
  gTimer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
  gTimer.initWithCallback({
    notify: function myworld_loadColumnLayoutTimer() {
      loadColumnLayout();
    }
  }, 0, Ci.nsITimer.TYPE_ONE_SHOT);

  _logger.debug("onLoad");

  document.getElementById("focusOnLoad").select();

  this.gMetrics.report("MyWorld-Opened", null);
  
  checkDefaultHomepage();  

  loadEngines();

  loadMyWorldTitle();

  resizeFunction(null);

  window.addEventListener("resize", resizeFunction, false);

  // This is a nescessary hack, please see
  // https://bugzilla.mozilla.org/show_bug.cgi?id=196057
  window.onload = null;
}

function checkDefaultHomepage() {
   var defhbox = document.getElementById("makedefault");

   // Remove to re-build
   while(defhbox.firstChild) {
      defhbox.removeChild(defhbox.firstChild);
   }

   // check if flock is in the default
   if(PREFS.getPrefType(PREF_BROWSER_STARTUP_HOMEPAGE)) {
      var homepage = PREFS.getComplexValue(PREF_BROWSER_STARTUP_HOMEPAGE, nsIPrefLocalizedString).data;
      if(homepage.indexOf("about:myworld") >= 0) {
        return; 
      }
   }
   
   // Add to page if nescessary
   var button = document.createElement("button");
   button.setAttribute("onclick", "makeDefaultHomepage()");
   button.setAttribute("class", "defaultHomepageBtn");
   
   var label = document.createElement("label");
   label.setAttribute("class", "myworldHomepage");
   label.setAttribute("value", flockGetString("start/myworld", "flock.myworld.homepage.set"));
   
   button.appendChild(label);
   defhbox.appendChild(button);
}

function loadEngines() {
  // Get a list of all search engines
  var searchService = Cc["@mozilla.org/browser/search-service;1"].getService(Ci.nsIBrowserSearchService);
  if (!searchService) return;

  var engines = searchService.getVisibleEngines({ });

  var eMenu = document.getElementById("engineMenuID");

  // delete list before re-creating it
  var existingChildrenCount = eMenu.childNodes.length;
  for(var z=existingChildrenCount;z>0;z--)
  {
    var existChild = eMenu.childNodes[z-1];
    eMenu.removeChild(existChild);
  }

  var currentEngine;
  // Get current myworld engine
  if (PREFS.getPrefType(PREF_MYWORLD_CURRENT_ENGINE)) {
    currentEngine = PREFS.getCharPref(PREF_MYWORLD_CURRENT_ENGINE);
  }

  if (!currentEngine || !searchService.getEngineByName(currentEngine)) {
    // If myworld engine doesnt exist, use current default engine
    if (!searchService.currentEngine) return;
    currentEngine = searchService.currentEngine.name;
  }

  PREFS.setCharPref(PREF_MYWORLD_CURRENT_ENGINE, currentEngine);

    // Add them to the menu
  for(var i=engines.length-1;i>=0;--i) {
    var menuitem = document.createElementNS(kXULNS, "menuitem");
    menuitem.setAttribute("label", engines[i].name);
    menuitem.setAttribute("id", engines[i].name);
    menuitem.setAttribute("class", "engine-menuitem");
    menuitem.setAttribute("engine", engines[i].name);
    menuitem.setAttribute("image", engines[i].iconURI);
    menuitem.setAttribute("oncommand", "selectEngine(this)");

    if(engines[i].name == currentEngine) {
      menuitem.setAttribute("selected", "true");
      currentEng = engines[i];
    }

    if(engines[i].iconURI) {
      menuitem.setAttribute("image", engines[i].iconURI.spec);

      // Also set the image of the selected item
      if(engines[i].name == currentEngine) {
        var engineImageID = document.getElementById("engineImageID");
        engineImageID.setAttribute("src",engines[i].iconURI.spec);
      }
    }

    eMenu.insertBefore(menuitem, eMenu.firstChild);
    menuitem.engine = engines[i];
  }
}

function toggleEngineMenu() {
  var popup = document.getElementById("engineMenuID");
  var popButton = document.getElementById("popButton");

  popup.showPopup(popButton, -1, -1, "popup", "bottomleft", "topleft");
}

function selectEngine(engine) {
  var engineName  = engine.getAttribute("engine");
  var engineImage = engine.getAttribute("image");
  this.gMetrics.report("MyWorld-SearchEngineChange", engineName);

  if(engineImage != "" && engineImage != null) {
    var engineImageID = document.getElementById("engineImageID");
    engineImageID.setAttribute("src",engineImage);
  }

  // Set current engine
  var searchService = Cc["@mozilla.org/browser/search-service;1"].getService(Ci.nsIBrowserSearchService);
  var engines = searchService.getVisibleEngines({ });

  for(var i=engines.length-1;i>=0;--i) {
    if(engines[i].name == engineName) {
      currentEng = engines[i];
      PREFS.setCharPref(PREF_MYWORLD_CURRENT_ENGINE, currentEng.name);
      return;
    }
  }
}

function openService(node, event)
{
  if(event.button == 2) {
    return;
  }

  // Open the service URL in a new tab
  var svcID = node.getAttribute("serviceId");
  var svc = Cc[svcID].getService(Ci.flockIWebService);
  this.gMetrics.report("MyWorld-ServiceClick", svc.shortName);
  var wm = Cc["@mozilla.org/appshell/window-mediator;1"].getService(Ci.nsIWindowMediator);
  var win = wm.getMostRecentWindow('navigator:browser');
  if (win) {
    win.openUILinkIn(svc.url, "tab");
  }

  // Tell the Web Service Manager to automatically keep any accounts
  // created for this service in the next while...
  var wsm = Cc["@flock.com/webservice-manager;1"].getService(Ci.flockIWebServiceManager);
  wsm.autoKeepAccountForService(node.getAttribute("urn"));
}

function refreshDateTime()
{
  var timeLabel = document.getElementById("timeUpdated");

  var time = new Date();

  var dtFormat = flockGetString("start/myworld", "flock.myworld.dateTimeFormat");

  // see bug 9830#31
  dtFormat = dtFormat.replace("%e", time.getDate());

  var thedate = time.toLocaleFormat(dtFormat);

  timeLabel.setAttribute("value", thedate);

  this.os.notifyObservers(null, "refresh-myworld-date", null);
  setTimeout("refreshDateTime()", (60 - time.getSeconds() + 1) * 1000);
}

function uploadPhotos(event)
{
  if(event.button == 2) {
    return; 
  }

  this.gMetrics.report("Uploader-Open", "MyWorldUploadButton");
  var wm = Cc["@mozilla.org/appshell/window-mediator;1"].getService(Ci.nsIWindowMediator);
  var win = wm.getMostRecentWindow('navigator:browser');
  if (win) {
    win.FlockMedia.getUploadSvc().launchUploader();
  }
}

function createBlogPost(event)
{
  if(event.button == 2) {
    return; 
  }
  
  this.gMetrics.report("BlogDialog-Open", "MyWorldBlogButton");
  BLOG_SVC.openEditor("","","");
}

function searchboxKeyPress(event,query) {
  if (event.keyCode == KeyEvent.DOM_VK_RETURN) {
    searchFunction(query);
  }
}

function searchFunction(query)
{
  var wm = Cc["@mozilla.org/appshell/window-mediator;1"].getService(Ci.nsIWindowMediator);
  var win = wm.getMostRecentWindow('navigator:browser');
  if (win) {
    currentEng.QueryInterface(Ci.flockISearchEngine);
    var submission = currentEng.getSubmissionByContext(query, null, "w");
    if (!submission) { return; }

    this.gMetrics.report("MyWorld-Search", currentEng.name);

    // Check to see where we should open.
    var openLocation;
    if (PREFS.getPrefType(PREF_OPEN_SEARCH_IN)) {
      openLocation = PREFS.getCharPref(PREF_OPEN_SEARCH_IN);
    } else {
      openLocation = "tab";
      PREFS.setCharPref(PREF_OPEN_SEARCH_IN, openLocation);
    }

    // Double check to see if it's one of the 3 choices.  Default to tab.
    switch (openLocation) {
      case "tab":
      case "window":
      case "current":
      break;

      default: {
        openLocation = "tab";
        PREFS.setCharPref(PREF_OPEN_SEARCH_IN, openLocation);
      }
      break;
    }

    win.openUILinkIn(submission.uri.spec, openLocation);
  }
}

function makeDefaultHomepage() {
  // Make My World the default homepage
  PREFS.setCharPref(PREF_BROWSER_STARTUP_HOMEPAGE, "about:myworld");

  checkDefaultHomepage();

  var notificationBox = getNotificationBox();
  if(notificationBox) {
    var message = flockGetString("start/myworld", "flock.myworld.homepage");
    notificationBox.appendNotification( message,
                                        "myworldadded",
                                        "chrome://flock/skin/common/Info.png",
                                        notificationBox.FLOCK_PRIORITY_MEDIUM,
                                        null,
                                        null);
  
  }
}

function getNotificationBox() {

  var wm = Cc["@mozilla.org/appshell/window-mediator;1"]
           .getService(Ci.nsIWindowMediator);
  var win = wm.getMostRecentWindow('navigator:browser');
  if (win) {
    var gBrowser = win.getBrowser();
    var notificationBox = gBrowser.getNotificationBox();

    return notificationBox;
  }

  return null;
}

function initWidgetObservers() {
  // Add observer for refresh
  var inst = this;
  if (!gObsrvr) {
    gObsrvr = {};
    gObsrvr.observe = function engineManager_observer(aEngine, aTopic, aVerb) {
      if(aTopic == "browser-search-engine-modified") {
        aEngine.QueryInterface(Ci.nsISearchEngine);
        switch (aVerb) {
          case "engine-added":
          case "engine-removed":
            setTimeout("loadEngines()", 1000);
        }
      }
      //for each widget, an observer is added with the form
      //refresh-myworld-{widget.label}
      if (aTopic.match(/^refresh-myworld/)) {
        _logger.debug("resizeWidgetContainer");
        inst.resizeWidgetContainer();
      }
    }
    // Add observer for engines added
    this.os = Cc["@mozilla.org/observer-service;1"].getService(Ci.nsIObserverService);
    this.os.addObserver(gObsrvr, "browser-search-engine-modified", false);
    this.os.addObserver(gObsrvr, "refresh-myworld-size", false);
  
    setTimeout("refreshDateTime()", 500);

    //resize window to account for iframe content
    setTimeout("resizeWidgetContainer()", 500);
  }
}

function removeWidgetObservers() {
  this.os.removeObserver(gObsrvr, "browser-search-engine-modified");
  this.os.removeObserver(gObsrvr, "refresh-myworld-size");
}

/**
 * Gets the column layout from the pref store
 */
function getColumnLayout() {
  if (PREFS.getPrefType(PREF_MYWORLD_WIDGETS_LAYOUT)) {
    return PREFS.getComplexValue(PREF_MYWORLD_WIDGETS_LAYOUT,
                                 Ci.nsIPrefLocalizedString).data;
  } else {
    var columns = [];

    columns.push({
      label: "topfeeds",
      url: "chrome://flock/content/start/topfeeds.xul",
      properties_url: "chrome://flock/locale/start/myworld.properties",
      show: "true"
    });
    columns.push({
      label: "topfriends",
      url: "chrome://flock/content/start/topfriends.xul",
      properties_url: "chrome://flock/locale/start/myworld.properties",
      show: "true"
    });
    columns.push({
      label: "topmedia",
      url: "chrome://flock/content/start/topmedia.xul",
      properties_url: "chrome://flock/locale/start/myworld.properties",
      show: "true"
    });
    columns.push({
      label: "topsites",
      url: "chrome://flock/content/start/topsites.xul",
      properties_url: "chrome://flock/locale/start/myworld.properties",
      show: "false"
    });

    var nsJSON = Cc["@mozilla.org/dom/json;1"].createInstance(Ci.nsIJSON);
    var layoutString = nsJSON.encode(columns);
    setColumnLayout(layoutString);
    return layoutString;
  }
}

/**
 * Sets the column layout in the pref store
 */
function setColumnLayout(aLayout) {
  PREFS.setCharPref(PREF_MYWORLD_WIDGETS_LAYOUT, aLayout);
}

/**
 * Loads column layout
 */
function loadColumnLayout() {
  var nsJSON = Cc["@mozilla.org/dom/json;1"].createInstance(Ci.nsIJSON);
  var layoutObj = nsJSON.decode(getColumnLayout());
  gMyworldWidgets = eval(layoutObj);
  //add additional observers for each widget
  initWidgetObservers();
    
  var containerElement = document.getElementById("ftcontainerID");

  //clean up existing dom within container if exists
  while (containerElement.hasChildNodes()) {
    containerElement.removeChild(containerElement.lastChild);
  }

  var colIndex = 0;

  var updateToLayout = false;
  var sbs = Cc["@mozilla.org/intl/stringbundle;1"]
            .getService(Ci.nsIStringBundleService);

  gMyworldWidgetsVisible = 0;

  for each (var column in gMyworldWidgets) {
    // Migration check: make sure properties_url is specified
    var widgetId = column.label + "Id";
    if (!column.properties_url) {
      column.properties_url = "chrome://flock/locale/start/myworld.properties";

      updateToLayout = true;
    }

    var sb = sbs.createBundle(column.properties_url);
    var titleStr = sb.GetStringFromName("flock.myworld." + column.label
                                                         + ".title");
    //create menuitem for widget entry
    var widgetMenuPopup = document.getElementById("widgets-selector");
    var widgetMenuItem = document.createElement("menuitem");
    widgetMenuItem.setAttribute("label", titleStr);
    widgetMenuItem.setAttribute("id", widgetId + "-toggleItem");
    widgetMenuItem.setAttribute("class", "menuitem-iconic");
    widgetMenuItem.setAttribute("type", "checkbox");
    var command = "doWidgetToggle('" + column.label + "');";
    widgetMenuItem.setAttribute("oncommand", command);
    widgetMenuPopup.appendChild(widgetMenuItem);
    if (eval(column.show)) {
      openWidget(column, sb);
    }
  }

  checkDiscoveryBox(gMyworldWidgetsVisible);

  refreshMoveItems();
  if (updateToLayout) {
    saveColumnLayout();
  }
}

/**
 * Saves column layout
 */
function saveColumnLayout() {
  var nsJSON = Cc["@mozilla.org/dom/json;1"].createInstance(Ci.nsIJSON);
  var newLayout = nsJSON.encode(gMyworldWidgets);

  var metricsValue = [];
  for (var i = 0; i < gMyworldWidgets.length; i++) {
    metricsValue.push({name: gMyworldWidgets[i].label,
                       position: i,
                       visible: (gMyworldWidgets[i].show == "true")});
  }
  // .report expects the value to be in an [ ] if it is a complex object,
  // even if it's already an [ ].
  gMetrics.report("MyWorld-LayoutChange", [metricsValue]);

  setColumnLayout(newLayout);
}

/**
 * Moves column left/right
 * aDirection = left/right
 */
function moveColumn(widgetId, aDirection) {
  var moveTarget = document.getElementById(widgetId);
  var parent = moveTarget.parentNode;
  var colIndex = moveTarget.getAttribute("col_index");
  var column = gMyworldWidgets[colIndex];
  var sibling2swap;

  switch (aDirection) {
    case "left": {
      sibling2swap = moveTarget.previousSibling;
      if (!sibling2swap || colIndex == 0) {
        // We can't go left anymore
        return false;
      }
      parent.removeChild(moveTarget);
      parent.insertBefore(moveTarget, sibling2swap);
    } break;
    case "right": {
      sibling2swap = moveTarget.nextSibling;
      if (!sibling2swap || gMyworldWidgets.length == colIndex) {
        return false;
      }
      parent.removeChild(moveTarget);
      parent.insertBefore(moveTarget, sibling2swap.nextSibling);
    } break;
    default: {
      // Not used yet or invalid choice
      return false;
    }
  }
  var newIndex = sibling2swap.getAttribute("col_index");
  gMyworldWidgets.splice(colIndex, 1);
  gMyworldWidgets.splice(newIndex, 0, column);
  this.updateCurrentColumnLayout();
  return true;
}

function updateCurrentColumnLayout() {
  checkDiscoveryBox(gMyworldWidgetsVisible);
  this.refreshMoveItems();
  this.saveColumnLayout();
}

function checkDiscoveryBox(aWidgetsVisible) {
  if (aWidgetsVisible == 0) {
    // Show the discovery box
    document.getElementById("noContentID").setAttribute("hidden", "false");
  } else {
    // Hide the discovery box
    document.getElementById("noContentID").setAttribute("hidden", "true");
  }
}

function closeWidget(aWidgetId) {
  var metrics = Components.classes["@flock.com/metrics-service;1"]
                .getService(Components.interfaces.flockIMetricsService)
  metrics.report("MyWorld-HideWidget", "close");

  var node = document.getElementById(aWidgetId);
  var colIndex = node.getAttribute("col_index");
  var column = gMyworldWidgets[colIndex];
  column.show = "false";
  gMyworldWidgetsVisible--;

  var parent = node.parentNode;
  parent.removeChild(node);
  var nodeMenuItem = document.getElementById(aWidgetId + "-toggleItem");
  nodeMenuItem.setAttribute("checked", "false");

  this.updateCurrentColumnLayout();
}

function openWidget(aColumn, aBundle) {
  var widgetId = aColumn.label + "Id";
  gMyworldWidgetsVisible++;

  var containerElement = document.getElementById("ftcontainerID");
  // iframe content for respective xul page
  var iframe = document.createElement("iframe");
  iframe.setAttribute("flex", "1");
  iframe.setAttribute("src", aColumn.url);

  // separator
  var columnEnclosure = document.createElement("vbox");
  columnEnclosure.setAttribute("class", "col_enclosure");

  // widget container
  var widgetVbox = document.createElement("vbox");
  widgetVbox.setAttribute("label_val", aColumn.label);
  widgetVbox.setAttribute("iframe_src", aColumn.url);

  widgetVbox.setAttribute("properties_url", aColumn.properties_url);
  widgetVbox.setAttribute("widgetContainer", "true");
  widgetVbox.setAttribute("id", widgetId);
  widgetVbox.setAttribute("class", "col_hdr_container");
  widgetVbox.setAttribute("hidden", "false");

  var hboxContainer = document.createElement("hbox");
  var vboxContainer = document.createElement("vbox");
  hboxContainer.appendChild(vboxContainer);

  // widget title
  var titleHbox = document.createElement("hbox");
  titleHbox.setAttribute("align", "start");
  var title = document.createElement("label");
  title.setAttribute("class", "favHeaderLabel");
  title.setAttribute("onclick", this.doWidgetAction(aColumn.label));

  var titleStr = aBundle.GetStringFromName("flock.myworld."
                                           + aColumn.label + ".title");
  title.setAttribute("value", titleStr);
  title.setAttribute("tooltiptext", titleStr);
  var menupopup = document.createElement("menupopup");
  menupopup.setAttribute("id", aColumn.label + "ID");
  menupopup.setAttribute("class", "whatsThisPopup");
  menupopup.setAttribute("position", "after_start");

  titleHbox.appendChild(title);
  titleHbox.appendChild(menupopup);
  vboxContainer.appendChild(titleHbox);

  // widget sub title and help image
  var subHbox = document.createElement("hbox");
  subHbox.setAttribute("class", "col_hdr_bottom");
  var image = document.createElement("image");
  image.setAttribute("class", "whatsThis");
  image.setAttribute("descID", aColumn.label + "ID");
  var tooltip = "flock.myworld." + aColumn.label + ".tooltip";
  image.setAttribute("desc", aBundle.GetStringFromName(tooltip));
  var whatsThis = "flock.myworld.whatsThis";
  image.setAttribute("value", aBundle.GetStringFromName(whatsThis));
  image.setAttribute("onclick", "popupWhatsThis(this)");
  var subTitle = document.createElement("label");
  subTitle.setAttribute("class", "col_sub_hdr");
  var subTitleName = "flock.myworld." + aColumn.label + ".subTitle2";
  subTitle.setAttribute("value", aBundle.GetStringFromName(subTitleName));
  subTitle.setAttribute("onclick",
                        "popupWhatsThis(this.parentNode.childNodes[0])");

  subHbox.appendChild(image);
  subHbox.appendChild(subTitle);
  vboxContainer.appendChild(subHbox);

  // widget settings
  var widgetSettings = document.createElement("vbox");
  widgetSettings.setAttribute("class", "col_hdr_settings");
  var widgetSettingsHbox = document.createElement("hbox");
  widgetSettings.appendChild(widgetSettingsHbox);

  // create move button
  var moveLeftBtn = document.createElement("toolbarbutton");
  moveLeftBtn.setAttribute("class", "widgetMoveBtn-left");
  var mvLeftTT = aBundle.GetStringFromName("flock.myworld.widgets.moveLeft");
  moveLeftBtn.setAttribute("tooltiptext", mvLeftTT);
  var moveCommand = "moveColumn('" + widgetId + "',";
  moveLeftBtn.setAttribute("oncommand", moveCommand + "'left');");
  moveLeftBtn.setAttribute("id", widgetId + "-moveLeft");

  var moveRightBtn = document.createElement("toolbarbutton");
  moveRightBtn.setAttribute("class", "widgetMoveBtn-right");
  var mvRightTT = aBundle.GetStringFromName("flock.myworld.widgets.moveRight");
  moveRightBtn.setAttribute("tooltiptext", mvRightTT);
  moveRightBtn.setAttribute("oncommand", moveCommand + "'right');");
  moveRightBtn.setAttribute("id", widgetId + "-moveRight");

  // attach move button
  widgetSettingsHbox.appendChild(moveLeftBtn);
  widgetSettingsHbox.appendChild(moveRightBtn);

  // close button
  var closeBtn = document.createElement("toolbarbutton");
  closeBtn.setAttribute("oncommand", "closeWidget('" + widgetId + "');");
  closeBtn.setAttribute("class", "widgetCloseBtn");
  var close = "flock.myworld.widgets.close";
  closeBtn.setAttribute("tooltiptext", aBundle.GetStringFromName(close));
  widgetSettingsHbox.appendChild(closeBtn);

  // append elements
  hboxContainer.appendChild(widgetSettings);
  widgetVbox.appendChild(hboxContainer);
  widgetVbox.appendChild(iframe);
  containerElement.appendChild(widgetVbox);

  var nodeMenuItem = document.getElementById(widgetId + "-toggleItem");
  nodeMenuItem.setAttribute("checked", "true");
}

function doWidgetToggle(aColumnLabel) {
  var widgetId = aColumnLabel + "Id";
  var node = document.getElementById(widgetId);
  if (!node) {
    var rightColumnIndex = -1;
    for (var colIndex = gMyworldWidgets.length-1; colIndex >= 0; colIndex--) {
      var column = gMyworldWidgets[colIndex];
      if (eval(column.show)) {
        if (rightColumnIndex < 0) {
          rightColumnIndex = colIndex;
        }
      } else if (column.label == aColumnLabel) {
        // If rightColumnIndex == -1, this means the first widget we ran into
        // was the right-most widget in the gMyworldWidgets array, which is
        // set NOT to show.
        if (rightColumnIndex != -1) {
          gMyworldWidgets.splice(colIndex, 1);
          gMyworldWidgets.splice(rightColumnIndex, 0, column);
        }
        column.show = "true";
        var sbs = Cc["@mozilla.org/intl/stringbundle;1"]
                  .getService(Ci.nsIStringBundleService);
        openWidget(column, sbs.createBundle(column.properties_url));
        this.updateCurrentColumnLayout();
        break;
      }
    }
  } else {
    this.closeWidget(widgetId);
    gMetrics.report("MyWorld-HideWidget", "uncheck");
  }
}

function refreshMoveItems() {
  var colIndex = 0;
  var moveRight = null;
  for each (var column in gMyworldWidgets) {
    var widgetId = column.label + "Id";
    var node = document.getElementById(widgetId);
    if (node) {
      node.setAttribute("col_index", colIndex);
      var moveLeft = document.getElementById(widgetId + "-moveLeft");
      if (moveRight) {
        // this is not the first column
        moveLeft.setAttribute("hidden", "false");
        moveRight.setAttribute("hidden", "false");
      } else {
        moveLeft.setAttribute("hidden", "true");
      }
      moveRight = document.getElementById(widgetId + "-moveRight");
    }
    colIndex++;
  }
  if (moveRight) {
    // handle the last column
    moveRight.setAttribute("hidden", "true");
  }
}

function doWidgetAction(aAction) {
  switch (aAction) {
    case "topsites": {
      return "openFavoritesManager(event);";
    } break;
    case "topfeeds": {
      return "openNewsSidebar(null, event, 'FaveHeaderLabel');";
    } break;
    case "topmedia": {
      return "openMediaBar(null, event);";
    } break;
    case "topfriends": {
      return "openPeopleSidebar(event);";
    } break;
  }
  return null;
}

/**
 * Load MyWorld Title
 */
function loadMyWorldTitle() {
  var titleElement = document.getElementById("myworld-title");
  if (PREFS.getPrefType(PREF_MYWORLD_TITLE)) {
    var myTitle = PREFS.getComplexValue(PREF_MYWORLD_TITLE,
                                        nsIPrefLocalizedString).data;
  } else {
    var myTitle = flockGetString("start/myworld", "flock.myworld.title");
  }
  titleElement.setAttribute("value", myTitle);
  titleElement.setAttribute("originalTitle", myTitle);

  // Get computed style for title to determine min-width
  var view = titleElement.ownerDocument.defaultView;
  var titleStyle = view.getComputedStyle(titleElement, "");
  gMyworldTitleMinWidth = (parseInt(titleStyle
                                    .getPropertyCSSValue("min-width")
                                    .cssText) || 0);
  //init title resize
  this.resizeMyWorldTitle(titleElement);

  document.title = myTitle;
}

function focusMyWorldTitle(obj) {
  obj.setAttribute("edit","true");
  obj.removeAttribute("readonly");
  obj.select();
}

/**
 * Handle KeyPress Events for the MyWorld Title
 */
function keypressMyWorldTitle(aEvent, obj) {
  switch (aEvent.keyCode) {
    case KeyEvent.DOM_VK_RETURN:
      document.commandDispatcher.advanceFocus();
      this.saveMyWorldTitle(obj);
      break;
    case KeyEvent.DOM_VK_TAB:
      this.saveMyWorldTitle(obj);
      break;
    case KeyEvent.DOM_VK_ESCAPE:
      obj.value = obj.getAttribute("originalTitle");
      document.commandDispatcher.advanceFocus();
      this.resizeMyWorldTitle(obj);
      break;
    default:
      //resize textbox
      this.resizeMyWorldTitle(obj);
      break;
   }
}

/**
 * Resize MyWorld Title
 */
function resizeMyWorldTitle(obj) {
  var strValue = obj.value;
  var strLength = strValue.length;
  if (strLength) {
    //need scaling factor for font-size used
    var scale = 0;
    var str = strValue.split("");
    for (var i = 0; i < str.length; i++) {
      if(str[i] != str[i].toLowerCase()) {
        scale += MYWORLD_TITLE_FONTSCALE_UC;
      } else {
        scale += MYWORLD_TITLE_FONTSCALE_LC;
      }
    }
    scale = (scale / str.length).toFixed();
    var newWidth = (strLength * scale).toFixed();
    if (newWidth > gMyworldTitleMinWidth) {
      obj.style.width = newWidth + "px";
    } else {
      obj.style.width = gMyworldTitleMinWidth + "px";
    }
  }

  // Make sure we place the search box properly after any resize to the title.
  resizeFunction(null);
}

/**
 * Saves MyWorld Title
 */
function saveMyWorldTitle(obj) {
  var origTitle = obj.getAttribute("originalTitle");

  // Get current myworld title
  if (PREFS.getPrefType(PREF_MYWORLD_TITLE)) {
    origTitle = PREFS.getComplexValue(PREF_MYWORLD_TITLE,
                                      nsIPrefLocalizedString).data;
  }
  if (origTitle == null) {
    // No pref and has never been changed, therefore we should use the existing
    // document title
    origTitle = document.title;
  }

  // Save and reset css states to appear unfocused
  if (obj.value != null) {
    obj.setAttribute("originalTitle", obj.value);
    obj.removeAttribute("edit");
    obj.setAttribute("readonly", "true");
    var str = Cc["@mozilla.org/supports-string;1"]
              .createInstance(Ci.nsISupportsString);
    str.data = obj.value;
    PREFS.setComplexValue(PREF_MYWORLD_TITLE, Ci.nsISupportsString, str);
    gMetrics.report("MyWorld-NameChange", null);
    document.title = obj.value;
  }
}

//ensure column container renders the height of the iframe content
function resizeWidgetContainer() {
  //reset height for changes in size
  var columnContainerHeight = 0;

  //calculate max height of the widgets
  for each (var column in gMyworldWidgets) {
    var widget = document.getElementById(column.label + "Id");
    if (widget && widget.childNodes[1] &&
        widget.getAttribute("hidden") != "true") {
      var colDoc = widget.childNodes[1].contentDocument.documentElement;
      if (colDoc.lastChild.boxObject) {
        var colHeight = colDoc.lastChild.boxObject.height;
        if (colHeight > columnContainerHeight) {
          //take into account the height for the widget header
          var headerOffset = widget.firstChild.boxObject.height;
          columnContainerHeight = colHeight + headerOffset;
        }
      }
    }
  }
  //set the size of the container
  document.getElementById("ftcontainerID")
          .setAttribute("style", "min-height:" + columnContainerHeight + "px");
}

function resizeFunction(aEvent) {
  var windowWidth = document.defaultView.innerWidth;

  var titleWidth = document.getElementById("titleID").boxObject.width;
  var searchBoxWidth = document.getElementById("searchBoxID").boxObject.width;

  if (windowWidth > (titleWidth + searchBoxWidth)) {
    var differenceWidth = windowWidth - (titleWidth + searchBoxWidth);
    document.getElementById("titleSpacerID")
            .setAttribute("width", differenceWidth - 80);
  } else {
    document.getElementById("titleSpacerID").setAttribute("width", "0");
  }
}

window.onload = onLoad;
window.onunload = onUnload;

