#ifndef STATUS_CHANGER
#define STATUS_CHANGER

#include <qobject.h>

#include "status.h"

class QTimer;

/**
	@class StatusChanger
	@brief Podstawowa klasa suca do zmiany statusu uytkownika.

	Implementacje klasy dokonuj zmiany statusu uytkownikw na podstawie dowolnych, zalenych od
	klasy parametrw. Zmianie moe ulega zarwno opis jak i stan dostpnoci.
 **/
class StatusChanger : public QObject
{
	Q_OBJECT

	int Priority;

public:
	/**
		Podstawowy kontruktor. Tworzy modyfikator statusu z zadanym priorytetem.
		Modyfikatory aplikowane s w kolejnoci rosncych priorytetw.

		@arg priority priorytet modyfikatora
	 **/
	StatusChanger(int priority);
	virtual ~StatusChanger();

	int priority();

	/**
		Metoda zmieniajca status.

		@arg status aktualny status uytkownika, metoda moe zmieni warto argumentu
	 **/
	virtual void changeStatus(UserStatus &status) = 0;

signals:
	void statusChanged();

};

/**
	@class UserStatusChanger
	@brief Klasa umoliwiajca uytkownikowi ustawienie wasnego statusu.

	Klasa posiada pole userStatus okrelajce status, jaki uytkownik sam sobie ustawi.
	Klasa modyfikuje aktualny status uytkownika ustawiajc go na warto pola userStatus.
	Priorytet modyfikatora: 0.
 **/
class UserStatusChanger : public StatusChanger
{
	Q_OBJECT

	UserStatus userStatus;

public:
	UserStatusChanger();
	virtual ~UserStatusChanger();

	virtual void changeStatus(UserStatus &status);
	const UserStatus & status() { return userStatus; }

public slots:
	/**
		Zapisuje status, jaki uytkownik sobie ustawi.

		@param status status uytkownika
	 **/
	void userStatusSet(UserStatus &status);

};

/**
	@class SplitStatusChanger
	@brief Klasa dzielca dugie opisy na krtsze.

	Klasa dzieli dugi opis na kilka krtszych, ktre wywietlane s w odstpie 30 sekund od siebie.
	Dla przykadu, opis "ten opis ma wicej ni dwadziecia znakw" przy maksymalnej dugoci opisu
	wynoszcej dwadziecia znakw zostanie podzielony na nastpujce pod-opisy:

		"ten opis ma wi --->"
		"<--- cej ni dw --->"
		"<--- adziecia  --->"
		"<--- znakw"

	W Gadu-Gadu limit dugoci wynosi 70 znakw.

	Priorytet: 1000.
 **/
class SplitStatusChanger : public StatusChanger
{
	Q_OBJECT

	unsigned int splitSize;

	unsigned int descriptionSplitBegin;
	unsigned int descriptionSplitLength;

	QTimer *splitTimer;
	QString lastDescription;

	bool enabled;

private slots:
	void timerInvoked();

public:
	/**
		Tworzy nowy obiekt dzielcy opisy na pod-opisy o maksymalnej dugoci splitSize znakw.

		@arg splitSize maksymalna dugo pod-opisu
	 **/
	SplitStatusChanger(unsigned int splitSize);
	virtual ~SplitStatusChanger();

	virtual void changeStatus(UserStatus &status);

	void enable();
	void disable();

};

/**
	@class StatusChangerManager
	@brief Manager modyfikatorw statusu.

	Klasa zarzdza modyfikatorami statusu. Docza i odcza na danie nowy modyfikatory.
	Wywouje poszczeglne modyfikatory w kolejnoci ich rosncych priorytetw.

	Po stworzeniu i dodaniu pierwszych modyfikatorw wymaga uruchomienia za pomoc metody enable.
 **/
class StatusChangerManager : public QObject
{
	Q_OBJECT

	QValueList<StatusChanger *> statusChangers;
	bool enabled;

public:
	static void initModule();
	static void closeModule();

	StatusChangerManager();
	virtual ~StatusChangerManager();

	/**
		Aktywuje modyfikatory.
	 **/
	void enable();

	void registerStatusChanger(StatusChanger *statusChanger);
	void unregisterStatusChanger(StatusChanger *statusChanger);

public slots:
	/**
		Slot wywoywany, gdy jeden z modyfikatorw dokona zmiany opisu.
		Wywoanie powoduje ponowne przetworzenie statusu przez wszystkie modyfikatory
		i wysanie sygnau statusChanged(UserStatus).
	 **/
	void statusChanged();

signals:
	/**
		Sygna wysyany, gdy zostanie ustalony ostateczny status uytkownika.
	 **/
	void statusChanged(UserStatus);

};

extern StatusChangerManager *status_changer_manager;

#endif // STATUS_CHANGER
