/***************************************************************************
                          account.cpp  -  description
                             -------------------
    begin                : Sun Jan 5 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "account.h"

#include "msnobject.h"

#include <qdir.h>
#include <qregexp.h>

#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
#include <kmdcodec.h>
#include <kstandarddirs.h>
#include <kapplication.h>

#include "kmessdebug.h"

#ifdef KMESSDEBUG_ACCOUNT
  #define KMESSDEBUG_ACCOUNT_GENERAL
  #define KMESSDEBUG_ACCOUNT_DIRTY
#endif

// The constructor
Account::Account()
 : autoreplyMessage_( i18n("I'm away from my computer") ),
   chatStyle_( "Fresh" ),
   contactFontColor_("#000000"),
   customImageIndex_(0),
   deleted_(false),
   dirty_(false),
   doNotifyContactsOnline_(true),
   doNotifyContactsStatus_(false),
   doNotifyContactsChatStart_(true),
   doNotifyContactsChatMessages_(false),
   doNotifyContactsOffline_(true),
   doNotifyEmails_(true),
   emailCommand_("kmail"),
   emailSupported_(true),  // Assume true (for new accounts), until the msn server proves the opposite
   emoticonStyle_("KMess-new"),
   fontColor_("#000000"),
   friendlyName_( i18n("Your name") ),
   guestAccount_(true),
   groupFollowupMessages_(true),
   handle_( i18n("you@hotmail.com") ),
   idleTime_(5),
   initialStatus_("NLN"),
   notificationDuration_(10),
   saveChats_(false),
   savedChatDirectoryStructure_( 0 ),
   shakeNudge_(true),
   showAllowedContacts_(false),
   showContactsByGroup_(true),
   showEmail_(true),
   showMessageTime_(true),
   showNowListening_(false),
   showListPictures_(false),
   showOfflineContacts_(true),
   showOtherFolders_(true),
   showRemovedContacts_(false),
   showSidebar_(true),
   useAutologin_(false),
   useContactFont_(false),
   useEmoticons_(true),
   useFontEffects_(true),
   useHotmail_(true),
   useIdleTimer_(true),

   verified_(true),

   showImage_(true),
   msnObject_(0)
{
  QDir homeDir;

  font_.setFamily("Arial");
  font_.setBold(false);
  font_.setItalic(false);
  font_.setUnderline(false);
  font_.setPointSize(12);

  contactFont_.setFamily("Arial");
  contactFont_.setBold(false);
  contactFont_.setItalic(false);
  contactFont_.setUnderline(false);

  homeDir = QDir::home();
  saveChatPath_ = homeDir.absPath();
}



// The destructor
Account::~Account()
{
  if ( msnObject_ )
    delete msnObject_;

#ifdef KMESSDEBUG_ACCOUNT_GENERAL
  kdDebug() << "DESTROYED Account [handle=" << handle_ << "]" << endl;
#endif
}



// Copy an account
void Account::copyAccount( const Account *account )
{
  // This method is called to write information from a global Account
  // object to the CurrentAccount and visa-versa.
  // See also: copyAccountUISettings() for properties only changed by the GUI.

  setLoginInformation        ( account->getHandle(),
                               account->getFriendlyName(),
                               account->getPassword() );
  setShowAllowedContacts     ( account->getShowAllowedContacts() );
  setShowContactsByGroup     ( account->getShowContactsByGroup() );
  setShowOfflineContacts     ( account->getShowOfflineContacts() );
  setShowRemovedContacts     ( account->getShowRemovedContacts() );
  setShowListPictures        ( account->getShowListPictures()    );
  setEmoticonStyle           ( account->getEmoticonStyle()       );

  setAlertsInformation       ( account->getUseIdleTimer(),
                               account->getIdleTime(),
                               account->getShakeNudge(),
                               account->getShowNowListening() );
  setNotificationsInformation( account->getNotifyContactsOnline(),
                               account->getNotifyContactsStatus(),
                               account->getNotifyContactsChatStart(),
                               account->getNotifyContactsChatMessages(),
                               account->getNotifyContactsOffline(),
                               account->getNotifyEmails(),
                               account->getNotificationDuration() );
  setChatInformation         ( account->getUseContactFont(),
                               account->getUseEmoticons(),
                               account->getUseFontEffects(),
                               account->getShowMessageTime(),
                               account->getGroupFollowupMessages(),
                               account->getChatStyle()                     );
  setChatLoggingInformation  ( account->getSaveChats(),
                               account->getSaveChatPath(),
                               account->getSavedChatDirectoryStructure()   );
  setEmailInformation        ( account->getUseHotmail(),
                               account->getEmailCommand(),
                               account->getShowEmail(),
                               account->getShowOtherFolders()              );
  setEmailSupported          ( account->getEmailSupported()                );
  setFontInformation         ( account->getFont(),
                               account->getFontColor(),
                               account->getContactFont(),
                               account->getContactFontColor()              );
  setPersonalMessage         ( account->getPersonalMessage()               );

  setUseAutologin            ( account->getUseAutologin()                  );
  setAutoreplyMessage        ( account->getAutoreplyMessage()              );
  setShowSidebar             ( account->getShowSidebar()                   );
  setInitialStatus           ( account->getInitialStatus()                 );
  setGuestAccount            ( account->isGuestAccount()                   );

  setShowImage               ( account->getShowImage()                     );
  setTemporaryPassword       ( account->getTemporaryPassword()             );
  setVerified                ( account->isVerified()                       );

  dirty_            = account->isDirty();
  customImageIndex_ = account->customImageIndex_;
  updateMsnObject();
}



// Copy an account's user-interface, not settings, changable properties
void Account::copyAccountUISettings( const Account *account )
{
  // All properties that are modified through the main gui are listed here.
  setPersonalMessage    ( account->getPersonalMessage()     );
  setShowAllowedContacts( account->getShowAllowedContacts() );
  setShowOfflineContacts( account->getShowOfflineContacts() );
  setShowRemovedContacts( account->getShowRemovedContacts() );
  setShowContactsByGroup( account->getShowContactsByGroup() );
  setShowListPictures   ( account->getShowListPictures()    );
  setAutoreplyMessage   ( account->getAutoreplyMessage()    );
  setEmailSupported     ( account->getEmailSupported()      );
  setGuestAccount       ( account->isGuestAccount()         );
  setShowSidebar        ( account->getShowSidebar()         );
  setVerified           ( account->isVerified()             );

  password_         = account->getPassword();
  dirty_            = account->isDirty();
  customImageIndex_ = account->customImageIndex_;
}



// Delete the account data
void Account::deleteAccountData()
{
  // Can be called twice, avoid warnings.
  if( deleted_ )
  {
    return;
  }

#ifdef KMESSDEBUG_KMESS
  kdDebug() << "Account::deleteAccountData: deleting configuration fields" << endl;
#endif

  KConfig     *config = kapp->config();
  QString      groupName;
  QStringList  groups = config->groupList();

  // Delete Profile_<handle>
  if( ! config->deleteGroup("Profile_" + handle_) )
  {
    kdWarning() << "Account::deleteAccountData: Could not remove 'Profile_" << handle_ << "' from the config file!" << endl;
  }

  // Delete Contact_<handle>/<contacthandle>
  QString contactPrefix = "Contact_" + handle_ + "/";
  for( QStringList::Iterator it = groups.begin(); it != groups.end(); ++it )
  {
    groupName = *it;
    if( groupName.startsWith(contactPrefix) )
    {
      if( ! config->deleteGroup(groupName) )
      {
        kdWarning() << "Account::deleteAccountData: Could not remove '" << groupName << "' from the config file!" << endl;
      } 
    }
  }

  // Mark as deleted, so other classes avoid saving the settings again.
  deleted_      = true;
  guestAccount_ = true;
}



// Read the auto-reply message.
const QString& Account::getAutoreplyMessage() const
{
  return autoreplyMessage_;
}



// Return the chat style name.
const QString& Account::getChatStyle() const
{
  return chatStyle_;
}



// Return the font used for contact messages, if forced to.
const QFont& Account::getContactFont() const
{
  return contactFont_;
}



// Return the color of the forced contact font.
const QString& Account::getContactFontColor() const
{
  return contactFontColor_;
}



// Read the email command used when not using hotmail
const QString& Account::getEmailCommand() const
{
  return emailCommand_;
}


// Return whether email notifications are supported
bool Account::getEmailSupported() const
{
  return emailSupported_;
}


// Return the user message font.
const QFont& Account::getFont() const
{
  return font_;
}



// Return the color of the user message font.
const QString& Account::getFontColor() const
{
  return fontColor_;
}



// Return the user's friendlyName
QString Account::getFriendlyName() const
{
  return friendlyName_;
}



// Whether follow-up messages from the contact should be grouped.
bool Account::getGroupFollowupMessages() const
{
  return groupFollowupMessages_;
}



// Return the user's handle
QString Account::getHandle() const
{
  return handle_;
}



// Return the length of time waited before the user goes idle.
const int& Account::getIdleTime() const
{
  return idleTime_;
}



// Return the path for a new the custom display image.
QString Account::getNewCustomImagePath()
{
  customImageIndex_++;  // Ensures the image name is always unique, so the ChatWindow's KHTMLPart doesn't cache the image.
  QString imageRoot  = ::locateLocal("data", "kmess");
  QString safeHandle = getHandle().lower().replace( QRegExp( "[^a-z0-9_.@]" ), "-" );
  return imageRoot + "/msnpicture-" + safeHandle + "-" + QString::number(customImageIndex_)  + ".png";
}



// Return the time popup windows are shown.
const int& Account::getNotificationDuration() const
{
  return notificationDuration_;
}



// Return if popup notifications will be shown for contacts which come online.
bool Account::getNotifyContactsOnline() const
{
  return doNotifyContactsOnline_;
}



// Return if popup notifications will be shown for contacts which change their msn status.
bool Account::getNotifyContactsStatus() const
{
  return doNotifyContactsStatus_;
}



// Return if popup notifications will be shown for contacts which start a chat with you.
bool Account::getNotifyContactsChatStart() const
{
  return doNotifyContactsChatStart_;
}



// Return if popup notifications will be shown for all received chat messages.
bool Account::getNotifyContactsChatMessages() const
{
  return doNotifyContactsChatMessages_;
}



// Return if popup notifications will be shown for contacts which go offline.
bool Account::getNotifyContactsOffline() const
{
  return doNotifyContactsOffline_;
}



// Return if popup notifications will be shown for incoming mail.
bool Account::getNotifyEmails() const
{
  return doNotifyEmails_;
}



// Return whether now listening information should be displayed.
bool Account::getShowNowListening() const
{
  return showNowListening_;
}



// Return the user's password
QString Account::getPassword() const
{
  return password_;
}



// Return the personal message
QString Account::getPersonalMessage() const
{
  return personalMessage_;
}



// Return the selected emoticon's style
const QString& Account::getEmoticonStyle() const
{
  return emoticonStyle_;
}



// Return the status to set upon connection
QString Account::getInitialStatus() const
{
  return initialStatus_;
}



// Return whether or not to save chats.
bool Account::getSaveChats() const
{
  return saveChats_;
}



// Return the base directory to which chats are saved.
const QString& Account::getSaveChatPath() const
{
  return saveChatPath_;
}



// Return the code of the structure used to save chats.
const int& Account::getSavedChatDirectoryStructure() const
{
  return savedChatDirectoryStructure_;
}



// Read whether the chat window should shake when a nudge is received or sent.
bool Account::getShakeNudge() const
{
  return shakeNudge_;
}



// Read whether or not allowed contacts are shown.
bool Account::getShowAllowedContacts() const
{
  return showAllowedContacts_;
}



// Read whether contacts are shown by group or by online/offline.
bool Account::getShowContactsByGroup() const
{
  return showContactsByGroup_;
}



// Read whether the email information should be shown in the main view.
bool Account::getShowEmail() const
{
  return showEmail_;
}



// Read whether message times should be shown.
bool Account::getShowMessageTime() const
{
  return showMessageTime_;
}



// Read whether offline contacts should be shown.
bool Account::getShowOfflineContacts() const
{
  return showOfflineContacts_;
}



// Read property of bool showOtherFolders.
bool Account::getShowOtherFolders() const
{
  return showOtherFolders_;
}



// Read whether or not removed (reverse) contacts are shown.
bool Account::getShowRemovedContacts() const
{
  return showRemovedContacts_;
}



// Read whether or not the sidebar should be shown in chats
bool Account::getShowSidebar() const
{
  return showSidebar_;
}



// Return the a temporary password for the account for login attempts from the initialview dialog
const QString & Account::getTemporaryPassword() const
{
  return temporaryPassword_;
}



// Return whether or not to autologin with this profile.
bool Account::getUseAutologin() const
{
  return useAutologin_;
}



// Return whether or not to show contact messages in the stored font.
bool Account::getUseContactFont() const
{
  return useContactFont_;
}



// Return whether or not to show emoticons in chats.
bool Account::getUseEmoticons() const
{
  return useEmoticons_;
}



// Return whether or not to use the bold/italic/underline effects.
bool Account::getUseFontEffects() const
{
  return useFontEffects_;
}



// Read whether the user uses hotmail to read email or a set command
bool Account::getUseHotmail() const
{
  return useHotmail_;
}



// Return whether the idle timer should be used.
bool Account::getUseIdleTimer() const
{
  return useIdleTimer_;
}



// Returns whether or not to send our display image
bool Account::getShowImage() const
{
  return showImage_;
}


// Returns whether display pictures should be shown in contact list
bool Account::getShowListPictures() const
{
  return showListPictures_;
}



// Return the path of the custom display image.
const QString Account::getCustomImagePath() const
{
  QString safeHandle = getHandle().lower().replace( QRegExp( "[^a-z0-9_.@]" ), "-" );
  QString suffix     = ( customImageIndex_ > 0 ? "-" + QString::number(customImageIndex_) : QString::null );

  return ::locateLocal("data", "kmess/msnpicture-" + safeHandle + suffix + ".png");
}



// Return the path to the *current* display image
const QString Account::getImagePath() const
{
  KStandardDirs *dirs = KGlobal::dirs();
  QString imagePath;

  // return image path if user has selected to show it else return QString::null
  if( showImage_ )
  {
    // if the user has selected a custom image it will be located (usually) in
    // ~/.kde/share/apps/kmess/msnpicture-handle.png
    imagePath = getCustomImagePath();

    // If the file can't be found, fallback to the default image.
    if(! QFile::exists(imagePath))
    {
      imagePath = dirs->findResource( "data", "kmess/pics/kmesspic.png"  );
    }

    return imagePath;
  }
  else
  {
    return QString::null;
  }
}


// Returns the string representation of the MsnObject
const QString Account::getMsnObjectString() const
{
  if ( msnObject_ )
    return msnObject_->objectString();
  else
    return "";
}



// Whether the account is marked to be deleted
bool Account::isDeleted() const
{
  return deleted_;
}



// Whether the account has unsaved settings changes
bool Account::isDirty() const
{
  return dirty_;
}



// Whether the account is a guest account, not permanently saved on the computer
bool Account::isGuestAccount() const
{
  return guestAccount_;
}



// Return whether the account's passport is verified
bool Account::isVerified() const
{
  return verified_;
}



// Read in account properties
void Account::readProperties(KConfig *config, QString handle)
{
  handle_ = handle;
  QDir    homeDir;
  QString defaultPath;

  homeDir = QDir::home();
  defaultPath = homeDir.absPath();

  // Go the the profile's group
  config->setGroup( "Profile_" + handle_ );

  autoreplyMessage_              = config->readEntry    ( "autoreplymessage",             i18n("I'm away from my computer") );
  chatStyle_                     = config->readEntry    ( "chatStyle",                    "Fresh  "                         );
// contactFont_.setFamily done below
  contactFont_.setBold           ( config->readBoolEntry( "contactfontbold",              false                           ) );
  contactFont_.setItalic         ( config->readBoolEntry( "contactfontitalic",            false                           ) );
  contactFont_.setUnderline      ( config->readBoolEntry( "contactfontunderline",         false                           ) );
  contactFont_.setPointSize      ( config->readNumEntry ( "contactfontpointsize",         10                              ) );
  contactFontColor_              = config->readEntry    ( "contactfontcolor" ,            "#000000"                         );
  customImageIndex_              = config->readNumEntry ( "customimageindex",             0                                 );
  doNotifyContactsOnline_        = config->readBoolEntry( "doNotifyContactsOnline",       true                              );
  doNotifyContactsStatus_        = config->readBoolEntry( "doNotifyContactsStatus",       false                             );
  doNotifyContactsChatStart_     = config->readBoolEntry( "doNotifyContactsChatStart",    true                              );
  doNotifyContactsChatMessages_  = config->readBoolEntry( "doNotifyContactsChatMessages", false                             );
  doNotifyContactsOffline_       = config->readBoolEntry( "doNotifyContactsOffline",      true                              );
  doNotifyEmails_                = config->readBoolEntry( "doNotifyEmails",               true                              );
  emailCommand_                  = config->readEntry    ( "emailCommand",                 "kmail"                           );
  emoticonStyle_                 = config->readEntry    ( "emoticonStyle",                "KMess-new"                       );
//font_.setFamily done below
  font_.setBold                  ( config->readBoolEntry( "fontbold",                     false                           ) );
  font_.setItalic                ( config->readBoolEntry( "fontitalic",                   false                           ) );
  font_.setPointSize             ( config->readNumEntry ( "fontpointsize",                12                              ) );
  font_.setUnderline             ( config->readBoolEntry( "fontunderline",                false                           ) );
  fontColor_                     = config->readEntry    ( "fontColor" ,                   "#000000"                         );
  friendlyName_                  = config->readEntry    ( "friendlyName" ,                QString::null                     );
  groupFollowupMessages_         = config->readBoolEntry( "groupfollowupmessages",        true                              );
  idleTime_                      = config->readNumEntry ( "idleTime",                     5                                 );
  initialStatus_                 = config->readEntry    ( "initialStatus",                "NLN"                             );
  notificationDuration_          = config->readNumEntry ( "notification",                 5                                 );
  password_                      = config->readEntry    ( "password" ,                    QString::null                     );
  personalMessage_               = config->readEntry    ( "personalMessage",              QString::null                     );
  saveChatPath_                  = config->readEntry    ( "saveChatPath",                 defaultPath                       );
  saveChats_                     = config->readBoolEntry( "saveChats",                    false                             );
  savedChatDirectoryStructure_   = config->readNumEntry ( "savedChatDirectoryStructure",  0                                 );
  shakeNudge_                    = config->readBoolEntry( "shakeNudge",                   true                              );
  showAllowedContacts_           = config->readBoolEntry( "showAllowed",                  false                             );
  showContactsByGroup_           = config->readBoolEntry( "showByGroup",                  true                              );
  showEmail_                     = config->readBoolEntry( "showEmail",                    true                              );
  showImage_                     = config->readBoolEntry( "showImage",                    true                              );
  showListPictures_              = config->readBoolEntry( "showListPictures",             false                             );
  showMessageTime_               = config->readBoolEntry( "showMessageTime",              true                              );
  showNowListening_              = config->readBoolEntry( "showNowListening",             false                             );
  showOfflineContacts_           = config->readBoolEntry( "showOfflineContacts",          true                              );
  showOtherFolders_              = config->readBoolEntry( "showOtherFolders",             true                              );
  showRemovedContacts_           = config->readBoolEntry( "showRemoved",                  false                             );
  showSidebar_                   = config->readBoolEntry( "showSidebar",                  true                              );
  useAutologin_                  = config->readBoolEntry( "useAutologin",                 false                             );
  useContactFont_                = config->readBoolEntry( "useContactFont",               false                             );
  useEmoticons_                  = config->readBoolEntry( "useEmoticons",                 true                              );
  useFontEffects_                = config->readBoolEntry( "useFontEffects",               true                              );
  useHotmail_                    = config->readBoolEntry( "useHotmail",                   true                              );
  useIdleTimer_                  = config->readBoolEntry( "useIdleTimer",                 true                              );

  updateMsnObject();


  // Read font setting, remove accel character if it's still there.
  // That's a KMess 1.3 issue, but still visible in 1.4 if you never changed font settings.
  QString normalFont  = config->readEntry("fontfamily", "Arial");
  QString contactFont = config->readEntry( "contactfontfamily", "Arial");
  font_.setFamily( normalFont.replace("&", QString::null) );
  contactFont_.setFamily( contactFont.replace("&", QString::null) );

  // After reading options from file, consider "dirty" to still be false
  guestAccount_ = false;
  dirty_ = false;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account::readProperties(): Resetting 'dirty' to false." << endl;
#endif

  // If a chat style is removed, fall back to the default.
  // This default is "Fresh" as of 1.5 final, to improve the first impression a user gets.
  // The previous "Default" theme has been renamed to "Classic".
  // This code also makes the migration from 1.5-pre2 to 1.5 (and beyond) easier.
  if( KGlobal::dirs()->findResource( "data", "kmess/styles/" + chatStyle_ + "/" + chatStyle_ + ".xsl" ).isEmpty() )
  {
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::readProperties(): Auto-corrected chat & setting, setting 'dirty' to true." << endl;
#endif

    // Fall back to our new default.
    chatStyle_ = "Fresh";
    groupFollowupMessages_ = true;  // Make sure Fresh looks good.
    dirty_ = true;
  }

  // If the fonts were auto-corrected, make sure the modified setting is saved in the kmessrc.
  // Avoids questions and discussions like "yes, it's there but that's not a problem". Better just fix it.
  if( normalFont.contains("&") || contactFont.contains("&") )
  {
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::readProperties(): Auto-corrected font & setting, setting 'dirty' to true." << endl;
#endif
    dirty_ = true;
  }
}



// Save account properties
void Account::saveProperties(KConfig *config)
{
  // Protect against unwanted usage
  if( guestAccount_ )
  {
    kdWarning() << "Account::saveProperties: Application attempted to save a guest account!" << endl;
  }
  if( deleted_ )
  {
    kdWarning() << "Account::saveProperties: Application attempted to save the deleted account again!" << endl;
  }

  if ( dirty_ )
  {
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::saveProperties(): Account " << handle_ << " is 'dirty', save properties." << endl;
#endif
    // Go the the profile's group
    config->setGroup( "Profile_" + handle_ );

    // Remove old chatFormat, replaced by chatStyle
    config->deleteEntry( "chatFormat" );

    config->writeEntry( "autoreplymessage",             autoreplyMessage_             );
    config->writeEntry( "chatStyle",                    chatStyle_                    );
    config->writeEntry( "contactfontfamily",            contactFont_.family()         );
    config->writeEntry( "contactfontbold",              contactFont_.bold()           );
    config->writeEntry( "contactfontitalic",            contactFont_.italic()         );
    config->writeEntry( "contactfontunderline",         contactFont_.underline()      );
    config->writeEntry( "contactfontpointsize",         contactFont_.pointSize()      );
    config->writeEntry( "contactfontcolor",             contactFontColor_             );
    config->writeEntry( "customimageindex",             customImageIndex_             );
    config->writeEntry( "doNotifyContactsOnline",       doNotifyContactsOnline_       );
    config->writeEntry( "doNotifyContactsStatus",       doNotifyContactsStatus_       );
    config->writeEntry( "doNotifyContactsChatStart",    doNotifyContactsChatStart_    );
    config->writeEntry( "doNotifyContactsChatMessages", doNotifyContactsChatMessages_ );
    config->writeEntry( "doNotifyContactsOffline",      doNotifyContactsOffline_      );
    config->writeEntry( "doNotifyEmails",               doNotifyEmails_       );
    config->writeEntry( "emailCommand",                 emailCommand_                 );
    config->writeEntry( "emoticonStyle",                emoticonStyle_                );
    config->writeEntry( "fontfamily",                   font_.family()                );
    config->writeEntry( "fontbold",                     font_.bold()                  );
    config->writeEntry( "fontitalic",                   font_.italic()                );
    config->writeEntry( "fontunderline",                font_.underline()             );
    config->writeEntry( "fontpointsize",                font_.pointSize()             );
    config->writeEntry( "fontColor",                    fontColor_                    );
    config->writeEntry( "friendlyName",                 friendlyName_                 );
    config->writeEntry( "groupfollowupmessages",        groupFollowupMessages_        );
    config->writeEntry( "idleTime",                     idleTime_                     );
    config->writeEntry( "notification",                 notificationDuration_         );
    config->writeEntry( "password",                     password_                     );
    config->writeEntry( "personalMessage",              personalMessage_              );
    config->writeEntry( "saveChatPath",                 saveChatPath_                 );
    config->writeEntry( "saveChats",                    saveChats_                    );
    config->writeEntry( "savedChatDirectoryStructure",  savedChatDirectoryStructure_  );
    config->writeEntry( "shakeNudge",                   shakeNudge_                   );
    config->writeEntry( "showAllowed",                  showAllowedContacts_          );
    config->writeEntry( "showByGroup",                  showContactsByGroup_          );
    config->writeEntry( "showEmail",                    showEmail_                    );
    config->writeEntry( "showListPictures",             showListPictures_             );
    config->writeEntry( "showMessageTime",              showMessageTime_              );
    config->writeEntry( "showNowListening",             showNowListening_             );
    config->writeEntry( "showOfflineContacts",          showOfflineContacts_          );
    config->writeEntry( "showOtherFolders",             showOtherFolders_             );
    config->writeEntry( "showRemoved",                  showRemovedContacts_          );
    config->writeEntry( "showSidebar",                  showSidebar_                  );
    config->writeEntry( "initialStatus",                initialStatus_                );
    config->writeEntry( "useAutologin",                 useAutologin_                 );
    config->writeEntry( "useContactFont",               useContactFont_               );
    config->writeEntry( "useEmoticons",                 useEmoticons_                 );
    config->writeEntry( "useFontEffects",               useFontEffects_               );
    config->writeEntry( "useHotmail",                   useHotmail_                   );
    config->writeEntry( "useIdleTimer",                 useIdleTimer_                 );
    config->writeEntry( "showImage",                    showImage_                    );
  }
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  else
  {
    kdDebug() << "Account::saveProperties(): Account " << handle_ << " is NOT 'dirty', don't save properties." << endl;
  }
#endif

  // Save to disk the changes now, to avoid losing data.
  config->sync();
}



// The login succeeded with the new password. Save the temporary password as new account password.
void Account::saveTemporaryPassword()
{
  if( ! temporaryPassword_.isEmpty() && temporaryPassword_ != password_ )
  {
    password_ = temporaryPassword_;
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::saveTemporaryPassword(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set common alerts information
void Account::setAlertsInformation( bool useIdleTimer, int idleTime, bool shakeNudge, bool showNowListening )
{
  useIdleTimer_ = useIdleTimer;
  idleTime_ = idleTime;
  shakeNudge_ = shakeNudge;

  emit changedNotificationsSettings();
  emit changedTimerSettings();

  if( showNowListening_ != showNowListening )
  {
#ifdef KMESSDEBUG_ACCOUNT_GENERAL
    kdDebug() << "Now listening settings changed, notifying listeners." << endl;
#endif

    showNowListening_ = showNowListening;
    emit changedNowListeningSettings();
  }

  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setAlertsInformation(): Setting 'dirty' to true." << endl;
#endif
}



// Set notification alerts information
void Account::setNotificationsInformation( bool doNotifyContactsOnline, bool doNotifyContactsStatus, bool doNotifyContactsChatStart,
                                           bool doNotifyContactsChatMessages, bool doNotifyContactsOffline, bool doNotifyEmails, int notificationDuration )
{
  doNotifyContactsOnline_ = doNotifyContactsOnline;
  doNotifyContactsStatus_ = doNotifyContactsStatus;
  doNotifyContactsChatStart_ = doNotifyContactsChatStart;
  doNotifyContactsChatMessages_ = doNotifyContactsChatMessages;
  doNotifyContactsOffline_ = doNotifyContactsOffline;
  doNotifyEmails_ = doNotifyEmails;
  notificationDuration_ = notificationDuration;

  emit changedNotificationsSettings();

  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setNotificationsInformation(): Setting 'dirty' to true." << endl;
#endif
}



// Set the auto-reply message.
void Account::setAutoreplyMessage( const QString& autoreplyMessage)
{
  if ( autoreplyMessage_ != autoreplyMessage )
  {
    autoreplyMessage_ = autoreplyMessage;
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setAutoreplyMessage(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set chat information
void Account::setChatInformation( bool useContactFont, bool useEmoticons, bool useFontEffects,
                                  bool showMessageTime, bool groupFollowupMessages, const QString& chatStyle )
{
  bool changedSettings = (useContactFont_        != useContactFont)
                      || (useEmoticons_          != useEmoticons)
                      || (useFontEffects_        != useFontEffects)
                      || (showMessageTime_       != showMessageTime)
                      || (groupFollowupMessages_ != groupFollowupMessages)
                      || (chatStyle_             != chatStyle);

  // Avoid signals because the chat windows will regenerate all messages.
  if( ! changedSettings )
  {
    return;
  }

  useContactFont_        = useContactFont;
  useEmoticons_          = useEmoticons;
  useFontEffects_        = useFontEffects;
  showMessageTime_       = showMessageTime;
  groupFollowupMessages_ = groupFollowupMessages;
  chatStyle_             = chatStyle;
  emit changedChatStyleSettings();
  dirty_ = true;

#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setChatInformation(): Setting 'dirty' to true." << endl;
#endif
}



// Set chat logging information
void Account::setChatLoggingInformation( bool saveChats, const QString& saveChatPath, const int& directoryStructure )
{
  if( saveChats_                   != saveChats
  ||  saveChatPath_                != saveChatPath
  ||  savedChatDirectoryStructure_ != directoryStructure )
  {
    saveChats_                   = saveChats;
    saveChatPath_                = saveChatPath;
    savedChatDirectoryStructure_ = directoryStructure;
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setChatLoggingInformation(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set email information
void Account::setEmailInformation( bool useHotmail, const QString& emailCommand, bool showEmail, bool showOtherFolders )
{
  if( useHotmail_       != useHotmail
  ||  emailCommand_     != emailCommand
  ||  showEmail_        != showEmail
  ||  showOtherFolders_ != showOtherFolders )
  {
    useHotmail_       = useHotmail;
    emailCommand_     = emailCommand;
    showEmail_        = showEmail;
    showOtherFolders_ = showOtherFolders;

    emit changedEmailDisplaySettings();
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setEmailInformation(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Update the email-supported state (protected method)
void Account::setEmailSupported(bool emailSupported)
{
  emailSupported_ = emailSupported; 
}



// Set the font
void Account::setFont( const QFont& font )
{
  if( font != font_ )
  {
    font_ = font;
    emit changedFontSettings();
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setFont(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set the font color
void Account::setFontColor( const QString& fontColor )
{
  if( fontColor != fontColor_ )
  {
    fontColor_ = fontColor;
    emit changedFontSettings();
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setFontColor(): Setting 'dirty' to true." << endl;
#endif
  }
}




// Set the fonts and font colors
void Account::setFontInformation( const QFont& font, const QString &fontColor, const QFont& contactFont, const QString &contactFontColor)
{
  bool changedSettings = (font_ != font)
                      || (fontColor_ != fontColor)
                      || (contactFont_ != contactFont)
                      || (contactFontColor_ != contactFontColor);

  // Avoid signals because the chat windows will regenerate all messages.
  if( ! changedSettings )
  {
    return;
  }

  font_             = font;
  fontColor_        = fontColor;
  contactFont_      = contactFont;
  contactFontColor_ = contactFontColor;
  emit changedFontSettings();
  dirty_ = true;

#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setFontInformation(): Setting 'dirty' to true." << endl;
#endif
}



// Set the emoticon path
void Account::setEmoticonStyle( const QString& emoticonStyle )
{
  if( emoticonStyle != emoticonStyle_ )
  {
    // Set the new style, and signal EmoticonManager about it.
    emoticonStyle_ = emoticonStyle;
    emit changedEmoticonSettings();

    // Tell the account settings changed and need to be saved
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setEmoticonStyle(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Change the friendly name
void Account::setFriendlyName(const QString& newName)
{
  if ( ( newName != "" ) && ( newName != friendlyName_ ) )
  {
    friendlyName_ = newName;
    emit changedFriendlyName();
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setFriendlyName(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Whether the account is a guest account, not permanently saved on the computer
void Account::setGuestAccount( bool guestAccount )
{
  guestAccount_ = guestAccount;

  // Undelete when user did restore the account.
  if( ! guestAccount_ )
  {
    deleted_ = false;
  }
}



// Change the status to set upon connection
void Account::setInitialStatus( QString status )
{
  if( initialStatus_ == status )
  {
    return;
  }

  initialStatus_ = status;
  dirty_ = true;

#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account::setInitialStatus(): Setting 'dirty' to true." << endl;
#endif
}



// Set basic account data
void Account::setLoginInformation( QString handle, QString friendlyName, QString password )
{
#ifdef KMESSTEST
  ASSERT( handle.contains("@") );
  ASSERT( handle != "" );
#endif

  if( handle_       != handle.lower()
  ||  friendlyName_ != friendlyName
  ||  ( ! password.isEmpty() && password != password_ ) )
  {
    handle_       = handle.lower();

    // Never change the nickname if the account is not verified. We would receive a server error
    if( verified_ )
    {
      friendlyName_ = friendlyName;
      emit changedFriendlyName();
    }

    if( ! password.isEmpty() )
    {
      password_ = password;
    }

    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setLoginInformation(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set whether to show popup notifications when contacts come online.
void Account::setNotifyContactsOnline( bool doNotifyContactsOnline )
{
  doNotifyContactsOnline_ = doNotifyContactsOnline;
}



// Set whether to show popup notifications when contacts change their msn status.
void Account::setNotifyContactsStatus( bool doNotifyContactsStatus )
{
  doNotifyContactsStatus_ = doNotifyContactsStatus;
}



// Set whether to show popup notifications when contacts start a chat with you.
void Account::setNotifyContactsChatStart( bool doNotifyContactsChatStart )
{
  doNotifyContactsChatStart_ = doNotifyContactsChatStart;
}



// Set whether to show popup notifications for all received chat messages.
void Account::setNotifyContactsChatMessages( bool doNotifyContactsChatMessages )
{
  doNotifyContactsChatMessages_ = doNotifyContactsChatMessages;
}



// Set whether to show popup notifications when contacts go offline.
void Account::setNotifyContactsOffline( bool doNotifyContactsOffline )
{
  doNotifyContactsOffline_ = doNotifyContactsOffline;
}



// Set whether to show popup notifications for incoming mail.
void Account::setNotifyEmails( bool doNotifyEmails )
{
  doNotifyEmails_ = doNotifyEmails;
}


// Set the personal message
void Account::setPersonalMessage(const QString& newMessage)
{
  if( newMessage != personalMessage_ )
  {
    personalMessage_ = newMessage;
    emit changedPersonalMessage();

    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setPersonalMessage(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set whether or not allowed contacts are shown.
void Account::setShowAllowedContacts( bool showAllowedContacts )
{
  if( showAllowedContacts != showAllowedContacts_ )
  {
    showAllowedContacts_ = showAllowedContacts;
    emit changedDisplaySettings();
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setShowAllowedContacts(): Setting 'dirty' to true." << endl;
#endif
  }
}




// Set whether contacts are shown by group or by online/offline.
void Account::setShowContactsByGroup( bool showContactsByGroup )
{
  if ( showContactsByGroup != showContactsByGroup_ )
  {
    showContactsByGroup_ = showContactsByGroup;
    emit changedViewMode();
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setShowContactsByGroup(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set whether offline contacts should be shown.
void Account::setShowOfflineContacts( bool showOfflineContacts )
{
  if( showOfflineContacts != showOfflineContacts_ )
  {
    showOfflineContacts_ = showOfflineContacts;
    emit changedDisplaySettings();
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setShowOfflineContacts(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set whether or not removed (reverse) contacts are shown.
void Account::setShowRemovedContacts( bool showRemovedContacts )
{
  if( showRemovedContacts != showRemovedContacts_ )
  {
    showRemovedContacts_ = showRemovedContacts;
    emit changedDisplaySettings();
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setShowRemovedContacts(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set whether or not the sidebar should be shown in chats
void Account::setShowSidebar( bool showSidebar )
{
  if ( showSidebar_ != showSidebar )
  {
    showSidebar_ = showSidebar;
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setShowSidebar(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set a temporary password for the account for login attempts from the initialview dialog
void Account::setTemporaryPassword( const QString &temporaryPassword )
{
  temporaryPassword_ = temporaryPassword;
}



// Set whether or not to login with this account.
void Account::setUseAutologin( bool useAutologin)
{
  if( useAutologin != useAutologin_ )
  {
    useAutologin_ = useAutologin;
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setUseAutologin(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set whether or not to show emoticons in chats.
void Account::setUseEmoticons( bool useEmoticons)
{
  useEmoticons_ = useEmoticons;
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setUseEmoticons(): Setting 'dirty' to true." << endl;
#endif
}



// Set whether the account's passport is verified
void Account::setVerified( bool verified )
{
  verified_ = verified;
}



// Set whether or not to send our display image
void Account::setShowImage( bool showImage )
{
  if( showImage != showImage_ )
  {
    showImage_ = showImage;
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account::setShowImage(): Setting 'dirty' to true." << endl;
#endif
  }
}



// Set whether the display pictures of the contacts should be shown in list
void Account::setShowListPictures( bool showPictures)
{
  // No need to update
  if( showPictures == showListPictures_ )
  {
    return;
  }

  showListPictures_ = showPictures;
  emit changedDisplaySettings();
  dirty_ = true;

#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account::setShowListPictures(): Setting 'dirty' to true." << endl;
#endif
}



// Creates the MsnObject only if the user wants to show a display picture
void Account::updateMsnObject()
{
  QString oldMsnString;
  QString newMsnString;

  if( msnObject_ )
  {
    oldMsnString = msnObject_->objectString();
    delete msnObject_;
    msnObject_ = 0;
  }

#ifdef KMESSDEBUG_ACCOUNT_GENERAL
  kdDebug() << "Account::updateMsnObject: Show Image: " << showImage_ << endl;
#endif

  if ( showImage_ )
  {
#ifdef KMESSDEBUG_ACCOUNT_GENERAL
    kdDebug() << "Account::updateMsnObject: Image source: " << getImagePath() << endl;
#endif

    QFile iFile( getImagePath() );
    if ( iFile.open( IO_ReadOnly ) )
    {
      QByteArray data = iFile.readAll();
      iFile.close();

      msnObject_   = new MsnObject( handle_, "KMess.tmp", "", MsnObject::DISPLAYPIC, data );
      newMsnString = msnObject_->objectString();
    }
    else
    {
      kdWarning() << "Account::updateMsnObject() - Image could not be loaded: " << getImagePath() << endl;
    }
  }

  // Detect if there was a change.
  if(oldMsnString != newMsnString)
  {
#ifdef KMESSDEBUG_ACCOUNT_GENERAL
    kdDebug() << "Account::updateMsnObject: result is: '" << newMsnString << "'" << endl;
#endif
    emit changedMsnObject();
  }
}


#include "account.moc"
