/***************************************************************************
                          account.h  -  description
                             -------------------
    begin                : Sun Jan 5 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef ACCOUNT_H
#define ACCOUNT_H

#include <qfont.h>
#include <qwidget.h>

// Forward declarations
class KConfig;
class KMessTest;
class MsnObject;

/**
 * @brief Data class for user account settings.
 *
 * The properties of this class correspond with the profile section in the <code>kmessrc</code> file.
 * The data is read with the readProperties() method and saved with the saveProperties() method.
 * When the user switches to another account, the data is copied to the CurrentAccount class with
 * the copyAccount() method. The <code>set*</code> methods are issued from the SettingsDialog code.
 * When the account is disconnected, the data is copied back and saved to disk.
 *
 * @author Mike K. Bennett
 * @ingroup Root
 */
class Account : public QObject
{
  Q_OBJECT

  friend class KMessTest;

  public: // Public methods
    // Directory structure constants
    enum ChatDirectoryStructure { SINGLEDIRECTORY = 0
                                , BYYEAR          = 1
                                , BYMONTH         = 2
                                , BYDAY           = 3
                                };

    // The constructor
    Account();
    // The destructor
    virtual ~Account();
    // Copy an account
    virtual void                 copyAccount( const Account *account );
    // Copy an account's user-interface, not settings, changable properties
    void                         copyAccountUISettings( const Account *account );
    // Delete the account data
    void                         deleteAccountData();
    // Read the auto-reply message.
    const QString&               getAutoreplyMessage() const;
    // Return the chat style name.
    const QString&               getChatStyle() const;
    // Return the font used for contact messages, if forced to.
    const QFont&                 getContactFont() const;
    // Return the color of the forced contact font.
    const QString&               getContactFontColor() const;
    // Read the email command used when not using hotmail
    const QString&               getEmailCommand() const;
    // Return whether email notifications are supported
    bool                         getEmailSupported() const;
    // Return the user message font.
    const QFont&                 getFont() const;
    // Return the color of the user message font.
    const QString&               getFontColor() const;
    // Return the user's friendlyName
    QString                      getFriendlyName() const;
    // Whether follow-up messages from the contact should be grouped.
    bool                         getGroupFollowupMessages() const;
    // Return the user's handle
    QString                      getHandle() const;
    // Return the length of time waited before the user goes idle.
    const int&                   getIdleTime() const;
    // Return the status to set upon connection
    QString                      getInitialStatus() const;
    // Return the time popup windows are shown.
    const int&                   getNotificationDuration() const;
    // Return if popup notifications will be shown for contacts which come online.
    bool                         getNotifyContactsOnline() const;
    // Return if popup notifications will be shown for contacts which change their msn status.
    bool                         getNotifyContactsStatus() const;
    // Return if popup notifications will be shown for contacts which start a chat with you.
    bool                         getNotifyContactsChatStart() const;
    // Return if popup notifications will be shown for all received chat messages.
    bool                         getNotifyContactsChatMessages() const;
    // Return if popup notifications will be shown for contacts which go offline.
    bool                         getNotifyContactsOffline() const;
    // Return if popup notifications will be shown for incoming mail.
    bool                         getNotifyEmails() const;
    // Return the user's password
    QString                      getPassword() const;
    // Return the personal message
    QString                      getPersonalMessage() const;
   // Return whether or not to save chats.
    bool                         getSaveChats() const;
    // Return the base directory to which chats are saved.
    const QString&               getSaveChatPath() const;
    // Return the code of the structure used to save chats.
    const int&                   getSavedChatDirectoryStructure() const;
    // Read whether or not allowed contacts are shown.
    bool                         getShowAllowedContacts() const;
    // Read whether contacts are shown by group or by online/offline.
    bool                         getShowContactsByGroup() const;
    // Read whether the email information should be shown in the main view.
    bool                         getShowEmail() const;
    // Returns whether display pictures should be shown in contact list.
    bool                         getShowListPictures() const;
    // Read the selected emoticon's list number
    const QString&               getEmoticonStyle() const;
    // Read whether the chat window should shake when a nudge is received or sent.
    bool                         getShakeNudge() const;
    // Read whether offline contacts should be shown.
    bool                         getShowOfflineContacts() const;
    // Read property of bool showMessageTime.
    bool                         getShowMessageTime() const;
    // Return whether now listening information should be displayed.
    bool                         getShowNowListening() const;
    // Read property of bool showOtherFolders.
    bool                         getShowOtherFolders() const;
    // Read whether or not removed (reverse) contacts are shown.
    bool                         getShowRemovedContacts() const;
    // Read whether or not the sidebar should be shown in chats
    bool                         getShowSidebar() const;
    // Return the a temporary password for the account for login attempts from the initialview dialog
    const QString &              getTemporaryPassword() const;
    // Return whether or not to autologin with this profile.
    bool                         getUseAutologin() const;
    // Return whether or not to show contact messages in the stored font.
    bool                         getUseContactFont() const;
    // Return whether or not to show emoticons in chats.
    bool                         getUseEmoticons() const;
    // Return whether or not to use bold/italic/underline effects in chats.
    bool                         getUseFontEffects() const;
    // Read whether the user uses hotmail to read email or a set command
    bool                         getUseHotmail() const;
    // Return whether the idle timer should be used.
    bool                         getUseIdleTimer() const;
    // Returns whether or not to send our display image
    bool                         getShowImage() const;
    // Return the path of the custom display image.
    const QString                getCustomImagePath() const;
    // Return the path to the *current* display image
    const QString                getImagePath() const;
    // Returns the string representation of the MsnObject
    const QString                getMsnObjectString() const;
    // Return the path for a new the custom display image.
    QString                      getNewCustomImagePath();
    // Whether the account is marked to be deleted
    bool                         isDeleted() const;
    // Whether the account has unsaved settings changes
    bool                         isDirty() const;
    // Whether the account is a guest account, not permanently saved on the computer
    bool                         isGuestAccount() const;
    // Return whether the account's passport is verified
    bool                         isVerified() const;
    // Read in account properties
    void                         readProperties(KConfig *config, QString  handle);
    // Save account properties
    void                         saveProperties(KConfig *config);
    // The login succeeded with the new password. Save the temporary password as new account password.
    void                         saveTemporaryPassword();
    // Set alert information
    void                         setAlertsInformation( bool useIdleTimer, int idleTime, bool shakeNudge, bool showNowListening );
    // Set the auto-reply message.
    void                         setAutoreplyMessage( const QString&  autoreplyMessage);
    // Set chat information
    void                         setChatInformation( bool useContactFont, bool useEmoticons, bool useFontEffects, bool showMessageTime, bool groupFollowupMessages, const QString& chatStyle );
    // Set chat logging information
    void                         setChatLoggingInformation( bool saveChats, const QString& saveChatPath, const int& directoryStructure );
    // Set email information
    void                         setEmailInformation( bool useHotmail, const QString&  emailCommand, bool showEmail, bool showOtherFolders );
    // Set the font
    void                         setFont( const QFont& font );
    // Set the font color
    void                         setFontColor( const QString&  fontColor );
    // Set the fonts and font colors
    void                         setFontInformation( const QFont& font, const QString  &fontColor, const QFont& contactFont, const QString  &contactFontColor);
    // Change the friendly name
    void                         setFriendlyName( const QString&  newName );
    // Set the emoticon folder
    void                         setEmoticonStyle( const QString& emoticonStyle );
    // Whether the account is a guest account, not permanently saved on the computer
    void                         setGuestAccount( bool guestAccount );
    // Change the status to set upon connection
    void                         setInitialStatus( QString status );
    // change the personal message
    void                         setPersonalMessage( const QString& newMessage );
    // Set basic account data
    void                         setLoginInformation( QString  handle, QString  friendlyName, QString  password = "" );
    // Set notification alerts information
    void                         setNotificationsInformation( bool doNotifyContactsOnline, bool doNotifyContactsStatus,
                                                              bool doNotifyContactsChatStart, bool doNotifyContactsChatMessages,
                                                              bool doNotifyContactsOffline, bool doNotifyEmails, int notificationDuration );
    // Set whether to show popup notifications when contacts come online.
    void                         setNotifyContactsOnline( bool doNotifyContactsOnline );
    // Set whether to show popup notifications when contacts change their msn status.
    void                         setNotifyContactsStatus( bool doNotifyContactsStatus );
    // Set whether to show popup notifications when contacts start a chat with you.
    void                         setNotifyContactsChatStart( bool doNotifyContactsChatStart );
    // Set whether to show popup notifications for all received chat messages.
    void                         setNotifyContactsChatMessages( bool doNotifyContactsChatMessages );
    // Set whether to show popup notifications when contacts go offline.
    void                         setNotifyContactsOffline( bool doNotifyContactsOffline );
    // Set whether to show popup notifications for incoming mail.
    void                         setNotifyEmails( bool doNotifyEmails );
    // Set whether or not allowed contacts are shown.
    void                         setShowAllowedContacts( bool showAllowedContacts );
    // Set whether contacts are shown by group or by online/offline.
    void                         setShowContactsByGroup( bool showContactsByGroup );
    // Set whether the display pictures of the contacts should be shown in list
    void                         setShowListPictures( bool showPictures);
    // Set whether offline contacts should be shown.
    void                         setShowOfflineContacts( bool showOfflineContacts );
    // Set whether or not removed (reverse) contacts are shown.
    void                         setShowRemovedContacts( bool showRemovedContacts );
    // Set whether or not the sidebar should be shown in chats
    void                         setShowSidebar( bool showSidebar );
    // Set a temporary password for the account for login attempts from the initialview dialog
    void                         setTemporaryPassword( const QString &temporaryPassword );
    // Set whether or not to login with this account.
    void                         setUseAutologin( bool useAutologin);
    // Set whether or not to show emoticons in chats.
    void                         setUseEmoticons( bool useEmoticons);
    // Set whether the account's passport is verified
    void                         setVerified( bool verified );
    // Set whether or not to send our display image
    void                         setShowImage( bool showImage );
    // Creates the MsnObject only if the user wants to show a display picture
    void                         updateMsnObject();

  protected: // Protected methods
    // Update the email-supported state (protected method)
    void                         setEmailSupported(bool emailSupported);

  private: // Private attributes
    // The message used to automatically reply to messages when the user is away.
    QString                      autoreplyMessage_;
    // The chat style
    QString                      chatStyle_;
    // The message font to be used by contact messages.
    QFont                        contactFont_;
    // The font color to be applied to contact messages.
    QString                      contactFontColor_;
    // The number of the custom image
    int                          customImageIndex_;
    // Whether the account is marked to be deleted
    bool                         deleted_;
    // Whether the account has been changed at all
    bool                         dirty_;
    // Show a notification when contacts come online?
    bool                         doNotifyContactsOnline_;
    // Show a notification when contacts change their status?
    bool                         doNotifyContactsStatus_;
    // Show a notification when contacts start a chat?
    bool                         doNotifyContactsChatStart_;
    // Show a notification everywhen contacts send you messages?
    bool                         doNotifyContactsChatMessages_;
    // Show a notification when contacts go offline?
    bool                         doNotifyContactsOffline_;
    // Show a notification when receiving new mail?
    bool                         doNotifyEmails_;
    // The external command used, optionally, to open email
    QString                      emailCommand_;
    // Whether the user can receive email notifications (hotmail.com or msn.com account)
    bool                         emailSupported_;
    // The directory of a selected emoticon
    QString                      emoticonStyle_;
    // The user's message font.
    QFont                        font_;
    // The user's message font color.
    QString                      fontColor_;
    // The user's friendly name, i.e. "Mike"
    QString                      friendlyName_;
    // Whether the account is a "guest account", not permanently saved on the computer
    bool                         guestAccount_;
    // Whether follow-up messages from the contact should be grouped.
    bool                         groupFollowupMessages_;
    // The user's handle, i.e. mkb137b@hotmail.com
    QString                      handle_;
    // The number of minutes of inactivity before the user status is
    //  changed to idle.
    int                          idleTime_;
    // The status to connect as
    QString                      initialStatus_;
    // The duration in seconds that a notification balloon is shown.
    int                          notificationDuration_;
    // The user's password
    QString                      password_;
    // The path to which chat files are automatically saved.
    QString                      saveChatPath_;
    // Whether or not to save chat files.
    bool                         saveChats_;
    // Whether saved chats are organized by year, month, day, or all in a single directory
    int                          savedChatDirectoryStructure_;
    // Whether the chat window shakes for nudges
    bool                         shakeNudge_;
    // Whether or not allowed contacts should be visible.
    bool                         showAllowedContacts_;
    // Whether contacts should be shown by group (true) or by online/offline status (false).
    bool                         showContactsByGroup_;
    // Whether or not to show any email notifications (the user may not
    //  have a hotmail account).
    bool                         showEmail_;
    // Whether or not time information is shown for chat messages.
    bool                         showMessageTime_;
    // Whether now listening information should be shown or not.
    bool                         showNowListening_;
    // Whether or not display pictures are shown.
    bool                         showListPictures_;
    // Whether or not offline contacts are shown.
    bool                         showOfflineContacts_;
    // Whether or not notifications should be shown for "other folder" email.
    bool                         showOtherFolders_;
    // Whether or not removed (or reverse) contacts should be visible.
    bool                         showRemovedContacts_;
    // Whether or not the sidebar should be shown in chats
    bool                         showSidebar_;
    // A temporary password for the account for login attempts from the initialview dialog
    QString                      temporaryPassword_;
    // Whether or not to autologin with this user.
    bool                         useAutologin_;
    // Whether or not the contact's font should be forced to some value.
    bool                         useContactFont_;
    // Whether or not emoticons should be used.
    bool                         useEmoticons_;
    // Whether or not to use bold/italic/underline effects in chats.
    bool                         useFontEffects_;
    // Whether to use Hotmail as the mail client or not (ie, use command given)
    bool                         useHotmail_;
    // Whether or not the user wants to go idle after a certain time.
    bool                         useIdleTimer_;
    // Whether the account's passport is verified
    bool                         verified_;
    // the user's personal message
    QString                      personalMessage_;
    // Whether or not to send our display image
    bool                         showImage_;
    // MsnObject representing our display picture if we have one
    MsnObject                    *msnObject_;

  signals: // Public signals
    // Signal that the chat style settings have been changed
    void                         changedChatStyleSettings();
    // Signal that group/contact display settings have been changed
    void                         changedDisplaySettings();
    // Signal that the user changed email display settings
    void                         changedEmailDisplaySettings();
    // Signal that the emoticon settings changed
    void                         changedEmoticonSettings();
    // Signal that font settings have changed
    void                         changedFontSettings();
    // Signal that the friendly name has changed
    void                         changedFriendlyName();
    // Signal that the msn object has changed.
    void                         changedMsnObject();
    // Signal that the now listening settings have changed
    void                         changedNowListeningSettings();
    // Signal that the personal message has changed
    void                         changedPersonalMessage();
    // Signal that timer settings were changed
    void                         changedTimerSettings();
    // Signal that the view mode has changed
    void                         changedViewMode();
    // Signal that tray notifications settings were changed
    void                         changedNotificationsSettings();
};

#endif
