/***************************************************************************
                          chatwindow.h  -  description
                             -------------------
    begin                : Wed Jan 15 22:41:32 CST 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CHATWINDOW_H
#define CHATWINDOW_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qdatetime.h>
#include <qmap.h>
#include <qptrlist.h>
#include <qstringlist.h>
#include <qtimer.h>

#include <kapp.h>

#include "chatmessage.h"
#include "chatmaster.h"
#include "chatwindowinterface.h"

// Forward declarations
class ChatInformation;
class ChatView;
class ChatMessage;
class ContactBase;
class ContactAction;
class ContactSidebar;
class CurrentAccount;
class EmoticonSidebar;
class KMessTest;
class MsnSwitchboardConnection;
class QDir;
class QLabel;
class WebSidebar;

/* ChatWindow is the main window for a chat with one or more contacts */
class ChatWindow : public ChatWindowInterface
{
  Q_OBJECT

  friend class KMessTest;

  public:
    // The construtor
                       ChatWindow( QWidget* parent=0, const char *name=0 );
    // The destructor
                      ~ChatWindow();
    // Disable the parts of the window which allow user interaction, to allow viewing conversations from disconnected accounts
    void               disable();
    // Filter to catch window activation events
    bool               eventFilter( QObject *obj, QEvent *event );
    // Initialize the object
    bool               initialize( QString handle );
    // Return the switchboard connection used by the chat window.
    MsnSwitchboardConnection * getSwitchboardConnection() const;
    // Return whether or not the contact is in this chat.
    bool               isContactInChat( const QString &handle );
    // Return whether or not this is an exclusive chat with the given contact
    bool               isExclusiveChatWithContact( const QString& handle );
    // Return whether or not the chat is at its first incoming message
    bool               isChatFirstMessage();
    // Start a chat
    void               startChat();
    // Show a status message temporary in the status dialog
    void               showStatusMessage( QString message, int duration = 5000 );
    // Show a wink received by a contact
    void               showWink( const QString &handle, const QString &filename, const QString &animationName );
    // Update the messages which contain custom emoticons
    void               updateCustomEmoticon( const QString &handle, const QString &code );

  public slots:
    // A message was received from a contact.
    void               receivedMessage(const ChatMessage &message);
    // Change the switchboard connection used by the chat window.
    void               setSwitchboardConnection( MsnSwitchboardConnection *newConnection = 0 );
    // An event requires the window to update its properties
    void               update();

  protected:
    // Restore the window properties (called by KMainWindow)
    virtual void       readProperties(KConfig *config);
    // Save the window properties (called by KMainWindow)
    virtual void       saveProperties(KConfig *config);

  protected slots:
    // put the marked text/object into the clipboard and remove
    //	it from the document
    void               editCut();
    // put the marked text/object into the clipboard
    void               editCopy();
    // Bring up a dialog to change the message font color.
    void               editFontColor();
    // Bring up a dialog to change the message font color.
    void               editFont();
    // paste the clipboard into the document
    void               editPaste();
    // enlarge the font size
    void               editZoomIn();
    // decrease the font size
    void               editZoomOut();
    // "Show status bar" was toggled.
    void               showStatusBar();
    // Send a nudge
    void               slotSendNudge();
    // Send a file to a contact.
    void               slotStartFileTransfer( QStringList fileList = QStringList() );
    //Voice conversation
    void               startConversation();
    // Start GnomeMeeting with a contact.
    void               startMeeting();
    // Change the status bar message.
    void               statusMessage(QString message);
    // Called when the "use emoticons" action is called.
    void               toggleEmoticons(bool useEmoticons);
    // Called when the "show sidebar" action is called.
    void               toggleSidebar();
    // Called when the "use spell checking" action is called.
    void               toggleSpellCheck( bool useSpellCheck );

  private: // Private methods
    // Check if the user enabled an auto-reply, and send it
    void               checkAutoReply();
    // Toggle the sidebar based on account preferences
    void               checkSidebarPreferences();
    // Choose the contact to start an invitation with.
    const QString &    chooseContact();
    // Create the contact actions
    void               createContactActions();
    // Return the contact action with the given handle
    ContactAction*     getContactActionByHandle(const QString& handle);
    // Set up the main chat view
    bool               initializeChatView();
    // Set up the contact sidebar
    bool               initializeContactSidebar();
    // Set up the current account
    bool               initializeCurrentAccount();
    // Initialize the emoticon chooser
    bool               initializeEmoticonSidebar();
    // Set up the switchboard connection
    bool               initializeSwitchboardConnection();
    // Set up the contact sidebar
    bool               initializeWebSidebar();
    // The application is exiting
    bool               queryExit();
    // The chat window is closing
    bool               queryClose();
    // Save the chat if necessary
    void               saveChatAutomatically();
    // Shake the window, for nudge/buzzer effect.
    void               shakeWindow();
    // Start blinking the caption
    void               startBlink();

  private slots: // Private slots
    // Make the caption blink if the window still doesn't have focus
    void               blinkCaption();
    // A contact joined the chat
    void               contactJoined(QString handle, QString friendlyName);
    // A contact left the chat
    void               contactLeft(QString handle, bool isChatIdle);
    // A contact is typing
    void               contactTyping(QString handle, QString friendlyName);
    // The emoticon button was pressed.
    void               emoticonButtonPressed();
    // Forward the application command to the ChatMaster
    void               forwardAppCommand(QString cookie, QString contact, QString command);
    // Return a list of the contacts in the chat to be used as window caption
    QString            getCaption();
    // Invite a contact to the chat
    void               inviteContact(QString handle);
    // Notify the user of a received nudge
    void               slotReceivedNudge(const QString &handle);
    // Save the chat according to the user's request
    void               saveChat();
    // Signal that a message could not be sent to the switchboard
    void               slotSendingFailed( QString message );
    // Update the status label text
    void               updateTypingMessage();
    // The user sent a new message
    void               userSentMessage();

  private: // Private attributes
    // A timer used to make the caption blink on new messages.
    QTimer             blinkTimer_;
    // Whether to blink to upper (true) or lower (false) case
    bool               blinkToUpper_;
    // The stored window caption
    QString            caption_;
    // The chat view widget
    ChatView          *chatView_;
    // The contact actions in the invite menu
    QPtrList<ContactAction> contactActions_;
    // The contact sidebar, which goes in the qtoolbox widget
    ContactSidebar    *contactSidebar_;
    // A pointer to the instance of the current account
    CurrentAccount    *currentAccount_;
    // The custom emoticons sidebar
    EmoticonSidebar   *customEmoticonSidebar_;
    // Whether or not the incoming message is the first incoming message
    bool               firstMessage_;
    // Whether or not the object was initialized
    bool               initialized_;
    // The last contact that left the chat.
    QString            lastContact_;
    // The last time the window was shaked
    QTime              lastShake_;
    // The object that connects to the switchboard server.
    MsnSwitchboardConnection *msnSwitchboardConnection_;
    // The list of participants to this chat
    QStringList        participants_;
    // The standard emoticons sidebar
    EmoticonSidebar   *standardEmoticonSidebar_;
    // The date/time the chat started
    QDate              startDate_;
    QTime              startTime_;
    // The status label
    QLabel            *statusLabel_;
    // A timer to time the duration that a status message is shown
    QTimer             statusTimer_;
    // Whether the sidebar should only be temporary visible
    bool               temporarySidebar_;
    // The list of users which are currently typing
    QMap<QString,QTime> typingContacts_;
    // The web sidebar, which goes in the qtoolbox widget
    WebSidebar        *webSidebar_;


  signals: // Public signals
    // Signal that there's an application command
    void               appCommand(QString cookie, QString contact, QString method);
    // Signal that the window is about to close.
    void               closing(QObject *object);
    // Signal the presence of a new incoming chat message
    void               newChatMessage( const ChatMessage &message, ChatWindow *chatWindow );
    // Signal that the ChatMaster should start a filetransfer.
    void               requestFileTransfer(const QString &handle, const QString &filename);
    // Signal that the ChatMaster should start a newmeeting session.
    void               requestNetMeeting(const QString &handle);
    // The user wants to add or remove a contact.
    void               setContactAdded( QString handle, bool isAdded );
    // The user wants to allow a contact to see his/hers online status.
    void               setContactAllowed( QString handle );
    // The user wants to block or unblock a contact. 
    void               setContactBlocked( QString handle, bool isBlocked );
};

#endif
