/***************************************************************************
                          systemtraywidget.cpp  -  description
                             -------------------
    begin                : Sun Dec 29 2002
    copyright            : (C) 2002 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "systemtraywidget.h"

#include <qdockwindow.h>
#include <qpainter.h>
#include <qtooltip.h>

#include <kdebug.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kmanagerselection.h>
#include <kmessagebox.h>

#include "config.h"
#include "currentaccount.h"


// The constructor
SystemTrayWidget::SystemTrayWidget(QWidget *parent, const char *name )
 : KSystemTray(parent,name)
{
}


/**
 * Display an Hide On Close dialog window with a trayicon screenshot
 *
 * This method, taken from the Basket Note Pads project (see the related links below), is used when
 * the user closes the main KMess contact list window. Then a custom dialog window is shown to notify
 * that the app will still run in background. This useful method adds to it a screenshot of the system
 * tray area of the users' desktop, highlighting the application's icons.
 *
 * @see http://basket.kde.org/ - Official site of the Basket Note Pads project
 * @see http://basket.kde.org/systemtray-on-close-info.php - Specific tray dialog page
 */
void SystemTrayWidget::displayCloseMessage( QString /*fileMenu*/ )
{
#if KDE_IS_VERSION( 3, 1, 90 )
  // Don't do all the computations if they are unneeded:
  if ( ! KMessageBox::shouldBeShownContinue("hideOnCloseInfo") )
      return;
#endif

  // Some values we need:
  QPoint g = mapToGlobal(pos());
  int desktopWidth  = kapp->desktop()->width();
  int desktopHeight = kapp->desktop()->height();
  int tw = width();
  int th = height();

  // We are trying to make a live screenshot of the systray icon to circle it
  //  and show it to the user. If no systray is used or if the icon is not visible,
  //  we should not show that screenshot but only a text!

  // 1. Determine if the user use a system tray area or not:
  QCString screenstr;
  screenstr.setNum(qt_xscreen());
  QCString trayatom = "_NET_SYSTEM_TRAY_S" + screenstr;
  bool useSystray = (KSelectionWatcher(trayatom).owner() != 0L);

  // 2. And then if the icon is visible too (eg. this->show() has been called):
  useSystray = useSystray && isVisible();

  // 3. Kicker (or another systray manager) can be visible but masked out of
  //    the screen (ie. on right or on left of it). We check if the icon isn't
  //    out of screen.
  QRect deskRect(0, 0, desktopWidth, desktopHeight);
  if( useSystray && ( ! deskRect.contains( g.x(), g.y() ) ) && ( ! deskRect.contains( g.x() + tw, g.y() + th ) ) )
  {
    useSystray = false;
  }


  QString message = i18n( "Closing the main window will keep KMess running in the"
                          " system tray. Use 'Quit' from the 'File' menu to quit the application." );

  // If the system tray is not visible, just show the text message
  if( ! useSystray )
  {
#ifdef KMESSDEBUG_SYSTEMTRAY
    kdDebug() << "SystemTrayWidget::displayCloseMessage() - Displaying simple dialog with no screenshot." << endl;
#endif

    KMessageBox::information( kapp->activeWindow(), message,
                              i18n( "Docking in System Tray" ),
                              "hideOnCloseInfo" );
    return;
  }

  // We are sure the systray icon is visible: ouf!

#ifdef KMESSDEBUG_SYSTEMTRAY
    kdDebug() << "SystemTrayWidget::displayCloseMessage() - Displaying dialog with tray icon screenshot: generating image..." << endl;
#endif

  // Compute size and position of the pixmap to be grabbed:
  int w = desktopWidth / 4;
  int h = desktopHeight / 9;
  int x = g.x() + tw/2 - w/2; // Center the rectange in the systray icon
  int y = g.y() + th/2 - h/2;
  if (x < 0)                 x = 0; // Move the rectangle to stay in the desktop limits
  if (y < 0)                 y = 0;
  if (x + w > desktopWidth)  x = desktopWidth - w;
  if (y + h > desktopHeight) y = desktopHeight - h;

  // Grab the desktop and draw a circle arround the icon:
  const int CIRCLE_MARGINS = 6;
  const int CIRCLE_WIDTH   = 3;
  const int SHADOW_OFFSET  = 1;
  const int IMAGE_BORDER   = 1;
  int ax = g.x() - x - CIRCLE_MARGINS - 1;
  int ay = g.y() - y - CIRCLE_MARGINS - 1;
  QPixmap shot = QPixmap::grabWindow(qt_xrootwin(), x, y, w, h);
  QPainter painter(&shot);

  painter.setPen( QPen(KApplication::palette().active().dark(), CIRCLE_WIDTH) );
  painter.drawArc(ax + SHADOW_OFFSET, ay + SHADOW_OFFSET,
                  tw + 2*CIRCLE_MARGINS, th + 2*CIRCLE_MARGINS, 0, 16*360);
  painter.setPen( QPen(Qt::red, CIRCLE_WIDTH) );
  painter.drawArc(ax, ay, tw + 2*CIRCLE_MARGINS, th + 2*CIRCLE_MARGINS, 0, 16*360);

  // Draw the pixmap over the screenshot in case a window hide the icon:
  painter.drawPixmap(g.x(), g.y(), *pixmap());
  painter.end();

  // Then, we add a border arround the image to make it more visible:
  QPixmap finalShot(w + 2*IMAGE_BORDER, h + 2*IMAGE_BORDER);
  finalShot.fill(KApplication::palette().active().foreground());

  painter.begin(&finalShot);
  painter.drawPixmap(IMAGE_BORDER, IMAGE_BORDER, shot);
  painter.end();

#ifdef KMESSDEBUG_SYSTEMTRAY
  kdDebug() << "SystemTrayWidget::displayCloseMessage() - ...Done, showing the dialog." << endl;
#endif

  // Associate source to image and show the dialog:
  QMimeSourceFactory::defaultFactory()->setPixmap("systray_shot", finalShot);

  KMessageBox::information(kapp->activeWindow(),
      "<qt><p>" + message + "</p><p><center><img source=\"systray_shot\"></center></p></qt>",
      i18n("Docking in System Tray"), "hideOnCloseInfo");

  QMimeSourceFactory::defaultFactory()->setData("systray_shot", 0L);
}



// The destructor
SystemTrayWidget::~SystemTrayWidget()
{
#ifdef KMESSDEBUG_SYSTEMTRAY
  kdDebug() << "DESTROYED SystemTrayWidget" << endl;
#endif
}



// Initialize the class
bool SystemTrayWidget::initialize()
{
  CurrentAccount *currentAccount = CurrentAccount::instance();

  connect( currentAccount, SIGNAL(       changedStatus() ),
           this,           SLOT  (       statusChanged() ) );
  connect( currentAccount, SIGNAL(       accountOnline() ),
           this,           SLOT  (       statusChanged() ) );
  connect( currentAccount, SIGNAL(    accountInvisible() ),
           this,           SLOT  (       statusChanged() ) );
  connect( currentAccount, SIGNAL(      accountOffline() ),
           this,           SLOT  (       statusChanged() ) );
  connect( currentAccount, SIGNAL( changedFriendlyName() ),
           this,           SLOT  (       statusChanged() ) );

  // Force changing the icon
  statusChanged();

  return true;
}



// Return the context menu
KPopupMenu* SystemTrayWidget::menu() const
{
  return contextMenu();
}



// Change the icon when the user's status changes
void SystemTrayWidget::statusChanged()
{
  QString     status;
  QString     newTooltip;
  QString     iconName, statusName;
  KIconLoader *loader = KGlobal::iconLoader();
  CurrentAccount *currentAccount = CurrentAccount::instance();

  status = currentAccount->getStatus();

  // Update the onlineStatusLabel
  if ( status == "AWY" )
  {
    iconName = "away";
    statusName = i18n("Away");
  }
  else if ( status == "BRB" )
  {
    iconName = "berightback";
    statusName = i18n("Be Right Back");
  }
  else if ( status == "BSY" )
  {
    iconName = "busy";
    statusName = i18n("Busy");
  }
  else if ( status == "FLN" )
  {
    iconName = "offline";
    statusName = i18n("Offline");
  }
  else if ( status == "HDN" )
  {
    iconName = "invisible";
    statusName = i18n("Invisible");
  }
  else if ( status == "IDL" )
  {
    iconName = "away";
    statusName = i18n("Idle");
  }
  else if ( status == "LUN" )
  {
    iconName = "lunch";
    statusName = i18n("Out to Lunch");
  }
  else if ( status == "NLN" )
  {
    iconName = "online";
    statusName = i18n("Online");
  }
  else if ( status == "PHN" )
  {
    iconName = "onthephone";
    statusName = i18n("On the Phone");
  }
  else
  {
    kdWarning() << "SystemTrayWidget::statusChanged() - The account had an invalid status: " << status << "." << endl;
    return;
  }

  // Update the icon
  setPixmap( loader->loadIcon( iconName, KIcon::User ) );

  // Change the tooltip
  QToolTip::remove( this );

  newTooltip = i18n( "%1 %2" ).arg( I18N_NOOP("KMess") ).arg ( VERSION );

  if( status != "FLN" )
  {
    newTooltip += i18n( "<br><b>%1</b> (%2)" ).arg( currentAccount->getFriendlyName() ).arg( statusName );
  }

  QToolTip::add( this, newTooltip );
}



#include "systemtraywidget.moc"
