/***************************************************************************
   Copyright (C) 2006
   by Marco Gulino <marco@kmobiletools.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/



#include <kmainwindow.h>
#include <qprocess.h>
#include <ktextedit.h>
#include <kprogress.h>
#include <kdebug.h>
#include <qtimer.h>
#include <qregexp.h>
#include <kpushbutton.h>
#include <kactivelabel.h>
#include <qfile.h>
#include <kurlrequester.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <qlabel.h>

#include "kbluetoothpairingwizard.h"
#include "btservice.h"

#define HCITOOL_CMD "hcitool"
#define SDPTOOL_CMD "sdptool"
#define RFCOMM_CMD "rfcomm"

KBluetoothPairingWizard::KBluetoothPairingWizard()
    : KBluetoothPairingWizardWidgetBase( 0, "KBluetoothPairingWizard" ), hcitool(0), sdptool(0), progressDialog(0),
    exenotfoundmsg(i18n("<qt>Cannot start the <b>%1</b> process.<br>Check if the <u><b>%2</b></u> package if installed property. The wizard will now exit.</q>")),
    exitedErrorMsg(i18n("<qt>Process <b>%1</b> exited with errors. Cannot continue.</qt>"))
{
    connect(hciscanStart, SIGNAL(clicked()), this, SLOT(scanDevices() ) );
    connect(sdpscanStart, SIGNAL(clicked()), this, SLOT(scanServices() ) );
    connect(this, SIGNAL(selected(const QString &)), this, SLOT(selected( const QString& ) ) );
    connect(foundList, SIGNAL(currentChanged ( QListViewItem * ) ), this, SLOT(currentChanged( QListViewItem* ) ) );
    connect(foundList, SIGNAL(clicked ( QListViewItem * ) ), this, SLOT(currentChanged( QListViewItem* ) ) );
    connect(svcListView, SIGNAL(currentChanged ( QListViewItem * ) ), this, SLOT(currentChanged( QListViewItem* ) ) );
    connect(svcListView, SIGNAL(clicked ( QListViewItem * ) ), this, SLOT(currentChanged( QListViewItem* ) ) );
    connect(rfcommURL, SIGNAL(textChanged (const QString &) ), this, SLOT(fileChanged( const QString& ) ) );
    QPixmap wizardLogoPixmap;
    wizardLogoPixmap.load( KGlobal::dirs ()->findResource("data", "kbluetoothpairingwizard/kmobilebtwizard.png") );
    wizardLogo1->setPixmap( wizardLogoPixmap );
    wizardLogo2->setPixmap( wizardLogoPixmap );
    wizardLogo3->setPixmap( wizardLogoPixmap );
}

KBluetoothPairingWizard::~KBluetoothPairingWizard()
{
}

#include "kbluetoothpairingwizard.moc"

void KBluetoothPairingWizard::scanServices()
{
    sdptool=new QProcess(QString(SDPTOOL_CMD), this);
    sdptool->addArgument("records");
    sdptool->addArgument( curDevice->address() );
    connect(sdptool, SIGNAL(processExited ()), this, SLOT(servicesFound() ) );
//     kdDebug() << "Starting process " << sdptool->args() << ":" << sdptool->start( QProcess::NotifyOnExit, QProcess::All ) << endl;
    if(!sdptool->start())
    {
        KMessageBox::error( this, exenotfoundmsg.arg(SDPTOOL_CMD).arg("Bluez") );
        done( Rejected );
        return;
    }
    progressDialog=new KProgressDialog(this, 0, i18n("Searching"), i18n("Please wait while searching for services") );
    progressDialog->setAllowCancel( true );
    progressDialog->progressBar()->setTotalSteps( 0 );
    progressDialog->progressBar()->setFormat(" ");
    progressDialog->progressBar()->setPercentageVisible(false);
    discoverycancelled=false;
    hciscanStart->setEnabled(false);
    sdpscanStart->setEnabled(false);
    progressDialog->show();
    scanTimer=new QTimer();
    connect(scanTimer, SIGNAL(timeout() ), this, SLOT(advanceScan() ) );
    scanTimer->start( 5, false);
}

/*!
    \fn KBluetoothPairingWizard::scanDevices()
 */
void KBluetoothPairingWizard::scanDevices()
{
    hcitool=new QProcess(QString(HCITOOL_CMD), this);
    hcitool->addArgument("scan");
    connect(hcitool, SIGNAL(processExited ()), this, SLOT(devicesFound() ) );
//     kdDebug() << "Starting process " << hcitool->args() << ":" << hcitool->start( QProcess::NotifyOnExit, QProcess::All ) << endl;
    if (! hcitool->start() )
    {
        KMessageBox::error( this, exenotfoundmsg.arg(HCITOOL_CMD).arg("Bluez") );
        done( Rejected );
        return;
    }
    scanTimer=new QTimer();
    connect(scanTimer, SIGNAL(timeout() ), this, SLOT(advanceScan() ) );
    scanTimer->start( 5, false);
}

void KBluetoothPairingWizard::advanceScan()
{
    if( ( hcitool && hcitool->isRunning() ) || ( sdptool && sdptool->isRunning() ) )
    {
        if(!progressDialog)
        {
            progressDialog=new KProgressDialog(this, 0, i18n("Searching"), i18n("Please wait while scanning bluetooth devices") );
            progressDialog->setAllowCancel( true );
            progressDialog->progressBar()->setTotalSteps( 0 );
            progressDialog->progressBar()->setFormat(" ");
            progressDialog->progressBar()->setPercentageVisible(false);
            discoverycancelled=false;
            progressDialog->show();
            hciscanStart->setEnabled(false);
            sdpscanStart->setEnabled(false);
        }
        if( progressDialog->wasCancelled() )
        {
            discoverycancelled=true;
            if( hcitool && hcitool->isRunning() )
                hcitool->kill();
            if( sdptool && sdptool->isRunning() )
                sdptool->kill();
            kdDebug() << "W000t! progressDialog was cancelled! :(\n";
            return;
        }
        progressDialog->progressBar()->advance(2);
    }
    else
    {
        hciscanStart->setEnabled(true);
        sdpscanStart->setEnabled(true);
        if(!progressDialog) return;
        disconnect(scanTimer, 0, 0, 0);
        scanTimer->stop();
        progressDialog->close();
        delete progressDialog;
        progressDialog=0;
    }
}

void KBluetoothPairingWizard::servicesFound()
{
    if(!sdptool->normalExit() )
    {
        if(discoverycancelled) return;
        KMessageBox::error( this, exitedErrorMsg.arg(SDPTOOL_CMD) );
        done( Rejected );
        return;
    }
    svcListView->clear();
    curDevice->serviceList()->clear();
    kdDebug() << "KBluetoothPairingWizard::servicesFound()\n";
    QString sbuffer(sdptool->readStdout());
    kdDebug() <<" command buffer: for " << sdptool->arguments().join(" ") << "\n" << sbuffer << endl;
    if(sbuffer.isEmpty())return;
    kdDebug() << sbuffer << endl;
    QRegExp regexp, regexp2;
    regexp.setPattern("[\\s]*Service Name: ([^\\n]*)");
    regexp2.setPattern( "[\\s]*Channel: ([\\d]*)");
    QStringList lines=QStringList::split( '\n', sbuffer);
    BTService newsvc;
    bool foundsvcname=false;
    for(QStringList::Iterator it=lines.begin(); it!=lines.end(); ++it)
    {
        if(regexp.search( *it )!=-1)
        {
            foundsvcname=true;
            kdDebug() << "Found! svc name=" << regexp.cap( 1) << endl;
            newsvc.setSvcName( regexp.cap( 1));
        }
        if(regexp2.search( *it) != -1 && foundsvcname )
        {
            kdDebug() << "svc channel=" << regexp2.cap( 1) << endl;
            newsvc.setChannel( regexp2.cap( 1).toInt() );
            foundsvcname=false;
            curDevice->serviceList()->append( newsvc );
        }
    }
//     for(QValueList<BlueDevice>::Iterator dit=devicesList.begin(); dit!=devicesList.end(); ++dit)
//         new BlueListViewItem((*dit), foundList );
    KListViewItem *tempitem;
    for (QValueList<BTService>::ConstIterator sit=curDevice->serviceList()->constBegin(); sit!=curDevice->serviceList()->constEnd(); ++sit)
    {
        tempitem=new ServiceListViewItem(*sit, svcListView );
        if( (*sit).svcName().contains("Dial-up") ) svcListView->setCurrentItem(tempitem);
    }
    if( ! svcListView->currentItem() ) setNextEnabled ( currentPage(), false );
    else setNextEnabled ( currentPage(), true );
}


/*!
    \fn KBluetoothPairingWizard::devicesFound(QProcess *proc, char *buffer, int buflen)
 */
void KBluetoothPairingWizard::devicesFound()
{
    if(! hcitool->normalExit())
    {
        if(discoverycancelled) return;
        KMessageBox::error( this, exitedErrorMsg.arg(HCITOOL_CMD) );
        done( Rejected );
        return;
    }
    foundList->clear();
    devicesList.clear();
    kdDebug() << "KBluetoothPairingWizard::devicesFound()\n";
//     if(!buflen) return;
    QString sbuffer(hcitool->readStdout());
    kdDebug() <<" command buffer: for " << hcitool->arguments().join(" ") << "\n" << sbuffer << endl;
    if(sbuffer.isEmpty())return;
    kdDebug() << sbuffer << endl;
    QRegExp regexp;
    regexp.setPattern("(([A-Fa-f\\d]{2,2}:?){6,6})[\\s\\t]*(.*)");
    QStringList lines=QStringList::split( '\n', sbuffer);
    for(QStringList::Iterator it=lines.begin(); it!=lines.end(); ++it)
    {
        kdDebug() << "Watching line:" << *it << ";\n" ;
        if(regexp.search( *it)==-1) continue;
        kdDebug() << "Found! cap=" << regexp.capturedTexts() << endl;
        devicesList.append( BlueDevice(regexp.cap( 3), regexp.cap( 1)) );
    }
    for(QValueList<BlueDevice>::Iterator dit=devicesList.begin(); dit!=devicesList.end(); ++dit)
        new BlueListViewItem((*dit), foundList );
}



/*!
    \fn KBluetoothPairingWizard::selected(const QString &)
 */
void KBluetoothPairingWizard::selected(const QString &)
{
    kdDebug() << "KBluetoothPairingWizard::selected()\n" << indexOf(currentPage() ) << endl;
    switch( indexOf(currentPage() ) ){
        case 0:
            if(!foundList->currentItem() )
                setNextEnabled ( currentPage(), false );
            break;
        case 1:
            scanServices();
            setNextEnabled ( currentPage(), false );
            break;
        default:
            QString summaryText=i18n("<qt>Device name: <b>%1</b><br>\n\
                    Bluetooth address: <b>%2</b><br>\n\
                    Channel: <b>%3</b> (%4)");
            summLabel->setText(summaryText.arg(curDevice->devName()).arg(curDevice->address()).arg(curSVC->channel()).arg(curSVC->svcName() ) );
//             rfcommURL->setURL("/tmp/rfcomm.conf");
            fileChanged(rfcommURL->url());
    }
}


/*!
    \fn KBluetoothPairingWizard::currentChanged ( QListViewItem * )
 */
void KBluetoothPairingWizard::currentChanged ( QListViewItem * item)
{
    if(!item)
    {
        setNextEnabled ( currentPage(), false );
        return;
    }
    ServiceListViewItem *s_item;
    BlueListViewItem *b_item;
    switch( indexOf(currentPage() ) ){
        case 0:
            kdDebug() << "KBluetoothPairingWizard::currentChanged()\n";
            b_item=(BlueListViewItem*)item;
            curDevice=b_item->device();
            setNextEnabled ( currentPage(), true );
            break;
        case 1:
            s_item=(ServiceListViewItem*)item;
            curSVC=s_item->service();
            setNextEnabled ( currentPage(), true );
            break;
    }

}


/*!
    \fn KBluetoothPairingWizard::fileChanged (const QString &)
 */
void KBluetoothPairingWizard::fileChanged (const QString &url)
{
    if( ! url.isEmpty() ) setFinishEnabled ( currentPage(), true );
    else setFinishEnabled ( currentPage(), false );
}


/*!
    \fn KBluetoothPairingWizard::done ( int r )
 */
void KBluetoothPairingWizard::done ( int r )
{
    if(r==Rejected)
    {
        QWizard::done(r);
        return;
    }
    QFile rfcommfile(rfcommURL->url() );
    if(! rfcommfile.open( IO_ReadWrite | IO_Append) )
    {
        KMessageBox::error( this, i18n("<qt>Couldn't open <b>%1</b> for read/write.</b>").arg(rfcommfile.name()) );
        return;
    }
    rfcommfile.at(0);
    QTextStream stream(&rfcommfile);
    QString line, tempLine;
    QStringList devices;
    QRegExp rfcommBusy("^[\\t\\s]*rfcomm([\\d]+)[\\s\\t]*");
    QRegExp addrBusy("^[\\t\\s]*device[\\s\\t]*(([A-Fa-f\\d]{1,2}:?){6,6})[\\s\\t]*;");
    QRegExp channelBusy("^[\\t\\s]*channel[\\s\\t]*([\\d]+)[\\s\\t]*;");
    QRegExp endDev("[^#]*\\}");
    QString dev, addr, ch;
    while (! stream.atEnd() )
    {
        dev=QString::null; addr=QString::null; ch=QString::null;
        line=stream.readLine();
        kdDebug() << "Line:" << line << endl;
        if( rfcommBusy.search(line) != -1 )
        {
            dev=rfcommBusy.cap(1);
            do {
                line=stream.readLine();
                if(addrBusy.search(line)!=-1) addr=addrBusy.cap(1);
                if(channelBusy.search(line)!=-1) ch=channelBusy.cap(1);
            } while (endDev.search(line)==-1);
            if(addr.isNull() ) continue;
            tempLine="%1|%2|ch=%3";
            tempLine=tempLine.arg(dev).arg(addr).arg(ch.toInt() );
            devices+=tempLine;
        }
    }
    kdDebug() << "Devices list:" << devices << endl;
    if( ! devices.grep( curDevice->address(), false ).grep( QString("ch=%1").arg(curSVC->channel() ) ).isEmpty() )
    {
        KMessageBox::information( this, i18n("<qt>The device <b>%1</b> with address <b>%2</b> and channel <b>%3 %4</b> is already configured as <b>rfcomm%5</b>.</qt>")
                .arg( curDevice->devName() ) .arg( curDevice->address() ).arg(curSVC->channel() ).arg(curSVC->svcName() )
                        .arg ( (devices.grep( curDevice->address(), false ).first()).section('|', 0, 0) )
                                );
        rfcommfile.close();
//         QWizard::done( Rejected );
        return;
    }
    devices= devices.gres( QRegExp("([\\d]+)\\|.*"), "\\1" );
    devices.sort();
    kdDebug() << "Index list:" << devices << endl;
    int i=0;
    for(QStringList::Iterator it=devices.begin(); it!=devices.end(); ++it)
    {
        if((*it).toInt() != i ) break;
        i++;
    }
    tempLine="rfcomm%1 {\n\tbind yes;\n\tdevice %2;\n\tchannel %3;\n\tcomment \"%4\";\n}\n";
    tempLine=tempLine.arg(i)
            .arg( curDevice->address() )
            .arg( curSVC->channel() )
            .arg( curDevice->devName() );
    stream << tempLine;
    rfcommfile.close();
    QProcess *rfcomm=new QProcess (QString(RFCOMM_CMD) );
    rfcomm->addArgument( "bind" );
    rfcomm->addArgument( QString::number( i ) );
    rfcomm->addArgument( curDevice->address() );
    rfcomm->addArgument( QString::number( curSVC->channel() ) ) ;
    if(! rfcomm->start() )
    {
        KMessageBox::error( this, exenotfoundmsg.arg(RFCOMM_CMD).arg("Bluez") );
        done( Rejected );
        return;
    }

    QDialog::done( Accepted );
    return;
}
