/***************************************************************************
   Copyright (C) 2007
   by Marco Gulino <marco@kmobiletools.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/
#include "importphonebookdlg.h"

#include <klocale.h>
#include <kabc/stdaddressbook.h>
#include <kabc/resourcefile.h>
#include <kdebug.h>
#include <qlabel.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klistview.h>
#include <qbuttongroup.h>
#include <kmessagebox.h>

#include "kmobiletoolsengine.h"

#define COL_MEMSLOT 2
#define COL_INTMEMSLOT 3
#define COL_ID 4

importPhonebookDlg::importPhonebookDlg(int availPBSlots, QWidget *parent, const char *name)
    : KDialogBase(parent, name, true, i18n("Import Addressbook"), Ok| Cancel )
{
    p_addresseeList=0;
    m_widget=new importPhonebook(this);
    setMainWidget(m_widget);
    m_widget->abcLoad->setIconSet(KGlobal::iconLoader()->loadIconSet("key_enter",KIcon::Small) );
//     connect(m_widget->fileUrl, SIGNAL(textChanged(const QString&)), this, SLOT(slotUrlChanged(const QString& )));
    connect(m_widget->fileUrl, SIGNAL(urlSelected(const QString&)), this, SLOT(slotUrlChanged(const QString& )));
    connect(m_widget->fileUrl, SIGNAL(returnPressed(const QString&)), this, SLOT(slotUrlChanged(const QString& )));
    connect(m_widget->lv_abc, SIGNAL(selectionChanged() ), this, SLOT(enableButtons() ) );
    connect(m_widget->abcLoad, SIGNAL(clicked() ), this, SLOT(slotLoadABC() ) );
    connect(m_widget->toDataCard, SIGNAL(clicked() ), this, SLOT(slotToDataCard() ) );
    connect(m_widget->toPhone, SIGNAL(clicked() ), this, SLOT(slotToPhone() ) );
    connect(m_widget->toSim, SIGNAL(clicked() ), this, SLOT(slotToSim() ) );
    connect(m_widget->dontimport, SIGNAL(clicked() ), this, SLOT(slotDontImport() ) );
    connect(m_widget->importFrom, SIGNAL(clicked( int ) ), this, SLOT(slotImportFromChanged(int ) ) );
    enableButtonOK(false);
    enableButtons();
    i_availPBSlots=availPBSlots;
    resize(550,450);
    slotImportFromChanged( m_widget->importFrom->selectedId () );
}


importPhonebookDlg::~importPhonebookDlg()
{
}


#include "importphonebookdlg.moc"


/*!
    \fn importPhonebookDlg::slotUrlChanged(const QString&)
 */
void importPhonebookDlg::slotUrlChanged(const QString& text)
{
    KABC::AddressBook *res_abc=new KABC::AddressBook();
    KABC::Resource *resfile=new KABC::ResourceFile(text);
    res_abc->addResource (resfile);
    if( ! res_abc->load() || !res_abc->allAddressees().count() >0 )
    {
        p_addresseeList=0;
        m_widget->formatLabel->setText( i18n("File format not recognized or file not found.") );
        enableButtonOK(false);
        return;
    }
    loadAddressBook(res_abc);
}

void importPhonebookDlg::slotDontImport()
{
    QListViewItemIterator it( m_widget->lv_abc, QListViewItemIterator::Selected );
    while( it.current() ){
        setListViewItemSlot( -1, it.current() );
        ++it;
    }
}

/*!
    \fn importPhonebookDlg::slotOk()
 */
void importPhonebookDlg::slotOk()
{
    if(!p_addresseeList) return;
    if( m_widget->clearPhoneBook->isChecked() )
        if(KMessageBox::warningYesNo(this, i18n("This will erase your mobile phonebook.\nAre you sure you want to continue?") ) == KMessageBox::No ) return;
    /// TODO make some more checks to avoid dummy user actions
    KDialogBase::slotOk();
}


/*!
    \fn importPhonebookDlg::slotLoadABC()
 */
void importPhonebookDlg::slotLoadABC()
{
    slotUrlChanged( m_widget->fileUrl->url() );
}


/*!
    \fn importPhonebookDlg::phoneNumbers(QValueList<KABC::PhoneNumber> list)
 */
QStringList importPhonebookDlg::phoneNumbers(QValueList<KABC::PhoneNumber> list)
{
    QStringList retval;
    QValueListIterator<KABC::PhoneNumber> it;
    for ( it = list.begin(); it != list.end(); ++it )
            retval.append( (*it).number() );
    return retval;
}


/*!
    \fn importPhonebookDlg::enableButtons()
 */
void importPhonebookDlg::enableButtons()
{
    QListViewItemIterator it( m_widget->lv_abc, QListViewItemIterator::Selected );
    if(it.current() )
    {
        if( i_availPBSlots & kmobiletoolsEngine::PB_SIM) m_widget->toSim->setEnabled(true);
        if( i_availPBSlots & kmobiletoolsEngine::PB_Phone) m_widget->toPhone->setEnabled(true);
        if( i_availPBSlots & kmobiletoolsEngine::PB_DataCard) m_widget->toDataCard->setEnabled(true);
    }
    else {
        m_widget->toSim->setEnabled(false);
        m_widget->toPhone->setEnabled(false);
        m_widget->toDataCard->setEnabled(false);
    }
}


/*!
    \fn importPhonebookDlg::slotToDataCard()
 */
void importPhonebookDlg::slotToDataCard()
{
    QListViewItemIterator it( m_widget->lv_abc, QListViewItemIterator::Selected );
    while( it.current() ){
        setListViewItemSlot( kmobiletoolsEngine::PB_DataCard, it.current() );
        ++it;
    }
}


/*!
    \fn importPhonebookDlg::slotToPhone()
 */
void importPhonebookDlg::slotToPhone()
{
    QListViewItemIterator it( m_widget->lv_abc, QListViewItemIterator::Selected );
    while( it.current() ){
        setListViewItemSlot( kmobiletoolsEngine::PB_Phone, it.current() );
        ++it;
    }
}


/*!
    \fn importPhonebookDlg::slotToSim()
 */
void importPhonebookDlg::slotToSim()
{
    QListViewItemIterator it( m_widget->lv_abc, QListViewItemIterator::Selected );
    while( it.current() ){
        setListViewItemSlot( kmobiletoolsEngine::PB_SIM, it.current() );
        ++it;
    }
}

void importPhonebookDlg::setListViewItemSlot(int memslot, QListViewItem *item)
{
    switch( memslot ){
        case -1:
            item->setText(COL_MEMSLOT, i18n( "Don't import phonenumber", "Skip"));
            break;
        case kmobiletoolsEngine::PB_SIM:
            item->setText(COL_MEMSLOT, i18n( "Short SIM memory slot descriptor", "SIM"));
            break;
        case kmobiletoolsEngine::PB_DataCard:
            item->setText(COL_MEMSLOT, i18n( "Short Datacard memory slot descriptor", "DataCard" ));
            break;
//         case kmobiletoolsEngine::PB_Phone:
        default: // Let's make PB_Phone the default choice if memslot isn't found ;-)
            item->setText(COL_MEMSLOT, i18n( "Short Phone memory slot descriptor", "Phone" ));
            break;
    }
    item->setText(COL_INTMEMSLOT, QString::number(memslot));
    QValueListIterator<KABC::Addressee> it;
    for( it=p_addresseeList->begin(); it!=p_addresseeList->end(); ++it)
        if( (*it).uid() == item->text(COL_ID) ) (*it).insertCustom("KMobileTools", "memslot", QString::number(memslot) );
}



/*!
    \fn importPhonebookDlg::slotImportFromChanged()
 */
void importPhonebookDlg::slotImportFromChanged( int id )
{
    if(id)
    {
        m_widget->fileUrl->setEnabled(true);
        m_widget->abcLoad->setEnabled(true);
    } else
    {
        m_widget->fileUrl->setEnabled(false);
        m_widget->abcLoad->setEnabled(false);
        loadAddressBook(KABC::StdAddressBook::self());
    }
}

void importPhonebookDlg::loadAddressBook(KABC::AddressBook *addressBook)
{
    if(! addressBook->allAddressees().count() ) return;
    m_widget->lv_abc->clear();
    enableButtonOK(true);
    p_addresseeList=new KABC::AddresseeList(addressBook->allAddressees() );
    m_widget->formatLabel->setText( i18n("%2 addressbook, with %1 contacts.").arg(p_addresseeList->count() ) );
    if(p_addresseeList->first().custom("KMobileTools","memslot").toInt() )
    {
        m_widget->formatLabel->setText(m_widget->formatLabel->text().arg("KMobileTools") );
        b_kmobiletoolsFormat=true;
    }
    else
    {
        m_widget->formatLabel->setText(m_widget->formatLabel->text().arg("KDE") ) ;
        b_kmobiletoolsFormat=false;
    }
    KListViewItem *item;
    QValueListIterator<KABC::Addressee> it;
    for ( it = p_addresseeList->begin(); it != p_addresseeList->end(); ++it )
    {
        item=new KListViewItem( m_widget->lv_abc, (*it).formattedName(), phoneNumbers( (*it).phoneNumbers() ).join(",") );
        setListViewItemSlot( (*it).custom("KMobileTools","memslot").toInt(), item );
        item->setText(COL_ID, (*it).uid() );
    }
}

