/* This file is part of the KDE project
   Copyright (c) 2004 Kevin Ottens <ervin ipsquad net>
   Rearranged by Marco Gulino <marco@kmobiletools.org> for the mobile protocol

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <qcstring.h>
#include <qsocket.h>
#include <qdatetime.h>
#include <qbitarray.h>

#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include "deviceIFace_stub.h"

#include <kapplication.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kinstance.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <kurl.h>
#include <ksock.h>
#include <kmobiletools_devices.h>

#include "mobile.h"

using namespace KIO;


kio_mobileProtocol::kio_mobileProtocol(const QCString &pool_socket, const QCString &app_socket)
    : SlaveBase("kio_mobile", pool_socket, app_socket)
{
    kdDebug() << "kio_mobileProtocol::kio_mobileProtocol()" << endl;
    m_dcop.attach();
}


kio_mobileProtocol::~kio_mobileProtocol()
{
    kdDebug() << "kio_mobileProtocol::~kio_mobileProtocol()" << endl;
}

// From here, it's copied and adapted from kdebase/kioslave/system
void kio_mobileProtocol::listDir(const KURL &url)
{
    kdDebug() << "MobileProtocol::listDir: " << url << endl;

    if ( url.path().length() <= 1 )
    {
        listRoot();
        return;
    }
    QString host=QStringList::split("/",url.path() ).first();
    KMobileTools::DevicesConfig *hostCFG=KMobileTools::DevicesConfig::prefs(host);
    hostCFG->readConfig();
    kdDebug() << "Reading config\n";
    kdDebug() << "FSType:" << hostCFG->fstype() << endl;
    KURL target;
    switch( hostCFG->fstype() ){
        case 1:
            kdDebug() << "P2K Filesystem\n";
            target.setProtocol("p2k");
            target.setUser( host);
            target.setHost(QString("%1.%2")
                    .arg(hostCFG->p2k_vendor())
                    .arg(hostCFG->p2k_prodid()) );
            break;
        case 2:
            kdDebug() << "ObexFTP Filesystem\n";
            target.setProtocol("obex2");
            switch( hostCFG->obex_transp() ){
                case 0:
                    target.setHost( "irda"); break;
                case 1:
                    target.setHost("inet"); break;
                case 2:
                    target.setHost("custom"); break;
                case 3:
                default:
                    target.setHost("bluetooth"); break;
                case 4:
                    target.setHost("usb"); break;
            }
            kdDebug() << "Obex transport: " << hostCFG->obex_transp() << "; device: " << hostCFG->obex_device() << "; port: " << hostCFG->obex_port() << endl;
            target.setUser(hostCFG->obex_device().replace("/", "\\") );
            if(hostCFG->obex_port()) target.setPort(hostCFG->obex_port());
            break;
    }
    kdDebug() << "Resulting URL: \"" << target << "\"; is valid: " << target.isValid() << endl;
//     target.setProtocol("p2k");
//     target.setHost(host);
//     kdDebug() << "Path: " << url.path() << "; host: " << url.host() << endl;
//     kdDebug() << "possible redirection target : " << target << endl;
    if( target.isValid() )
    {
        DeviceIFace_stub dcopDevice(&m_dcop, QCString("kmobiletools"), host.utf8() );
        kdDebug() << "launching switch2filesystem() to `" << dcopDevice.app() << "` -- `" << dcopDevice.obj() << "`" << endl;
        dcopDevice.switch2filesystem();
        kdDebug() << "Call failed: " << (dcopDevice.status() == DCOPStub::CallFailed) << endl;
        redirection(target);
        finished();
        return;
    }

    error(KIO::ERR_MALFORMED_URL, url.prettyURL());
}

void kio_mobileProtocol::stat(const KURL &url)
{
    kdDebug() << "MobileProtocol::stat: " << url << endl;

    QString path = url.path();
    if ( path.isEmpty() || path == "/" )
    {
                // The root is "virtual" - it's not a single physical directory
        KIO::UDSEntry entry;
        m_impl.createTopLevelEntry( entry );
        statEntry( entry );
        finished();
        return;
    }


//     error(KIO::ERR_MALFORMED_URL, url.prettyURL());
}


void kio_mobileProtocol::listRoot()
{
    KIO::UDSEntry entry;

    KIO::UDSEntryList mobile_entries;
    bool ok = m_impl.listRoot(mobile_entries);

    if (!ok)
    {
        error( m_impl.lastErrorCode(), m_impl.lastErrorMessage() );
        return;
    }

    totalSize(mobile_entries.count()+1);

    m_impl.createTopLevelEntry(entry);
    listEntry(entry, false);

    KIO::UDSEntryListIterator it = mobile_entries.begin();
    KIO::UDSEntryListIterator end = mobile_entries.end();

    for(; it!=end; ++it)
    {
        listEntry(*it, false);
    }

    entry.clear();
    listEntry(entry, true);

    finished();
}

// end of the system kioslave part

extern "C"
{
    int kdemain(int argc, char **argv)
    {
        KInstance instance( "kio_mobile" );
        
        kdDebug(7101) << "*** Starting kio_mobile " << endl;
        
        if (argc != 4) {
            kdDebug(7101) << "Usage: kio_mobile  protocol domain-socket1 domain-socket2" << endl;
            exit(-1);
        }
        
        kio_mobileProtocol slave(argv[2], argv[3]);
        slave.dispatchLoop();
        
        kdDebug(7101) << "*** kio_mobile Done" << endl;
        return 0;
    }
} 
