/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <math.h>
#include "ipv4.h"

/* Table for cidr <-> netmask translation */
static unsigned int cidr2netmask[] =
  {
    0x00000000, /* 0  - 0.0.0.0         */
    0x00000080, /* 1  - 128.0.0.0       */
    0x000000C0, /* 2  - 192.0.0.0       */
    0x000000E0, /* 3  - 224.0.0.0       */
    0x000000F0, /* 4  - 240.0.0.0       */
    0x000000F8, /* 5  - 248.0.0.0       */
    0x000000FC, /* 6  - 252.0.0.0       */
    0x000000FE, /* 7  - 254.0.0.0       */
    0x000000FF, /* 8  - 255.0.0.0       */
    0x000080FF, /* 9  - 255.128.0.0     */
    0x0000C0FF, /* 10 - 255.192.0.0     */
    0x0000E0FF, /* 11 - 255.224.0.0     */
    0x0000F0FF, /* 12 - 255.240.0.0     */
    0x0000F8FF, /* 13 - 255.248.0.0     */
    0x0000FCFF, /* 14 - 255.252.0.0     */
    0x0000FEFF, /* 15 - 255.254.0.0     */
    0x0000FFFF, /* 16 - 255.255.0.0     */
    0x0080FFFF, /* 17 - 255.255.128.0   */
    0x00C0FFFF, /* 18 - 255.255.192.0   */
    0x00E0FFFF, /* 19 - 255.255.224.0   */
    0x00F0FFFF, /* 20 - 255.255.240.0   */
    0x00F8FFFF, /* 21 - 255.255.248.0   */
    0x00FCFFFF, /* 22 - 255.255.252.0   */
    0x00FEFFFF, /* 23 - 255.255.254.0   */
    0x00FFFFFF, /* 24 - 255.255.255.0   */
    0x80FFFFFF, /* 25 - 255.255.255.128 */
    0xC0FFFFFF, /* 26 - 255.255.255.192 */
    0xE0FFFFFF, /* 27 - 255.255.255.224 */
    0xF0FFFFFF, /* 28 - 255.255.255.240 */
    0xF8FFFFFF, /* 29 - 255.255.255.248 */
    0xFCFFFFFF, /* 30 - 255.255.255.252 */
    0xFEFFFFFF, /* 31 - 255.255.255.254 */
    0xFFFFFFFF, /* 32 - 255.255.255.255 */
  };

IPv4::IPv4()
{
}

IPv4::IPv4(const QString & s)
{
  set(s);
}

IPv4::~IPv4()
{
}

bool IPv4::isNetwork()
{
  if (m_addr.ar[3] == 0)
    return true;
  else
    return false;
}

bool IPv4::isBroadcast()
{
  if (m_addr.ar[3] == 255)
    return true;
  else
    return false;
}

QString IPv4::address() const
{
  return(QString( "%1.%2.%3.%4" )
	 .arg( m_addr.ar[0] )
	 .arg( m_addr.ar[1] )
	 .arg( m_addr.ar[2] )
	 .arg( m_addr.ar[3] ));
}

QString IPv4::netmask() const
{
  return(QString( "%1.%2.%3.%4" )
	 .arg( m_mask.ar[0] )
	 .arg( m_mask.ar[1] )
	 .arg( m_mask.ar[2] )
	 .arg( m_mask.ar[3] ));
}

QString IPv4::cidrstr(addrType atype) const
{
  QString addr;
  if (atype == Network)
    addr = network();
  else if (atype == Address)
    addr = address();
  else if (atype == Broadcast)
    addr = broadcast();
  return (QString("%1/%2").arg(addr).arg(m_clen));
}

QString IPv4::network() const
{
  return(QString( "%1.%2.%3.%4" )
	 .arg( m_netw.ar[0] )
	 .arg( m_netw.ar[1] )
	 .arg( m_netw.ar[2] )
	 .arg( m_netw.ar[3] ));
}

QString IPv4::broadcast() const
{
  return(QString( "%1.%2.%3.%4" )
	 .arg( m_brdc.ar[0] )
	 .arg( m_brdc.ar[1] )
	 .arg( m_brdc.ar[2] )
	 .arg( m_brdc.ar[3] ));
}

unsigned int IPv4::cidrlen() const
{
  return m_clen;
}

bool IPv4::containedBy(const IPv4 &i)
{
  if ((m_netw.whole & i.m_mask.whole) == i.m_netw.whole)
    {
      return true;
    }
  else
    {
      return false;
    }
}

bool IPv4::contains(const IPv4 &i)
{
  if ((i.m_netw.whole & m_mask.whole) == m_netw.whole)
    {
      return true;
    }
  else
    {
      return false;
    }
}

void IPv4::truncate()
{
  m_addr.whole = m_netw.whole;
}

void IPv4::lockMask()
{
  m_clen = 32;
  m_netw.whole = m_addr.whole;
  m_brdc.whole = m_addr.whole;
  m_mask.whole = cidr2netmask[32];
}

bool IPv4::set(const QString &sz, int clen) {
  QStringList l1, l2;
  bool ok;
  QString s = sz.simplifyWhiteSpace();
  l1 = QStringList::split(QRegExp(" */ *"), s);
  if (l1.count() == 2) {
    m_clen = l1[1].toUInt(&ok);
    if (!ok) {
      qWarning("Invalid CIDR suffix '%s'", l1[1].ascii());
      return FALSE;
    }
  } else {
    m_clen = 0;
  }

  if (clen != 0) // Override CLEN if specified (Host in a subnet)
    m_clen = clen;

  l2 = QStringList::split('.', l1[0]);
  if (l2.count() != 4) {
    qWarning("Invalid IPv4 address '%s'", l1[0].ascii());
    return FALSE;
  }

  m_addr.ar[0] = (unsigned char) l2[0].toUInt (&ok);
  if (!ok) {
    qWarning("Invalid IPv4 component '%s'", l2[0].ascii());
    return FALSE;
  }

  m_addr.ar[1] = (unsigned char) l2[1].toUInt (&ok);
  if (!ok) {
    qWarning("Invalid IPv4 component '%s'", l2[1].ascii());
    return FALSE;
  }

  m_addr.ar[2] = (unsigned char) l2[2].toUInt (&ok);
  if (!ok) {
    qWarning("Invalid IPv4 component '%s'", l2[2].ascii());
    return FALSE;
  }

  m_addr.ar[3] = (unsigned char) l2[3].toUInt (&ok);
  if (!ok) {
    qWarning("Invalid IPv4 component '%s'", l2[3].ascii());
    return FALSE;
  }

  if (m_clen == 0) {
    if (m_addr.ar[0] == 10)
      m_clen = 8;
    else if ((m_addr.ar[0] == 172) && (m_addr.ar[1] == 16))
      m_clen = 12;
    else if ((m_addr.ar[0] == 192) && (m_addr.ar[1] == 168))
      m_clen = 16;
    else if ((m_addr.ar[0] & 0x00000080) == 0x00000000) /* Class A */
      m_clen = 8;
    else if ((m_addr.ar[0] & 0x000000C0) == 0x00000080) /* Class B */
      m_clen = 16;
    else if ((m_addr.ar[0] & 0x000000E0) == 0x000000C0) /* Class C */
      m_clen = 24;
    else                                           /* Class D/E */
      m_clen = 32;
  }
  m_mask.whole = cidr2netmask[m_clen];
  m_netw.whole = m_addr.whole & m_mask.whole;
  m_brdc.whole = m_netw.whole ^ (m_mask.whole ^ 0xFFFFFFFF);

  return TRUE;
}

// IPv4 Wildcard expansion: (aka Perl-Like fun with Strings)
QStringList IPv4::expandIPTerm(const QString &term) // Expands a term such as * or 1,16-32
{
  QStringList ret;
  QRegExp isRange("^(\\d+)\\-(\\d+)$");
  if (term.isEmpty())
    return ret;
  if (term == "*") { // Damn (1..254) looks soo easy in Perl doesn't it?
    int i;
    for (i=1; i<=254; i++)
      ret.append(QString("%1").arg(i));
  } else {
    if (term.contains(',')) {
      QStringList terms = QStringList::split(',', term);
      for (QStringList::Iterator it=terms.begin(); it!=terms.end(); ++it) {
	if (isRange.search(*it) != -1) {
	  int i = isRange.cap(1).toInt();
	  int j = isRange.cap(2).toInt();
	  for (; i<=j; i++)
	    ret.append(QString("%1").arg(i));
	} else {
	  ret.append(QString("%1").arg((*it).toInt()));
	}
      }
    } else {
      if (isRange.search(term) != -1) {
	int i = isRange.cap(1).toInt();
	int j = isRange.cap(2).toInt();
	for (; i<=j; i++)
	  ret.append(QString("%1").arg(i));
      } else {
	ret.append(QString("%1").arg(term.toInt()));
      }
    }
  }
  return ret;
}

QStringList IPv4::expandIP(const QString &ip) // Pure convenience
{
  QStringList foo = QStringList::split('.', ip);
  if (foo.count() == 4) {
    return IPv4::expandIP(foo);
  } else {
    qWarning("IPv4::expandIP: Invalid IP wildcard expression: %s", ip.ascii());
    return 0;
  }
}

IPv4 *IPv4::firstAddress()
{
  IPv4 *ip = new IPv4();
  ip->m_addr = m_netw;
  ip->m_mask = m_mask;
  ip->m_netw = m_netw;
  ip->m_brdc = m_brdc;
  ip->m_clen = m_clen;

  ip->m_addr.whole++;

  return ip;
}

IPv4 *IPv4::lastAddress()
{
  IPv4 *ip = new IPv4();
  ip->m_addr = m_brdc;
  ip->m_mask = m_mask;
  ip->m_netw = m_netw;
  ip->m_brdc = m_brdc;
  ip->m_clen = m_clen;

  ip->m_addr.whole--;

  return ip;
}

int IPv4::useableCount() // const IPv4 &ip)
{
  return (int) pow(2, (float) 32-m_clen) - 2;
}

int IPv4::countIP(const QString &ip)
{
  static QRegExp isCidr("\\d+\\.\\d+\\.\\d+\\.\\d+\\/(\\d+)");
  if (isCidr.search(ip) != -1)
    {
      int hostBits = 32-isCidr.cap(1).toInt();
      return (int) pow(2, (float) hostBits) - 2;
    }
  else
    {
      QStringList foo = QStringList::split('.', ip);
      if (foo.count() == 4) {
	return IPv4::countIP(foo);
      } else {
	qWarning("IPv4::countIP: Invalid IP wildcard expression: %s", ip.ascii());
	return 0;
      }
    }
}

QStringList IPv4::expandIP(const QStringList &ip) // Expands an IP such as: 10.*.1,16-32.*
{                                                  // into a stringlist of proper IPv4 Strings.
  QStringList a, b, c, d, ret;
  QStringList::Iterator i, j, k, l;
  if (ip.count() != 4)                             // IP should be a 4 element stringlist 
    return ret;
  a = IPv4::expandIPTerm(ip[0]);
  b = IPv4::expandIPTerm(ip[1]);
  c = IPv4::expandIPTerm(ip[2]);
  d = IPv4::expandIPTerm(ip[3]);
  if ((a.empty()) || (b.empty()) || (c.empty()) || (d.empty()))
    {
      qWarning("IPv4::expandIP: Invalid term in IP");
      return ret;
    }
  if ((a.count() > 8) &&
      (b.count() > 8) &&
      (c.count() > 8) &&
      (d.count() > 8))
    {
      qWarning("IPv4::expandIP: Huh-uh, you don't need THAT many IPs!");
      return ret;
    }

  for (i=a.begin(); i!=a.end(); ++i)
    for (j=b.begin(); j!=b.end(); ++j)
      for (k=c.begin(); k!=c.end(); ++k)
	for (l=d.begin(); l!=d.end(); ++l)
	  ret.append(QString("%1.%2.%3.%4")
		     .arg(*i).arg(*j).arg(*k).arg(*l));
  return ret;
}

int IPv4::countIP(const QStringList &ip) // like above, but counts terms instead of making a list
{
  QStringList a, b, c, d;
  int ret=0;
  QStringList::Iterator i, j, k, l;
  if (ip.count() != 4)
    return ret;
  a = IPv4::expandIPTerm(ip[0]);
  b = IPv4::expandIPTerm(ip[1]);
  c = IPv4::expandIPTerm(ip[2]);
  d = IPv4::expandIPTerm(ip[3]);
  if ((a.empty()) || (b.empty()) || (c.empty()) || (d.empty()))
    {
      qWarning("IPv4::countIP: Invalid term in IP");
      return ret;
    }

  int ac = a.count();
  int bc = b.count();
  int cc = c.count();
  int dc = d.count();
  ret = ac*bc*cc*dc;

  return ret;
}
