/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef IPV4_H
#define IPV4_H

#include <qstring.h>
#include <qstringlist.h>
#include <qregexp.h>

class IPv4 {
public:
  IPv4();
  IPv4(const QString &);
  ~IPv4();
  typedef union {
    unsigned char ar[4];
    unsigned int  whole;
  } quad;

  enum addrType {
    Network=1,
    Address,
    Broadcast
  };

  bool set(const QString &, int clen=0);
  QString address() const;
  QString netmask() const;
  QString cidrstr(addrType=Address) const;
  QString network() const;
  QString broadcast() const;
  unsigned int cidrlen() const;
  void truncate();
  void lockMask();

  bool contains(const IPv4 &);
  bool containedBy(const IPv4 &);
  bool isNetwork();
  bool isBroadcast();

  int useableCount();
  IPv4 *firstAddress();
  IPv4 *lastAddress();

  static QStringList expandIP(const QString &);
  static QStringList expandIP(const QStringList &);
  static QStringList expandIPTerm(const QString &);
  static int countIP(const QString &);
  static int countIP(const QStringList &);

  friend inline bool operator<(const IPv4 &i1, const IPv4 &i2);
  friend inline bool operator<=(const IPv4 &i1, const IPv4 &i2);
  friend inline bool operator==(const IPv4 &i1, const IPv4 &i2);
  friend inline bool operator!=(const IPv4 &i1, const IPv4 &i2);
  friend inline bool operator>(const IPv4 &i1, const IPv4 &i2);
  friend inline bool operator>=(const IPv4 &i1, const IPv4 &i2);

private:
  quad m_addr;
  quad m_mask;
  quad m_netw;
  quad m_brdc;
  unsigned int m_clen;
};

inline bool operator<(const IPv4 &i1, const IPv4 &i2)
{
  return((i1.m_addr.ar[0] < i2.m_addr.ar[0] ||
	  i1.m_addr.ar[1] < i2.m_addr.ar[1] ||
	  i1.m_addr.ar[2] < i2.m_addr.ar[2] ||
	  i1.m_addr.ar[3] < i2.m_addr.ar[3]) ||
	 i1.m_clen > i2.m_clen);
}

inline bool operator<=(const IPv4 &i1, const IPv4 &i2)
{
  return((i1.m_addr.ar[0] <= i2.m_addr.ar[0] ||
	  i1.m_addr.ar[1] <= i2.m_addr.ar[1] ||
	  i1.m_addr.ar[2] <= i2.m_addr.ar[2] ||
	  i1.m_addr.ar[3] <= i2.m_addr.ar[3]) ||
	 i1.m_clen >= i2.m_clen);
}

inline bool operator==(const IPv4 &i1, const IPv4 &i2)
{
  return((i1.m_addr.ar[0] == i2.m_addr.ar[0] &&
	  i1.m_addr.ar[1] == i2.m_addr.ar[1] &&
	  i1.m_addr.ar[2] == i2.m_addr.ar[2] &&
	  i1.m_addr.ar[3] == i2.m_addr.ar[3]) &&
	 i1.m_clen == i2.m_clen);
}

inline bool operator!=(const IPv4 &i1, const IPv4 &i2)
{
  return((i1.m_addr.ar[0] != i2.m_addr.ar[0] ||
	  i1.m_addr.ar[1] != i2.m_addr.ar[1] ||
	  i1.m_addr.ar[2] != i2.m_addr.ar[2] ||
	  i1.m_addr.ar[3] != i2.m_addr.ar[3]) ||
	 i1.m_clen != i2.m_clen);
}

inline bool operator>(const IPv4 &i1, const IPv4 &i2)
{
  return((i1.m_addr.ar[0] > i2.m_addr.ar[0] ||
	  i1.m_addr.ar[1] > i2.m_addr.ar[1] ||
	  i1.m_addr.ar[2] > i2.m_addr.ar[2] ||
	  i1.m_addr.ar[3] > i2.m_addr.ar[3]) ||
	 i1.m_clen < i2.m_clen);
}

inline bool operator>=(const IPv4 &i1, const IPv4 &i2)
{
  return((i1.m_addr.ar[0] >= i2.m_addr.ar[0] ||
	  i1.m_addr.ar[1] >= i2.m_addr.ar[1] ||
	  i1.m_addr.ar[2] >= i2.m_addr.ar[2] ||
	  i1.m_addr.ar[3] >= i2.m_addr.ar[3]) ||
	 i1.m_clen <= i2.m_clen);
}

#endif // IPV4_H
