/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qobjectlist.h>
#include <qdir.h>
#include <qpainter.h>

#include <kdeversion.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <kconfig.h>
#include <kstdaction.h>
#include <kaction.h>
#include <kactionclasses.h>
#include <kkeydialog.h>
#include <kedittoolbar.h>
#include <kfiledialog.h>
#include <kio/netaccess.h>

#include "knetmap.h"
#include "knetmapview.h"
#include "knetmapdoc.h"
#include "knetmaphost.h"
#include "knetmapsubnet.h"
#include "knetmapconsole.h"
#include "knetmappref.h"
#include "knetmapnmapscanner.h"
#include "knetmapscanwidget.h"
#include "knetmapgeneralprefpage.h"

#define ID_STATUS_MSG 1

KNetmapApp::KNetmapApp(QWidget *,
		       const char *name)
  : KMainWindow(0, name)
{
  config = kapp->config();
  console = new KNetmapConsole();

  ///////////////////////////////////////////////////////////////////
  initStatusBar();

  // TODO this shouldn't be hardcoded
  (void)new KNetmapNmapScanner(this);

  doc = new KNetmapDoc(this);
  initActions();
  doc->newDocument();
  initView();

  readOptions();

  if (m_rootcmd.isEmpty())
    setRootCmd((const QString &)"kdesu -t --");

  ///////////////////////////////////////////////////////////////////
  // disable actions at startup
  // TODO replace these with <states> in knetmapui.rc
  fileSave->setEnabled(false);
  fileSaveAs->setEnabled(false);

  setConsoleShown(false);
}

KNetmapApp::~KNetmapApp()
{
}

// Accessors
bool KNetmapApp::autoopen() const { return m_lastdoc; }
const QStringList KNetmapApp::rootCmd() const { return m_rootcmd; }

void KNetmapApp::setAutoopen(bool b)
{
  m_lastdoc = b;
}

void KNetmapApp::setRootCmd(const QStringList &l)
{
  m_rootcmd = l;
}

void KNetmapApp::setRootCmd(const QString &s)
{
  if (s.isEmpty())
    m_rootcmd.clear();
  else
    m_rootcmd = QStringList::split(QRegExp("\\s+"), s);
}

void KNetmapApp::changeState(QString state)
{
  stateChanged(state);
  if (state == "subnet_selected")
    {
      menuBar()->setItemVisible(sub_menu_id,  true);
      menuBar()->setItemVisible(host_menu_id, false);
    }
  else if (state == "host_selected")
    {
      menuBar()->setItemVisible(sub_menu_id,  false);
      menuBar()->setItemVisible(host_menu_id, true);
    }
  else if (state == "default")
    {
      menuBar()->setItemVisible(sub_menu_id,  false);
      menuBar()->setItemVisible(host_menu_id, false);
    }
}

void KNetmapApp::setConsoleShown(bool b_conShow)
{
  viewConsole->setChecked(b_conShow);
  viewConsole->setEnabled(b_conShow);
  slotViewConsole();
}

void KNetmapApp::initActions()
{
  fileNew = KStdAction::openNew(this, SLOT(slotFileNew()), actionCollection());
  fileOpen = KStdAction::open(this, SLOT(slotFileOpen()), actionCollection());
  fileOpenRecent = KStdAction::openRecent(this, SLOT(slotFileOpenRecent(const KURL&)), actionCollection());
  fileSave = KStdAction::save(this, SLOT(slotFileSave()), actionCollection());
  fileSaveAs = KStdAction::saveAs(this, SLOT(slotFileSaveAs()), actionCollection());
  fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
  viewToolBar = KStdAction::showToolbar(this, SLOT(slotViewToolBar()), actionCollection());
  viewStatusBar = KStdAction::showStatusbar(this, SLOT(slotViewStatusBar()), actionCollection());
  viewConsole = new KToggleAction
    (i18n("Show &Console"), 0, this, SLOT(slotViewConsole()), actionCollection(), "view_console");
  deleteSelected = new KAction(i18n("&Delete Selected"), 0,
			       doc, SLOT(deleteSelected()),
			       actionCollection(), "deleteSelected");

  fileNew->setStatusText(i18n("Creates a new document"));
  fileOpen->setStatusText(i18n("Opens an existing document"));
  fileOpenRecent->setStatusText(i18n("Opens a recently used file"));
  fileSave->setStatusText(i18n("Saves the actual document"));
  fileSaveAs->setStatusText(i18n("Saves the actual document as..."));
  fileQuit->setStatusText(i18n("Quits the application"));
  viewToolBar->setStatusText(i18n("Enables/disables the toolbar"));
  viewStatusBar->setStatusText(i18n("Enables/disables the statusbar"));
  viewConsole->setStatusText("Shows/hides the console");

  KStdAction::keyBindings(this, SLOT(optionsConfigureKeys()), actionCollection());
  KStdAction::configureToolbars(this, SLOT(optionsConfigureToolbars()), actionCollection());
  KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection());

  m_scanWidget = new KNetmapScanWidget(this);
  scanAction = new KWidgetAction(m_scanWidget, "Scan", 0, 0, 0, actionCollection(), "scan_widget");
  scanAction->setAutoSized(true);

  // Now give each scanner a chance to build the gui
  QObjectList    *scanners = queryList("KNetmapScanner");
  QObjectListIt  sit(*scanners);
  KNetmapScanner *scanner;
  while ((scanner = (KNetmapScanner *)sit.current()) != 0)
    {
      ++sit;
      scanner->initActions();
    }
  delete scanners;

  updateGUI();

  /*
     <rantMode level="veryHigh">
       Damn but this is fucked up!
       Why couldn't we just <state><hide><Menu name="foo" /></hide></state> ???
       I mean, tell me I'm not the first to think of this!?!?
       Maybe there is a better way to do this, and I just haven't seen it
       wouldn't know since I've never been able to find strong docs on KXML*
       not even a nice page on what all I can put in my appui.rc file!
       everything in knetmapui.rc had to be gleaned from google and other projects.
     </rantMode>
  */

  int i, c;
  c = menuBar()->count();
  for(i=0; i<c; i++)
    {
      int id = menuBar()->idAt(i);
      if (id != -1)
	{
	  QMenuItem *item = menuBar()->findItem(id);
	  if (item)
	    {
	      if (item->text() == "S&ubnet")
		sub_menu_id = id;
	      else if (item->text() == "H&ost")
		host_menu_id = id;
	    }
	}
    }

  changeState("default");
}


void KNetmapApp::initStatusBar()
{
  statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG);
}

void KNetmapApp::initView()
{ 
  ////////////////////////////////////////////////////////////////////
  // create the main widget here that is managed by KTMainWindow's view-region and
  // connect the widget to your document to display document contents.

  view = new KNetmapView(this);
  doc->addView(view);
  setCentralWidget(view);	
  setCaption(doc->URL().fileName(),false);

}

void KNetmapApp::openDocumentFile(const KURL& url)
{
  slotStatusMsg(i18n("Opening file..."));

  doc->openDocument(url);
  fileOpenRecent->addURL(url);
  slotStatusMsg(i18n("Ready."));
}


KNetmapDoc *KNetmapApp::getDocument() const
{
  return doc;
}

void KNetmapApp::saveOptions()
{	
  config->setGroup("General Options");
  config->writeEntry("Version", VERSION);
  config->writeEntry("Geometry", size());
  config->writeEntry("Show Toolbar", viewToolBar->isChecked());
  config->writeEntry("Show Statusbar",viewStatusBar->isChecked());
  config->writeEntry("ToolBarPos", (int) toolBar("mainToolBar")->barPos());
  config->writeEntry("rootCommand", m_rootcmd, ' ');

  fileOpenRecent->saveEntries(config,"Recent Files");

  // Last document
  config->writeEntry("openLastFile", m_lastdoc);
  config->writeEntry("lastFile", doc->URL().url());

  m_scanWidget->saveConfig(config);

  // Now give each scanner a chance to save config
  QObjectList    *scanners = queryList("KNetmapScanner");
  QObjectListIt  sit(*scanners);
  KNetmapScanner *scanner;
  while ((scanner = (KNetmapScanner *)sit.current()) != 0)
    {
      ++sit;
      scanner->saveConfig(config);
    }
  delete scanners;
}


void KNetmapApp::readOptions()
{
	
  config->setGroup("General Options");

  // bar status settings
  bool bViewToolbar = config->readBoolEntry("Show Toolbar", true);
  viewToolBar->setChecked(bViewToolbar);
  slotViewToolBar();

  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
  viewStatusBar->setChecked(bViewStatusbar);
  slotViewStatusBar();

  // bar position settings
  KToolBar::BarPosition toolBarPos;
  toolBarPos=(KToolBar::BarPosition) config->readNumEntry("ToolBarPos", KToolBar::Top);
  toolBar("mainToolBar")->setBarPos(toolBarPos);
	
  // initialize the recent file list
  fileOpenRecent->loadEntries(config,"Recent Files");

  QSize size=config->readSizeEntry("Geometry");
  if(!size.isEmpty())
    {
      resize(size);
    }

  m_rootcmd = config->readListEntry("rootCommand", ' ');

  // Last document
  m_lastdoc = config->readBoolEntry("openLastFile");
  if (m_lastdoc && config->hasKey("lastFile"))
    {
      doc->openDocument(KURL(config->readEntry("lastFile")));
      setCaption(doc->URL().fileName(), false);
    }

  m_scanWidget->readConfig(config);

  // Nove give each scanner a chance to read config
  QObjectList    *scanners = queryList("KNetmapScanner");
  QObjectListIt  sit(*scanners);
  KNetmapScanner *scanner;
  while ((scanner = (KNetmapScanner *)sit.current()) != 0)
    {
      ++sit;
      scanner->readConfig(config);
    }
  delete scanners;
}

void KNetmapApp::saveProperties(KConfig *_cfg)
{
  if(doc->URL().fileName()!=i18n("Untitled") && !doc->isModified())
    {
      // saving to tempfile not necessary
    }
  else
    {
      KURL url=doc->URL();	
      _cfg->writeEntry("filename", url.url());
      _cfg->writeEntry("modified", doc->isModified());
      QString tempname = kapp->tempSaveName(url.url());
      QString tempurl= KURL::encode_string(tempname);
      KURL _url(tempurl);
      doc->saveDocument(_url);
    }
}


void KNetmapApp::readProperties(KConfig* _cfg)
{
  QString filename = _cfg->readEntry("filename", "");
  KURL url(filename);
  bool modified = _cfg->readBoolEntry("modified", false);
  if(modified)
    {
      bool canRecover;
      QString tempname = kapp->checkRecoverFile(filename, canRecover);
      KURL _url(tempname);
  	
      if(canRecover)
	{
	  doc->openDocument(_url);
	  doc->setModified();
	  setCaption(_url.fileName(),true);
	  QFile::remove(tempname);
	}
    }
  else
    {
      if(!filename.isEmpty())
	{
	  doc->openDocument(url);
	  setCaption(url.fileName(),false);
	}
    }
}		

bool KNetmapApp::queryClose()
{
  return doc->saveModified();
}

bool KNetmapApp::queryExit()
{
  saveOptions();
  return true;
}

void KNetmapApp::optionsConfigureKeys()
{
  KKeyDialog::configure(actionCollection());
}

void KNetmapApp::optionsConfigureToolbars()
{
  // use the standard toolbar editor
  saveMainWindowSettings( KGlobal::config(), autoSaveGroup() );
  KEditToolbar dlg(actionCollection());
  connect(&dlg, SIGNAL(newToolbarConfig()), this, SLOT(updateGUI()));
  dlg.exec();
}

void KNetmapApp::updateGUI()
{
  createGUI(QString::null, false);
  applyMainWindowSettings( KGlobal::config(), autoSaveGroup() );
}

void KNetmapApp::optionsPreferences()
{
  // popup some sort of preference dialog, here
  KNetmapPreferences dlg(this);

  // Add each scanner's prefPage:
  QObjectList    *scanners = queryList("KNetmapScanner");
  QObjectListIt  sit(*scanners);
  KNetmapScanner *scanner;
  (void) new KNetmapGeneralPrefPage(&dlg);
  while ((scanner = (KNetmapScanner *)sit.current()) != 0)
    {
      ++sit;
      scanner->prefPage(&dlg);
    }

  if (dlg.exec())
    {
      // redo your settings
    }
}

void KNetmapApp::slotFileNew()
{
  slotStatusMsg(i18n("Creating new document..."));

  if(doc->saveModified())
    {
      doc->newDocument();		
      setCaption(doc->URL().fileName(), false);
    }

  slotStatusMsg(i18n("Ready."));
}

void KNetmapApp::slotFileOpen()
{
  slotStatusMsg(i18n("Opening file..."));
	
  if(doc->saveModified())
    {	
      KURL url=KFileDialog::getOpenURL(QString::null,
				       i18n("*|All files"), this, i18n("Open File..."));
      if(!url.isEmpty())
	{
	  doc->openDocument(url);
	  setCaption(url.fileName(), false);
	  fileOpenRecent->addURL( url );
	}
    }
  slotStatusMsg(i18n("Ready."));
}

void KNetmapApp::slotFileOpenRecent(const KURL& url)
{
  slotStatusMsg(i18n("Opening file..."));
	
  if(doc->saveModified())
    {
      doc->openDocument(url);
      setCaption(url.fileName(), false);
    }

  slotStatusMsg(i18n("Ready."));
}

void KNetmapApp::slotFileSave()
{
  slotStatusMsg(i18n("Saving file..."));

  doc->saveDocument(doc->URL());

  slotStatusMsg(i18n("Ready."));
}

void KNetmapApp::slotFileSaveAs()
{
  slotStatusMsg(i18n("Saving file with a new filename..."));

  KURL url=KFileDialog::getSaveURL(QDir::currentDirPath(),
				   i18n("*|All files"), this, i18n("Save as..."));
  if(!url.isEmpty())
    {
      doc->saveDocument(url);
      fileOpenRecent->addURL(url);
      setCaption(url.fileName(),doc->isModified());
    }

  slotStatusMsg(i18n("Ready."));
}

void KNetmapApp::slotFileQuit()
{
  slotStatusMsg(i18n("Exiting..."));
  saveOptions();
  close();
}

void KNetmapApp::slotViewToolBar()
{
  slotStatusMsg(i18n("Toggling toolbar..."));
  // turn Toolbar on or off
  if(!viewToolBar->isChecked())
    {
      toolBar("mainToolBar")->hide();
    }
  else
    {
      toolBar("mainToolBar")->show();
    }		

  slotStatusMsg(i18n("Ready."));
}

void KNetmapApp::slotViewStatusBar()
{
  slotStatusMsg(i18n("Toggle the statusbar..."));
  //turn Statusbar on or off
  if(!viewStatusBar->isChecked())
    {
      statusBar()->hide();
    }
  else
    {
      statusBar()->show();
    }

  slotStatusMsg(i18n("Ready."));
}

void KNetmapApp::slotViewConsole()
{
  console->setShown(viewConsole->isChecked());
}

void KNetmapApp::slotStatusMsg(const QString &text)
{
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG);
}

#include "knetmap.moc"
