/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qdir.h>
#include <qwidget.h>
#include <qobject.h>
#include <qobjectlist.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include <ksavefile.h>
#include <kdeversion.h>

#include "knetmapdoc.h"
#include "knetmap.h"
#include "knetmapview.h"
#include "knetmapsubnet.h"
#include "knetmaphost.h"
#include "knetmapscanwidget.h"

QPtrList<KNetmapView> *KNetmapDoc::pViewList = 0L;

KNetmapDoc::KNetmapDoc(QWidget *parent, const char *name)
  : QObject(parent, name)
{
  m_ccount=0;
  if(!pViewList)
    pViewList = new QPtrList<KNetmapView>();
  pViewList->setAutoDelete(true);
  m_cur = 0;
}

KNetmapDoc::~KNetmapDoc()
{
}

const QPtrList<KNetmapHost> KNetmapDoc::selectedHosts() const
{
  return m_curHosts;
}

const QPtrList<KNetmapSubnet> KNetmapDoc::selectedSubnets() const
{
  return m_curSubs;
}

void KNetmapDoc::setSelectedHosts(const QPtrList<KNetmapHost> &l)
{
  m_curHosts = l;
}

void KNetmapDoc::setSelectedSubnets(const QPtrList<KNetmapSubnet> &l)
{
  m_curSubs = l;
}

KNetmapView *KNetmapDoc::findView(const QString &s)
{
  KNetmapView *v=0;
  for (v = pViewList->first(); v; v = pViewList->next())
    {
      if (v->name() == s) return v;
    }
  return 0;
}

void KNetmapDoc::childEvent(QChildEvent *e)
{
  if (e->inserted())
    {
      if (e->child()->inherits("KNetmapSubnet"))
	{
	  KNetmapSubnet *sub = (KNetmapSubnet *)e->child();
	  if (!m_ccount)
	    setModified(true);
	  else
	    --m_ccount;
	  emit subnetAdded(sub);
	  connect(sub, SIGNAL(modified()),
		  this, SLOT(slotSetModified()));
	}
    }
}

void KNetmapDoc::slotSetModified()
{
  setModified(true);
}

void KNetmapDoc::setModified(bool _m)
{
  KNetmapApp *app = getApp();
  modified=_m;
  if (app && app->fileSave && app->fileSaveAs)
    {
      app->fileSave->setEnabled(_m);
      app->fileSaveAs->setEnabled(_m);
    }
}

KNetmapSubnet *KNetmapDoc::subForIP(const IPv4 &ip, bool createOk)
{
  KNetmapSubnet *sub=0;
  QObjectList *l = queryList("KNetmapSubnet");
  QObjectListIt it(*l);
  QObject *o;

  while ((o=it.current()) != 0)
    {
      ++it;
      if (((KNetmapSubnet *)o)->ip().contains(ip))
	if (!sub)
	  sub = (KNetmapSubnet *)o;
	else
	  if (((KNetmapSubnet *)o)->ip() < sub->ip())
	    sub = (KNetmapSubnet *)o;
    }
  delete l;

  if (createOk)
    {
      if (!sub)
	sub = new KNetmapSubnet(this, ip.cidrstr(IPv4::Network));
      if (sub->ip().cidrlen() < 16)
	{
	  IPv4 i = IPv4(QString("%1/16").arg(ip.address()));
	  i.truncate();
	  KNetmapSubnet *temp = findSubnet(i);
	  if (temp)
	    sub = temp;
	  else
	    sub = new KNetmapSubnet(sub, i.cidrstr(IPv4::Network));
	}
      if (sub->ip().cidrlen() < 24)
	{
	  IPv4 i = IPv4(QString("%1/24").arg(ip.address()));
	  i.truncate();
	  KNetmapSubnet *temp = findSubnet(i);
	  if (temp)
	    sub = temp;
	  else
	    sub = new KNetmapSubnet(sub, i.cidrstr(IPv4::Network));
	}
    }
  return sub;
}

KNetmapHost *KNetmapDoc::addHost(const IPv4 &ip)
{
  KNetmapHost *host = findHost(ip);
  if (!host)
    {
      KNetmapSubnet *sub = subForIP(ip, true);
      host = new KNetmapHost(sub, ip.address());
    }
  return host;
}

KNetmapSubnet *KNetmapDoc::findSubnet(const IPv4 &ip)
{
  KNetmapSubnet *sub=0;
  QObjectList *l = queryList("KNetmapSubnet");
  QObjectListIt it(*l);
  QObject *o;

  while ((o=it.current()) != 0)
    {
      ++it;
      if (((KNetmapSubnet *)o)->ip() == ip)
	{
	  sub = (KNetmapSubnet *)o;
	  break;
	}
    }
  delete l;

  return sub;

}

KNetmapHost *KNetmapDoc::findHost(const IPv4 &i)
{
  KNetmapHost *host=0;
  QObjectList *l = queryList("KNetmapHost");
  QObjectListIt it(*l);
  QObject *o;
  IPv4 ip(i.cidrstr());
  ip.lockMask();

  while ((o=it.current()) != 0)
    {
      ++it;
      IPv4 a(((KNetmapHost *)o)->ip().cidrstr());
      a.lockMask();
      if (a == ip)
	{
	  host = (KNetmapHost *)o;
	  break;
	}
    }
  delete l;

  return host;
}

void KNetmapDoc::setCurrent(QObject *o)
{
  m_cur = o;
  emit currentChanged(o);
}

void KNetmapDoc::deleteSelected()
{
  setCurrent(0);
  KNetmapHost *host;
  if (!m_curHosts.isEmpty())
    {
      for (host = m_curHosts.first(); host; host = m_curHosts.next())
	{
	  host->deleteLater();
	}
    }
  KNetmapSubnet *sub;
  if (!m_curSubs.isEmpty())
    {
      for (sub = m_curSubs.first(); sub; sub = m_curSubs.next())
	{
	  sub->deleteLater();
	}
    }
}

void KNetmapDoc::deleteCurrent()
{
  delete m_cur;
  setCurrent(0);
}

void KNetmapDoc::addView(KNetmapView *view)
{
  pViewList->append(view);
}

void KNetmapDoc::removeView(KNetmapView *view)
{
  pViewList->remove(view);
}
void KNetmapDoc::setURL(const KURL &url)
{
  doc_url=url;
}

const KURL& KNetmapDoc::URL() const
{
  return doc_url;
}

void KNetmapDoc::slotUpdateAllViews(KNetmapView *sender)
{
  KNetmapView *w;
  if(pViewList)
  {
    for(w=pViewList->first(); w!=0; w=pViewList->next())
    {
      if(w!=sender)
        w->repaint();
    }
  }

}

bool KNetmapDoc::saveModified()
{
  bool completed=true;

  if(modified)
    {
      int want_save = KMessageBox::warningYesNoCancel(getApp(),
						      i18n("The current file has been modified.\n"
							   "Do you want to save it?"),
						      i18n("Warning"));
      switch(want_save)
	{
	case KMessageBox::Yes:
	  saveDocument(URL());
	  completed=true;
	  break;

	case KMessageBox::No:
	  setModified(false);
	  completed=true;
	  break;

	case KMessageBox::Cancel:
	  completed=false;
	  break;

	default:
	  completed=false;
	  break;
	}
    }

  return completed;
}

void KNetmapDoc::closeDocument()
{
  deleteContents();
}

bool KNetmapDoc::newDocument()
{
  doc_url.setFileName(i18n("Untitled"));
  deleteContents();
  setModified(false);

  return true;
}

bool KNetmapDoc::openDocument(const KURL& url)
{
  if (url.isEmpty())
    return false;
  else if (url.fileName() == i18n("Untitled"))
    return false;
  QString tmpfile;

  if (!KIO::NetAccess::download(url, tmpfile, getApp()))
    {
      qWarning("Failed to access url %s: %s",
	       url.url().ascii(), KIO::NetAccess::lastErrorString().ascii());
      return false;
    }

  QFile file(tmpfile);
  if (!file.open(IO_ReadOnly | IO_Translate))
    {
      qWarning("Failed to read file %s", tmpfile.ascii());
      KIO::NetAccess::removeTempFile(tmpfile);
      return false;
    }

  QDomDocument doc;
  QString errbuf;
  if (!doc.setContent(&file, &errbuf))
    {
      qWarning("Failed to parse XML from %s: %s",
	       url.url().ascii(), errbuf.ascii());
      file.close();
      KIO::NetAccess::removeTempFile(tmpfile);
      return false;
    }
  file.close();
  KIO::NetAccess::removeTempFile(tmpfile);

  QDomNode n = doc.documentElement().firstChild();
  while (!n.isNull())
    {
      if (n.nodeType() == QDomNode::ElementNode)
	{
	  QDomElement e = n.toElement();
	  if (e.tagName() == "Subnet")
	    {
	      KNetmapSubnet::deserialize(this, e);
	      m_ccount++;
	    }
	  else
	    {
	      qWarning("Unknown tag %s while reading %s",
		       e.tagName().ascii(), url.url().ascii());
	    }
	}
      n = n.nextSibling();
    }

  setURL(url);
  setModified(false);
  return true;
}

bool KNetmapDoc::saveDocument(const KURL& url)
{
  if (url.fileName() == i18n("Untitled"))
    {
      qWarning("Attempt to save Untitled document, kicing up to app");
      getApp()->slotFileSaveAs();
      return false;
    }
  if (url.url() != doc_url.url())
    doc_url = url;

  QDomDocument doc;
  QDomElement root = doc.createElement("KNetmap");
  doc.appendChild(root);

  const QObjectList *l = children();
  if (!l || l->isEmpty())
    {
      qWarning("Attempt to save empty document");
      return false;
    }

  QObjectListIt it(*l);
  QObject *o;
  while ((o=it.current()) != 0)
    {
      ++it;
      if (o->inherits("KNetmapSubnet"))
	((KNetmapSubnet *)o)->serialize(doc, root);
      else
	qWarning("Alien child of class: %s in document", o->className());
    }

  KSaveFile save(url.path());
  if (save.status())
    {
      qWarning("failed to create savefile, errno: %i", save.status());
      return false;
    }
  else
    {
      *(save.textStream()) << doc.toString();
      if (!save.close())
	{
	  qWarning("close failed, errno: %i", save.status());
	  return false;
	}
    }

  setModified(false);
  return true;
}

void KNetmapDoc::deleteContents()
{
  const QObjectList *l = children();
  if (!l || l->isEmpty())
    return;
  QObjectListIt it(*l);
  QObject *o;
  while ((o=it.current()) != 0)
    {
      ++it;
      if (o->inherits("KNetmapSubnet"))
	delete o;
    }
}
#include "knetmapdoc.moc"
