/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qobjectlist.h>
#include <kglobal.h>
#include <kiconloader.h>

#include "knetmaphost.h"
#include "knetmapsubnet.h"

KNetmapHost::KNetmapHost(KNetmapSubnet *parent, QString n, QString i)
  : QObject(parent)
{
  m_ccount=0;
  m_name = QString::null;
  m_tcpseq = QString::null;
  m_tcptsseq = QString::null;
  m_ipidseq = QString::null;
  m_icon = QString::null;
  if (m_ip.set(i, parent->ip().cidrlen()))
    {
      m_name = n;
      setName(n);
    }
}

KNetmapHost::KNetmapHost(KNetmapSubnet *parent, QString i)
  : QObject(parent)
{
  m_ccount=0;
  m_name = QString::null;
  m_tcpseq = QString::null;
  m_tcptsseq = QString::null;
  m_ipidseq = QString::null;
  m_icon = QString::null;
  if (m_ip.set(i, parent->ip().cidrlen()))
    {
      setName(m_ip.address());
    }
}

KNetmapHost::~KNetmapHost()
{
}

void KNetmapHost::setFingerprint(const QString &s)
{
  m_fingerprint = s;
  emit modified();
}

int KNetmapHost::compare(KNetmapHost *h,
			 bool)
{
  if (m_ip < h->m_ip)
    return -1;
  else if (m_ip == h->m_ip)
    return 0;
  else
    return 1;
}

KNetmapPort *KNetmapHost::findPort(KNetmapPort::Protocol protocol,
				   int number)
{
  KNetmapPort *ret=0, *port;
  QObjectList *l = queryList("KNetmapPort");
  QObjectListIt it(*l);
  while ((port = (KNetmapPort *)it.current()) != 0)
    {
      ++it;
      if ((port->protocol() == protocol) &&
	  (port->number() == number))
	{
	  ret = port;
	  break;
	}
    }
  delete l;
  return ret;
}

KNetmapOS *KNetmapHost::osMatch()
{
  KNetmapOS *ret=0, *os;
  int acc=0;
  QObjectList *l = queryList("KNetmapOS");
  QObjectListIt it(*l);
  while ((os = (KNetmapOS *)it.current()) != 0)
    {
      ++it;
      if (os->accuracy() > acc)
	{
	  acc = os->accuracy();
	  ret = os;
	}
    }
  delete l;
  return ret;
}

void KNetmapHost::clearPorts()
{
  QObjectList *l = queryList("KNetmapPort");
  QObjectListIt it(*l);
  QObject *o;
  while ((o=it.current()) != 0)
    {
      ++it;
      delete o;
    }
  delete l;
  emit modified();
}

void KNetmapHost::clearOS()
{
  QObjectList *l = queryList("KNetmapOS");
  QObjectListIt it(*l);
  QObject *o;
  while ((o=it.current()) != 0)
    {
      ++it;
      delete o;
    }
  delete l;
  emit modified();
}

void KNetmapHost::setIcon(const QString &p)
{
  m_icon = p;
  emit iconChanged(p);
}

void KNetmapHost::setHostname(const QString &name)
{
  m_name = name;
  emit nameChanged(m_name);
  emit modified();
}

void KNetmapHost::childEvent(QChildEvent *e)
{
  if (e->inserted())
    if (e->child()->inherits("KNetmapPort"))
      {
	KNetmapPort *port = (KNetmapPort *)e->child();
	emit portAdded(port);
	if (!m_ccount)
	  emit modified();
	else
	  --m_ccount;
	connect(port, SIGNAL(modified()),
		this, SIGNAL(modified()));
	connect(port, SIGNAL(destroyed()),
		this, SIGNAL(modified()));
      }
    else if (e->child()->inherits("KNetmapOS"))
      {
	KNetmapOS *os = (KNetmapOS *)e->child();
	emit osAdded(os);
	if (!m_ccount)
	  emit modified();
	else
	  --m_ccount;
	connect(os, SIGNAL(destroyed()),
		this, SIGNAL(modified()));
      }
}

void KNetmapHost::serialize(QDomDocument doc, QDomElement parent)
{
  QDomElement me = doc.createElement("Host");
  me.setAttribute("address", m_ip.address());
  if (!m_name.isEmpty())
    me.setAttribute("name", m_name);
  if (!m_tcpseq.isEmpty())
    me.setAttribute("tcpseq", m_tcpseq);
  if (!m_tcptsseq.isEmpty())
    me.setAttribute("tcptsseq", m_tcptsseq);
  if (!m_ipidseq.isEmpty())
    me.setAttribute("ipidseq", m_ipidseq);
  parent.appendChild(me);

  QObjectList *l = queryList("KNetmapOS");
  QObjectListIt it(*l);
  QObject *o;
  while ((o=it.current()) != 0)
    {
      ++it;
      ((KNetmapOS *)o)->serialize(doc, me);
    }
  delete l;

  l = queryList("KNetmapPort");
  it = QObjectListIt(*l);
  while ((o=it.current()) != 0)
    {
      ++it;
      ((KNetmapPort *)o)->serialize(doc, me);
    }
  delete l;
}

void KNetmapHost::deserialize(KNetmapSubnet *parent, QDomElement me)
{
  if (!me.hasAttribute("address"))
    {
      qWarning("Host element does not have an address attribute");
      return;
    }

  KNetmapHost *host;
  if (me.hasAttribute("name"))
    host = new KNetmapHost(parent, me.attribute("name"), me.attribute("address"));
  else
    host = new KNetmapHost(parent, me.attribute("address"));

  if (me.hasAttribute("tcpseq"))
    host->m_tcpseq = me.attribute("tcpseq");
  if (me.hasAttribute("tcptsseq"))
    host->m_tcptsseq = me.attribute("tcptsseq");
  if (me.hasAttribute("ipidseq"))
    host->m_ipidseq = me.attribute("ipidseq");

  QDomNode n = me.firstChild();
  while (!n.isNull())
    {
      if (n.nodeType() == QDomNode::ElementNode)
	{
	  QDomElement e = n.toElement();
	  if (e.tagName() == "Port")
	    {
	      KNetmapPort::deserialize(host, e);
	      ++host->m_ccount;
	    }
	  else if (e.tagName() == "OS")
	    {
	      KNetmapOS::deserialize(host, e);
	      ++host->m_ccount;
	    }
	  else
	    qWarning("Unknown element %s in host element",
		     e.tagName().ascii());
	}
      n = n.nextSibling();
    }
}

const IPv4 KNetmapHost::ip() const
{
  return m_ip;
}

const QString KNetmapHost::hostname() const
{
  return m_name;
}

const QString KNetmapHost::tcpSeq() const
{
  return m_tcpseq;
}

const QString KNetmapHost::tcpTSSeq() const
{
  return m_tcptsseq;
}

const QString KNetmapHost::ipidSeq() const
{
  return m_ipidseq;
}

const QString KNetmapHost::fingerprint() const
{
  return m_fingerprint;
}

void KNetmapHost::setTcpSeq(const QString &s)
{
  m_tcpseq = s;
  emit modified();
}

void KNetmapHost::setTcpTSSeq(const QString &s)
{
  m_tcptsseq = s;
  emit modified();
}

void KNetmapHost::setIpidSeq(const QString &s)
{
  m_ipidseq = s;
  emit modified();
}

const QString KNetmapHost::icon()
{
  if (m_icon.isEmpty())
    {
      QString s;
      QPixmap p;
      int a=0;
      KNetmapOS *o;
      QObjectList *l = queryList("KNetmapOS");
      QObjectListIt it(*l);
      while ((o = (KNetmapOS *)it.current()) != 0)
	{
	  ++it;
	  if (o->accuracy() > a)
	    {
	      a = o->accuracy();
	      s = QString("os_%1").arg(o->family().lower());
	      p = KGlobal::iconLoader()->loadIcon
		(s, KIcon::Small, 16, KIcon::DefaultState, 0L, true);
	      if (p.isNull())
		{
		  s = QString("os_%1").arg(o->vendor().lower());
		  p = KGlobal::iconLoader()->loadIcon
		    (s, KIcon::Small, 16, KIcon::DefaultState, 0L, true);
		}
	    }
	}
      delete l;

      if (p.isNull())
	{
	  s = QString(HOST_ICON);
	}
      else
	{
	  m_icon = s;
	  emit iconChanged(s);
	}
      return s;
    }
  else
    {
      return m_icon;
    }
}

////////////////////////////////////////////////////////////
// Port code
KNetmapPort::KNetmapPort(KNetmapHost *parent,
			 Protocol proto, int number, QString svc,
			 QString product, QString version, QString extrainfo)
  : QObject(parent)
{
  m_inDeser = false;
  m_proto = proto;
  m_num   = number;
  m_svc   = svc;
  m_prod  = product;
  m_ver   = version;
  m_extra = extrainfo;
  setName(toStr());
}

KNetmapPort::~KNetmapPort()
{
}

void KNetmapPort::setServiceInfo(QString service,
				 QString product,
				 QString version,
				 QString extra)
{
  m_svc   = service;
  m_prod  = product;
  m_ver   = version;
  m_extra = extra;
  emit updateInfo();
  if (!m_inDeser)
    emit modified();
}

void KNetmapPort::setIcon(QString icon)
{
  m_icon = icon;
}

QString KNetmapPort::protocolStr()
{
  if (m_proto == Tcp)
    return QString("TCP");
  else if (m_proto == Udp)
    return QString("UDP");
  return QString("");
}

QString KNetmapPort::toStr()
{
  QString s = QString("%1/%2 %3")
    .arg((QString)protocolStr())
    .arg(m_num)
    .arg(m_svc);

  QString pstr = productStr();
  if (!pstr.isEmpty())
    s += QString(" - %1").arg(pstr);
  return s;
}

QString KNetmapPort::productStr()
{
  QString s("");
  if (!m_prod.isEmpty())
    s += QString(" %1").arg(m_prod);
  if (!m_ver.isEmpty())
    s += QString(" - %1").arg(m_ver);
  if (!m_extra.isEmpty())
    s += QString(" %1").arg(m_extra);
  return s;
}

void KNetmapPort::serialize(QDomDocument doc, QDomElement parent)
{
  QDomElement me = doc.createElement("Port");
  switch (m_proto)
    {
    case Tcp:
      me.setAttribute("protocol", "TCP");
      break;
    case Udp:
      me.setAttribute("protocol", "UDP");
      break;
    };
  me.setAttribute("number", m_num);
  parent.appendChild(me);
  if (!m_svc.isEmpty())
    {
      QDomElement svc = doc.createElement("Service");
      svc.setAttribute("name", m_svc);
      if (!m_prod.isEmpty())
	svc.setAttribute("product", m_prod);
      if (!m_ver.isEmpty())
	svc.setAttribute("version", m_ver);
      if (!m_extra.isEmpty())
	svc.setAttribute("extra", m_extra);
      me.appendChild(svc);
    }
}

void KNetmapPort::deserialize(KNetmapHost *parent, QDomElement me)
{
  if (!me.hasAttribute("protocol"))
    {
      qWarning("Port element missing protocol attribute");
      return;
    }
  if (!me.hasAttribute("number"))
    {
      qWarning("Port element missing number attribute");
      return;
    }

  Protocol pr = Tcp;
  int num;
  bool ok;
  num = me.attribute("number").toInt(&ok);
  if (!ok)
    {
      qWarning("Invalid port number");
      return;
    }
  if (me.attribute("protocol") == "UDP")
    pr = Udp;

  KNetmapPort *port = new KNetmapPort(parent, pr, num);

  QDomNode svcnode = me.namedItem("Service");
  if ((!svcnode.isNull()) &&
      (svcnode.nodeType() == QDomNode::ElementNode))
    {
      port->m_inDeser=true;
      QDomElement svc = svcnode.toElement();
      port->setServiceInfo(svc.attribute("name"),
			   svc.attribute("product"),
			   svc.attribute("version"),
			   svc.attribute("extra"));
      port->m_inDeser=false;
    }
}

KNetmapPort::Protocol KNetmapPort::protocol()
{
  return m_proto;
}

int KNetmapPort::number()
{
  return m_num;
}

const QString KNetmapPort::service()
{
  return m_svc;
}

const QString KNetmapPort::product()
{
  return m_prod;
}

const QString KNetmapPort::version()
{
  return m_ver;
}

const QString KNetmapPort::extrainfo()
{
  return m_extra;
}

const QString KNetmapPort::icon()
{
  if (m_icon.isEmpty())
    {
      QString s;
      QPixmap p;
      s = QString("svc_%1").arg(m_svc.lower());
      p = KGlobal::iconLoader()->loadIcon
	(s, KIcon::Small, 16, KIcon::DefaultState, 0L, true);
      if (p.isNull())
	s = QString(PORT_ICON);
      else
	setIcon(s);
      return s;
    }
  else
    {
      return m_icon;
    }
}

////////////////////////////////////////////////////////////
// OS code
KNetmapOS::KNetmapOS(KNetmapHost *parent,
		     QString vendor, QString family,
		     QString gen,    QString type,
		     int accuracy)
  : QObject(parent)
{
  m_vendor = vendor;
  m_family = family;
  m_gen    = gen;
  m_type   = type;
  m_acc    = accuracy;
  setName(toStr());
}

KNetmapOS::~KNetmapOS()
{
}

QString KNetmapOS::toStr()
{
  QString s;
  if (m_vendor == m_family)
    {
      s = QString("%1% (%2) %3 %4")
	.arg(m_acc)
	.arg(m_type)
	.arg(m_vendor)
	.arg(m_gen);
    }
  else
    {
      s = QString("%1% (%2) %3 %4 %5")
	.arg(m_acc)
	.arg(m_type)
	.arg(m_vendor)
	.arg(m_family)
	.arg(m_gen);
    }
  return s;
}

QString KNetmapOS::shortString()
{
  QString s;
  if (m_vendor == m_family)
    {
      s = QString("%1 %2")
	.arg(m_vendor)
	.arg(m_gen);
    }
  else
    {
      s = QString("%1 %2 %3")
	.arg(m_vendor)
	.arg(m_family)
	.arg(m_gen);
    }
  return s;
}

void KNetmapOS::serialize(QDomDocument doc, QDomElement parent)
{
  QDomElement me = doc.createElement("OS");
  me.setAttribute("vendor", m_vendor);
  me.setAttribute("family", m_family);
  me.setAttribute("gen", m_gen);
  me.setAttribute("type", m_type);
  me.setAttribute("accuracy", m_acc);
  parent.appendChild(me);
}

void KNetmapOS::deserialize(KNetmapHost *parent, QDomElement me)
{
  if (!(me.hasAttribute("vendor") &&
	me.hasAttribute("family") &&
	me.hasAttribute("gen")    &&
	me.hasAttribute("type")   &&
	me.hasAttribute("accuracy")))
    {
      qWarning("Incomplete OS element");
      return;
    }
  int acc;
  bool ok;
  acc = me.attribute("accuracy").toInt(&ok);
  if (!ok)
    {
      qWarning("Invalid accuracy attribute on OS element");
      return;
    }
  (void) new KNetmapOS(parent,
		       me.attribute("vendor"),
		       me.attribute("family"),
		       me.attribute("gen"),
		       me.attribute("type"), acc);
}

KNetmapOSList::KNetmapOSList()
  : QPtrList<KNetmapOS>()
{
}

KNetmapOSList::~KNetmapOSList()
{
}

int KNetmapOSList::compareItems (QPtrCollection::Item a,
				 QPtrCollection::Item b)
{
  KNetmapOS *osa = (KNetmapOS *)b;
  KNetmapOS *osb = (KNetmapOS *)a;
  if (osa->accuracy() > osb->accuracy())
    return 1;
  else if (osa->accuracy() == osb->accuracy())
    return 0;
  else
    return -1;
}

#include "knetmaphost.moc"
