/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef KNETMAPHOST_H
#define KNETMAPHOST_H

#include <qobject.h>
#include <qstring.h>
#include <qptrlist.h>
#include <qdom.h>
#include "ipv4.h"

#define HOST_ICON "os_default"
#define PORT_ICON "svc_default"

class KNetmapSubnet;
class KNetmapOS;
class KNetmapPort;
class KNetmapHost;

class KNetmapPort : public QObject
{
  Q_OBJECT
public:
  typedef enum protocol_e
    {
      Tcp=1,
      Udp=2
    } Protocol;

  KNetmapPort(KNetmapHost *parent,
	      Protocol proto, int number,
	      QString svc=QString::null,
	      QString product=QString::null,
	      QString version=QString::null,
	      QString extrainfo=QString::null);
  ~KNetmapPort();

  void setServiceInfo(QString service, QString product, QString version, QString extra);
  Protocol protocol();
  int number();
  const QString service();
  const QString product();
  const QString version();
  const QString extrainfo();
  const QString icon();
  QString protocolStr();
  QString toStr();
  QString productStr();
  void setIcon(QString);

  void serialize(QDomDocument, QDomElement);
  static void deserialize(KNetmapHost *, QDomElement);

signals:
  void modified();
  void updateInfo();

private:
  Protocol m_proto;
  int m_num;
  QString m_svc;
  QString m_prod;
  QString m_ver;
  QString m_extra;
  QString m_icon;
  bool m_inDeser;
};

class KNetmapOS : public QObject
{
  Q_OBJECT
public:
  KNetmapOS(KNetmapHost *parent,
	    QString vendor, QString family,
	    QString gen,    QString type,
	    int accuracy=100);
  ~KNetmapOS();
  QString toStr();
  QString shortString();

  // Accessors
  QString vendor()   const {return m_vendor;};
  QString family()   const {return m_family;};
  QString gen()      const {return m_gen;};
  QString type()     const {return m_type;};
  int     accuracy() const {return m_acc;};

  void serialize(QDomDocument, QDomElement);
  static void deserialize(KNetmapHost *, QDomElement);

private:
  QString m_vendor;
  QString m_family;
  QString m_gen;
  QString m_type;
  int     m_acc;
};

class KNetmapOSList : public QPtrList<KNetmapOS>
{
public:
  KNetmapOSList();
  ~KNetmapOSList();

protected:
  virtual int compareItems (QPtrCollection::Item, QPtrCollection::Item);
};

class KNetmapHost : public QObject
{
  Q_OBJECT
public:
  KNetmapHost(KNetmapSubnet *parent, QString n, QString i);
  KNetmapHost(KNetmapSubnet *parent, QString i);
  virtual ~KNetmapHost();
  // Accessors
  const IPv4    ip() const;
  const QString hostname() const;
  const QString tcpSeq() const;
  const QString tcpTSSeq() const;
  const QString ipidSeq() const;
  const QString fingerprint() const;
  const QString icon();
  void setTcpSeq(const QString &);
  void setTcpTSSeq(const QString &);
  void setIpidSeq(const QString &);

  KNetmapPort *findPort(KNetmapPort::Protocol, int);
  KNetmapOS *osMatch(); // Returns the highest os match

  void serialize(QDomDocument, QDomElement);
  static void deserialize(KNetmapSubnet *, QDomElement);
  virtual int compare(KNetmapHost *, bool);

public slots:
  void setHostname(const QString &);
  void setIcon(const QString &);
  void clearPorts();
  void clearOS();
  void setFingerprint(const QString &);

signals:
  void osAdded(KNetmapOS *);
  void portAdded(KNetmapPort *);
  void nameChanged(const QString &);
  void iconChanged(const QString &);
  void modified();
  void infoUpdated();

protected:
  virtual void childEvent(QChildEvent *);

private:
  IPv4 m_ip;
  QString m_name;
  QString m_tcpseq;
  QString m_tcptsseq;
  QString m_ipidseq;
  QString m_icon;
  QString m_fingerprint;
  int m_ccount;
};

#endif // KNETMAPHOST_H
