/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qvgroupbox.h>
#include <qobjectlist.h>
#include <qhbox.h>
#include <kglobal.h>
#include <kiconloader.h>

#include "klistviewobject.h"
#include "knetmaphostpage.h"
#include "knetmaphost.h"
#include "knetmapview.h"
#include "knetmap.h"

KNetmapHostPage::KNetmapHostPage(QWidget *parent, const char *name)
  : QWidget(parent, name)
{
  m_host = 0;

  top_layout = new QVBoxLayout(this, 0, 3);
  top_layout->setAutoAdd(true);

  // The title row
  QHBox *row1 = new QHBox(this, "row1");
  m_icon = new QLabel(row1); // The Icon
  m_lbl  = new QLabel(row1);
  m_os   = new QLabel(QString::null, row1, "os");
  m_icon->setFixedWidth(52);
  m_icon->setPixmap(KGlobal::iconLoader()->loadIcon(HOST_ICON, KIcon::NoGroup, 48));
  row1->setStretchFactor(m_icon, 0);
  row1->setStretchFactor(m_lbl,  0);
  row1->setStretchFactor(m_os,   1);

  m_tabs = new QTabWidget(this, "tabs");

  // Ports
  QWidget *portw = new QWidget(this, "port_w");
  QVBoxLayout *portl = new QVBoxLayout(portw);
  portl->setAutoAdd(true);
  m_ports = new KListView(portw);
  m_ports->setFrameShape(QFrame::NoFrame);
  m_ports->addColumn("Protocol");
  m_ports->addColumn("Port");
  m_ports->addColumn("Service");
  m_ports->addColumn("Product");
  m_ports->setSorting(1, true);
  m_ports->setShowSortIndicator(true);

  // Extra Info
  QWidget *infow = new QWidget(this, "info_w");
  QVBoxLayout *infol = new QVBoxLayout(infow);
  infol->setAutoAdd(true);
  m_info = new KTextEdit(QString::null, QString::null, infow, "text");
  m_info->setReadOnly(true);
  m_info->setTextFormat(Qt::RichText);
  m_info->setFrameShape(QFrame::NoFrame);
  m_info->setText("No Host Selected (you shouldn't be seing this ;-)");

  m_tabs->addTab(portw, KGlobal::iconLoader()->loadIconSet("svc_default", KIcon::Small), "&Ports");
  m_tabs->addTab(infow, KGlobal::iconLoader()->loadIconSet("info", KIcon::Small), "&Info");

  connect(getApp()->getDocument(), SIGNAL(currentChanged(QObject *)),
	  this, SLOT(update(QObject *)));
}

KNetmapHostPage::~KNetmapHostPage()
{
}

KNetmapApp *KNetmapHostPage::getApp()
{
  return ((KNetmapView *)parent()->parent()->parent()->parent())->getApp();
}

void KNetmapHostPage::updateInfo()
{
  QString s = QString::null;
  if (m_host)
    {
      KNetmapOSList ol;
      QObjectList *l = m_host->queryList("KNetmapOS");
      if (!l->isEmpty())
	{
	  QObjectListIt it(*l);
	  KNetmapOS *os;
	  while ((os = (KNetmapOS *)it.current()) != 0)
	    {
	      ++it;
	      ol.append(os);
	    }
	  ol.sort();
	}
      delete l;

      int portc=0, udpc=0, tcpc=0;
      l = m_host->queryList("KNetmapPort");
      QObjectListIt it(*l);
      KNetmapPort *port;
      while ((port = (KNetmapPort *)it.current()) != 0)
	{
	  ++it;
	  portc++;
	  if (port->protocol() == KNetmapPort::Tcp)
	    tcpc++;
	  else if (port->protocol() == KNetmapPort::Udp)
	    udpc++;
	}
      delete l;

      s = QString("<p><h2><u>OS Fingerprints</u>:</h2>");
      if (!ol.isEmpty())
	{
	  KNetmapOS *os;
	  for (os = ol.first(); os; os = ol.next())
	    {
	      s += QString("&nbsp;&nbsp;&nbsp;%1<br />").arg(os->toStr());
	    }
	}
      if (!m_host->fingerprint().isEmpty())
	{
	  QStringList lines = QStringList::split(QString("\n"), m_host->fingerprint());
	  bool first=true;
	  for (QStringList::Iterator it=lines.begin(); it!=lines.end(); ++it)
	    {
	      if (first)
		{
		  s += QString("<br />&nbsp;&nbsp;&nbsp;<b><u>%1</u></b><br />").arg(*it);
		  first=false;
		}
	      else
		{
		  s += QString("&nbsp;&nbsp;&nbsp;%1<br />").arg(*it);
		}
	    }
	  s += QString("&nbsp;&nbsp;&nbsp;If you know what OS this host is running, please see http://www.insecure.org/cgi-bin/nmap-submit.cgi<br />");
	}

      s += QString("</p><p>"
		   "<h2><u>TCP/IP Stack Analysis</u>:</h3>"
		   "&nbsp;&nbsp;&nbsp;<span title=\"foo\"><b>TCP Sequence: </b>%1</span><br />"
		   "&nbsp;&nbsp;&nbsp;<b>TCP TS Sequence: </b>%2<br />"
		   "&nbsp;&nbsp;&nbsp;<b>IP ID Sequence: </b>%3<br />"
		   "</p><p>"
		   "<h2><u>Port Statistics</u>:</h3>"
		   "&nbsp;&nbsp;&nbsp;<b>Total: </b>%4<br />"
		   "&nbsp;&nbsp;&nbsp;<b>TCP: </b>%5<br />"
		   "&nbsp;&nbsp;&nbsp;<b>UDP: </b>%6<br />"
		   "</p>")
	.arg(m_host->tcpTSSeq())
	.arg(m_host->tcpSeq())
	.arg(m_host->ipidSeq())
	.arg(portc)
	.arg(tcpc)
	.arg(udpc);
    }
  else
    {
      s = QString("No Host Selected (you shouldn't be seing this ;-)");
    }
  m_info->setText(s);
}

void KNetmapHostPage::addOS(KNetmapOS *)
{
  KNetmapOS *t_os = m_host->osMatch();
  if (t_os)
    m_os->setText(QString(" running %1").arg(t_os->shortString()));
  else
    m_os->setText(QString::null);
}

void KNetmapHostPage::addPort(KNetmapPort *port)
{
  (void)new KNetmapPortItem(m_ports, port);
}

void KNetmapHostPage::setIcon(const QString &icon)
{
  if (icon.isEmpty())
    m_icon->setPixmap(KGlobal::iconLoader()->loadIcon(HOST_ICON, KIcon::Small, 48));
  else
    m_icon->setPixmap(KGlobal::iconLoader()->loadIcon(icon, KIcon::Small, 48));
}

void KNetmapHostPage::setName(const QString &name)
{
  QString title = QString("");
  if (name.isEmpty())
    title += m_host->ip().address();
  else
    {
      title += name;
      title += "(" + m_host->ip().address() + ")";
    }
  m_lbl->setText(title);
}

void KNetmapHostPage::update(QObject *o)
{
  KNetmapHost *host=0;
  if (o && o->inherits("KNetmapHost"))
    host = (KNetmapHost *) o;

  if (m_host)
    {
      disconnect(m_host, SIGNAL(osAdded(KNetmapOS *)),
  		 this,     SLOT(addOS  (KNetmapOS *)));
      disconnect(m_host, SIGNAL(portAdded(KNetmapPort *)),
		 this,     SLOT(addPort  (KNetmapPort *)));
      disconnect(m_host, SIGNAL(nameChanged(const QString &)),
		 this,     SLOT(setName    (const QString &)));
      disconnect(m_host, SIGNAL(iconChanged(const QString &)),
		 this,     SLOT(setIcon    (const QString &)));
      disconnect(m_host, SIGNAL(modified()),
		 this,     SLOT(updateInfo()));
    }
  m_host = host;
  if (m_host)
    {
      setIcon(m_host->icon());
      setName(m_host->hostname());
      KNetmapOS *os = host->osMatch();
      if (os)
	m_os->setText(QString(" running %1").arg(os->shortString()));
      else
	m_os->setText(QString::null);
      m_ports->clear();

      QObjectList *l = m_host->queryList("KNetmapPort");
      QObjectListIt it(*l);
      QObject *o;
      while ((o = it.current()) != 0)
	{
	  ++it;
	  addPort((KNetmapPort *)o);
	}
      delete l;

      connect(m_host, SIGNAL(osAdded(KNetmapOS *)),
  	      this,     SLOT(addOS  (KNetmapOS *)));
      connect(m_host, SIGNAL(portAdded(KNetmapPort *)),
	      this,     SLOT(addPort  (KNetmapPort *)));
      connect(m_host, SIGNAL(nameChanged(const QString &)),
	      this,     SLOT(setName    (const QString &)));
      connect(m_host, SIGNAL(iconChanged(const QString &)),
	      this,     SLOT(setIcon    (const QString &)));
      connect(m_host, SIGNAL(modified()),
	      this,     SLOT(updateInfo()));
    }
  updateInfo();
}

////////////////////////////////////////////////////////////
KNetmapPortItem::KNetmapPortItem(QListView *parent,
				 KNetmapPort *port)
  : KListViewObject(parent,
		    port->protocolStr(),
		    QString("%1").arg(port->number()),
		    (port->service().isEmpty()    ? QString("-") : port->service()),
		    (port->productStr().isEmpty() ? QString("-") : port->productStr()))
{
  setRepresents(port);
  setPixmap(2, KGlobal::iconLoader()->loadIcon
	    (port->icon(), KIcon::Small, 16,
	     KIcon::DefaultState, 0L, true));
  connect(port, SIGNAL(destroyed()),
	  this,   SLOT(deleteLater()));
  connect(port, SIGNAL(updateInfo()),
	  this,   SLOT(updateInfo()));
}

void KNetmapPortItem::updateInfo()
{
  KNetmapPort *port = (KNetmapPort *)m_rep;
  setText(2, (port->service().isEmpty()    ? QString("-") : port->service()));
  setText(3, (port->productStr().isEmpty() ? QString("-") : port->productStr()));
  setPixmap(2, KGlobal::iconLoader()->loadIcon
	    (port->icon(), KIcon::Small, 16,
	     KIcon::DefaultState, 0L, true));
}

int KNetmapPortItem::compare(QListViewItem *i,
			     int col,
			     bool ascending) const
{
  if (col == 1)
    {
      KNetmapPort *porta = (KNetmapPort *)m_rep;
      KNetmapPort *portb = (KNetmapPort *)((KNetmapPortItem *)i)->m_rep;
      if ((porta->protocol() == KNetmapPort::Tcp) &&
	  (portb->protocol() == KNetmapPort::Udp))
	return -1;
      else if ((porta->protocol() == KNetmapPort::Udp) &&
	       (portb->protocol() == KNetmapPort::Tcp))
	return 1;
      else
	{
	  int a = porta->number();
	  int b = portb->number();
	  if (a < b)
	    return -1;
	  else if (a == b)
	    return 0;
	  else
	    return 1;
	}
    }
  else
    {
      return this->KListViewObject::compare(i, col, ascending);
    }
}

#include "knetmaphostpage.moc"
