/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>

#include <qxml.h>
#include <qmessagebox.h>
#include <klocale.h>

#include "knetmapnmapparser.h"
#include "knetmapnmapscanner.h"
#include "knetmaphost.h"
#include "knetmapdoc.h"
#include "knetmap.h"

// The Parser
KNetmapNmapParser::KNetmapNmapParser(KNetmapNmapScan *parent)
  : QObject(parent, "nmap-xml-parser")
{
  xmlWarning = false;
  xmlError = false;
  xmlFatalError = false;
  theApp = parent->getApp();
  filename = parent->xmlFile();
  file = new QFile(filename);
  started = false;

  KDirWatch *watcher = KDirWatch::self();
  connect(watcher, SIGNAL(dirty    (const QString &)),
	  this,      SLOT(slotDirty(const QString &)));
  watcher->addFile(filename);
  watcher->startScan();

  xmlSource  = new QXmlInputSource(file);
  xmlReader  = new QXmlSimpleReader();
  xmlHandler = new KNetmapNmapXMLHandler(this);
  xmlReader->setFeature("http://trolltech.com/xml/features/report-whitespace-only-CharData", false);
  xmlReader->setContentHandler(xmlHandler);

  if (file->size() > 0)
    {
      if (!xmlReader->parse(xmlSource, true))
	{
	  qWarning("Failed to parse: %s", filename.ascii());
	  disconnect(watcher, SIGNAL(dirty    (const QString &)),
		     this,      SLOT(slotDirty(const QString &)));
	  emit parseFailed();
	  started = true;
	}
    }
}

KNetmapNmapParser::KNetmapNmapParser(KNetmapNmapScanner *parent,
				     const QString &_filename)
  : QObject(parent, "nmap-xml-parser")
{
  theApp   = parent->getApp();
  filename = _filename;
  file     = new QFile(filename);

  xmlSource  = new QXmlInputSource(file);
  xmlReader  = new QXmlSimpleReader();
  xmlHandler = new KNetmapNmapXMLHandler(this);
  xmlReader->setFeature("http://trolltech.com/xml/features/report-whitespace-only-CharData", false);
  xmlReader->setContentHandler(xmlHandler);

  if (!xmlReader->parse(xmlSource))
    {
      qWarning("Failed to parse %s", filename.ascii());
      emit parseFailed();
    }
}

KNetmapNmapParser::~KNetmapNmapParser()
{
  KDirWatch::self()->removeFile(filename);
  delete xmlReader;
  delete xmlHandler;
  delete xmlSource;
  delete file;
}

void KNetmapNmapParser::parse()
{
  while (file->at() < file->size())
    {
      if (!parseIter())
	break;
    }
}

void KNetmapNmapParser::doParseDone()
{
  disconnect(KDirWatch::self(), SIGNAL(dirty    (const QString &)),
	     this,                SLOT(slotDirty(const QString &)));
  
  // unsigned int dur = xmlHandler->endTime() - xmlHandler->startTime();
  emit parseDone();
}

void KNetmapNmapParser::slotDirty(const QString &path)
{
  if (path == filename)
    {
      while (file->at() < file->size())
	{
	  if (!parseIter())
	    {
	      if (xmlFatalError)
		disconnect(KDirWatch::self(), SIGNAL(dirty    (const QString &)),
			   this,                SLOT(slotDirty(const QString &)));
	      break;
	    }
	}
    }
  else
    {
      qWarning("got a dirty, for an unknown file %s", path.ascii());
    }
}

bool KNetmapNmapParser::parseIter()
{
  xmlWarning = false;
  xmlError = false;
  xmlFatalError = false;

  if (!started)
    {
      if (!xmlReader->parse(xmlSource, true))
	{
	  if (xmlFatalError)
	    emit parseFailed();
	  else
	    started = true;
	  return false;
	}
      started = true;
    }
  else
    {
      if (!xmlReader->parseContinue())
	{
	  if (xmlFatalError)
	    emit parseFailed();
	  return false;
	}
    }
  return true;
}

////////////////////////////////////////////////////////////
// XML Handler
KNetmapNmapXMLHandler::KNetmapNmapXMLHandler(KNetmapNmapParser *_parser)
  : QXmlDefaultHandler()
{
  parser  = _parser;
  host    = 0;
  port    = 0;
  inHost  = false;
  inPort  = false;
  hostUp  = false;
  firstOS = true;
}

unsigned int KNetmapNmapXMLHandler::startTime() const
{
  return m_stime;
}

unsigned int KNetmapNmapXMLHandler::endTime() const
{
  return m_etime;
}

bool KNetmapNmapXMLHandler::startElement(const QString &,
					 const QString &,
					 const QString &name,
					 const QXmlAttributes &attr)
{
  static QRegExp paren("^\\s*\\((.+)\\)\\s*$");

  if (inPort)
    {
      if ((name == "state") &&
	  (attr.value("state") != "open"))
	{
	  delete port;
	  inPort = false;
	}
      else if (name == "service")
	{
	  QString extrainfo = attr.value("extrainfo");
	  if (paren.search(extrainfo) != -1)
	    extrainfo = paren.cap(1);

	  int i = attr.index("tunnel");
	  if (i != -1)
	    {
	      if (extrainfo.isEmpty())
		extrainfo = QString("Through %1 tunnel").arg(attr.value(i));
	      else
		extrainfo += QString(", through %1 tunnel").arg(attr.value(i));
	    }
	  port->setServiceInfo(attr.value("name"),
			       attr.value("product"),
			       attr.value("version"),
			       extrainfo);
	}
    }
  else if (inHost)
    {
      if ((name == "status") &&
	  (attr.value("state") == "up"))
	{
	  hostUp = true;
	}
      else if ((name == "address") &&
	       hostUp              &&
	       (attr.value("addrtype") == "ipv4"))
	{
	  IPv4 ip;
	  if (ip.set(attr.value("addr")) &&
	      !(ip.isNetwork() || ip.isBroadcast()))
	    {
	      host = parser->getApp()->getDocument()->addHost(ip);
	    }
	}
      else if ((name == "hostname") && host)
	{
	  host->setHostname(attr.value("name"));
	}
      else if ((name == "addport") && host &&
	       (attr.value("state" == "open")))
	{
	  bool ok;
	  int number = attr.value("portid").toInt(&ok);
	  if (ok)
	    {
	      if ((attr.value("protocol") == "tcp") &&
		  !host->findPort(KNetmapPort::Tcp, number))
		(void) new KNetmapPort(host, KNetmapPort::Tcp, number);
	      else if ((attr.value("protocol") == "udp") &&
		       !host->findPort(KNetmapPort::Udp, number))
		(void) new KNetmapPort(host, KNetmapPort::Udp, number);
	    }
	}
      else if ((name == "port") && host)
	{
	  bool ok;
	  int number = attr.value("portid").toInt(&ok);
	  if (ok)
	    {
	      if ((attr.value("protocol") == "tcp") &&
		  ((port = host->findPort(KNetmapPort::Tcp, number)) == 0))
		port = new KNetmapPort(host, KNetmapPort::Tcp, number);
	      else if ((attr.value("protocol") == "udp") &&
		       ((port = host->findPort(KNetmapPort::Udp, number)) == 0))
		port = new KNetmapPort(host, KNetmapPort::Udp, number);
	      if (port)
		inPort = true;
	    }
	}
      else if ((name == "osclass") && host)
	{
	  if (firstOS)
	    {
	      host->clearOS();
	      firstOS = false;
	    }
	  bool ok;
	  int acc = attr.value("accuracy").toInt(&ok);
	  if (ok)
	    {
	      (void) new KNetmapOS(host,
				   attr.value("vendor"),
				   attr.value("osfamily"),
				   attr.value("osgen"),
				   attr.value("type"),
				   acc);
	    }
	}
      else if ((name == "tcpsequence") && host)
	{
	  host->setTcpSeq(QString("%1, %2 (index %3)")
			  .arg(attr.value("class"))
			  .arg(attr.value("difficulty"))
			  .arg(attr.value("index")));
	}
      else if ((name == "tcptssequence") && host)
	{
	  host->setTcpTSSeq(attr.value("class"));
	}
      else if ((name == "ipidsequence") && host)
	{
	  host->setIpidSeq(attr.value("class"));
	}
    }
  else if (name == "nmaprun")
    {
      bool ok=false;
      float ver = attr.value("xmloutputversion").toFloat(&ok);
      if (ok && (ver == 1.0))
	{
	  m_stime = attr.value("start").toUInt(&ok);
	  if (!ok)
	    m_stime=0;
	}
      else
	{
	  QString err("no xmloutputverison attribute of nmaprun.");
	  qWarning("%s", err.ascii());
	  parser->errstr = err;
	  parser->xmlFatalError = true;
	  return false;
	}
    }
  else if (name == "host")
    {
      inHost = true;
    }
  else if (name == "finished")
    {
      bool ok=false;
      m_etime = attr.value("time").toUInt(&ok);
    }
  return true;
}

bool KNetmapNmapXMLHandler::endElement(const QString &,
				       const QString &,
				       const QString &name)
{
  if (inPort)
    {
      if (name == "port")
	{
	  inPort = false;
	  port = 0;
	}
    }
  else if (inHost)
    {
      if (name == "host")
	{
	  if (host)
	    host->icon();
	  inHost  = false;
	  hostUp  = false;
	  firstOS = true;
	  host = 0;
	}
    }
  else if (name == "nmaprun")
    parser->doParseDone();
  return true;
}

bool KNetmapNmapXMLHandler::warning(const QXmlParseException &e)
{
  qWarning("XML Warning: at %i:%i %s (sysid %s pub id %s)",
	   e.lineNumber(), e.columnNumber(),
	   e.message().ascii(),
	   e.systemId().ascii(),
	   e.publicId().ascii());
  parser->xmlWarning = true;
  return true;
}

bool KNetmapNmapXMLHandler::error(const QXmlParseException &e)
{
  QString err =
    QString("XML Error: at %i:%i %s (sysid %s pub id %s)")
    .arg(e.lineNumber())
    .arg(e.columnNumber())
    .arg(e.message())
    .arg(e.systemId())
    .arg(e.publicId());
  qWarning("%s", err.ascii());
  parser->errstr = err;
  parser->xmlError = true;
  return true;
}

bool KNetmapNmapXMLHandler::fatalError(const QXmlParseException &e)
{
  QString err =
    QString("XML Fatal Error: at %i:%i %s (sysid %s pub id %s)")
    .arg(e.lineNumber())
    .arg(e.columnNumber())
    .arg(e.message())
    .arg(e.systemId())
    .arg(e.publicId());
  qWarning("%s", err.ascii());
  parser->errstr = err;
  parser->xmlFatalError = true;
  return false;
}

#include "knetmapnmapparser.moc"
