/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qlabel.h>
#include <qcheckbox.h>
#include <kurlrequester.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <kcombobox.h>
#include <klineedit.h>
#include <knuminput.h>

#include "klistviewobject.h"
#include "knetmappref.h"
#include "knetmapnmapprefpage.h"
#include "knetmapnmapscanner.h"
#include "knetmapnmapprofile.h"

KNetmapNmapPrefPage::KNetmapNmapPrefPage(KNetmapNmapScanner *scanner,
					 KNetmapPreferences *prefs)
  : KNetmapNmapPrefs(prefs->vboxPage("Nmap", i18n("Nmap Preferences"), "nmap"), "nmapPrefs")
{
  m_scanner = scanner;
  m_profile = 0;
  m_prefs = prefs;

  // Connections
  connect(m_scanner, SIGNAL(versionChanged(const QString &)), this, SLOT(updateVersion(const QString &)));
  connect(genPath, SIGNAL(textChanged(const QString &)), m_scanner, SLOT(setExecutable(const QString &)));
  connect(genPath, SIGNAL(urlSelected(const QString &)), m_scanner, SLOT(setExecutable(const QString &)));
  connect(genAutoStart, SIGNAL(toggled(bool)), m_scanner, SLOT(setAutostart(bool)));
  connect(multCombo, SIGNAL(activated(int)), m_scanner, SLOT(setScanmode(int)));
  connect(newProfile, SIGNAL(clicked()), this, SLOT(newProfile_clicked()));
  connect(delProfile, SIGNAL(clicked()), this, SLOT(delProfile_clicked()));
  connect(defProfile, SIGNAL(toggled(bool)), this, SLOT(defProfile_toggled(bool)));
  connect(profEntry, SIGNAL(textChanged(const QString &)), this, SLOT(profEntry_textChanged(const QString &)));
  connect(profileList, SIGNAL(selectionChanged(QListViewItem *)), this, SLOT(profileChanged(QListViewItem *)));
  connect(cliEntry, SIGNAL(lostFocus()), this, SLOT(cliEntry_lostFocus()));
  connect(sourceIfaceCheck, SIGNAL(toggled(bool)), this, SLOT(setSourceIface(bool)));
  connect(sourcePortCheck, SIGNAL(toggled(bool)), this, SLOT(setSourcePort(bool)));
  connect(sourceIPCheck, SIGNAL(toggled(bool)), this, SLOT(setSourceIP(bool)));
  connect(sourceIfaceValue, SIGNAL(textChanged(const QString &)), m_scanner, SLOT(setSourceIface(const QString &)));
  connect(sourcePortValue, SIGNAL(textChanged(const QString &)), m_scanner, SLOT(setSourcePort(const QString &)));
  connect(sourceIPValue, SIGNAL(textChanged(const QString &)), m_scanner, SLOT(setSourceIP(const QString &)));
  connect(decUse, SIGNAL(toggled(bool)), m_scanner, SLOT(setUseDecoys(bool)));
  connect(this, SIGNAL(decoysModified()), this, SLOT(updateDecoys()));
  connect(profProperties, SIGNAL(currentChanged(QWidget *)), this, SLOT(tabSwitched(QWidget *)));

  profileChanged((KNetmapNmapProfile *)0);
  setScannerUI();
}

KNetmapNmapPrefPage::~KNetmapNmapPrefPage()
{
  m_scanner->replugProfiles();
}

void KNetmapNmapPrefPage::setScannerUI()
{
  // General
  genPath->setURL(m_scanner->executable());
  genAutoStart->setChecked(m_scanner->autostart());
  multCombo->setCurrentItem(m_scanner->scanmode());
  updateVersion(m_scanner->version());
  // Profiles
  profileList->blockSignals(true);
  profileList->clear();
  profileList->blockSignals(false);
  QStringList profs = m_scanner->profiles();
  for (QStringList::Iterator it=profs.begin(); it!=profs.end(); ++it)
    {
      KListViewObject *i = new KListViewObject(profileList, *it);
      KNetmapNmapProfile *prof = m_scanner->profile(*it);
      i->setRepresents(prof);
      connect(prof, SIGNAL(destroyed()), i, SLOT(deleteLater()));
    }
  // Source
  sourceIfaceCheck->setChecked(!m_scanner->sourceIface().isEmpty());
  sourceIfaceValue->setText(m_scanner->sourceIface());
  sourcePortCheck->setChecked(!m_scanner->sourcePort().isEmpty());
  sourcePortValue->setText(m_scanner->sourcePort());
  sourceIPCheck->setChecked(!m_scanner->sourceIP().isEmpty());
  sourceIPValue->setText(m_scanner->sourceIP());
  // Decoys
  decUse->setChecked(m_scanner->useDecoys());
  setDecoyList(m_scanner->decoys());
}

void KNetmapNmapPrefPage::cliEntry_lostFocus()
{
  if (m_profile)
    {
      QString newstr = cliEntry->text();
      if (m_oldstr != newstr)
	{
	  QStringList extra;
	  QString s;
	  m_profile->setFromCLI(newstr, &extra);
	  if (!extra.empty())
	    {
	      s = extra.join(" ");
	      extra.clear();
	      m_scanner->setFromCLI(s, &extra);
	      if (!extra.empty())
		{
		  s = extra.join(" ");
		}
	    }
	  setScannerUI();
	  profileChanged(m_profile);
	}
    }
}

// Profile list code:
void KNetmapNmapPrefPage::newProfile_clicked()
{
  KListViewObject *o = new KListViewObject(profileList, "Unnamed");
  KNetmapNmapProfile *prof = new KNetmapNmapProfile(m_scanner, "Unnamed");
  o->setRepresents(prof);
  connect(prof, SIGNAL(destroyed()), o, SLOT(deleteLater()));
  profileList->setCurrentItem(o);
  profileList->setSelected(o, true);
  profileList_selectionChanged(o);
  profileChanged(o);
  profEntry->setText("Unnamed");
  profEntry->setFocus();
  profEntry->selectAll();
}

void KNetmapNmapPrefPage::delProfile_clicked()
{
  if (m_profile)
    {
      delete m_profile;
      m_profile = 0;
//       profileList->setCurrentItem(0);
//       profileList->clearSelection();
//       profileList_selectionChanged(0);
//       profileChanged((KNetmapNmapProfile *)0);
    }
}

void KNetmapNmapPrefPage::defProfile_toggled(bool b)
{
  if (m_profile) m_profile->setDefault(b);
}

void KNetmapNmapPrefPage::profEntry_textChanged(const QString &s)
{
  KListViewObject *o = (KListViewObject *) profileList->selectedItem();
  if (o)
    {
      KNetmapNmapProfile *prof = (KNetmapNmapProfile *) o->represents();
      prof->setName(s);
      o->setText(0, s);
    }
}

// Profile tab code:
void KNetmapNmapPrefPage::profileChanged(QListViewItem *i)
{
  KNetmapNmapProfile *p=0;
  if (i)
    p = (KNetmapNmapProfile *)((KListViewObject *)i)->represents();
  profileChanged(p);
}

void KNetmapNmapPrefPage::profileChanged(KNetmapNmapProfile *p)
{
  if (m_profile)
    { // Disconnect old profile
      disconnect(m_profile, SIGNAL(modified()), this, SLOT(updateCLI()));
      disconnect(scanTypeEntry, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setZombie(const QString &)));
      disconnect(discACKPorts, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setTCPPingPorts(const QString &)));
      disconnect(discSYNPorts, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setSYNPingPorts(const QString &)));
      disconnect(discUDPPorts, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setUDPPingPorts(const QString &)));
      disconnect(timCombo, SIGNAL(activated(int)), m_profile, SLOT(setTiming(int)));
      disconnect(timParMinValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimParMin(int)));
      disconnect(timParMaxValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimParMax(int)));
      disconnect(timRttInitValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimRttInit(int)));
      disconnect(timRttMinValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimRttMin(int)));
      disconnect(timRttMaxValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimRttMax(int)));
      disconnect(timTTLValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimTTL(int)));
      disconnect(timHTimeValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimHTime(int)));
      disconnect(timDelayValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimDelay(int)));
      disconnect(optPLoadValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setDatalen(int)));
      disconnect(optSockValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setMaxSockets(int)));
      disconnect(scanPortsRangeEntry, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setPortList(const QString &)));
      disconnect(tcpScanCombo, SIGNAL(activated(int)), this, SLOT(setScanType(int)));
      disconnect(scanUDP, SIGNAL(toggled(bool)), this, SLOT(scanUDP_toggled(bool)));
      disconnect(scanPortsCombo, SIGNAL(activated(int)), this, SLOT(setScanPorts(int)));
      disconnect(scanRPC, SIGNAL(toggled(bool)), this, SLOT(setRPCScan(bool)));
      disconnect(scanIdentd, SIGNAL(toggled(bool)), this, SLOT(setIdentdScan(bool)));
      disconnect(scanVersion, SIGNAL(toggled(bool)), this, SLOT(setVersionScan(bool)));
      disconnect(scanOS, SIGNAL(toggled(bool)), this, SLOT(setOSScan(bool)));
      disconnect(discDontPing, SIGNAL(toggled(bool)), this, SLOT(setNoPing(bool)));
      disconnect(discACKCheck, SIGNAL(toggled(bool)), this, SLOT(setACKPing(bool)));
      disconnect(discSYNCheck, SIGNAL(toggled(bool)), this, SLOT(setSYNPing(bool)));
      disconnect(discUDPCheck, SIGNAL(toggled(bool)), this, SLOT(setUDPPing(bool)));
      disconnect(discEchoCheck, SIGNAL(toggled(bool)), this,  SLOT(setEchoPing(bool)));
      disconnect(discTStampCheck, SIGNAL(toggled(bool)), this, SLOT(setTStampPing(bool)));
      disconnect(discNetmaskCheck, SIGNAL(toggled(bool)), this, SLOT(setMaskPing(bool)));
      disconnect(timParMinCheck, SIGNAL(toggled(bool)), this, SLOT(timParMinToggled(bool)));
      disconnect(timParMaxCheck, SIGNAL(toggled(bool)), this, SLOT(timParMaxToggled(bool)));
      disconnect(timRttInitCheck, SIGNAL(toggled(bool)), this, SLOT(timRttInitToggled(bool)));
      disconnect(timRttMinCheck, SIGNAL(toggled(bool)), this, SLOT(timRttMinToggled(bool)));
      disconnect(timRttMaxCheck, SIGNAL(toggled(bool)), this, SLOT(timRttMaxToggled(bool)));
      disconnect(timTTLCheck, SIGNAL(toggled(bool)), this, SLOT(timTTLToggled(bool)));
      disconnect(timHTimeCheck, SIGNAL(toggled(bool)), this, SLOT(timHTimeToggled(bool)));
      disconnect(timDelayCheck, SIGNAL(toggled(bool)), this, SLOT(timDelayToggled(bool)));
      disconnect(optRDNSCombo, SIGNAL(activated(int)), this, SLOT(setRDNS(int)));
      disconnect(optPLoadCheck, SIGNAL(toggled(bool)), this, SLOT(optPLoadToggled(bool)));
      disconnect(optSockCheck, SIGNAL(toggled(bool)), this, SLOT(optSockToggled(bool)));
      disconnect(optOSLimit, SIGNAL(toggled(bool)), this, SLOT(setOSLimit(bool)));
      disconnect(optFrag, SIGNAL(toggled(bool)), this, SLOT(setFrag(bool)));
      disconnect(optRandHost, SIGNAL(toggled(bool)), this, SLOT(setRandHost(bool)));
      disconnect(optRandPort, SIGNAL(toggled(bool)), this, SLOT(setRandPort(bool)));
    } 

  m_profile = p;

  if (m_profile)
    {
      defProfile->setOn(m_profile->isDefault());
      m_oldstr = m_profile->cmdline(KNetmapNmapProfile::hostScan, true).join(" ");
      cliEntry->setText(m_oldstr);

      // Scanning
      int i = m_profile->scan();// & (KNetmapNmapProfile::MethodMask & 0xffffffff);
      if (i & KNetmapNmapProfile::TCPScan)
	tcpScanCombo->setCurrentItem(ScanConnect);
      else if (i & KNetmapNmapProfile::SYNScan)
	tcpScanCombo->setCurrentItem(ScanSYN);
      else if (i & KNetmapNmapProfile::ACKScan)
	tcpScanCombo->setCurrentItem(ScanACK);
//       else if (i & KNetmapNmapProfile::FINACKScan)
// 	tcpScanCombo->setCurrentItem(ScanFINACK);
      else if (i & KNetmapNmapProfile::FINScan)
	tcpScanCombo->setCurrentItem(ScanFIN);
      else if (i & KNetmapNmapProfile::NullScan)
	tcpScanCombo->setCurrentItem(ScanNULL);
      else if (i & KNetmapNmapProfile::XMasScan)
	tcpScanCombo->setCurrentItem(ScanXMas);
      else if (i & KNetmapNmapProfile::WindowScan)
	tcpScanCombo->setCurrentItem(ScanTCPWin);
      else if (i & KNetmapNmapProfile::IdleScan)
	{
	  tcpScanCombo->setCurrentItem(ScanIdle);
	  scanTypeEntry->setText(m_profile->zombie());
	}
      scanUDP->setChecked(i & KNetmapNmapProfile::UDPScan);
      scanRPC->setChecked(i & KNetmapNmapProfile::RPCScan);
      scanVersion->setChecked(i & KNetmapNmapProfile::VersionScan);
      scanIdentd->setChecked(m_profile->flags() & KNetmapNmapProfile::Ident);
      scanOS->setChecked(m_profile->flags() & KNetmapNmapProfile::OS);
      tcpScanCombo_activated(tcpScanCombo->currentItem());

      i = m_profile->ports();
      switch (i)
	{
	case KNetmapNmapProfile::portsDefault:
	  scanPortsCombo->setCurrentItem(portsDefault);
	  break;
	case KNetmapNmapProfile::portsAll:
	  scanPortsCombo->setCurrentItem(portsAll);
	  break;
	case KNetmapNmapProfile::portsFast:
	  scanPortsCombo->setCurrentItem(portsFast);
	  break;
	case KNetmapNmapProfile::portsRange:
	  scanPortsCombo->setCurrentItem(portsRange);
	  scanPortsRangeEntry->setText(m_profile->portsList().join(","));
	  break;
	};
      scanPortsCombo_activated(scanPortsCombo->currentItem());

      // Discovery
      i = m_profile->ping();
      if (i & KNetmapNmapProfile::NoPing)
	{
	  discDontPing->setChecked(true);
	  discACKCheck->setChecked(false);
	  discSYNCheck->setChecked(false);
	  discUDPCheck->setChecked(false);
	  discEchoCheck->setChecked(false);
	  discTStampCheck->setChecked(false);
	  discNetmaskCheck->setChecked(false);
	  discACKPorts->setText(QString::null);
	  discSYNPorts->setText(QString::null);
	  discUDPPorts->setText(QString::null);
	}
      else
	{
	  discDontPing->setChecked(false);
	  discACKCheck->setChecked(i & KNetmapNmapProfile::TCPPing);
	  discSYNCheck->setChecked(i & KNetmapNmapProfile::SYNPing);
	  discUDPCheck->setChecked(i & KNetmapNmapProfile::UDPPing);
	  discEchoCheck->setChecked(i & KNetmapNmapProfile::ICMPEchoPing);
	  discTStampCheck->setChecked(i & KNetmapNmapProfile::ICMPTSPing);
	  discNetmaskCheck->setChecked(i & KNetmapNmapProfile::ICMPMaskPing);
	  if(i & KNetmapNmapProfile::TCPPing)
	    discACKPorts->setText(m_profile->TCPPingPortsString());
	  else
	    discACKPorts->setText(QString::null);
	  if(i & KNetmapNmapProfile::SYNPing)
	    discSYNPorts->setText(m_profile->SYNPingPortsString());
	  else
	    discSYNPorts->setText(QString::null);
	  if(i & KNetmapNmapProfile::UDPPing)
	    discUDPPorts->setText(m_profile->UDPPingPortsString());
	  else
	    discUDPPorts->setText(QString::null);
	}

      // Timing
      switch (m_profile->timing())
	{
	case KNetmapNmapProfile::Paranoid:
	  timCombo->setCurrentItem(paranoidTiming);
	  break;
	case KNetmapNmapProfile::Sneaky:
	  timCombo->setCurrentItem(sneakyTiming);
	  break;
	case KNetmapNmapProfile::Polite:
	  timCombo->setCurrentItem(politeTiming);
	  break;
	case KNetmapNmapProfile::Normal:
	  timCombo->setCurrentItem(normalTiming);
	  break;
	case KNetmapNmapProfile::Aggressive:
	  timCombo->setCurrentItem(aggressiveTiming);
	  break;
	case KNetmapNmapProfile::Insane:
	  timCombo->setCurrentItem(insaneTiming);
	  break;
	};

      //   Parallelism:
      if (m_profile->timParMin() == -1)
	timParMinCheck->setChecked(false);
      else
	{
	  timParMinCheck->setChecked(true);
	  timParMinValue->setValue(m_profile->timParMin());
	}
      if (m_profile->timParMax() == -1)
	timParMaxCheck->setChecked(false);
      else
	{
	  timParMaxCheck->setChecked(true);
	  timParMaxValue->setValue(m_profile->timParMax());
	}

      //   RTT:
      if (m_profile->timRttInit() == -1)
	timRttInitCheck->setChecked(false);
      else
	{
	  timRttInitCheck->setChecked(true);
	  timRttInitValue->setValue(m_profile->timRttInit());
	}
      if (m_profile->timRttInit() == -1)
	timRttInitCheck->setChecked(false);
      else
	{
	  timRttMinCheck->setChecked(true);
	  timRttMinValue->setValue(m_profile->timRttMin());
	}
      if (m_profile->timRttMax() == -1)
	timRttMaxCheck->setChecked(false);
      else
	{
	  timRttMaxCheck->setChecked(true);
	  timRttMaxValue->setValue(m_profile->timRttMax());
	}

      //   Misc:
      if (m_profile->timTTL() == -1)
	timTTLCheck->setChecked(false);
      else
	{
	  timTTLCheck->setChecked(true);
	  timTTLValue->setValue(m_profile->timTTL());
	}
      if (m_profile->timHTime() == -1)
	timHTimeCheck->setChecked(false);
      else
	{
	  timHTimeCheck->setChecked(true);
	  timHTimeValue->setValue(m_profile->timHTime());
	}
      if (m_profile->timDelay() == -1)
	timDelayCheck->setChecked(false);
      else
	{
	  timDelayCheck->setChecked(true);
	  timDelayValue->setValue(m_profile->timDelay());
	}

      // Misc options:
      //   RDNS:
      if (m_profile->flags() & KNetmapNmapProfile::AlwaysDNS)
	optRDNSCombo->setCurrentItem(rdnsAlways);
      else if (m_profile->flags() & KNetmapNmapProfile::NoDNS)
	optRDNSCombo->setCurrentItem(rdnsNever);
      else
	optRDNSCombo->setCurrentItem(rdnsReq);
      //   Payload:
      if (m_profile->datalen() == -1)
	optPLoadCheck->setChecked(false);
      else
	{
	  optPLoadCheck->setChecked(true);
	  optPLoadValue->setValue(m_profile->datalen());
	}
      //   Max Sockets:
      if (m_profile->maxSockets() == -1)
	optSockCheck->setChecked(false);
      else
	{
	  optSockCheck->setChecked(true);
	  optSockValue->setValue(m_profile->maxSockets());
	}
      optOSLimit->setChecked(m_profile->flags() & KNetmapNmapProfile::OSLimit);
      optFrag->setChecked(m_profile->flags() & KNetmapNmapProfile::Fragment);
      optRandHost->setChecked(m_profile->flags() & KNetmapNmapProfile::RandomizeHosts);
      optRandPort->setChecked(!(m_profile->flags() & KNetmapNmapProfile::OrderedPorts));

      // Connect to this profile:
      connect(m_profile, SIGNAL(modified()), this, SLOT(updateCLI()));
      connect(scanTypeEntry, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setZombie(const QString &)));
      connect(discACKPorts, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setTCPPingPorts(const QString &)));
      connect(discSYNPorts, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setSYNPingPorts(const QString &)));
      connect(discUDPPorts, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setUDPPingPorts(const QString &)));
      connect(timCombo, SIGNAL(activated(int)), m_profile, SLOT(setTiming(int)));
      connect(timParMinValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimParMin(int)));
      connect(timParMaxValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimParMax(int)));
      connect(timRttInitValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimRttInit(int)));
      connect(timRttMinValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimRttMin(int)));
      connect(timRttMaxValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimRttMax(int)));
      connect(timTTLValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimTTL(int)));
      connect(timHTimeValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimHTime(int)));
      connect(timDelayValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setTimDelay(int)));
      connect(optPLoadValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setDatalen(int)));
      connect(optSockValue, SIGNAL(valueChanged(int)), m_profile, SLOT(setMaxSockets(int)));
      connect(scanPortsRangeEntry, SIGNAL(textChanged(const QString &)), m_profile, SLOT(setPortList(const QString &)));
      connect(tcpScanCombo, SIGNAL(activated(int)), this, SLOT(setScanType(int)));
      connect(scanUDP, SIGNAL(toggled(bool)), this, SLOT(scanUDP_toggled(bool)));
      connect(scanPortsCombo, SIGNAL(activated(int)), this, SLOT(setScanPorts(int)));
      connect(scanRPC, SIGNAL(toggled(bool)), this, SLOT(setRPCScan(bool)));
      connect(scanIdentd, SIGNAL(toggled(bool)), this, SLOT(setIdentdScan(bool)));
      connect(scanVersion, SIGNAL(toggled(bool)), this, SLOT(setVersionScan(bool)));
      connect(scanOS, SIGNAL(toggled(bool)), this, SLOT(setOSScan(bool)));
      connect(discDontPing, SIGNAL(toggled(bool)), this, SLOT(setNoPing(bool)));
      connect(discACKCheck, SIGNAL(toggled(bool)), this, SLOT(setACKPing(bool)));
      connect(discSYNCheck, SIGNAL(toggled(bool)), this, SLOT(setSYNPing(bool)));
      connect(discUDPCheck, SIGNAL(toggled(bool)), this, SLOT(setUDPPing(bool)));
      connect(discEchoCheck, SIGNAL(toggled(bool)), this,  SLOT(setEchoPing(bool)));
      connect(discTStampCheck, SIGNAL(toggled(bool)), this, SLOT(setTStampPing(bool)));
      connect(discNetmaskCheck, SIGNAL(toggled(bool)), this, SLOT(setMaskPing(bool)));
      connect(timParMinCheck, SIGNAL(toggled(bool)), this, SLOT(timParMinToggled(bool)));
      connect(timParMaxCheck, SIGNAL(toggled(bool)), this, SLOT(timParMaxToggled(bool)));
      connect(timRttInitCheck, SIGNAL(toggled(bool)), this, SLOT(timRttInitToggled(bool)));
      connect(timRttMinCheck, SIGNAL(toggled(bool)), this, SLOT(timRttMinToggled(bool)));
      connect(timRttMaxCheck, SIGNAL(toggled(bool)), this, SLOT(timRttMaxToggled(bool)));
      connect(timTTLCheck, SIGNAL(toggled(bool)), this, SLOT(timTTLToggled(bool)));
      connect(timHTimeCheck, SIGNAL(toggled(bool)), this, SLOT(timHTimeToggled(bool)));
      connect(timDelayCheck, SIGNAL(toggled(bool)), this, SLOT(timDelayToggled(bool)));
      connect(optRDNSCombo, SIGNAL(activated(int)), this, SLOT(setRDNS(int)));
      connect(optPLoadCheck, SIGNAL(toggled(bool)), this, SLOT(optPLoadToggled(bool)));
      connect(optSockCheck, SIGNAL(toggled(bool)), this, SLOT(optSockToggled(bool)));
      connect(optOSLimit, SIGNAL(toggled(bool)), this, SLOT(setOSLimit(bool)));
      connect(optFrag, SIGNAL(toggled(bool)), this, SLOT(setFrag(bool)));
      connect(optRandHost, SIGNAL(toggled(bool)), this, SLOT(setRandHost(bool)));
      connect(optRandPort, SIGNAL(toggled(bool)), this, SLOT(setRandPort(bool)));
    }
  else
    {
      cliEntry->setText(QString::null);
      defProfile->setOn(false);
      tcpScanCombo->setCurrentItem(ScanConnect);
      scanTypeEntry->setText(QString::null);
      scanPortsCombo->setCurrentItem(portsDefault);
      scanPortsRangeEntry->setText(QString::null);
      scanRPC->setChecked(false);
      scanIdentd->setChecked(false);
      scanVersion->setChecked(false);
      scanOS->setChecked(false);
      discDontPing->setChecked(true);
      discACKCheck->setChecked(false);
      discSYNCheck->setChecked(false);
      discUDPCheck->setChecked(false);
      discEchoCheck->setChecked(false);
      discTStampCheck->setChecked(false);
      discNetmaskCheck->setChecked(false);
      discACKPorts->setText(QString::null);
      discSYNPorts->setText(QString::null);
      discUDPPorts->setText(QString::null);
      timCombo->setCurrentItem(normalTiming);
      timParMinCheck->setChecked(false);
      timParMinValue->setValue(0);
      timParMaxCheck->setChecked(false);
      timParMaxValue->setValue(0);
      timRttInitCheck->setChecked(false);
      timRttInitValue->setValue(0);
      timRttMinCheck->setChecked(false);
      timRttMinValue->setValue(0);
      timRttMaxCheck->setChecked(false);
      timRttMaxValue->setValue(0);
      timTTLCheck->setChecked(false);
      timTTLValue->setValue(0);
      timHTimeCheck->setChecked(false);
      timHTimeValue->setValue(0);
      timDelayCheck->setChecked(false);
      timDelayValue->setValue(0);
      optRDNSCombo->setCurrentItem(rdnsReq);
      optPLoadCheck->setChecked(false);
      optPLoadValue->setValue(0);
      optOSLimit->setChecked(false);
      optFrag->setChecked(false);
      optRandHost->setChecked(false);
      optRandPort->setChecked(false);
    }
  resizeMePlease();
}

void KNetmapNmapPrefPage::updateCLI()
{
  if (m_profile)
    cliEntry->setText(m_profile->cmdline(KNetmapNmapProfile::hostScan, true).join(" "));
}

void KNetmapNmapPrefPage::updateVersion(const QString &ver)
{
  if (ver.isEmpty())
    {
      genVersionIcon->setPixmap(KGlobal::iconLoader()->loadIcon("cancel", KIcon::Small, 16));
      genVersion->setText("Unknown, check binary.");
    }
  else
    {
      if (!m_scanner->nmapIsVersion(2, 54, 6))
	{
	  genVersionIcon->setPixmap(KGlobal::iconLoader()->loadIcon("cancel", KIcon::Small, 16));
	  genVersion->setText(QString("NMap is too old (version %1)").arg(ver));
	}
      else if (!m_scanner->nmapIsVersion(3, 48))
	{
	  genVersionIcon->setPixmap(KGlobal::iconLoader()->loadIcon("messagebox_warning", KIcon::Small, 16));
	  genVersion->setText(QString("You are running NMap %1").arg(ver));
	}
      else
	{
	  genVersionIcon->setPixmap(KGlobal::iconLoader()->loadIcon("ok", KIcon::Small, 16));
	  genVersion->setText(ver);
	}
    }
}

void KNetmapNmapPrefPage::setScanType(int v)
{
  if (!m_profile) return;
  int i = m_profile->scan() & (KNetmapNmapProfile::MethodMask ^ 0xffffffff);

  switch (v)
    {
    case ScanConnect:
      m_profile->setScan(i | KNetmapNmapProfile::TCPScan);
      break;
    case ScanSYN:
      m_profile->setScan(i | KNetmapNmapProfile::SYNScan);
      break;
    case ScanACK:
      m_profile->setScan(i | KNetmapNmapProfile::ACKScan);
      break;
    case ScanFIN:
      m_profile->setScan(i | KNetmapNmapProfile::FINScan);
      break;
//     case ScanFINACK:
//       m_profile->setScan(i | KNetmapNmapProfile::FINACKScan);
//       break;
    case ScanNULL:
      m_profile->setScan(i | KNetmapNmapProfile::NullScan);
      break;
    case ScanXMas:
      m_profile->setScan(i | KNetmapNmapProfile::XMasScan);
      break;
    case ScanTCPWin:
      m_profile->setScan(i | KNetmapNmapProfile::WindowScan);
      break;
    case ScanIdle:
      m_profile->setScan(i | KNetmapNmapProfile::IdleScan);
      break;
    };
}

void KNetmapNmapPrefPage::setScanPorts(int v)
{
  if (!m_profile) return;
  switch (v)
    {
    case portsDefault:
      m_profile->setPorts(KNetmapNmapProfile::portsDefault);
      break;
    case portsAll:
      m_profile->setPorts(KNetmapNmapProfile::portsAll);
      break;
    case portsFast:
      m_profile->setPorts(KNetmapNmapProfile::portsFast);
      break;
    case portsRange:
      m_profile->setPorts(KNetmapNmapProfile::portsRange);
      break;
    };
}

void KNetmapNmapPrefPage::setRPCScan(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addScan(KNetmapNmapProfile::RPCScan);
  else
    m_profile->remScan(KNetmapNmapProfile::RPCScan);
}

void KNetmapNmapPrefPage::setIdentdScan(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addFlag(KNetmapNmapProfile::Ident);
  else
    m_profile->remFlag(KNetmapNmapProfile::Ident);
}

void KNetmapNmapPrefPage::setVersionScan(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addScan(KNetmapNmapProfile::VersionScan);
  else
    m_profile->remScan(KNetmapNmapProfile::VersionScan);
}

void KNetmapNmapPrefPage::setOSScan(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addFlag(KNetmapNmapProfile::OS);
  else
    m_profile->remFlag(KNetmapNmapProfile::OS);
}

void KNetmapNmapPrefPage::setNoPing(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setPing(KNetmapNmapProfile::NoPing);
  else
    m_profile->setPing(KNetmapNmapProfile::DefaultPing);
}

void KNetmapNmapPrefPage::setACKPing(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addPing(KNetmapNmapProfile::TCPPing);
  else
    m_profile->remPing(KNetmapNmapProfile::TCPPing);
}

void KNetmapNmapPrefPage::setSYNPing(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addPing(KNetmapNmapProfile::SYNPing);
  else
    m_profile->remPing(KNetmapNmapProfile::SYNPing);
}

void KNetmapNmapPrefPage::setUDPPing(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addPing(KNetmapNmapProfile::UDPPing);
  else
    m_profile->remPing(KNetmapNmapProfile::UDPPing);
}

void KNetmapNmapPrefPage::setEchoPing(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addPing(KNetmapNmapProfile::ICMPEchoPing);
  else
    m_profile->remPing(KNetmapNmapProfile::ICMPEchoPing);
}

void KNetmapNmapPrefPage::setTStampPing(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addPing(KNetmapNmapProfile::ICMPTSPing);
  else
    m_profile->remPing(KNetmapNmapProfile::ICMPTSPing);
}

void KNetmapNmapPrefPage::setMaskPing(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addPing(KNetmapNmapProfile::ICMPMaskPing);
  else
    m_profile->remPing(KNetmapNmapProfile::ICMPMaskPing);
}

void KNetmapNmapPrefPage::timParMinToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setTimParMin(0);
  else
    m_profile->setTimParMin(-1);
}

void KNetmapNmapPrefPage::timParMaxToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setTimParMax(0);
  else
    m_profile->setTimParMax(-1);
}

void KNetmapNmapPrefPage::timRttInitToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setTimRttInit(0);
  else
    m_profile->setTimRttInit(-1);
}

void KNetmapNmapPrefPage::timRttMinToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setTimRttMin(0);
  else
    m_profile->setTimRttMin(-1);
}

void KNetmapNmapPrefPage::timRttMaxToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setTimRttMax(0);
  else
    m_profile->setTimRttMax(-1);
}

void KNetmapNmapPrefPage::timTTLToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setTimTTL(0);
  else
    m_profile->setTimTTL(-1);
}

void KNetmapNmapPrefPage::timHTimeToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setTimHTime(0);
  else
    m_profile->setTimHTime(-1);
}

void KNetmapNmapPrefPage::timDelayToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setTimDelay(0);
  else
    m_profile->setTimDelay(-1);
}

void KNetmapNmapPrefPage::setRDNS(int v)
{
  if (!m_profile) return;
  switch (v)
    {
    case rdnsAlways:
      m_profile->addFlag(KNetmapNmapProfile::AlwaysDNS);
      m_profile->remFlag(KNetmapNmapProfile::NoDNS);
      break;
    case rdnsReq:
      m_profile->remFlag(KNetmapNmapProfile::AlwaysDNS);
      m_profile->remFlag(KNetmapNmapProfile::NoDNS);
      break;
    case rdnsNever:
      m_profile->remFlag(KNetmapNmapProfile::AlwaysDNS);
      m_profile->addFlag(KNetmapNmapProfile::NoDNS);
      break;
    };
}

void KNetmapNmapPrefPage::optPLoadToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setDatalen(0);
  else
    m_profile->setDatalen(-1);
}

void KNetmapNmapPrefPage::optSockToggled(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->setMaxSockets(0);
  else
    m_profile->setMaxSockets(-1);
}

void KNetmapNmapPrefPage::setOSLimit(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addFlag(KNetmapNmapProfile::OSLimit);
  else
    m_profile->remFlag(KNetmapNmapProfile::OSLimit);
}

void KNetmapNmapPrefPage::setFrag(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addFlag(KNetmapNmapProfile::Fragment);
  else
    m_profile->remFlag(KNetmapNmapProfile::Fragment);
}

void KNetmapNmapPrefPage::setRandHost(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->addFlag(KNetmapNmapProfile::RandomizeHosts);
  else
    m_profile->remFlag(KNetmapNmapProfile::RandomizeHosts);
}

void KNetmapNmapPrefPage::setRandPort(bool v)
{
  if (!m_profile) return;
  if (v)
    m_profile->remFlag(KNetmapNmapProfile::OrderedPorts);
  else
    m_profile->addFlag(KNetmapNmapProfile::OrderedPorts);
}

void KNetmapNmapPrefPage::updateDecoys()
{
  m_scanner->setDecoys(decoyList());
}

void KNetmapNmapPrefPage::setSourceIface(bool b)
{
  if (!b) m_scanner->setSourceIface(QString::null);
}

void KNetmapNmapPrefPage::setSourcePort(bool b)
{
  if (!b) m_scanner->setSourcePort(QString::null);
}

void KNetmapNmapPrefPage::setSourceIP(bool b)
{
  if (!b) m_scanner->setSourceIP(QString::null);
}

void KNetmapNmapPrefPage::scanUDP_toggled(bool b)
{
  if (b) m_profile->addScan(KNetmapNmapProfile::UDPScan);
  else   m_profile->remScan(KNetmapNmapProfile::UDPScan);
}

void KNetmapNmapPrefPage::tabSwitched(QWidget *)
{
  resizeMePlease();
}

void KNetmapNmapPrefPage::resizeMePlease()
{
  adjustSize();
  parentWidget()->adjustSize();
  m_prefs->adjustSize();
}

#include "knetmapnmapprefpage.moc"
