/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qobjectlist.h>
#include <qregexp.h>

#include "knetmapnmapprofile.h"
#include "knetmapnmapscanner.h"

static QRegExp listsep("(,|\\s+)");

const QValueList<int> splitToIntList(const QRegExp &sep, const QString &s)
{
  QValueList<int> l;
  QStringList sl = QStringList::split(sep, s);
  bool ok;
  int i;
  for (QStringList::Iterator it=sl.begin(); it!=sl.end(); ++it)
    {
      i = (*it).toInt(&ok);
      if (ok) l.append(i);
    }
  return l;
}

QString joinIntList(QValueList<int> l, const QString &sep)
{
  QStringList sl;
  for ( QValueListIterator<int> it = l.begin(); it != l.end(); ++it )
    sl.append(QString("%1").arg(*it));
  return sl.join(sep);
}


KNetmapNmapProfile::KNetmapNmapProfile(KNetmapNmapScanner *parent,
				       const char         *name)
  : QObject(parent)
{
  if (parent->child(name, "KNetmapNmapProfile", false))
    {
      int i=2;
      QString n = QString("%1(%2)").arg(name).arg(i);
      while (parent->child(n, "KNetmapNmapProfile", false))
	{
	  i++;
	  n = QString("%1(%2)").arg(name).arg(i);
	}
      setName(n);
    }
  else
    setName(name);

  m_pingt      = TCPPing | ICMPEchoPing;
  m_scant      = TCPScan | VersionScan;
  m_flags      = OS | OSLimit;
  m_dlen       = -1;
  m_ttl        = -1;
  m_nsox       = -1;
  m_timing     = Normal;
  m_htimeout   = -1;
  m_init_rtt   = -1;
  m_min_rtt    = -1;
  m_max_rtt    = -1;
  m_scan_delay = -1;
  m_min_par    = -1;
  m_max_par    = -1;
  m_ports      = portsDefault;
  m_default    = false;
}

KNetmapNmapProfile::KNetmapNmapProfile(KNetmapNmapScanner *parent,
				       const char         *name,
				       const QString      &s_cmdline)
// 				       KConfig            *config)
  : QObject(parent)
{
  if (parent->child(name, "KNetmapNmapProfile", false))
    {
      int i=2;
      QString n = QString("%1(%2)").arg(name).arg(i);
      while (parent->child(n, "KNetmapNmapProfile", false))
	{
	  i++;
	  n = QString("%1(%2)").arg(name).arg(i);
	}
      setName(n);
    }
  else
    setName(name);

  m_pingt      = 0;
  m_scant      = 0;
  m_flags      = 0;
  m_timing     = Normal;
  m_dlen       = -1;
  m_ttl        = -1;
  m_nsox       = -1;
  m_htimeout   = -1;
  m_init_rtt   = -1;
  m_min_rtt    = -1;
  m_max_rtt    = -1;
  m_scan_delay = -1;
  m_min_par    = -1;
  m_max_par    = -1;
  m_default    = false;
  m_ports      = portsDefault;
//   readConfig(config);
  setFromCLI(s_cmdline);
}

KNetmapNmapProfile::~KNetmapNmapProfile()
{
}

int KNetmapNmapProfile::timParMin() const { return m_min_par; }
int KNetmapNmapProfile::timParMax() const { return m_max_par; }
int KNetmapNmapProfile::timRttMin() const { return m_min_rtt; }
int KNetmapNmapProfile::timRttMax() const { return m_max_rtt; }
int KNetmapNmapProfile::timRttInit() const { return m_init_rtt; }
int KNetmapNmapProfile::timTTL() const { return m_ttl; }
int KNetmapNmapProfile::timHTime() const { return m_htimeout; }
int KNetmapNmapProfile::timDelay() const { return m_scan_delay; }
int KNetmapNmapProfile::datalen() const { return m_dlen; }
int KNetmapNmapProfile::maxSockets() const { return m_nsox; }
int KNetmapNmapProfile::ports() const { return m_ports; }
int KNetmapNmapProfile::flags() const { return m_flags; }
int KNetmapNmapProfile::ping() const { return m_pingt; }
int KNetmapNmapProfile::scan() const { return m_scant; }
int KNetmapNmapProfile::timing() const { return m_timing; }
const QString &KNetmapNmapProfile::zombie() const { return m_zombie; }
QStringList KNetmapNmapProfile::portsList() { return m_portslist; }

void KNetmapNmapProfile::setTimParMin(int i)
{
  m_min_par = i;
  emit modified();
}

void KNetmapNmapProfile::setTimParMax(int i)
{
  m_max_par = i;
  emit modified();
}

void KNetmapNmapProfile::setTimRttMin(int i)
{
  m_min_rtt = i;
  emit modified();
}

void KNetmapNmapProfile::setTimRttMax(int i)
{
  m_max_rtt = i;
  emit modified();
}

void KNetmapNmapProfile::setTimRttInit(int i)
{
  m_init_rtt = i;
  emit modified();
}

void KNetmapNmapProfile::setTimTTL(int i)
{
  m_ttl = i;
  emit modified();
}

void KNetmapNmapProfile::setTimHTime(int i)
{
  m_htimeout = i;
  emit modified();
}

void KNetmapNmapProfile::setTimDelay(int i)
{
  m_scan_delay = i;
  emit modified();
}

void KNetmapNmapProfile::setDatalen(int i)
{
  m_dlen = i;
  emit modified();
}

void KNetmapNmapProfile::setMaxSockets(int i)
{
  m_nsox = i;
  emit modified();
}

void KNetmapNmapProfile::setPorts(int i)
{
  m_ports = (scanPorts)i;
  emit modified();
}

void KNetmapNmapProfile::setTiming(int i)
{
  m_timing = (timingPreset)i;
  emit modified();
}

void KNetmapNmapProfile::setZombie(const QString &s)
{
  m_zombie = s;
  emit modified();
}

void KNetmapNmapProfile::setFlags(int i)
{
  m_flags = i;
  emit modified();
}

void KNetmapNmapProfile::addFlag(int i)
{
  m_flags |= i;
  emit modified();
}

void KNetmapNmapProfile::remFlag(int i)
{
  m_flags &= (i ^ 0xffffffff);
  emit modified();
}

void KNetmapNmapProfile::setPing(int i)
{
  m_pingt = i;
  emit modified();
}

void KNetmapNmapProfile::addPing(int i)
{
  m_pingt |= i;
  emit modified();
}

void KNetmapNmapProfile::remPing(int i)
{
  m_pingt &= (i ^ 0xffffffff);
  emit modified();
}

void KNetmapNmapProfile::setScan(int i)
{
  m_scant = i;
  emit modified();
}

void KNetmapNmapProfile::addScan(int i)
{
  m_scant |= i;
  emit modified();
}

void KNetmapNmapProfile::remScan(int i)
{
  m_scant &= (i ^ 0xffffffff);
  emit modified();
}

void KNetmapNmapProfile::setPortList(const QString &s)
{
  m_portslist = QStringList::split(listsep, s);
  emit modified();
}

bool KNetmapNmapProfile::isDefault() const
{
  return m_default;
}

void KNetmapNmapProfile::setDefault(bool def)
{
  m_default = def;
  if (m_default)
    { // Make sure we're the only one ;-)
      QObjectList *profs = parent()->queryList("KNetmapNmapProfile");
      QObjectListIt it(*profs);
      KNetmapNmapProfile *brother; // or sister
      while ((brother = (KNetmapNmapProfile *)it.current()) != 0)
	{
	  ++it;
	  if ((brother != this) && brother->isDefault())
	    brother->setDefault(false);
	}
      delete profs;
    }
}

// void KNetmapNmapProfile::saveConfig(KConfig *config)
// {
//   if (m_default)
//     config->writeEntry("default", m_default);
//   config->writeEntry("ping_method", QString().sprintf("%04X", m_pingt));
//   config->writeEntry("scan_method", QString().sprintf("%04X", m_scant));
//   config->writeEntry("flags",       QString().sprintf("%04X", m_flags));

//   if (!m_tpport.empty())
//     config->writeEntry("tcp_ping_ports", m_tpport);
//   else if (config->hasKey("tcp_ping_ports"))
//     config->deleteEntry("tcp_ping_ports");

//   if (!m_spport.empty())
//     config->writeEntry("syn_ping_ports", m_spport);
//   else if (config->hasKey("syn_ping_ports"))
//     config->deleteEntry("syn_ping_ports");

//   if (!m_upport.empty())
//     config->writeEntry("udp_ping_ports", m_upport);
//   else if (config->hasKey("udp_ping_ports"))
//     config->deleteEntry("udp_ping_ports");

//   config->writeEntry("ports", m_ports);
//   if (m_ports == portsRange)
//     config->writeEntry("portslist", m_portslist);
//   else if (config->hasKey("portslist"))
//     config->deleteEntry("portslist");

//   if (m_dlen != -1)
//     config->writeEntry("datalen", m_dlen);
//   if (m_ttl != -1)
//     config->writeEntry("ttl", m_ttl);
//   if (m_nsox != -1)
//     config->writeEntry("max_sockets", m_nsox);
//   switch (m_timing)
//     {
//     case Paranoid:
//       config->writeEntry("timing", "Paranoid");
//       break;
//     case Sneaky:
//       config->writeEntry("timing", "Sneaky");
//       break;
//     case Polite:
//       config->writeEntry("timing", "Polite");
//       break;
//     case Normal:
//       config->writeEntry("timing", "Normal");
//       break;
//     case Aggressive:
//       config->writeEntry("timing", "Aggressive");
//       break;
//     case Insane:
//       config->writeEntry("timing", "Insane");
//       break;
//     };
//   if (m_htimeout != -1)
//     config->writeEntry("htimeout",   m_htimeout);
//   if (m_init_rtt != -1)
//     config->writeEntry("init_rtt",   m_init_rtt);
//   if (m_min_rtt != -1)
//     config->writeEntry("min_rtt",    m_min_rtt);
//   if (m_max_rtt != -1)
//     config->writeEntry("max_rtt",    m_max_rtt);
//   if (m_scan_delay != -1)
//     config->writeEntry("scan_delay", m_scan_delay);
//   if (m_min_par != -1)
//     config->writeEntry("min_par",    m_min_par);
//   if (m_max_par != -1)
//     config->writeEntry("max_par",    m_max_par);
// }

// void KNetmapNmapProfile::readConfig(KConfig *config)
// {
//   bool ok;
//   m_default    = config->readBoolEntry("default");
//   m_pingt = config->readEntry("ping_method").toInt(&ok, 16);
//   if (!ok)
//     m_pingt = TCPPing | ICMPEchoPing;

//   m_scant = config->readEntry("scan_method").toInt(&ok, 16);
//   if (!ok)
//     m_scant = TCPScan | VersionScan;

//   m_flags = config->readEntry("flags").toInt(&ok, 16);
//   if (!ok)
//     m_flags = OS | OrderedPorts | OSLimit;

//   m_tpport = config->readIntListEntry("tcp_ping_ports");
//   m_spport = config->readIntListEntry("syn_ping_ports");
//   m_upport = config->readIntListEntry("udp_ping_ports");
//   m_ports = (scanPorts)config->readNumEntry("ports", portsDefault);
//   m_portslist = config->readListEntry("portslist");

//   m_dlen = config->readNumEntry("datalen", -1);
//   m_ttl  = config->readNumEntry("ttl", -1);
//   m_nsox = config->readNumEntry("max_sockets", -1);

//   QString timing = config->readEntry("timing");
//   if (timing == "Paranoid")
//     m_timing = Paranoid;
//   else if (timing == "Sneaky")
//     m_timing = Sneaky;
//   else if (timing == "Polite")
//     m_timing = Polite;
//   else if (timing == "Normal")
//     m_timing = Normal;
//   else if (timing == "Aggressive")
//     m_timing = Aggressive;
//   else if (timing == "Insane")
//     m_timing = Insane;
//   else
//     m_timing = Normal;

//   m_htimeout   = config->readNumEntry("htimeout", -1);
//   m_max_rtt    = config->readNumEntry("max_rtt", -1);
//   m_min_rtt    = config->readNumEntry("min_rtt", -1);
//   m_init_rtt   = config->readNumEntry("init_rtt", -1);
//   m_scan_delay = config->readNumEntry("scan_delay", -1);
//   m_max_par    = config->readNumEntry("max_par", -1);
//   m_min_par    = config->readNumEntry("min_par", -1);
// }

bool KNetmapNmapProfile::needRoot()
{
  KNetmapNmapScanner *scanner = (KNetmapNmapScanner *)parent();
  if (((m_pingt & (SYNPing|UDPPing)) ||
       (m_scant & (SYNScan|UDPScan|FINScan|XMasScan|NullScan|ACKScan|WindowScan)) ||
       (m_flags & (OS|Fragment))) &&
      !scanner->isSuid())
    return true;
  else
    return false;
}

void KNetmapNmapProfile::setTCPPingPorts(const QString &s)
{
  m_tpport = splitToIntList(listsep, s);
  emit modified();
}

void KNetmapNmapProfile::setSYNPingPorts(const QString &s)
{
  m_spport = splitToIntList(listsep, s);
  emit modified();
}

void KNetmapNmapProfile::setUDPPingPorts(const QString &s)
{
  m_upport = splitToIntList(listsep, s);
}

QString KNetmapNmapProfile::TCPPingPortsString()
{
  return joinIntList(m_tpport, ",");
}

QString KNetmapNmapProfile::SYNPingPortsString()
{
  return joinIntList(m_spport, ",");
}

QString KNetmapNmapProfile::UDPPingPortsString()
{
  return joinIntList(m_upport, ",");
}

void KNetmapNmapProfile::setFromCLI(const QString &s,
				    QStringList *extra)
{
  QStringList l = QStringList::split(QRegExp("\\s+"), s);
  static QRegExp r_ping("^\\-P(T|S|U)(.*)$");
  static QRegExp r_timing("^\\-T(\\d)$");

  if (l.empty()) return;
  m_pingt  = 0;
  m_scant  = 0;
  m_flags  = 0;
  m_timing = Normal;
  m_dlen       = -1;
  m_ttl        = -1;
  m_min_par    = -1;
  m_max_par    = -1;
  m_nsox       = -1;
  m_htimeout   = -1;
  m_min_rtt    = -1;
  m_max_rtt    = -1;
  m_init_rtt   = -1;
  m_scan_delay = -1;

  m_ports = portsDefault;

  for (QStringList::Iterator it=l.begin(); it!=l.end(); ++it)
    {
      if (*it == "-sP")
	m_scant |= PingScan;
      else if (*it == "-sS")
	m_scant |= SYNScan;
      else if (*it == "-sA")
	m_scant |= ACKScan;
      else if (*it == "-sT")
	m_scant |= TCPScan;
      else if (*it == "-sF")
	m_scant |= FINScan;
      else if (*it == "-sX")
	m_scant |= XMasScan;
      else if (*it == "-sN")
	m_scant |= NullScan;
      else if (*it == "-sU")
	m_scant |= UDPScan;
      else if (*it == "-sW")
	m_scant |= WindowScan;
      else if (*it == "-sI")
	{
	  m_scant |= IdleScan;
	  ++it;
	  if (it == l.end()) return;
	  m_zombie = *it;
	}
      else if (*it == "-sV")
	m_scant |= VersionScan;
      else if (*it == "-sR")
	m_scant |= RPCScan;
      else if (*it == "-sM")
	m_scant |= FINACKScan;
      else if (*it == "-P0")
	m_pingt |= NoPing;
      else if (*it == "-PE")
	m_pingt |= ICMPEchoPing;
      else if (*it == "-PP")
	m_pingt |= ICMPTSPing;
      else if (*it == "-PM")
	m_pingt |= ICMPMaskPing;
      else if (r_ping.search(*it) != -1)
	{
	  QValueList<int> ports;
	  if (!r_ping.cap(2).isEmpty())
	    ports = splitToIntList(QRegExp(","), r_ping.cap(2));

	  if (r_ping.cap(1) == "T")
	    {
	      m_pingt |= TCPPing;
	      m_tpport = ports;
	    }
	  else if (r_ping.cap(1) == "S")
	    {
	      m_pingt |= SYNPing;
	      m_spport = ports;
	    }
	  else if (r_ping.cap(1) == "U")
	    {
	      m_pingt |= UDPPing;
	      m_upport = ports;
	    }
	}
      else if (*it == "-p-")
	m_ports = portsAll;
      else if (*it == "-F")
	m_ports = portsFast;
      else if (*it == "-p")
	{
	  m_ports = portsRange;
	  ++it;
	  if (it == l.end()) return;
	  m_portslist = QStringList::split(",", *it);
	}
      else if (*it == "-O")
	m_flags |= OS;
      else if (*it == "--osscan_limit")
	m_flags |= OSLimit;
      else if (*it == "--osscan_guess")
	m_flags |= OSGuess;
      else if (*it == "-A")
	m_flags |= AggressiveMode;
      else if (*it == "-I")
	m_flags |= Ident;
      else if (*it == "-r")
	m_flags |= OrderedPorts;
      else if (*it == "-f")
	m_flags |= Fragment;
      else if (*it == "-n")
	m_flags |= NoDNS;
      else if (*it == "-R")
	m_flags |= AlwaysDNS;
      else if (*it == "--randomize_hosts")
	m_flags |= RandomizeHosts;
      else if (*it == "--data_length")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_dlen = (*it).toInt(&ok);
	  if (!ok) m_dlen = -1;
	}
      else if (*it == "--ttl")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_ttl = (*it).toInt(&ok);
	  if (!ok) m_ttl = -1;
	}
      else if (*it == "--min_parallelism")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_min_par = (*it).toInt(&ok);
	  if (!ok) m_min_par = -1;
	}
      else if (*it == "--max_parallelism")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_max_par = (*it).toInt(&ok);
	  if (!ok) m_max_par = -1;
	}
      else if (*it == "-M")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_nsox = (*it).toInt(&ok);
	  if (!ok) m_nsox = -1;
	}
      else if (*it == "--host_timeout")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_htimeout = (*it).toInt(&ok);
	  if (!ok) m_htimeout = -1;
	}
      else if (*it == "--max_rtt_timeout")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_max_rtt = (*it).toInt(&ok);
	  if (!ok) m_max_rtt = -1;
	}
      else if (*it == "--min_rtt_timeout")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_min_rtt = (*it).toInt(&ok);
	  if (!ok) m_min_rtt = -1;
	}
      else if (*it == "--initial_rtt_timeout")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_init_rtt = (*it).toInt(&ok);
	  if (!ok) m_init_rtt = -1;
	}
      else if (*it == "--scan_delay")
	{
	  bool ok;
	  ++it;
	  if (it == l.end()) return;
	  m_scan_delay = (*it).toInt(&ok);
	  if (!ok) m_scan_delay = -1;
	}
      else if (r_timing.search(*it) != -1)
	{
	  switch (r_timing.cap(1).toInt())
	    {
	    case 0:
	      m_timing = Paranoid;
	      break;
	    case 1:
	      m_timing = Sneaky;
	      break;
	    case 2:
	      m_timing = Polite;
	      break;
	    case 3:
	      m_timing = Normal;
	      break;
	    case 4:
	      m_timing = Aggressive;
	      break;
	    case 5:
	      m_timing = Insane;
	      break;
	    };
	}
      else if (extra)
	extra->append(*it);
    }
}

QStringList KNetmapNmapProfile::cmdline(int mode,
					bool argsOnly)
{
  KNetmapNmapScanner *scanner = (KNetmapNmapScanner *)parent();
  QStringList l;
  if (!argsOnly)
    l = QStringList(scanner->cmdline());

  // Ping Type:
  if (m_pingt & NoPing)
    l.append("-P0");
  if (m_pingt & TCPPing)
    l.append(QString("-PT%1").arg(joinIntList(m_tpport, ",")));
  if (m_pingt & SYNPing)
    l.append(QString("-PS%1").arg(joinIntList(m_spport, ",")));
  if (scanner->nmapIsVersion(3, 25) &&
      (m_pingt & UDPPing))
    l.append(QString("-PU%1").arg(joinIntList(m_upport, ",")));
  if (m_pingt & ICMPEchoPing)
    l.append("-PE");
  if (scanner->nmapIsVersion(2, 54, 31))
    {
      if (m_pingt & ICMPTSPing)
	l.append("-PP");
      if (m_pingt & ICMPMaskPing)
	l.append("-PM");
    }

  // Scan Type:
  if (mode == hostScan)
    {
      if (m_scant & SYNScan)
	l.append("-sS");
      if (scanner->nmapIsVersion(2, 30, 17))
	{
	  if (m_scant & ACKScan)
	    l.append("-sA");
	}
      if (m_scant & TCPScan)
	l.append("-sT");
      if (m_scant & FINScan)
	l.append("-sF");
      if (m_scant & XMasScan)
	l.append("-sX");
      if (m_scant & NullScan)
	l.append("-sN");
      if (scanner->nmapIsVersion(3, 40, 8) &&
	  (m_scant & VersionScan))
	l.append("-sV");
      if (m_scant & UDPScan)
	l.append("-sU");
      if (m_scant & WindowScan)
	l.append("-sW");
      if (scanner->nmapIsVersion(2, 54, 26) &&
	  (m_scant & IdleScan) &&
	  !m_zombie.isEmpty())
	{
	  l.append("-sI");
	  l.append(m_zombie);
	}
      if (m_scant & RPCScan)
	l.append("-sR");
      if (m_scant & FINACKScan)
	l.append("-sM");

      // Ports
      switch (m_ports)
	{
	case portsDefault:
	  break;
	case portsAll:
	  l.append("-p-");
	  break;
	case portsFast:
	  l.append("-F");
	  break;
	case portsRange:
	  if (!m_portslist.empty())
	    {
	      l.append("-p");
	      l.append(m_portslist.join(","));
	    }
	  break;
	}
    }
  else
    l.append("-sP");

  // Flags
  if (mode == hostScan)
    {
      if (m_flags & OS)
	{
	  l.append("-O");
	  if (m_flags & OSLimit)
	    l.append("--osscan_limit");
	  if (m_flags & OSGuess)
	    l.append("--osscan_guess");
	}
      if (scanner->nmapIsVersion(3, 40, 8))
	{
	  if (m_flags & AggressiveMode)
	    l.append("-A");
	}
      if (m_flags & Ident)
	l.append("-I");
      if (m_flags & OrderedPorts)
	l.append("-r");
    }
  if (m_flags & Fragment)
    l.append("-f");
  if (m_flags & NoDNS)
    l.append("-n");
  if (m_flags & AlwaysDNS)
    l.append("-R");
  if (m_flags & RandomizeHosts)
    l.append("--randomize_hosts");

  // Misc
  if (m_dlen != -1) {
    l.append("--data_length");
    l.append(QString("%1").arg(m_dlen));
  }
  if (m_ttl != -1) {
    l.append("--ttl");
    l.append(QString("%1").arg(m_ttl));
  }
  if (scanner->nmapIsVersion(3, 10, 3) &&
      (m_min_par != -1)) {
    l.append("--min_parallelism");
    l.append(QString("%1").arg(m_min_par));
  }
  if (m_max_par != -1) {
    l.append("--max_parallelism");
    l.append(QString("%1").arg(m_max_par));
  }
  if (m_nsox != -1) {
    l.append("-M");
    l.append(QString("%1").arg(m_nsox));
  }

  // Timing
  switch(m_timing)
    {
    case Paranoid:
      l.append("-T0");
      break;
    case Sneaky:
      l.append("-T1");
      break;
    case Polite:
      l.append("-T2");
      break;
    case Normal:
      // l.append("-T3");
      break;
    case Aggressive:
      l.append("-T4");
      break;
    case Insane:
      l.append("-T5");
      break;
    };

  if (m_htimeout != -1) {
    l.append("--host_timeout");
    l.append(QString("%1").arg(m_htimeout));
  }
  if (m_max_rtt != -1) {
    l.append("--max_rtt_timeout");
    l.append(QString("%1").arg(m_max_rtt));
  }
  if (m_min_rtt != -1) {
    l.append("--min_rtt_timeout");
    l.append(QString("%1").arg(m_min_rtt));
  }
  if (m_init_rtt != -1) {
    l.append("--initial_rtt_timeout");
    l.append(QString("%1").arg(m_init_rtt));
  }
  if (m_scan_delay != -1) {
    l.append("--scan_delay");
    l.append(QString("%1").arg(m_scan_delay));
  }

  // Whew!
  return l;
}

int KNetmapNmapProfile::scanStageCount() const
{
  int stages=0;
  if (m_scant & SYNScan)
    stages++;
  if (m_scant & ACKScan)
    stages++;
  if (m_scant & TCPScan)
    stages++;
  if (m_scant & FINScan)
    stages++;
  if (m_scant & XMasScan)
    stages++;
  if (m_scant & NullScan)
    stages++;
  if (m_scant & VersionScan)
    stages++;
  if (m_scant & UDPScan)
    stages++;
  if (m_scant & WindowScan)
    stages++;
//   if (m_scant & RPCScan)
//     stages++;
  if (m_flags & OS)
    stages++;

  return stages;
}

#include "knetmapnmapprofile.moc"
