/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <qheader.h>
#include <qcheckbox.h>
#include <qobjectlist.h>
#include <qregexp.h>
#include <kiconloader.h>
#include <kpushbutton.h>
#include <kio/netaccess.h>

#include "ipv4.h"
#include "knetmapnmapscanner.h"
#include "knetmapnmapparser.h"
#include "knetmapnmapprofile.h"
#include "knetmapnmapprefpage.h"
#include "knetmap.h"
#include "knetmapscanwidget.h"
#include "knetmapdoc.h"
#include "knetmaphost.h"
#include "knetmapsubnet.h"

KNetmapNmapScanner::KNetmapNmapScanner(KNetmapApp *app)
  : KNetmapScanner(app, "nmap-scanner")
{
  b_decoys = false;
  b_autostart = true;
  b_closedone = true;
  m_scanmode = separateScans;
  m_executable = QString::null;
  m_version    = QString::null;
  m_datadir    = QString::null;
  m_proc       = (KProcIO *)0;
  m_suid       = false;
  m_ver        = 0;
}

KNetmapNmapScanner::~KNetmapNmapScanner()
{
}

const QStringList KNetmapNmapScanner::decoys() const { return m_decoys; }
bool KNetmapNmapScanner::isSuid() const { return m_suid; }
bool KNetmapNmapScanner::autostart() const { return b_autostart; }
bool KNetmapNmapScanner::closedone() const { return b_closedone; }
bool KNetmapNmapScanner::useDecoys() const { return b_decoys; }
const QString KNetmapNmapScanner::executable() const { return m_executable; }
const QString KNetmapNmapScanner::version() const { return m_version; }
const QString KNetmapNmapScanner::datadir() const { return m_datadir; }
const QString KNetmapNmapScanner::sourceIP() const { return m_ip; }
const QString KNetmapNmapScanner::sourceIface() const { return m_iface; }
const QString KNetmapNmapScanner::sourcePort() const { return m_port; }
int KNetmapNmapScanner::scanmode() const { return m_scanmode; }

void KNetmapNmapScanner::setSourceIP(const QString &s) { m_ip = s; }
void KNetmapNmapScanner::setSourceIface(const QString &s) { m_iface = s; }
void KNetmapNmapScanner::setSourcePort(const QString &s) { m_port = s; }
void KNetmapNmapScanner::setAutostart(bool b) { b_autostart = b; }
void KNetmapNmapScanner::setClosedone(bool b) { b_closedone = b; }
void KNetmapNmapScanner::setScanmode(int m) { m_scanmode = m; }
void KNetmapNmapScanner::setUseDecoys(bool b) { b_decoys = b; }
void KNetmapNmapScanner::setDecoys(const QStringList &l) { m_decoys = l; }

void KNetmapNmapScanner::setFromCLI(const QString &s,
				    QStringList *extra)
{
  static QRegExp r_verb("^\\-v+$");
  QStringList l = QStringList::split(QRegExp("\\s+"), s);
  if (l.empty()) return;
  for (QStringList::Iterator it=l.begin(); it!=l.end(); ++it)
    {
      if (r_verb.search(*it) != -1)
	{}
      else if (*it == "--datadir")
	{
	  ++it;
	  if (it == l.end()) return;
	  m_datadir = *it;
	}
      else if (*it == "-D")
	{
	  ++it;
	  if (it == l.end()) return;
	  m_decoys = QStringList::split(",", *it);
	  b_decoys = !(m_decoys.empty());
	}
      else if (*it == "-S")
	{
	  ++it;
	  if (it == l.end()) return;
	  m_ip = *it;
	}
      else if (*it == "-e")
	{
	  ++it;
	  if (it == l.end()) return;
	  m_iface = *it;
	}
      else if (*it == "-g")
	{
	  ++it;
	  if (it == l.end()) return;
	  m_port = *it;
	}
      else if (extra)
	extra->append(*it);
    }
}

void KNetmapNmapScanner::setExecutable(const QString &s)
{
  m_executable = s;
  probeExecutable();
}

void KNetmapNmapScanner::setDatadir(const QString &s)
{
  m_datadir = s;
}

bool KNetmapNmapScanner::nmapIsVersion(int maj,
				       int min,
				       int subver) const
{
  return (m_ver >= MAKE_NMAP_VERSION(maj, min, subver));
}

bool KNetmapNmapScanner::nmapIsVersion(const QString &s) const
{
  static QRegExp ver("^nmap.*((\\d+)\\.(\\d+)(?:(?:ALPHA|BETA|PVT)(\\d+))?)");
  if (ver.search(s) != -1)
    return (m_ver >= MAKE_NMAP_VERSION
	    (ver.cap(2).toInt(), ver.cap(3).toInt(),
	     ((ver.numCaptures() == 4) ? ver.cap(4).toInt() : 0)));
  else
    return false;
}

void KNetmapNmapScanner::parserDone()
{
  KNetmapNmapParser *parser = (KNetmapNmapParser *)sender();
  delete parser;
}

void KNetmapNmapScanner::parserFailed()
{
  KNetmapNmapParser *parser = (KNetmapNmapParser *)sender();
  delete parser;
}


KNetmapNmapProfile *KNetmapNmapScanner::profile(const QString &name)
{
  if (name == QString::null)
    { // Find default profile
      KNetmapNmapProfile *def = 0;
      QObjectList *profs = queryList("KNetmapNmapProfile");
      QObjectListIt it(*profs);
      KNetmapNmapProfile *p;
      while ((p = (KNetmapNmapProfile *)it.current()) != 0)
	{
	  ++it;
	  if (p->isDefault())
	    {
	      def = p;
	      break;
	    }
	}
      if (!def)
	def = (KNetmapNmapProfile *)profs->getFirst();
      delete profs;
      return def;
    }
  else
    return (KNetmapNmapProfile *)child(name, "KNetmapNmapProfile", false);
}

QStringList KNetmapNmapScanner::profiles()
{
  QStringList names;
  QObjectList *profiles = queryList("KNetmapNmapProfile");
  QObjectListIt it(*profiles);
  QObject *o;

  while ((o = it.current()) != 0)
    {
      ++it;
      names.append(o->name());
    }
  delete profiles;
  return names;
}

void KNetmapNmapScanner::prefPage(KNetmapPreferences *prefs)
{
  (void) new KNetmapNmapPrefPage(this, prefs);
}

void KNetmapNmapScanner::initActions()
{
  KNetmapApp *theApp = (KNetmapApp *)parent();
  KAction *import = new KAction(i18n("&Import NMap XML"), 0,
				this, SLOT(import()),
				theApp->actionCollection(), "file_importNmap");
  KAction *scansub = new KAction(i18n("&Scan Subnet"), 0,
				 this, SLOT(internalScanSlot()),
				 theApp->actionCollection(), "subnet_scan");
  KAction *scanhost = new KAction(i18n("&Scan Host"), 0,
				  this, SLOT(internalScanSlot()),
				  theApp->actionCollection(), "host_scan");

  import->setStatusText(i18n("Loads NMap XML output into the document."));
  scansub->setStatusText(i18n("Scan the subnet for alive hosts."));
  scanhost->setStatusText(i18n("Scan the host with the default profile."));
}

void KNetmapNmapScanner::plugScanMenu(QPtrList<KAction> *menu)
{
  QObjectList *profiles = queryList("KNetmapNmapProfile");
  QObjectListIt it(*profiles);
  QStringList names;
  KNetmapNmapProfile *p;
  while ((p = (KNetmapNmapProfile *)it.current()) !=0)
    {
      ++it;
      names.append(p->name());
    }
  delete profiles;
  names.sort();
  for (QStringList::Iterator it=names.begin(); it!=names.end(); ++it)
    {
      p = (KNetmapNmapProfile *) child(*it, 0, false);
      KAction *act = new KAction(QString("NMap: %1").arg(p->name()),
				 "nmap", 0,
				 this, SLOT(scanWidgetItemActivated()),
				 this, p->name());
      menu->append(act);
    }
}

void KNetmapNmapScanner::replugProfiles()
{
  KNetmapApp *theApp = (KNetmapApp *)parent();

  QPtrList<KAction> host_scans, ping_scans;
  QObjectList *profiles = queryList("KNetmapNmapProfile");
  QObjectListIt it(*profiles);
  QStringList names;
  KNetmapNmapProfile *p;
  while ((p = (KNetmapNmapProfile *)it.current()) !=0)
    {
      ++it;
      names.append(p->name());
    }
  delete profiles;
  names.sort();
  for (QStringList::Iterator it=names.begin(); it!=names.end(); ++it)
    {
      p = (KNetmapNmapProfile *) child(*it, 0, false);
      KAction *act = new KAction(p->name(), "launch", 0,
				 this, SLOT(internalScanSlot()),
				 this, p->name());
      act->setStatusText(QString("%1: %2")
			 .arg(i18n("Scan host with profile"))
			 .arg(p->name()));
      host_scans.append(act);
      if (p->canPing())
	{
	  KAction *pact = new KAction(p->name(), "launch", 0,
				      this, SLOT(internalScanSlot()),
				      this, p->name());
	  pact->setStatusText(QString("%1: %2")
			      .arg(i18n("Scan subnet with profile"))
			      .arg(p->name()));
	  ping_scans.append(pact);
	}
    }

  theApp->unplugActionList("host_scanwith");
  theApp->unplugActionList("subnet_scanwith");
  theApp->plugActionList  ("host_scanwith",   host_scans);
  theApp->plugActionList  ("subnet_scanwith", ping_scans);

  // Tell the scan widget to rebuild its menu
  theApp->scanWidget()->updateMenu();
}

void KNetmapNmapScanner::readConfig(KConfig *config)
{
  config->setGroup("NMap");

  m_executable = config->readEntry("executable", "/usr/bin/nmap");
  probeExecutable();
  m_datadir    = config->readEntry("datadir");
  m_scanmode = config->readNumEntry("scanMode", groupedScan);
  m_ip     = config->readEntry("sourceIP");
  m_port   = config->readEntry("sourcePort");
  m_iface  = config->readEntry("sourceIface");
  b_decoys = config->readBoolEntry("useDecoys");
  b_autostart = config->readBoolEntry("autostartscans", true);
  b_closedone = config->readBoolEntry("closedone", true);
  m_decoys = config->readListEntry("decoys");
  // Read the profiles
  config->setGroup("NMapProfiles");
  int c = config->readNumEntry("count", -1);
  QString defprof = config->readEntry("default");
  for (int i=1; i<=c; i++)
    {
      QString name = config->readEntry(QString("prof%1name").arg(i));
      QString cmdl = config->readEntry(QString("prof%1cmdl").arg(i));
      if ((!name.isEmpty()) && (!cmdl.isEmpty()))
	{
	  KNetmapNmapProfile *prof =
	    new KNetmapNmapProfile(this, name, cmdl);
	  prof->setDefault(name == defprof);
	}
    }
  // Update the scanwidget
  replugProfiles();
}

void KNetmapNmapScanner::saveConfig(KConfig *config)
{
  QObjectList   *profos = queryList("KNetmapNmapProfile");
  QObjectListIt it(*profos);
  KNetmapNmapProfile *defprof=NULL, *prof=NULL;
  int i=1;
  defprof = (KNetmapNmapProfile *)(profos->getFirst());

  config->setGroup("NMapProfiles");
  config->writeEntry("count", profos->count());
  while ((prof = (KNetmapNmapProfile *)(it.current())) != 0)
    {
      if (prof->isDefault())
	defprof = prof;
      config->writeEntry
	(QString("prof%1name").arg(i), prof->name());
      config->writeEntry
	(QString("prof%1cmdl").arg(i),
	 prof->cmdline(KNetmapNmapProfile::hostScan, true).join(" "));
      i++;
      ++it;
    }
  delete profos;

  if (defprof)
    config->writeEntry("default", defprof->name());

  config->setGroup("NMap");
  config->writeEntry("executable", m_executable);
  if (!m_datadir.isNull())
    config->writeEntry("datadir", m_datadir);
  config->writeEntry("autostartscans", b_autostart);
  config->writeEntry("closedone", b_closedone);
  config->writeEntry("scanMode", m_scanmode);
  config->writeEntry("sourceIP", m_ip);
  config->writeEntry("sourcePort", m_port);
  config->writeEntry("sourceIface", m_iface);
  config->writeEntry("useDecoys", b_decoys);
  config->writeEntry("decoys", m_decoys);
}

void KNetmapNmapScanner::scanWidgetItemActivated()
{
  KNetmapNmapProfile *p = profile(sender()->name());
  if (!p)
    return;

  QString s_target = getApp()->scanWidget()->targetString();

  if (m_scanmode == groupedScan)
    {
      QStringList targets = QStringList::split(QRegExp("\\s+"), s_target);
      KNetmapNmapScan *scan = new KNetmapNmapScan(this, p, targets);
      if (b_autostart)
	scan->start();
    }
  else if (m_scanmode == separateScans)
    {
      QStringList targets = QStringList::split(QRegExp("\\s+"), s_target);

      for (QStringList::Iterator it=targets.begin(); it!=targets.end(); ++it)
	{
	  QStringList l(*it);
	  KNetmapScan *scan = new KNetmapNmapScan(this, p, l);
	  if (b_autostart)
	    scan->start();
	}
    }
}

void KNetmapNmapScanner::internalScanSlot()
{
  QString n(sender()->name());
  KNetmapNmapProfile *p = 0;
  QObject *currentItem = ((KNetmapApp *)parent())->getDocument()->current();
  QStringList targets;
  KNetmapScan::scanMode mode;

  if ((n == "subnet_scan") ||
      (n == "host_scan"))
    p = profile();
  else
    p = profile(n);
  if (!p || !currentItem) return;

  if (currentItem->inherits("KNetmapHost"))
    {
      QPtrList<KNetmapHost> l = ((KNetmapApp *)parent())->getDocument()->selectedHosts();
      KNetmapHost *h;
      mode = KNetmapScan::hostScan;
      if (m_scanmode == groupedScan)
	{
	  for (h = l.first(); h; h = l.next())
	    targets.append(h->ip().address());
	}
      else if (m_scanmode == separateScans)
	{
	  for (h = l.first(); h; h = l.next())
	    {
	      QStringList target(h->ip().address());
	      KNetmapNmapScan *scan = new KNetmapNmapScan(this, p, target, mode);
	      if (b_autostart) scan->start();
	    }
	}
    }
  else if (currentItem->inherits("KNetmapSubnet"))
    {
      QPtrList<KNetmapSubnet> l = ((KNetmapApp *)parent())->getDocument()->selectedSubnets();
      KNetmapSubnet *s;
      mode = KNetmapScan::subnetScan;
      if (m_scanmode == groupedScan)
	{
	  for (s = l.first(); s; s = l.next())
	    targets.append(s->ip().cidrstr(IPv4::Network));
	}
      else if (m_scanmode == separateScans)
	{
	  for (s = l.first(); s; s = l.next())
	    {
	      QStringList target(s->ip().cidrstr(IPv4::Network));
	      KNetmapNmapScan *scan = new KNetmapNmapScan(this, p, target, mode);
	      if (b_autostart) scan->start();
	    }
	}
    }
  else
    return;

  if (m_scanmode == groupedScan)
    {
      KNetmapNmapScan *scan = new KNetmapNmapScan(this, p, targets, mode);
      if (b_autostart) scan->start();
    }
}

void KNetmapNmapScanner::import()
{
  KURL url = KFileDialog::getOpenURL
    (QString::null,
     "*.xml",
     getApp(), i18n("Import NMap XML"));
  if (!url.isEmpty())
    {
      getApp()->slotStatusMsg(QString("%1: %2 ...")
			      .arg(i18n("Reading NMap XML from"))
			      .arg(url.path()));
      QString tmpFile;
      if (KIO::NetAccess::download(url, tmpFile, (QWidget *)parent()))
	{
	  getApp()->slotStatusMsg(i18n("Parsing NMap XML..."));
	  KNetmapNmapParser *parser = new KNetmapNmapParser(this, tmpFile);
	  connect(parser, SIGNAL(parseDone()),
		  this,   SLOT(parserDone()));
	  connect(parser, SIGNAL(parseFailed()),
		  this,   SLOT(parserFailed()));
      	  KIO::NetAccess::removeTempFile( tmpFile );
	}
    }
  getApp()->slotStatusMsg(i18n("Ready."));
}

bool KNetmapNmapScanner::needRoot()
{
  if (b_decoys ||
      !m_ip.isEmpty() ||
      !m_iface.isEmpty())
    return true;
  return false;
}

const QStringList KNetmapNmapScanner::cmdline() const
{
  QStringList l;
  l.append(m_executable);
  if (nmapIsVersion(3, 15, 2) &&
      !m_datadir.isEmpty()) {
    l.append("--datadir");
    l.append(m_datadir);
  }
  l.append("-vvv");
  if (b_decoys && !m_decoys.empty())
    {
      l.append("-D");
      l.append(m_decoys.join(","));
    }
  if (!m_ip.isEmpty())
    {
      l.append("-S");
      l.append(m_ip);
    }
  if (!m_port.isEmpty())
    {
      l.append("-g");
      l.append(m_port);
    }
  if (!m_iface.isEmpty())
    {
      l.append("-e");
      l.append(m_iface);
    }
  return l;
}

bool KNetmapNmapScanner::probeExecutable()
{
  struct stat buff;
  m_version = QString::null;
  m_ver     = 0;
  if (!m_executable.isEmpty() &&
      (stat(m_executable.ascii(), &buff) != -1) &&
      S_ISREG(buff.st_mode))
    {
      bool canExec=false;
      // Can anyone, or am I the onwer and it's executable?
      if ((buff.st_mode & S_IXOTH) ||
	  ((getuid() == buff.st_uid) &&
	   buff.st_mode & S_IXUSR))
	{
	  canExec=true;
	}
      else if (buff.st_mode & S_IXGRP)   // Group executable
	{
	  if (getgid() == buff.st_gid)   // My main group can
	    {
	      canExec=true;
	    }
	  else
	    {
	      gid_t groups[255];
	      int i;
	      i = getgroups(255, groups);
	      if (i)
		{
		  int c;
		  for (c=0; c<i; c++)
		    {
		      if (groups[c] == buff.st_gid)
			{
			  canExec=true; // One of my suplementary groups can
			  break;
			}
		    }
		}
	    }
	}
      
      if (canExec)
	{
	  // Owned by UID 0 and SUID!!! 
	  m_suid = ((buff.st_uid == 0) && buff.st_mode & S_ISUID);
	  // Now find out what version NMap is.
	  getExecVersion();
	  return true;
	}
      else
	{
	  emit versionChanged(QString::null);
	  return false;
	}
    }
  emit versionChanged(QString::null);
  return false;
}

void KNetmapNmapScanner::getExecVersion()
{
  if (!m_proc)
    {
      m_proc = new KProcIO();
      *m_proc << m_executable;
      *m_proc << "-V";
      if (!m_proc->start(KProcess::NotifyOnExit, true))
	{
	  delete m_proc;
	  m_proc = 0;
	}
      else
	{
	  connect(m_proc, SIGNAL(processExited(KProcess *)),
		  this,     SLOT(processExited(KProcess *)));
	  connect(m_proc, SIGNAL(readReady(KProcIO *)),
		  this,     SLOT(gotOutput(KProcIO *)));
	}
    }
}

void KNetmapNmapScanner::processExited(KProcess *proc)
{
  if (m_proc)
    {
      gotOutput((KProcIO *)proc);
      delete m_proc;
      m_proc = 0;
      if (m_version.isEmpty())
	{
	  qWarning("Failed to get version from nmap (%s).", m_executable.ascii());
	  emit versionChanged(QString::null);
	}
    }
}

void KNetmapNmapScanner::gotOutput(KProcIO *)
{
  static QRegExp ver("^nmap.*((\\d+)\\.(\\d+)(?:(?:ALPHA|BETA|PVT)(\\d+))?)");
  if (m_proc)
    {
      QString l;
      while (m_proc->readln(l) != -1)
	{
	  if (ver.search(l) != -1)
	    {
	      m_version = ver.cap(1);
	      m_ver = MAKE_NMAP_VERSION
		(ver.cap(2).toInt(), ver.cap(3).toInt(),
		 ((ver.numCaptures() == 4) ? ver.cap(4).toInt() : 0));
	      emit versionChanged(m_version);
	    }
	}
    }
}

////////////////////////////////////////////////////////////
KNetmapNmapScan::KNetmapNmapScan(KNetmapNmapScanner *scanner,
				 KNetmapNmapProfile *profile,
				 const QStringList &targets,
				 scanMode mode)
  : KNetmapScan(scanner, targets, mode)
{
  gotOS = false;
  osFPR = false;
  sigged = false;
  m_profile = profile;
  m_console = new KNetmapNmapConsole(getApp()->getConsole(), this);
  m_tmpfile = new KTempFile();
  m_tmpfile->close();
  m_xmlFile = m_tmpfile->name();
  parseStatus = parseNone;
  int stag = m_profile->scanStageCount();
  int targ = 0;
  int total = 0;
  for (QStringList::Iterator it=m_targets.begin(); it!=m_targets.end(); ++it)
    {
      targ += IPv4::countIP(*it);
    }
  total = targ * (stag+1);
  m_console->progress()->setTotalSteps(total);
  m_perstage = stag;

  QStringList args;

  if ((profile->needRoot() || scanner->needRoot())
      && !scanner->isSuid())
    args += getApp()->rootCmd();
  m_asroot = (profile->needRoot() || scanner->needRoot() || scanner->isSuid());

  m_mode = mode;
  if (m_mode == hostScan)
    args += profile->cmdline(KNetmapNmapProfile::hostScan);
  else if (m_mode == subnetScan)
    args += profile->cmdline(KNetmapNmapProfile::subnetScan);

  args.append("-oX");
  args.append(m_xmlFile);
  args += m_targets;

  m_text.append(QString("Invocation: %1").arg(args.join(" ")));

  m_proc = new KProcIO();

  *m_proc << args;

  parser = new KNetmapNmapParser(this);
  parseStatus = parseRunning;

  connect(parser, SIGNAL(parseDone()),
	  this,     SLOT(parserDone()));
  connect(parser, SIGNAL(parseFailed()),
	  this,     SLOT(parserFailed()));
  connect(m_proc, SIGNAL(processExited(KProcess *)),
	  this,     SLOT(procExitSlot(KProcess *)));
  connect(m_proc, SIGNAL(readReady(KProcIO *)),
	  this,     SLOT(gotOutput(KProcIO *)));
}

KNetmapNmapScan::~KNetmapNmapScan()
{
  delete m_console;
  delete m_tmpfile;
}


void KNetmapNmapScan::procExitSlot(KProcess *proc)
{
  // We'll take it from here ;-)
  disconnect(m_proc, SIGNAL(readReady(KProcIO *)),
	     this,     SLOT(gotOutput(KProcIO *)));
  m_state = scanDone;
  if (proc->normalExit())
    {
      int exitStat = proc->exitStatus();
      if (exitStat)
	{
	  if (!sigged)
	    m_console->append(KNetmapNmapConsole::errorIcon,
			      QString("NMap failed (exit code %1)")
			      .arg(exitStat));
	  emit failed();
	}
      else if (parseStatus == parseRunning)
	parser->parse();
      else if (parseStatus == parseDone)
	emit finished();
      else if (parseStatus == parseFailed)
	emit failed();
      else if (parseStatus == parseNone)
	qWarning("seems like the parser was never created *thwack*");

      gotOutput((KProcIO *)proc);
    }
  else
    {
      m_console->append(KNetmapNmapConsole::errorIcon,
			QString("NMap exited abnormally"));
      emit failed();
    }
  proc->deleteLater();
}

void KNetmapNmapScan::gotOutput(KProcIO *proc)
{
  QString l;
  while (proc->readln(l) != -1)
    handleNmapLine(l);
}

void KNetmapNmapScan::handleNmapLine(const QString &l)
{
  static QRegExp r_scanstart("^Starting nmap (?:V\\. )?([\\d.]+(?:(?:ALPHA|BETA|PVT)\\d+)?) \\(.+\\)(?: at (.+))?$");
  static QRegExp r_hostLine("^Host (.+) appears to be (\\w+).*$");
  static QRegExp r_scanLine("^Initiating (.+) against (.+)(?: at (.*))?$");
  static QRegExp r_scanDoneLine("^The (.+) took (.+) to scan (.+)\\.$");
  static QRegExp r_nmapDone("^Nmap run completed \\-\\- (\\d+) IP address \\((\\d+) hosts? up\\) scanned in (.+)$");
  static QRegExp r_osscan("^For OSScan assuming that port (\\d+) is open and port (\\d+) is closed and neither are firewalled$");
  static QRegExp r_sigexit("^caught (\\w+) signal, cleaning up$");
  static QRegExp r_noosmat("^No exact OS matches for host \\(If you know what OS is running on it, see (.+)\\)\\.$");
  static QRegExp r_noosscan("^Skipping OS Scan");

  l.stripWhiteSpace();
  if (l.isEmpty())
    {
      if (osFPR)
	{
	  if (hostip.isEmpty())
	    qWarning("Got an OS fingerprint w/o a host!");
	  else
	    {
	      KNetmapHost *h = getApp()->getDocument()->findHost(IPv4(hostip));
	      if (h)
		h->setFingerprint(fingerprint);
	      else
		qWarning("Couldn't find a host for %s", hostip.ascii());
	    }
	  fingerprint = "";
	  osFPR=false;
	}
    }
  else
    {
      if (r_scanstart.search(l) != -1)
	{
	  if (!r_scanstart.cap(2).isEmpty())
	    m_console->append(KNetmapNmapConsole::infoIcon,
			      QString("NMap %1 scan started at %2.")
			      .arg(r_scanstart.cap(1))
			      .arg(r_scanstart.cap(2)));
	  else
	    m_console->append(KNetmapNmapConsole::infoIcon,
			      QString("NMap %1 scan started.")
			      .arg(r_scanstart.cap(1)));
	}
      else if (r_noosmat.search(l) != -1)
	{
	  m_console->append(KNetmapNmapConsole::errorIcon,
			    QString("Unknown OS, see info tab for more details."));
	  osFPR=true;
	}
      else if (r_noosscan.search(l) != -1)
	{
	  m_console->append(KNetmapNmapConsole::errorIcon, "Skipping OS Scan", 1);
	}
      else if (r_sigexit.search(l) != -1)
	{
	  m_console->append(KNetmapNmapConsole::errorIcon,
			    QString("NMap exiting due to signal %1")
			    .arg(r_sigexit.cap(1)));
	  sigged=true;
	}
      else if (r_hostLine.search(l) != -1)
	{
	  QString host = r_hostLine.cap(1);
	  QString status = r_hostLine.cap(2);
	  gotOS = false;
	  if (status == "up")
	    m_console->append(KNetmapNmapConsole::hostIcon,
			      QString("Host %1 is up scanning...").arg(host), 1);
	  else
	    m_console->append(KNetmapNmapConsole::hostIcon,
			      QString("Host %1 is down, skipping.").arg(host), m_perstage+1);
	  QRegExp withDNS("^.*\\(([\\d.]+)\\)$");
	  if (withDNS.search(host) != -1)
	    hostip = withDNS.cap(1);
	  else
	    hostip = host;
	}
      else if (r_scanLine.search(l) != -1)
	{
	  if (!r_scanLine.cap(3).isEmpty())
	    m_console->append(KNetmapNmapConsole::scanIcon,
			      QString("%1ing %2 at %3...")
			      .arg(r_scanLine.cap(1))
			      .arg(r_scanLine.cap(2))
			      .arg(r_scanLine.cap(3)), 1);
	  else
	    m_console->append(KNetmapNmapConsole::scanIcon,
			      QString("%1ing %2...")
			      .arg(r_scanLine.cap(1))
			      .arg(r_scanLine.cap(2)), 1);
	}
      else if (r_scanDoneLine.search(l) != -1)
	{
	  m_console->append(KNetmapNmapConsole::timeIcon, l);
	}
      else if (r_nmapDone.search(l) != -1)
	{
	  int hostc = r_nmapDone.cap(2).toInt();
	  QString time = r_nmapDone.cap(3);
	  m_console->append(KNetmapNmapConsole::doneIcon,
			    QString("Scan done, scanned %1 hosts in %2.")
			    .arg(hostc).arg(time));
	}
      else if ((r_osscan.search(l) != -1) && !gotOS)
	{
	  int openport = r_osscan.cap(1).toInt();
	  int closedport = r_osscan.cap(2).toInt();
	  m_console->append(KNetmapNmapConsole::scanIcon,
			    QString("Fingerprinting OS, using ports %1 and %2...")
			    .arg(openport).arg(closedport), 1);
	  gotOS = true;
	}
      else if (osFPR)
	{
	  fingerprint += l + "\n";
	}
      m_text.append(l);
      emit textChanged(nmapOutput());
    }
}

void KNetmapNmapScan::start()
{
  if (m_state == scanStopped)
    {
      if (!m_proc->start(KProcess::NotifyOnExit, true))
	{
	  emit failed();
	  m_state = scanDone;
	}
      else
	{
	  emit started();
	  m_state = scanRunning;
	}
    }
}

void KNetmapNmapScan::stop()
{
  if (m_state == scanRunning)
    {
      if (m_asroot)
	{
	  KProcIO *killer = new KProcIO();
	  *killer << getApp()->rootCmd();
	  *killer << "kill";
	  *killer << QString("%1").arg(m_proc->pid());
	  if (!killer->start(KProcess::NotifyOnExit, true))
	    {
	      delete killer;
	      qWarning("Failed to start killer process, root process will remain.");
	    }
	  else
	    {
	      connect(killer, SIGNAL(processExited(KProcess *)), this, SLOT(killerExit(KProcess *)));
	    }
	}
      else
	{
	  m_proc->kill();
	}
    }
}

void KNetmapNmapScan::killerExit(KProcess *p)
{
  delete p;
}

void KNetmapNmapScan::parserDone()
{
  parseStatus = parseDone;
  if (m_state == scanDone)
    {
      emit finished();
      parser->deleteLater();
    }
}

void KNetmapNmapScan::parserFailed()
{
  parseStatus = parseFailed;
  if (m_state == scanDone)
    {
      m_console->append(KNetmapNmapConsole::errorIcon,
			parser->errorString());
      emit failed();
      parser->deleteLater();
    }
}

QString KNetmapNmapScan::nmapOutput()
{
  return m_text.join("\n");
}

KNetmapNmapScanner *KNetmapNmapScan::scanner() const
{
  return (KNetmapNmapScanner *)parent();
}

KNetmapApp *KNetmapNmapScan::getApp() const
{
  return scanner()->getApp();
}

QString KNetmapNmapScan::xmlFile() const
{
  return m_xmlFile;
}

////////////////////////////////////////////////////////////
KNetmapNmapConsole::KNetmapNmapConsole(KNetmapConsole *parent, KNetmapNmapScan *scan)
  : KNetmapConsolePage(parent,
		       QString("NMap: %1")
		       .arg((scan->targets().count() == 1) ?
			    scan->targets().first() :
			    QString("%1 %2")
			    .arg(scan->targets().count())
			    .arg((scan->getScanMode() == KNetmapScan::hostScan) ? i18n("hosts") : i18n("subnets"))),
		       QString("%1 %2").arg(i18n("NMap scan of"))
		       .arg((scan->targets().count() == 1) ?
			    scan->targets().first() :
			    QString("%1 %2")
			    .arg(scan->targets().count())
			    .arg((scan->getScanMode() == KNetmapScan::hostScan) ? i18n("hosts") : i18n("subnets"))),
		       "nmap_scan", "nmap")
{
  last = 0;
  m_scan = scan;
  m_list = new KListView(this, "list");
  m_list->addColumn("Scan Output");
  m_list->header()->hide();
  m_list->setSorting(-1);
  m_list->setSelectionMode(QListView::NoSelection);

  m_text = new KTextEdit(this, "text");
  m_text->setReadOnly(true);
  m_text->hide();

  m_prgs = new KProgress(this, "progress");
  m_prgs->setTextEnabled(true);

  QHBox *row = new QHBox(this, "row");
  QCheckBox *closedone = new QCheckBox("&Close console when scan finishes.", row, "closedone");
  closedone->setChecked(m_scan->scanner()->closedone());
  connect(closedone,       SIGNAL(toggled(bool)),
	  m_scan->scanner(), SLOT(setClosedone(bool)));
  m_butt = new KPushButton(KGlobal::iconLoader()->loadIconSet("run", KIcon::Small),
			   "&Start Scan", row, "butt");
  KPushButton *text = new KPushButton(KGlobal::iconLoader()->loadIconSet("info", KIcon::Small),
			   "Text Output", row, "text");
  text->setToggleButton(true);

  row->setStretchFactor(closedone, 1);
  row->setStretchFactor(m_butt, 0);

  m_buttstate = 0;
  connect(m_butt, SIGNAL(clicked()),
	  this,     SLOT(butt_clicked()));
  connect(text, SIGNAL(toggled(bool)),
	  this, SLOT(showStdout(bool)));
  connect(m_scan, SIGNAL(stopped()),
	  this,     SLOT(scanDone()));
  connect(m_scan, SIGNAL(finished()),
	  this,     SLOT(scanDone()));
  connect(m_scan, SIGNAL(failed()),
	  this,     SLOT(scanFailed()));
  connect(m_scan, SIGNAL(started()),
	  this,     SLOT(scanStarted()));
  addToConsole();
}

KNetmapNmapConsole::~KNetmapNmapConsole()
{
}

void KNetmapNmapConsole::showStdout(bool b)
{
  if (b)
    {
      m_list->hide();
      m_text->show();
      m_text->setText(m_scan->nmapOutput());
      connect(m_scan, SIGNAL(textChanged(const QString &)),
	      m_text, SLOT(setText(const QString &)));
    }
  else
    {
      m_list->show();
      m_text->hide();
      disconnect(m_scan, SIGNAL(textChanged(const QString &)),
		 m_text, SLOT(setText(const QString &)));
    }
}

void KNetmapNmapConsole::butt_clicked()
{
  switch (m_buttstate)
    {
    case 0:
      m_scan->start();
      break;
    case 1:
      m_scan->stop();
      break;
    case 2:
      m_scan->deleteLater();
      break;
    };
}

void KNetmapNmapConsole::append(const iconType _icon,
				const QString &text,
				const int prginc)
{
  QString icon;
  switch (_icon)
    {
    case errorIcon:
      icon = "error";
      break;
    case infoIcon:
      icon = "info";
      break;
    case hostIcon:
      icon = HOST_ICON;
      break;
    case subnetIcon:
      icon = SUBNET_ICON;
      break;
    case portIcon:
      icon = PORT_ICON;
      break;
    case scanIcon:
      icon = "launch";
      break;
    case timeIcon:
      icon = "clock";
      break;
    case doneIcon:
      icon = "finish";
      break;
    };
  append(icon, text, prginc);
  while (m_list->childCount() > 99)
    {
      delete m_list->firstChild();
    }
}

void KNetmapNmapConsole::append(const QString &icon,
				const QString &text,
				const int prginc)
{
  KListViewItem *i;
  if (last)
    i = new KListViewItem(m_list, last, text);
  else
    i = new KListViewItem(m_list, text);
  last = i;
  i->setPixmap(0, KGlobal::iconLoader()->loadIcon(icon, KIcon::Small, 16));
  m_list->ensureItemVisible(i);
  if (prginc)
    m_prgs->advance(prginc);
}

KProgress *KNetmapNmapConsole::progress() const
{
  return m_prgs;
}

KListView *KNetmapNmapConsole::listView() const
{
  return m_list;
}

void KNetmapNmapConsole::scanStarted()
{
  m_butt->setIconSet(KGlobal::iconLoader()->loadIconSet("stop", KIcon::Small));
  m_butt->setText("&Stop Scan");
  m_buttstate = 1;
}

void KNetmapNmapConsole::scanFailed()
{
  m_butt->setIconSet(KGlobal::iconLoader()->loadIconSet("fileclose", KIcon::Small));
  m_butt->setText("&Close");
  m_buttstate = 2;
}

void KNetmapNmapConsole::scanDone()
{
  m_butt->setIconSet(KGlobal::iconLoader()->loadIconSet("fileclose", KIcon::Small));
  m_butt->setText("&Close");
  m_buttstate = 2;
  if (m_scan->scanner()->closedone())
    {
      m_scan->deleteLater();
    }
}

#include "knetmapnmapscanner.moc"
