/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qlayout.h>
#include <qlabel.h>
#include <qobjectlist.h>
#include <qpopupmenu.h>
#include <ktoolbarbutton.h>
#include <kcompletion.h>
#include <kpopupmenu.h>
#include <klocale.h>

#include "knetmapscanwidget.h"
#include "knetmapscanner.h"
#include "knetmap.h"
#include "ipv4.h"

KNetmapScanWidget::KNetmapScanWidget(KNetmapApp *theApp)
  : KToolBar(0, "scanWidget")
{
  setMovingEnabled(false);
  setFullSize(true);
  // menuActions.setAutoDelete(true);
  m_app = theApp;

  insertWidget (1, 1, new QLabel(i18n("Scan"), this, "label"));
  insertCombo  (QString::null, 2, true,
		SIGNAL(returnPressed()), this, SLOT(scan()),
		true, "Scan this target", -1, -1, QComboBox::NoInsertion);
  scanButton  = new KActionMenu("Scan", this, "scan_button");
  defaultScan = new KSelectAction("&Default Scan", 0, scanButton, "default");
  scanButton->setIcon("launch");
  scanButton->plug(this);
  setItemAutoSized(2);

  combo = getCombo(2);
  combo->setDuplicatesEnabled(false);
  combo->setInsertionPolicy(QComboBox::NoInsertion);
  comp = new TermCompletion();
  combo->setCompletionObject(comp);
  combo->clear();

  connect(combo, SIGNAL(returnPressed(const QString &)), comp, SLOT(addItem(const QString&)));

  connect(combo, SIGNAL(returnPressed()), this, SLOT(scan()));
  connect(scanButton, SIGNAL(activated()), this, SLOT(scan()));
  connect(defaultScan, SIGNAL(activated(const QString &)), this, SLOT(updateDefault(const QString &)));
  connect(scanButton->popupMenu(), SIGNAL(activated(int)), this, SLOT(scanMenuActivated(int)));

  updateMenu();
}

KNetmapScanWidget::~KNetmapScanWidget()
{
}

const QString KNetmapScanWidget::targetString()
{
  return m_tstr;
}

QStringList KNetmapScanWidget::targets()
{
  QStringList t = QStringList::split(QRegExp("\\s+"), m_tstr);
  QStringList expn;
  for (QStringList::Iterator it=t.begin(); it!=t.end(); ++it)
    expn += IPv4::expandIP(*it);
  return expn;
}

void KNetmapScanWidget::updateMenu()
{
  KAction *act;
  QStringList defItems;

  if (!menuActions.isEmpty())
    { // Remove the previous ones
      for (act=menuActions.first(); act; act=menuActions.next())
	{
	  scanButton->remove(act);
	  if (act != defaultScan)
	    delete act;
	}
      menuActions.clear();
    }

  QObjectList *l = getApp()->queryList("KNetmapScanner");
  QObjectListIt it(*l);
  KNetmapScanner *scanner;
  while ((scanner = (KNetmapScanner *)it.current()) != 0)
    {
      ++it;
      scanner->plugScanMenu(&menuActions);
    }
  delete l;
  menuActions.append(defaultScan); // The menu for selecting the default one.

  for (act=menuActions.first(); act; act=menuActions.next())
    {
      scanButton->insert(act);
      if (act != defaultScan)
	defItems.append(act->text());
    }
  defaultScan->setItems(defItems);

  if (!defItems.isEmpty())
    {
      if ((!m_default.isEmpty()) && (defItems.findIndex(m_default) != -1))
	defaultScan->setCurrentItem(defItems.findIndex(m_default));
      else
	defaultScan->setCurrentItem(0);
    }
}

void KNetmapScanWidget::addTarget(const QString &s)
{
  m_tstr = combo->lineEdit()->text();
  if (m_tstr.isEmpty())
    m_tstr = s;
  else
    m_tstr += " " + s;

  combo->lineEdit()->setText(m_tstr);
}

void KNetmapScanWidget::addTargetsToHistory()
{
  QStringList targ = targets();
  QStringList items;
  for (int i=0; i<combo->count(); i++)
    {
      items.append(combo->text(i));
    }
  for (QStringList::Iterator it=targ.begin(); it!=targ.end(); ++it)
    {
      if (items.find(*it) == items.end())
	{
	  comp->addItem(*it);
	  combo->insertItem(*it);
	}
    }
}

void KNetmapScanWidget::scanMenuActivated(int)
{
  m_tstr = combo->lineEdit()->text();
  if (!m_tstr.isEmpty())
    {
      addTargetsToHistory();
      combo->clearEdit();
    }
}

void KNetmapScanWidget::scan()
{
  m_tstr = combo->lineEdit()->text();
  if (!m_tstr.isEmpty())
    {
      addTargetsToHistory();

      KAction *act;
      QString it = defaultScan->currentText();
      for (act=menuActions.first(); act; act=menuActions.next())
	{
	  if (act->text() == it)
	    {
	      act->activate();
	      break;
	    }
	}
      combo->clearEdit();
    }
}

void KNetmapScanWidget::updateDefault(const QString &text)
{
  m_default = text;
}

void KNetmapScanWidget::readConfig(KConfig *config)
{
  m_default = config->readEntry("scanWidgetDefault", QString::null);
  if (config->hasGroup("ScanWidgetItems"))
    {
      QString oldg = config->group();
      config->setGroup("ScanWidgetItems");
      QStringList l;
      for (int i=0; i<config->readNumEntry("count"); i++)
	l.append(config->readEntry(QString("item%1").arg(i)));
      combo->clear();
      comp->clear();
      combo->insertStringList(l);
      comp->insertItems(l);
      combo->clearEdit();
      config->setGroup(oldg);
    }
}

void KNetmapScanWidget::saveConfig(KConfig *config)
{
  if (!m_default.isEmpty())
    config->writeEntry("scanWidgetDefault", m_default);
  if (combo->count())
    {
      QString oldg = config->group();
      config->setGroup("ScanWidgetItems");
      config->writeEntry("count", combo->count());
      for (int i=0; i<combo->count(); i++)
	config->writeEntry(QString("item%1").arg(i), combo->text(i));
      config->setGroup(oldg);
    }
}

TermCompletion::TermCompletion()
  : KCompletion()
{
}

TermCompletion::~TermCompletion()
{
}

QString TermCompletion::makeCompletion(const QString& string)
{
  QStringList terms =
    QStringList::split(QRegExp("\\s+"), string);
  if (terms.count() > 0)
    {
      QString term = terms.last();
      terms.pop_back();
      if (terms.count() > 0)
	prefix = QString("%1 ").arg(terms.join(" "));
      else
	prefix = "";
      return KCompletion::makeCompletion(term);
    }
  else
    {
      return QString::null;
    }
}

void TermCompletion::postProcessMatch(QString *match) const
{
  if (!prefix.isEmpty())
    match->prepend(prefix);
}

void TermCompletion::postProcessMatches(QStringList *matches) const
{
  if (!prefix.isEmpty())
    {
      QStringList::Iterator it;
      for (it=matches->begin(); it!=matches->end(); ++it)
	{
	  (*it).prepend(prefix);
	}
    }
}

void TermCompletion::postProcessMatches(KCompletionMatches *matches) const
{
  if (!prefix.isEmpty())
    {
      KCompletionMatches::iterator it;
      for (it=matches->begin(); it!=matches->end(); ++it)
	{
	  (*it).value().prepend(prefix);
	}
    }
}

#include "knetmapscanwidget.moc"
