/* KNetmap - KDE Network Mapper
 *
 * Copyright (C) 2003 Joshua T. Corbin <jcorbin@linuxmail.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qwidget.h>
#include <qheader.h>
#include <qpainter.h>
#include <qvaluelist.h>
#include <qobjectlist.h>
#include <qpopupmenu.h>
#include <klistview.h>
#include <kglobal.h>
#include <kiconloader.h>

#include "knetmapview.h"
#include "knetmapdoc.h"
#include "knetmap.h"
#include "knetmapsubnet.h"
#include "knetmaphost.h"
#include "knetmapsubnetpage.h"
#include "knetmaphostpage.h"
#include "knetmapconsole.h"

#ifndef __QT_VERSION_CHECK__
#define __QT_VERSION_CHECK__
#include <qglobal.h>
#ifndef QT_IS_VERSION
#define QT_IS_VERSION(a,b,c) ( QT_VERSION >= (((a) << 16) | ((b) << 8) | (c)) )
#endif
#if QT_IS_VERSION(3,2,0)
#define QTV32
#endif
#endif // __QT_VERSION_CHECK__

KNetmapView::KNetmapView(QWidget *parent)
  : QWidget(parent, "treeView")
{
  top_layout = new QVBoxLayout(this);
  top_layout->setAutoAdd(true);

  vsplit = new QSplitter(this);
  vsplit->setOrientation(Qt::Vertical);
  // Top Half (Tree and Main)
  hsplit = new QSplitter(vsplit);
  tree   = new KNetmapTree(hsplit);
  main   = new QWidgetStack(hsplit);
  bpage  = new QWidget(main);
  spage  = new KNetmapSubnetPage(main);
  hpage  = new KNetmapHostPage(main);
  KNetmapConsole *con = getConsole();
  con->reparent(vsplit, QPoint());

  hsplit->setOpaqueResize(true);
  vsplit->setOpaqueResize(true);

  tree->setMinimumWidth(100);
  tree->setMaximumWidth(300);

  con->setMinimumHeight(200);
  con->setMaximumHeight(400);

  connect(getDocument(), SIGNAL(currentChanged(QObject *)),
	  this, SLOT(updatePage(QObject *)));
  connect(tree, SIGNAL(contextMenu(KListView *, QListViewItem *, const QPoint &)),
	  this, SLOT(slotContextMenu(KListView *, QListViewItem *, const QPoint &)));
}

KNetmapView::~KNetmapView()
{
}

KNetmapApp *KNetmapView::getApp() const
{
  KNetmapApp *theApp = (KNetmapApp *) parentWidget();
  return theApp;
}

void KNetmapView::slotContextMenu(KListView *,
				  QListViewItem *l,
				  const QPoint &p)
{
  QPopupMenu *menu = (QPopupMenu *)0;

  if (!l)
    return;

  if (l->rtti() == KNetmapTree::hostRTTI)
    menu = (QPopupMenu *)getApp()->factory()->container("host_popup", getApp());
  else if (l->rtti() == KNetmapTree::subnetRTTI)
    menu = (QPopupMenu *)getApp()->factory()->container("subnet_popup", getApp());
  else
    return;

  if (menu)
    menu->popup(p);
}

void KNetmapView::updatePage(QObject *o)
{
  if (!o)
    main->raiseWidget(bpage);
  else if (o->inherits("KNetmapSubnet"))
    main->raiseWidget(spage);
  else if (o->inherits("KNetmapHost"))
    main->raiseWidget(hpage);
}

KNetmapConsole *KNetmapView::getConsole() const
{
  return getApp()->getConsole();
}

KNetmapDoc *KNetmapView::getDocument() const
{
  return getApp()->getDocument();
}

////////////////////////////////////////////////////////////
KNetmapTree::KNetmapTree(QWidget *parent)
  : KListView(parent, "nettree")
{
  addColumn("Network");
  header()->hide();
  setRootIsDecorated(true);
  setSelectionMode(QListView::Extended);

  connect(getView()->getDocument(), SIGNAL(subnetAdded(KNetmapSubnet *)),
	  this, SLOT(insertSubnet(KNetmapSubnet *)));
  connect(this, SIGNAL(currentChanged(QListViewItem *)),
	  this, SLOT(currentChangedSlot(QListViewItem *)));
  connect(this, SIGNAL(selectionChanged()),
	  this, SLOT(selectionChangedSlot()));
}

KNetmapTree::~KNetmapTree()
{
}

void KNetmapTree::selectionChangedSlot()
{
  m_selHosts.clear();
  m_selSubs.clear();
  QPtrList<QListViewItem> sel = selectedItems();
  QListViewItem *i;
  for (i = sel.first(); i; i = sel.next())
    {
      if (i->rtti() == subnetRTTI)
	m_selSubs.append(((KNetmapSubnetTreeItem *)i)->subnet());
      else if (i->rtti() == hostRTTI)
	m_selHosts.append(((KNetmapHostTreeItem *)i)->host());
    }
  getView()->getDocument()->setSelectedHosts(m_selHosts);
  getView()->getDocument()->setSelectedSubnets(m_selSubs);
}

void KNetmapTree::currentChangedSlot(QListViewItem *i)
{
  if (i)
    {
      if (i->rtti() == subnetRTTI)
	{
	  KNetmapSubnet *sub = ((KNetmapSubnetTreeItem *) i)->subnet();
	  getView()->getDocument()->setCurrent(sub);
	  getView()->getApp()->changeState("subnet_selected");
	}
      else if (i->rtti() == hostRTTI)
	{
	  KNetmapHost *host = ((KNetmapHostTreeItem *) i)->host();
	  getView()->getDocument()->setCurrent(host);
	  getView()->getApp()->changeState("host_selected");
	}
    }
  else
    {
      getView()->getDocument()->setCurrent(0);
      getView()->getApp()->changeState("default");
    }
}

KNetmapView *KNetmapTree::getView() const
{
  return((KNetmapView *) parent()->parent()->parent());
}

void KNetmapTree::insertSubnet(KNetmapSubnet *sub)
{
  (void)new KNetmapSubnetTreeItem(this, sub);
}

void KNetmapTree::removeSubnet(KNetmapSubnet *sub)
{
  QListViewItem *i;
  i = firstChild();
  while (i)
    {
      if (i->rtti() == subnetRTTI)
	{
	  KNetmapSubnetTreeItem *s = (KNetmapSubnetTreeItem *) i;
	  if (s->subnet() == sub)
	    {
	      delete s;
	      return;
	    }
	}
      i = i->nextSibling();
    }
}

////////////////////////////////////////////////////////////
// Subnet Tree Item code
KNetmapSubnetTreeItem::KNetmapSubnetTreeItem(KNetmapTree *parent, KNetmapSubnet *sub)
  : KListViewItem(parent, sub->name())
{
  m_sub = sub;
  setPixmap(0, KGlobal::iconLoader()->loadIcon(SUBNET_ICON, KIcon::Small, 16));

  if (m_sub)
    {
      connect((KNetmapSubnet *)m_sub, SIGNAL(subnetAdded(KNetmapSubnet *)),
	      this,  SLOT(insertSubnet(KNetmapSubnet *)));
      connect((KNetmapSubnet *)m_sub, SIGNAL(hostAdded(KNetmapHost *)),
	      this,  SLOT(insertHost(KNetmapHost *)));
      connect((KNetmapSubnet *)m_sub, SIGNAL(destroyed()),
	      this,  SLOT(deleteLater()));
    }
}

KNetmapSubnetTreeItem::KNetmapSubnetTreeItem(KNetmapSubnetTreeItem *parent, KNetmapSubnet *sub)
  : KListViewItem(parent, sub->name())
{
  m_sub = sub;
  setPixmap(0, KGlobal::iconLoader()->loadIcon(SUBNET_ICON, KIcon::Small, 16));

  if (m_sub)
    {
      connect((KNetmapSubnet *)m_sub, SIGNAL(subnetAdded(KNetmapSubnet *)),
	      this,  SLOT(insertSubnet(KNetmapSubnet *)));
      connect((KNetmapSubnet *)m_sub, SIGNAL(hostAdded(KNetmapHost *)),
	      this,  SLOT(insertHost(KNetmapHost *)));
      connect((KNetmapSubnet *)m_sub, SIGNAL(destroyed()),
	      this,  SLOT(deleteLater()));
    }
}

KNetmapSubnetTreeItem::~KNetmapSubnetTreeItem()
{
}

void KNetmapSubnetTreeItem::insertSubnet(KNetmapSubnet *sub)
{
  (void)new KNetmapSubnetTreeItem(this, sub);
  setOpen(true);
}

void KNetmapSubnetTreeItem::insertHost(KNetmapHost *host)
{
  (void)new KNetmapHostTreeItem(this, host);
  setOpen(true);
}

////////////////////////////////////////////////////////////
// Host Tree Item code
KNetmapHostTreeItem::KNetmapHostTreeItem(KNetmapSubnetTreeItem *parent, KNetmapHost *host)
  : KListViewItem(parent, host->name())
{
  m_host = host;
  if (m_host)
    {
      connect((KNetmapHost *)m_host, SIGNAL(iconChanged(const QString &)),
	      this,     SLOT(setIcon    (const QString &)));
      connect((KNetmapHost *)m_host, SIGNAL(nameChanged(const QString &)),
	      this,     SLOT(setName    (const QString &)));
      connect((KNetmapHost *)m_host, SIGNAL(destroyed()),
	      this,     SLOT(deleteLater()));
      setIcon(m_host->icon());
    }
}

KNetmapHostTreeItem::~KNetmapHostTreeItem()
{
}

void KNetmapHostTreeItem::setName(const QString &name)
{
  setText(0, name);
}

void KNetmapHostTreeItem::setIcon(const QString &icon)
{
  if (icon.isEmpty())
    setPixmap(0, KGlobal::iconLoader()->loadIcon(HOST_ICON, KIcon::Small, 16));
  else
    setPixmap(0, KGlobal::iconLoader()->loadIcon(icon,      KIcon::Small, 16));
}

// Sorting code
int KNetmapHostTreeItem::compare(QListViewItem *i,
				 int,
				 bool asc) const
{
  if (i->rtti() == KNetmapTree::subnetRTTI)
    return -1;
  else if (i->rtti() == KNetmapTree::hostRTTI)
    return m_host->compare((KNetmapHost *)((KNetmapHostTreeItem *)i)->m_host, asc);
  else
    return 0;
}

int KNetmapSubnetTreeItem::compare(QListViewItem *i,
				 int,
				 bool asc) const
{
  if (i->rtti() == KNetmapTree::subnetRTTI)
    return m_sub->compare((KNetmapSubnet *)((KNetmapSubnetTreeItem *)i)->m_sub, asc);
  else if (i->rtti() == KNetmapTree::hostRTTI)
    return 1;
  else
    return 0;
}

#include "knetmapview.moc"
