/***************************************************************************
 *   Copyright (C) 2004 by Matthias Reif                                   *
 *   matthias.reif@informatik.tu-chemnitz.de                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kprofilewidget.h"

#include "wwidget.h"

KProfileWidget::KProfileWidget( WWidget* wWidget, QWidget *parent, const char *name ) : QWidget( parent, name )
{
	this->wWidget = wWidget;
	KIconLoader iconLoader;
	listView = new KListView( this );
	buttonBox = new QVBox( this );
	buttonBox->setSpacing( KWaveControl::SPACING );

	loadButton = new KPushButton( iconLoader.loadIconSet( "fileopen", KIcon::Small ), i18n( "Load" ), buttonBox );
	saveButton = new KPushButton( iconLoader.loadIconSet( "filesave", KIcon::Small ), i18n( "Save" ), buttonBox );
	removeButton = new KPushButton( iconLoader.loadIconSet( "editdelete", KIcon::Small ), i18n( "Remove" ), buttonBox );
	buttonBox->adjustSize();

	connect( loadButton, SIGNAL( clicked() ), this, SLOT( loadSlot() ) );
	connect( saveButton, SIGNAL( clicked() ), this, SLOT( saveSlot() ) );
	connect( removeButton, SIGNAL( clicked() ), this, SLOT( removeSlot() ) );

	loadButton->setEnabled( false );
	removeButton->setEnabled( false );


	listView->setRootIsDecorated( false );
	listView->setSelectionModeExt( KListView::Single );
	listView->move( 10, 10 );
	nameCol = listView->addColumn( i18n( "Name" ) );
	essidCol = listView->addColumn( i18n( "ESSID" ) );
	ipCol = listView->addColumn( i18n( "IP" ) );
	gatewayCol = listView->addColumn( i18n( "Gateway" ) );
	dhcpCol = listView->addColumn( i18n( "DHCP" ) );
	modeCol = listView->addColumn( i18n( "Mode" ) );
	channelCol = listView->addColumn( i18n( "Channel" ) );
	encCol = listView->addColumn( i18n( "Encryption" ) );
	keyCol = listView->addColumn( i18n( "Key" ) );
	connect( listView, SIGNAL( selectionChanged() ), this, SLOT( selectionChangedSlot() ) );

	restoreProfiles();
}


KProfileWidget::~KProfileWidget()
{
}


/*!
	\fn KProfileWidget::loadSlot()
 */
void KProfileWidget::loadSlot()
{
	QString name = listView->selectedItem()->text( nameCol );
	KGlobal::config()->setGroup( "kwavecontrol_profiles" );
	QStringList list = KGlobal::config()->readListEntry( name );
	if( list.size() > Profile::FREQUENCY )
	{
		Profile profile;
		profile.essid = list[ Profile::ESSID ];
		profile.ip = list[ Profile::IP ];
		profile.mode = list[ Profile::MODE ].toInt();
		profile.frequency = list[ Profile::FREQUENCY ].toDouble();
		// just for compatibility
		if( list.size() > Profile::GATEWAY )
		{
			profile.gateway = list[ Profile::GATEWAY ];
		}
		else
		{
			profile.gateway = "";
		}
		// just for compatibility
		if( list.size() > Profile::KEY_INDEX )
		{
			profile.security = list[ Profile::SECURITY ].toInt();
			profile.key = list[ Profile::KEY ];
			profile.keyIndex = list[ Profile::KEY_INDEX ].toInt();
		}
		else
		{
			profile.security = -1;
			profile.key = "";
			profile.keyIndex = -1;
		}
		// just for compatibility
		if( list.size() > Profile::DHCP )
		{
			profile.dhcp = ( list[ Profile::DHCP ] == "yes" );
		}
		else
		{
			profile.dhcp = false;
		}
		wWidget->setProfile( profile );
	}
}


/*!
	\fn KProfileWidget::saveSlot()
 */
void KProfileWidget::saveSlot()
{
	bool ok;
	QString name = KInputDialog::getText( i18n( "Profile name" ), i18n( "Enter name for new profile:" ), QString::null, &ok );
	if( ok && ( name != "kwavecontrol_profile_names" ) )
	{
		Profile profile = wWidget->getProfile();
		QStringList list;
		int i;
		for( i = 0; i <= Profile::MAX; i++ )
		{
			switch( i )
			{
				case Profile::ESSID:
					list.push_back( profile.essid );
					break;
				case Profile::IP:
					list.push_back( profile.ip );
					break;
				case Profile::MODE:
					list.push_back( QString::number( profile.mode ) );
					break;
				case Profile::FREQUENCY:
					list.push_back( QString::number( profile.frequency ) );
					break;
				case Profile::GATEWAY:
					list.push_back( profile.gateway );
					break;
				case Profile::SECURITY:
					list.push_back( QString::number( profile.security ) );
					break;
				case Profile::KEY:
					list.push_back( profile.key );
					break;
				case Profile::KEY_INDEX:
					list.push_back( QString::number( profile.keyIndex ) );
					break;
				case Profile::DHCP:
					list.push_back( profile.dhcp?"yes":"no" );
					break;
			}
		}

		KGlobal::config()->setGroup( "kwavecontrol_profiles" );
		KGlobal::config()->writeEntry( name, list, ',', true, false, false );
		
		list = KGlobal::config()->readListEntry( "kwavecontrol_profile_names" );
		list.push_back( name );
		KGlobal::config()->writeEntry( "kwavecontrol_profile_names", list, ',', true, false, false );
	}
	
	restoreProfiles();
}


/*!
	\fn KProfileWidget::removeSlot()
 */
void KProfileWidget::removeSlot()
{
	QString name = listView->selectedItem()->text( nameCol );
	if( KMessageBox::questionYesNo( this, i18n( "Do you realy want to remove the profile?" ) + " \"" + name + "\"?", i18n( "Remove profile" ) ) == KMessageBox::Yes )
	{
		KGlobal::config()->setGroup( "kwavecontrol_profiles" );
		KGlobal::config()->deleteEntry( name );
		
		QStringList profiles;
		KGlobal::config()->setGroup( "kwavecontrol_profiles" );
		profiles = KGlobal::config()->readListEntry( "kwavecontrol_profile_names" );
		profiles.remove( name );
		KGlobal::config()->writeEntry( "kwavecontrol_profile_names", profiles );

		listView->clearSelection();
	}
	restoreProfiles();
}


/*!
	\fn KProfileWidget::selectionChangedSlot()
 */
void KProfileWidget::selectionChangedSlot()
{
	loadButton->setEnabled( listView->selectedItem() );
	removeButton->setEnabled( listView->selectedItem() );
}


/*!
    \fn KProfileWidget::restoreProfiles()
 */
void KProfileWidget::restoreProfiles()
{
	listView->clear();

	QStringList profiles, profileValues;
	KGlobal::config()->setGroup( "kwavecontrol_profiles" );
	profiles = KGlobal::config()->readListEntry( "kwavecontrol_profile_names" );
	
	unsigned int i;
	for( i = 0; i < profiles.size(); i++ )
	{
		profileValues = KGlobal::config()->readListEntry( profiles[ i ] );
		KListViewItem* profileItem = new KListViewItem( listView, profiles[ i ] );

		profileItem->setText( essidCol, profileValues[ Profile::ESSID ] );
		profileItem->setText( ipCol, profileValues[ Profile::IP ] );

		// gateway: compatibility
		if( profileValues.size() > Profile::GATEWAY )
		{
			profileItem->setText( gatewayCol, profileValues[ Profile::GATEWAY ] );
		}

		profileItem->setText( modeCol, QString( iw_operation_mode[ profileValues[ Profile::MODE ].toInt() ] ) );
		
		double channel = profileValues[ Profile::FREQUENCY ].toDouble();
		if( channel < WCard::KILO )
			profileItem->setText( channelCol, QString::number( channel ) );
		else
			profileItem->setText( channelCol, WWidget::frequency2QString( profileValues[ Profile::FREQUENCY ].toDouble() ) );

		if( profileValues.size() > Profile::KEY_INDEX )
		{
			switch( profileValues[ Profile::SECURITY ].toInt() )
			{
				case IW_ENCODE_DISABLED:
					profileItem->setText( encCol, "off" );
					break;
				case IW_ENCODE_OPEN:
					profileItem->setText( encCol, "open" );
					break;
				case IW_ENCODE_RESTRICTED:
					profileItem->setText( encCol, "restricted" );
					break;
			}
			profileItem->setText( keyCol, profileValues[ Profile::KEY ] );
// 			profileItem->setText( 8, profileValues[ 7 ] );
		}

		if( profileValues.size() > Profile::DHCP )
		{
			profileItem->setText( dhcpCol, profileValues[ Profile::DHCP ]?"yes":"no" );
		}
		else
		{
			profileItem->setText( dhcpCol, "no" );
		}
		
		listView->insertItem( profileItem );
// 		profileItem->setRenameEnabled( 0, true );
	}
	
}


/*!
    \fn KProfileWidget::resizeEvent ( QResizeEvent * )
 */
void KProfileWidget::resizeEvent ( QResizeEvent * )
{
	buttonBox->move( this->width() - buttonBox->width() - 10, 10 );
	listView->resize( buttonBox->x() - listView->x() - 15, this->height() - 20 );
}



#include "kprofilewidget.moc"
