/***************************************************************************
 *   Copyright (C) 2004 by Matthias Reif                                   *
 *   matthias.reif@informatik.tu-chemnitz.de                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kscanwidget.h"

#include "wwidget.h"

KScanWidget::KScanWidget( WWidget* wWidget, QWidget* parent, const char *name ) : QHBox( parent, name )
{
	this->wWidget = wWidget;
	KIconLoader iconLoader;
	this->setMargin( 10 );
	this->setSpacing( 15 );
	listView = new KListView( this );
	QVBox* buttonBox = new QVBox( this );
	buttonBox->setSpacing( 15 );
	this->setStretchFactor( listView, 1 );

	scanButton = new KPushButton( iconLoader.loadIconSet( "find", KIcon::Small ), i18n( "Scan" ), buttonBox );
	connectButton = new KPushButton( iconLoader.loadIconSet( "connect_creating", KIcon::Small ), i18n( "Connect" ), buttonBox );
	connectDhcpButton = new KPushButton( iconLoader.loadIconSet( "connect_creating", KIcon::Small ), i18n( "Connect\n+ dhcp" ), buttonBox );
	optionsButton = new KPushButton( iconLoader.loadIconSet( "configure", KIcon::Small ), i18n( "Options" ), buttonBox );
	QWidget* dummy = new QWidget( buttonBox );

	buttonBox->setStretchFactor( dummy, 1 );

	connectButton->setEnabled( false );
	connectDhcpButton->setEnabled( false );

	connect( connectButton, SIGNAL( clicked() ), this, SLOT( connectSlot() ) );
	connect( connectDhcpButton, SIGNAL( clicked() ), this, SLOT( connectDhcpSlot() ) );
	connect( scanButton, SIGNAL( clicked() ), this, SLOT( scanSlot() ) );
	connect( scanButton, SIGNAL( clicked() ), wWidget, SLOT( scanSlot() ) );
	connect( optionsButton, SIGNAL( clicked() ), this, SLOT( optionsSlot() ) );


	sorting[ SIOCGIWESSID ] = "1";
	sorting[ SIOCGIWRATE ] = "2";
	sorting[ SIOCGIWENCODE ] = "3";
	sorting[ IWEVQUAL ] = "4";
	sorting[ SIOCGIWMODE ] = "5";
	sorting[ SIOCGIWFREQ ] = "6";
	sorting[ SIOCGIWAP ] = "7";
	sorting[ SIOCGIWNWID ] = "8";
	sorting[ SIOCGIWNAME ] = "9";
	sorting[ IWEVCUSTOM ] = "a";
	
	listView->setRootIsDecorated( true );
	listView->setSelectionModeExt( KListView::Single );
	listView->move( KWaveControl::SPACING, KWaveControl::SPACING );
	connect( listView, SIGNAL( selectionChanged() ), this, SLOT( selectionChangedSlot() ) );
	
	listView->addColumn( i18n( "Results" ) );    // name
	listView->addColumn( "" );                   // value
	listView->addColumn( "" );                   // sorting
	listView->addColumn( "" );                   // keys
	
	scanningLabel = new QLabel( i18n( "scanning" ) + "...", listView );
	scanningLabel->adjustSize();
	scanningLabel->move( ( listView->width() - scanningLabel->width() ) / 2 , ( listView->height() - scanningLabel->height() ) / 2 );
	scanningLabel->setPaletteBackgroundColor( listView->paletteBackgroundColor() );
	scanningLabel->hide();

	scanOptionsWidget = new KScanOptionsWidget( this );
	connect( scanOptionsWidget, SIGNAL( okClicked() ), this, SLOT( optionsApplySlot() ) );
	connect( scanOptionsWidget, SIGNAL( applyClicked() ), this, SLOT( optionsApplySlot() ) );

	manualScan = false;
}


KScanWidget::~KScanWidget()
{
}


/*!
	\fn KScanWidget::connectSlot()
 */
void KScanWidget::connectSlot()
{
	connectMeta( false );
}


/*!
	\fn KScanWidget::connectDhcpSlot()
 */
void KScanWidget::connectDhcpSlot()
{
	connectMeta( true );
}


/*!
	\fn KScanWidget::scanSlot()
 */
void KScanWidget::scanSlot()
{
	manualScan = true;
	setScanning();
}


/*!
	\fn KScanWidget::selectionChangedSlot()
 */
void KScanWidget::selectionChangedSlot()
{
	connectButton->setEnabled( listView->selectedItem() );
	connectDhcpButton->setEnabled( listView->selectedItem() );
}


/*!
    \fn KScanWidget::getKey( const unsigned char* data, int length, int flags )
 */
QString KScanWidget::getKey( const unsigned char* data, int length, int flags )
{
	QString key;
	int i;
	if( flags & IW_ENCODE_NOKEY )
	{
		if( length <= 0 )
		{
			key = i18n( "on" );
		}
		else
		{
			key= "**";
			for( i = 1; i < length; i++ )
			{
				if( i % 2 == 0 )
					key += "-";
				key += "**";
			}
		}
	}
	else
	{
		key = QString( "%1" ).arg( data[ 0 ], 2, 16 );
		for( i = 1; i < length; i++ )
		{
			if( i % 2 == 0 )
				key += "-";
			key += QString( "%1" ).arg( data[ i ], 2, 16 );
		}
	}
	return key;
}


/*!
    \fn KScanWidget::update( WCard* card, APInfos apInfos )
 */
void KScanWidget::update( WCard* card, APInfos apInfos )
{
	listView->clear();
	frequencies.clear();
	
	KIconLoader iconLoader;

	char* macAddr = new char[ 6*3 + 1 ];
	lastApItem = NULL;
	
	KListViewItem* lastBitrateItem = NULL;
	int maxBitrate = -1;

	QMap< QString, KnownCell > newCells;
	QMap< QString, KnownCell >::iterator lastNewCell;
		
	unsigned int i, j;
	for( i=0; i<apInfos.size(); i++ )
	{
		APInfo apInfo = apInfos.at( i );
		for( j=0; j<apInfo.size(); j++ )
		{
			struct iw_event event = apInfo.at( j );
			newSubItems.clear();
			switch( event.cmd )
			{
				case SIOCGIWAP:
					{
					const struct ether_addr* eth = (const struct ether_addr *)event.u.ap_addr.sa_data;
					sprintf( macAddr, "%02X:%02X:%02X:%02X:%02X:%02X",
						eth->ether_addr_octet[0], eth->ether_addr_octet[1],
						eth->ether_addr_octet[2], eth->ether_addr_octet[3],
						eth->ether_addr_octet[4], eth->ether_addr_octet[5] );
					QString mac( macAddr );
					listView->insertItem( lastApItem = new KListViewItem( listView, i18n( "Access Point" ) ) );
					lastApItem->setSelectable( true );
					
					newSubItems.push_back( new KListViewItem( lastApItem, "Mac", mac ) );
					
					lastBitrateItem = NULL;
					maxBitrate = -1;
					
					lastNewCell = newCells.insert( mac, KnownCell() );
					}
					break;
				case SIOCGIWNWID:
					{
					if( event.u.nwid.disabled )
						newSubItems.push_back( new KListViewItem( lastApItem, "NWID", i18n( "off/any" ) ) );
					else
						newSubItems.push_back( new KListViewItem( lastApItem, "NWID", QString::number( event.u.nwid.value ) ) );
					}
					break;
				case SIOCGIWFREQ:
					{
						double freq = ((double) event.u.freq.m) * pow( 10.0, event.u.freq.e );
						QString string = "";
						if( freq > WCard::GIGA )
							string = QString::number(freq/WCard::GIGA, 'f', 3) + "GHz";
						else if( freq > WCard::MEGA )
							string = QString::number(freq/WCard::MEGA, 'f', 3) + "MHz";
						else if( freq > WCard::KILO )
							string = QString::number(freq/WCard::KILO, 'f', 3) + "kHz";
						else
							string = QString::number(freq, 'f', 3) + "Hz";
						newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Frequency" ), string ) );
						int channel = card->getChannel( freq );
						if( channel >= 0 )
						{
							newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Channel" ), QString::number( channel ) ) );
						}
						frequencies[ lastApItem ] = freq;
					}
					break;
				case SIOCGIWMODE:
					{
						newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Mode" ), iw_operation_mode[ event.u.mode ] ) );
					}
					break;
				case SIOCGIWNAME:
					{
						newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Protocol" ), event.u.name ) );
					}
					break;
				case SIOCGIWESSID:
					{
						QString essid;
						if( ( event.u.essid.pointer ) && ( event.u.essid.length ) )
						{
							essid = QString( (char*)event.u.essid.pointer );
							essid.truncate( event.u.essid.length );
						}
						if( event.u.essid.flags )
						{
							if((event.u.essid.flags & IW_ENCODE_INDEX) > 1)
							{
								essid += " [" + QString::number( event.u.essid.flags & IW_ENCODE_INDEX ) + "]";
							}
						}
						else
						{
							essid = i18n( "off/any" );
						}
						newSubItems.push_back( new KListViewItem( lastApItem, i18n( "ESSID" ), essid ) );
						lastApItem->setText( 0, essid );
						lastNewCell.data().setEssid( essid );
					}
					break;
				case SIOCGIWENCODE:
					{
						QString key;
						QString security;
						
						if( ! event.u.data.pointer )
							event.u.data.flags |= IW_ENCODE_NOKEY;
						
						if( event.u.data.flags & IW_ENCODE_DISABLED )
						{
							key = i18n( "off" );
							//lastApItem->setPixmap( 1, iconLoader.loadIcon( "decrypted", KIcon::Small ) );
						}
						else
						{
							key = getKey( (unsigned char*)event.u.data.pointer, event.u.data.length, event.u.data.flags );
							lastApItem->setPixmap( 1, iconLoader.loadIcon( "encrypted", KIcon::Small ) );
							
							if( ( event.u.data.flags & IW_ENCODE_INDEX) > 1 )
							{
								key += " [" + QString::number( event.u.data.flags & IW_ENCODE_INDEX ) + "]";
							}
							
							if( event.u.data.flags & IW_ENCODE_RESTRICTED )
							{
								security = i18n( "restricted" );
							}
							if( event.u.data.flags & IW_ENCODE_OPEN )
							{
								security = i18n( "open" );
							}
						}
						
						newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Encryption key" ), key ) );
						if( ! security.isEmpty() )
						{
							newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Security mode" ), security ) );
						}
					}
					break;
				case SIOCGIWRATE:
					{
						if( lastBitrateItem == NULL )
						{
							newSubItems.push_back( lastBitrateItem = new KListViewItem( lastApItem, i18n( "Bit Rates" ), "" ) );
						}
						QString sortKey = QString::number( event.u.bitrate.value );
						while( sortKey.length() < 10 )
						{
							sortKey = "0" + sortKey;
						}
						
						newSubItems.push_back( new KListViewItem( lastBitrateItem, "", WWidget::bitrate2String( event.u.bitrate.value ), sortKey) );
						
						if( event.u.bitrate.value > maxBitrate )
						{
							maxBitrate = event.u.bitrate.value;
							lastBitrateItem->setText( 1, i18n( "max" ) + " " + WWidget::bitrate2String( maxBitrate ) );
						}
					}
					break;
				case IWEVQUAL:
					{
						iwqual* qual = &event.u.qual;
						if( card->has.range && ( qual->level != 0 ) )
						{
							if( ! ( qual->updated & IW_QUAL_QUAL_INVALID ) )
							{
								newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Quality" ), QString::number( qual->qual ) + "/" + QString::number( card->range->max_qual.qual ), "1" ) );
								float q = (float)qual->qual / (float)card->range->max_qual.qual;
								
							}
							if( qual->level > card->range->max_qual.level)
							{
								if( ! ( qual->updated & IW_QUAL_LEVEL_INVALID ) )
								{
									newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Signal Level" ), QString::number( qual->level - 0x100 ) + " dBm", "2" ) );
								}
								if( ! ( qual->updated & IW_QUAL_NOISE_INVALID ) )
								{
									newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Noise Level" ), QString::number( qual->noise - 0x100 ) + " dBm", "3" ) );
								}
							}
							else
							{
								if( ! ( qual->updated & IW_QUAL_LEVEL_INVALID ) )
								{
									newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Signal Level" ), QString::number( qual->level ) + "/" + QString::number( card->range->max_qual.level ), "2" ) );
								}
								if( ! ( qual->updated & IW_QUAL_NOISE_INVALID ) )
								{
									newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Noise Level" ), QString::number( qual->noise ) + "/" + QString::number( card->range->max_qual.noise ), "3" ) );
								}
							}
						}
						else
						{
							newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Quality" ), QString::number( qual->qual ), "1" ) );
							newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Signal Level" ), QString::number( qual->level ), "2" ) );
							newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Noise Level" ), QString::number( qual->noise ), "3" ) );
						}
					}
					break;
				case IWEVCUSTOM:
					{
						newSubItems.push_back( new KListViewItem( lastApItem, i18n( "Extra" ), QString( (const char*)event.u.data.pointer ) ) );
					}
					break;
			}
			unsigned int i;
			for( i = 0; i < newSubItems.size(); i++ )
			{
				newSubItems[ i ]->setSelectable( false );
				newSubItems[ i ]->setText( 2, sorting[ event.cmd ] + newSubItems[ i ]->text( 2 ) );
				newSubItems[ i ]->setText( 3, QString::number( event.cmd ) );
			}
			newSubItems.clear();
		}
	}
	
	listView->setSorting( 2 );
	hideCols();
	
	delete macAddr;

	checkNewCells( newCells );
	
	setScanning( false );
}


/*!
	\fn KScanWidget::setScanning( bool scanning )
 */
void KScanWidget::setScanning( bool scanning )
{
	if( scanning )
		listView->clear();
	else
		manualScan = false;
	scanButton->setEnabled( !scanning );
	scanningLabel->setShown( scanning );
	listView->setDisabled( scanning );
	scanningLabel->move( ( listView->width() - scanningLabel->width() ) / 2 , ( listView->height() - scanningLabel->height() ) / 2 );
}


/*!
    \fn KScanWidget::hideCols()
 */
void KScanWidget::hideCols()
{
	int i; 
	for( i=2; i<=3; i++ )
	{
		listView->hideColumn( i );
		listView->header()->setResizeEnabled( false, i );
		listView->header()->setStretchEnabled( false, i );
		listView->setColumnWidthMode( i, QListView::Manual );
		listView->hideColumn( i );
	}
}


/*!
    \fn KScanWidget::setScanningEnabled( bool enabled )
 */
void KScanWidget::setScanningEnabled( bool enabled )
{
	scanButton->setEnabled( enabled );
}


/*!
    \fn KScanWidget::optionsSlot()
 */
void KScanWidget::optionsSlot()
{
	scanOptionsWidget->setScanOptions( readScanOptions() );
	scanOptionsWidget->exec();
}


/*!
    \fn KScanWidget::optionsApplySlot()
 */
void KScanWidget::optionsApplySlot()
{
	optionsChanged( scanOptionsWidget->getScanOptions() );
}


/*!
    \fn KScanWidget::checkNewCells( QMap< QString, KnownCell > newCells )
 */
void KScanWidget::checkNewCells( QMap< QString, KnownCell > newCells )
{
	KGlobal::config()->setGroup( "KWaveControl_Scanning" );
	bool inform = KGlobal::config()->readBoolEntry( "inform", true );
	bool remember = KGlobal::config()->readBoolEntry( "remember", true );
	
	if( inform )
	{
		// remove old cells
		int rememberTime = KGlobal::config()->readNumEntry( "rememberTime", 10 );
		QTime x = QTime::currentTime().addSecs( rememberTime * -60 );
		QMap< QString, KnownCell >::iterator cell;
		for( cell = knownCells.begin(); cell != knownCells.end(); ++cell )
		{
			if( cell.data() < x )
			{
				knownCells.remove( cell );
			}
		}

		// check new cells
		if( remember )
		{
			QMap< QString, KnownCell >::iterator newCell;
			bool exists = false;
			for( newCell = newCells.begin(); newCell != newCells.end(); ++newCell )
			{
				exists = knownCells.contains( newCell.key() );
				knownCells.insert( newCell.key(), newCell.data() );
				if( exists )
				{
					newCells.remove( newCell );
				}
			}
		}

		if( ! newCells.empty() && ! manualScan )
		{
			newCellsFound( newCells );
		}
	}
	else
	{
		knownCells.clear();
	}
}


#include "kscanwidget.moc"


/*!
    \fn KScanWidget::readScanOptions()
 */
ScanOptions KScanWidget::readScanOptions()
{
	KGlobal::config()->setGroup( "KWaveControl_Scanning" );
	ScanOptions scanOptions;
	scanOptions.startScan = KGlobal::config()->readBoolEntry( "startScan", false );
	scanOptions.autoScan = KGlobal::config()->readBoolEntry( "autoScan", false );
	scanOptions.interval = KGlobal::config()->readNumEntry( "interval", 60 );
	scanOptions.inform = KGlobal::config()->readBoolEntry( "inform", true );
	scanOptions.remember = KGlobal::config()->readBoolEntry( "remember", true );
	scanOptions.rememberTime = KGlobal::config()->readNumEntry( "rememberTime", 10 );
	return scanOptions;
}


/*!
    \fn KScanWidget::connectMeta( bool dhcp )
 */
void KScanWidget::connectMeta( bool dhcp )
{
	QListViewItem* item = listView->selectedItem();
	if( item != 0 )
	{
		double frequency;
		QString essid;
		QString mode;
		bool enc = false;
		int key;
		QListViewItem* child = item->firstChild();
		while( child != 0 )
		{
			key = child->text( 3 ).toInt();
			switch( key )
			{
				case SIOCGIWESSID:
					essid = child->text( 1 );
					break;
				case SIOCGIWMODE:
					mode = child->text( 1 );
				case SIOCGIWENCODE:
					enc = child->text( 1 ) != "off";
					break;
			}
			child = child->nextSibling();
		}
		frequency = frequencies[ item ];
/*		if( enc )
		{
			KKeyDialog keyDialog( "", this );
			keyDialog.exec();
		}*/
		wWidget->connectToAp( essid, frequency, mode, enc, dhcp );
	}
}
