/***************************************************************************
 *   Copyright (C) 2004 by Matthias Reif                                   *
 *   matthias.reif@informatik.tu-chemnitz.de                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "ktraffictab.h"

KTrafficTab::KTrafficTab( QWidget *parent, const char *name ) : QHBox( parent, name )
{
	this->setMargin( 10 );
	this->setSpacing( 15 );
	QVBox* leftBox = new QVBox( this );
	QVBox* rightBox = new QVBox( this );
	this->setStretchFactor( leftBox, 1 );
	leftBox->setSpacing( 10 );
	rightBox->setSpacing( 10 );
	trafficWidget = new KTrafficWidget( 100, leftBox );

	QHBox* maxWidget = new QHBox( leftBox );
	maxWidget->setSpacing( 10 );
	maxValueLabel = new QLabel( maxWidget );
	maxValueLabel->setText( i18n( "Maximum" ) + ":" );

	maxValueEdit = new KLineEdit( maxWidget );
	maxValueEdit->setFixedWidth( 50 );
	maxValueEdit->setFixedHeight( maxValueLabel->height() + 4 );
	maxValueEdit->setAlignment( Qt::AlignRight );
	
	maxValueSelect = new KComboBox( maxWidget );
	maxValueSelect->insertItem( "Mbyte/s", 0 );
	maxValueSelect->insertItem( "kbyte/s", 1 );
	maxValueSelect->insertItem( "MBit/s", 2 );
	maxValueSelect->insertItem( "kBit/s", 3 );
	maxWidget->setStretchFactor( new QWidget( maxWidget ), 1 );

	gridWidget = new QWidget( rightBox );
	gridLayout = new QGridLayout( gridWidget );
	gridLayout->setSpacing( 10 );
	gridLayout->setColStretch( 2, 2 );
	
	recvInscr = new QLabel( i18n( "Received" ) + ":", gridWidget );
	recvValue = new QLabel( "?", gridWidget );
	sentInscr = new QLabel( i18n( "Sent" ) + ":", gridWidget );
	sentValue = new QLabel( "?", gridWidget );

	recvLed = new KLed( Qt::red, gridWidget );
	sentLed = new KLed( Qt::blue, gridWidget );

	new KSeparator( KSeparator::Horizontal, rightBox );
	
	discardWidget = new QWidget( rightBox );
	discardLayout = new QGridLayout( discardWidget );
	discardLayout->setSpacing( 10 );
	discardLayout->setColStretch( 1, 2 );

	nwidInscr = new QLabel( i18n( "Rx invalid nwid" ) + ":", discardWidget );
	nwidValue = new QLabel( "?", discardWidget );
	codeInscr = new QLabel( i18n( "Rx invalid crypt" ) + ":", discardWidget );
	codeValue = new QLabel( "?", discardWidget );
	fragmentInscr = new QLabel( i18n( "Rx invalid frag" ) + ":", discardWidget );
	fragmentValue = new QLabel( "?", discardWidget );
	retriesInscr = new QLabel( i18n( "Tx retries" ) + ":", discardWidget );
	retriesValue = new QLabel( "?", discardWidget );
	miscInscr = new QLabel( i18n( "Invalid misc" ) + ":", discardWidget );
	miscValue = new QLabel( "?", discardWidget );
	missedBeaconInscr = new QLabel( i18n( "Missed beacon" ) + ":", discardWidget );
	missedBeaconValue = new QLabel( "?", discardWidget );

	gridLayout->addWidget( recvLed, 0, 1, Qt::AlignLeft | Qt::AlignVCenter );
	gridLayout->addWidget( recvInscr, 0, 2, Qt::AlignLeft | Qt::AlignVCenter );
	gridLayout->addWidget( recvValue, 0, 3, Qt::AlignLeft | Qt::AlignVCenter );
	gridLayout->addWidget( sentLed, 1, 1, Qt::AlignLeft | Qt::AlignVCenter );
	gridLayout->addWidget( sentInscr, 1, 2, Qt::AlignLeft | Qt::AlignVCenter );
	gridLayout->addWidget( sentValue, 1, 3, Qt::AlignLeft | Qt::AlignVCenter );
	
	discardLayout->addWidget( nwidInscr, 0, 0, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( nwidValue, 0, 1, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( codeInscr, 1, 0, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( codeValue, 1, 1, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( fragmentInscr, 2, 0, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( fragmentValue, 2, 1, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( retriesInscr, 3, 0, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( retriesValue, 3, 1, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( miscInscr, 4, 0, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( miscValue, 4, 1, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( missedBeaconInscr, 5, 0, Qt::AlignLeft | Qt::AlignVCenter );
	discardLayout->addWidget( missedBeaconValue, 5, 1, Qt::AlignLeft | Qt::AlignVCenter );


	leftBox->setStretchFactor( trafficWidget, 1 );
	rightBox->setStretchFactor( new QWidget( rightBox ), 1 );
	
	KGlobal::config()->setGroup( "KWaveControl" );
	maxValueEdit->setText( KGlobal::config()->readEntry( "trafficMax", "500" ) );
	maxValueSelect->setCurrentItem( KGlobal::config()->readNumEntry( "trafficUnit", 1 ) );
	
	maxValueApply();
	
	connect( maxValueEdit, SIGNAL( returnPressed() ), this, SLOT( maxValueApply() ) );
	connect( maxValueEdit, SIGNAL( textChanged( const QString& ) ), this, SLOT( maxValueApply() ) );
	connect( maxValueSelect, SIGNAL( activated( int ) ), this, SLOT( maxValueApply() ) );
	
	QWhatsThis::add( trafficWidget, QWhatsThisStrings::traffic() );
	QWhatsThis::add( gridWidget, QWhatsThisStrings::sentRecv() );
	QWhatsThis::add( maxWidget, QWhatsThisStrings::trafficMaximum() );
	QWhatsThis::add( discardWidget, QWhatsThisStrings::discarded() );
}


KTrafficTab::~KTrafficTab()
{
}

void KTrafficTab::maxValueApply()
{
	float f = 1.0;
	switch( maxValueSelect->currentItem() )
	{
		case 0:
			f = MEGA_BYTE;
			break;
		case 1:
			f = KILO_BYTE;
			break;
		case 2:
			f = MEGA_BYTE / 8.0;
			break;
		case 3:
			f = KILO_BYTE / 8.0;
			break;
	}
	bool ok;
	float v = maxValueEdit->text().toFloat( &ok );
	if( ok )
	{
		trafficWidget->setMaxValue( (unsigned long)(f * v) );
		KGlobal::config()->writeEntry( "trafficMax", v );
	}
	KGlobal::config()->writeEntry( "trafficUnit", maxValueSelect->currentItem() );
}


/*!
    \fn KTrafficTab::setTraffic( double recv, double sent )
 */
void KTrafficTab::setTraffic( double recv, double sent )
{
	actRecv = recv - prevRecv;
	actSent = sent - prevSent;
	trafficWidget->push( (long)actSent, (long)actRecv );
	prevSent = sent;
	prevRecv = recv;
	
	QString recvUnit, sentUnit;
	
	if( recv >= GIGA_BYTE )
	{
		recv /= GIGA_BYTE;
		recvUnit = " GB";
	}
	else if( recv >= MEGA_BYTE )
	{
		recv /= MEGA_BYTE;
		recvUnit = " MB";
	}
	else if( recv >= KILO_BYTE )
	{
		recv /= KILO_BYTE;
		recvUnit = " kB";
	}
	else
	{
		recvUnit = " B";
	}
	
	if( sent >= GIGA_BYTE )
	{
		sent /= GIGA_BYTE;
		sentUnit = " GB";
	}
	else if( sent >= MEGA_BYTE )
	{
		sent /= MEGA_BYTE;
		sentUnit = " MB";
	}
	else if( sent >= KILO_BYTE )
	{
		sent /= KILO_BYTE;
		sentUnit = " kB";
	}
	else
	{
		sentUnit = " B";
	}
	
	recvValue->setText( QString::number( recv,  'f', 2 ) + recvUnit );
	sentValue->setText( QString::number( sent,  'f', 2 ) + sentUnit );

	recvValue->adjustSize();
	sentValue->adjustSize();
	gridWidget->adjustSize();
	
	if( actRecv > 0.0 )
		recvLed->setState( KLed::On );
	else
		recvLed->setState( KLed::Off );
	if( actSent > 0.0 )
		sentLed->setState( KLed::On );
	else
		sentLed->setState( KLed::Off );
}


/*!
    \fn KTrafficTab::clear( double recv, double sent )
 */
void KTrafficTab::clear( double recv, double sent )
{
	prevSent = sent;
	prevRecv = recv;
	trafficWidget->clear();
}


/*!
    \fn KTrafficTab::setDiscarded( iw_discarded discarded, int missedBeacon )
 */
void KTrafficTab::setDiscarded( iw_discarded discarded, int missedBeacon )
{
	nwidValue->setText( QString::number( discarded.nwid ) );
	codeValue->setText( QString::number( discarded.code ) );
	fragmentValue->setText( QString::number( discarded.fragment ) );
	retriesValue->setText( QString::number( discarded.retries ) );
	miscValue->setText( QString::number( discarded.misc ) );
	missedBeaconValue->setText( QString::number( missedBeacon ) );
}


/*!
    \fn KTrafficTab::setDiscardedUnknown()
 */
void KTrafficTab::setDiscardedUnknown()
{
	nwidValue->setText( "?" );
	codeValue->setText( "?" );
	fragmentValue->setText( "?" );
	retriesValue->setText( "?" );
	miscValue->setText( "?" );
	missedBeaconValue->setText( "?" );
}


#include "ktraffictab.moc"
