/***************************************************************************
                          wwidget.cpp  -  description
                             -------------------
    begin                : Die Nov 12 2002
    copyright            : (C) 2002 by Matthias Reif
    email                : matthias.reif@informatik.tu-chemnitz.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "wwidget.h"

/*!
	\fn WWidget::WWidget( WCard *card, QString cmd, QWidget *parent, const char *name )
*/
WWidget::WWidget( WCard *card, QString cmd, QWidget *parent, const char *name ) : KDialogBase( parent, name, false, QString::null, KDialogBase::Ok | KDialogBase::User1 | KDialogBase::User2 )
{
	this->card = card;
	this->cmd = cmd;

	buffer = new char[ 1024 ];

	// i'm optimist
	isWave = true;
	
	initTabs();
	adjustEntries();

	scanThread = new QScanThread( card );
	scanning = false;

	dhcpTimer = new QTimer( this );
	dhcpThread = new QDhcpThread( card );
	dhcpThreadRunning = false;
	
	gateway = "";
	
	QToolButton* whatsThisButton = QWhatsThis::whatsThisButton( this );
	KGuiItem whatsItem( i18n( "What's this?" ), whatsThisButton->iconSet(), i18n( "Show extended Tooltips" ) );
	setButtonGuiItem( KDialogBase::User1, whatsItem );
	delete whatsThisButton;

	KGuiItem rootItem( i18n( "Restart as root" ), KGlobal::iconLoader()->loadIcon( "launch", KIcon::Small ), i18n( "Restart the application with root privileges" ), QWhatsThisStrings::root() );
	setButtonGuiItem( KDialogBase::User2, rootItem );
	// show only if user isn't root
	showButton( KDialogBase::User2, strcmp( getpwuid( getuid() )->pw_name, "root" ) );

	connectSignals();
	setStaticInfo();

	setWhatsThis();
}


/*!
	\fn WWidget::~WWidget()
*/
WWidget::~WWidget()
{
	delete [] buffer;
	delete scanThread;
	delete dhcpThread;
}


/*!
    \fn WWidget::initTabs()
 */
void WWidget::initTabs()
{
	tab = new KTabCtl( this );

	firstTab = new KSingleTabWidget( tab, "first" );
	experiencedTab = new KExperiencedTab( tab, "Traffic" );
	infoTab = new KInfoWidget( tab );
	scanWidget = new KScanWidget( this, tab );
	profileWidget = new KProfileWidget( this, tab );
	trafficTab = new KTrafficTab( tab );
	
	tab->addTab( firstTab, i18n( "Main" ) );
	tab->addTab( experiencedTab, i18n( "Experienced" ) );
	tab->addTab( trafficTab, i18n( "Traffic" ) );
	tab->addTab( scanWidget, i18n( "Scanning" ) );
	tab->addTab( profileWidget, i18n( "Profiles" ) );
	tab->addTab( infoTab, i18n( "Info" ) );
	
	deviceSelection = new KWCEntrySelection( i18n( "Device" ), true, firstTab );
	dhcpEntry = new KWCEntryDhcp( firstTab );
	IPEdit = new KWCEntryEdit( i18n( "IP Address"), firstTab );
	gatewayEdit = new KWCEntryEdit( i18n( "Gateway"), firstTab );

	essidEdit = new KWCEntryEdit( i18n( "ESSID" ), firstTab );
	
	modeSelection = new KWCEntrySelection( i18n( "Mode" ), false, firstTab);
	modeSelection->insertItem( "Auto" , IW_MODE_AUTO );
	modeSelection->insertItem( "Ad-hoc" , IW_MODE_ADHOC );
	modeSelection->insertItem( "Infrastructure", IW_MODE_INFRA);
	modeSelection->insertItem( "Master", IW_MODE_MASTER );
	modeSelection->insertItem( "Repeater", IW_MODE_REPEAT );
	modeSelection->insertItem( "Second", IW_MODE_SECOND );
	
	linkBar = new KWCEntryBar( i18n( "Link Quality" ), firstTab );
	signalBar = new KWCEntryBar( i18n( "Signal Level" ), firstTab );
	noiseBar = new KWCEntryBar( i18n( "Noise Level" ), firstTab );
	encodingLabel = new KWCEntryLabel( i18n( "Encoding" ), firstTab );
	pollingSpin = new KWCEntrySpin( i18n( "Polling" ), 0.1, 99.0, 0.1, 1, " sec", firstTab );
	
	protocolLabel = new KWCEntryLabel( i18n( "Protocol" ), firstTab );
	bitrateSelection = new KWCEntrySelection( i18n( "Bit Rate" ), false, firstTab );
	channelSelection = new KWCEntrySelection( i18n( "Channel" ), false, firstTab );
	sensitivityLabel = new KWCEntryLabel( i18n( "Sensitivity" ), firstTab );
	
	macAPLabel = new KWCEntryLabel( i18n( "AP Mac" ), firstTab );
	macAPLabel->setValue( "00:00:00:00:00:00" );
	nickEdit =  new KWCEntryEdit( i18n( "Nickname" ), firstTab);
	
	encGroup = new KEncryptionGroup( i18n( "Encryption" ), experiencedTab->leftBox  );
	rtsGroup = new KThresholdGroup( i18n( "RTS threshold" ), experiencedTab->leftBox );
	fragGroup = new KThresholdGroup( i18n( "Fragmentation threshold" ), experiencedTab->leftBox );
	txPowerGroup = new KTxPowerGroup( i18n( "Tx Power" ), experiencedTab->rightBox  );
	powerGroup = new KPowerGroup( i18n( "Power Management" ), experiencedTab->rightBox );
	experiencedTab->leftBox->setStretchFactor( new QWidget( experiencedTab->leftBox ), 1 );
	experiencedTab->rightBox->setStretchFactor( new QWidget( experiencedTab->rightBox ), 1 );

	experiencedTab->rightBox->adjustSize();
	experiencedTab->adjustSize();
	
	authorLabel = new KWCEntryLabel( i18n( "Author" ), infoTab->gridWidget );
	authorLabel->setValue( "Matthias Reif"  );
	homepageLabel = new KWCEntryURLLabel( i18n( "Homepage" ), "kwavecontrol.org", "http://kwavecontrol.org", infoTab->gridWidget );
	emailLabel = new KWCEntryURLLabel( i18n( "Email" ), "kwc@progeln.de", "kwc@progeln.de", infoTab->gridWidget );
	
	int weVersion = WCard::getWeVersion();
	QString weVersionString;
	if( weVersion < 0 )
		weVersionString = "?";
	else
		weVersionString = QString::number( WCard::getWeVersion() );
	weVersionLabel = new KWCEntryURLLabel( i18n( "WE version" ), weVersionString, "http://www.hpl.hp.com/personal/Jean_Tourrilhes/Linux/Tools.html#wext", infoTab->gridWidget );
	licenseLabel = new KWCEntryLabel( i18n( "License" ), infoTab->gridWidget );
	licenseLabel->setValue( "GPL"  );
	versionLabel = new KWCEntryLabel( i18n( "Version" ), infoTab->gridWidget );
	versionLabel->setValue( VERSION );

	getWirelessDevices( QString( card->getDevice() ) );
	
	KGlobal::config()->setGroup( "KWaveControl" );
	pollingSpin->setValue( KGlobal::config()->readDoubleNumEntry( "polling", 0.5 ) );

	if( card->getTraffic( sent, recv ) )
	{
		trafficTab->clear( recv, sent );
	}

	this->setMainWidget( tab );
}


/*!
	\fn WWidget::adjustEntries()
 */
void WWidget::adjustEntries()
{
	deviceSelection->move( 0, 0 );
	dhcpEntry->move( 3, 0 );
	IPEdit->move( 1, 0 );
	gatewayEdit->move( 2, 0 );
	essidEdit->move( 4, 0 );
	nickEdit->move( 5, 0 );
	modeSelection->move( 6, 0 );
	bitrateSelection->move( 7, 0 );
	channelSelection->move( 8 , 0);
	
	linkBar->move( 0, 1 );
	signalBar->move( 1, 1 );
	noiseBar->move( 2, 1 );
	encodingLabel->move( 3, 1 );
	protocolLabel->move( 4, 1 );
	macAPLabel->move( 5, 1 );
	sensitivityLabel->move( 6, 1 );

	pollingSpin->move( 8, 1 );

	authorLabel->move( 0, 0 );
	homepageLabel->move( 1, 0 );
	emailLabel->move( 2, 0 );
	licenseLabel->move( 3, 0 );
	versionLabel->move( 4, 0 );
	weVersionLabel->move( 5, 0 );
}


/*!
	\fn WWidget::connectSignals()
 */
void WWidget::connectSignals()
{
	connect( tab, SIGNAL( tabSelected( int ) ), this, SLOT( tabSelectedSlot( int ) ) );

	connect( deviceSelection, SIGNAL( textChanged( const QString & ) ), this, SLOT( deviceSlot( const QString & ) ) );
	connect( IPEdit, SIGNAL( returnPressed( const QString & ) ), this, SLOT( IPChangedSlot( const QString & ) ) );
	connect( gatewayEdit, SIGNAL( returnPressed( const QString & ) ), this, SLOT( gatewaySlot( const QString & ) ) );
	connect( nickEdit, SIGNAL( returnPressed( const QString & ) ), this, SLOT( nickSlot( const QString & ) ) );
	connect( essidEdit, SIGNAL( returnPressed( const QString & ) ), this, SLOT( essidSlot( const QString & ) ) );
	connect( modeSelection, SIGNAL( activated( int ) ), this, SLOT( modeSlot( int) ) );
	connect( bitrateSelection, SIGNAL( activated( int ) ), this, SLOT( bitrateSlot( int ) ) );
	connect( channelSelection, SIGNAL( activated( int ) ), this, SLOT( channelSlot( int ) ) );
	connect( pollingSpin, SIGNAL( valueChanged( double ) ), this, SLOT( slotPolling( double ) ) );
	connect( powerGroup, SIGNAL( changed( bool, int ) ), this, SLOT( powerSlot( bool, int ) ) );
	connect( txPowerGroup, SIGNAL( changed( bool, bool, int, int ) ), this, SLOT( txPowerSlot(bool, bool, int, int ) ) );
	connect( rtsGroup, SIGNAL( applied( bool, int ) ), this, SLOT( rtsSlot( bool, int ) ) );
	connect( fragGroup, SIGNAL( applied( bool, int ) ), this, SLOT( fragSlot( bool, int ) ) );

	connect( encGroup, SIGNAL( changed( int, unsigned char*, int ) ), this, SLOT( encryptionSlot( int, unsigned char*, int ) ) );
	connect( encGroup, SIGNAL( changed( int ) ), this, SLOT( encryptionSlot( int ) ) );
	connect( encGroup, SIGNAL( modeChanged( int ) ), this, SLOT( encryptionModeSlot( int ) ) );

	connect( dhcpEntry, SIGNAL( buttonClicked() ), this, SLOT( dhcpSlot() ) );
	connect( dhcpTimer, SIGNAL( timeout() ), this, SLOT( dhcpSlot() ) );

	connect( this, SIGNAL( user1Clicked() ), this, SLOT( whatsThisSlot() ) );
	connect( this, SIGNAL( user2Clicked() ), this, SLOT( rootSlot() ) );
}


/*!
	\fn WWidget::basicTimerSlot()
*/
void WWidget::basicTimerSlot()
{
	if( isVisible() )
	{
		if( card->isWave() )
		{
			if( ! isWave )
			{
				setWave();
			}

			if( ! essidEdit->hasFocus() )
			{
				card->getESSID( buffer );
				essidEdit->setValue( buffer );
			}
			
			modeSelection->setCurrentItem( card->getMode() );
			
			if( !nickEdit->hasFocus() )
			{
				if( card->getNickName( buffer ) >= 0 )
				{
					nickEdit->setValue( buffer );
				}
				else
				{
					nickEdit->setValue( "" );
				}
			}
			
			if( card->has.stats )
			{
				card->getLinkQual(link, avg, signal, noise, maxQual, maxSignal, maxNoise);
				
				if( card->has.range && ( signal != 0 ) )
				{
					if( maxQual == 0)
					{
						linkBar->showBar( false );
						linkBar->setValue( QString::number( link ) + "/" + QString::number( maxQual ) );
					}
					else
					{
						linkBar->showBar();
						linkBar->setValue( (100*link) / maxQual );
					}
					
					if( signal > maxSignal )
					{
						signalBar->showBar( false );
						signalBar->setValue( QString::number( signal - 0x100 ) + " dBm" );
						noiseBar->showBar( false );
						noiseBar->setValue( QString::number( noise - 0x100 ) + " dBm" );
					}
					else
					{
						if( maxSignal == 0 )
						{
							signalBar->showBar( false );
							signalBar->setValue( QString::number( signal ) + "/" + QString::number( maxSignal ) );
						}
						else
						{
							signalBar->showBar();
							signalBar->setValue( (100*signal) / maxSignal );
						}
						if( maxNoise == 0 )
						{
							noiseBar->showBar( false );
							noiseBar->setValue( QString::number( noise ) + "/" + QString::number( maxNoise ) );
						}
						else
						{
							noiseBar->showBar();
							noiseBar->setValue( (100*noise) / maxNoise );
						}
					}
					
				}
				else
				{
					linkBar->showBar( false );
					linkBar->setValue( QString::number( link ) );
					signalBar->showBar( false );
					signalBar->setValue( QString::number( signal ) );
					noiseBar->showBar( false );
					noiseBar->setValue( QString::number( noise ) );
				}
			}
			
			card->getProtocol(buffer);
			protocolLabel->setValue(buffer);

			if(card->getEncode(buffer))
			{
				encodingLabel->setValue(i18n("enabled"));
			}
			else
			{
				encodingLabel->setValue(i18n("disabled"));
			}

			encGroup->setCurrentKey( card->getEncoding() );

			if( card->has.txPower )
			{
				int value, unit;
				bool disabled, fixed;
				value = card->getTransmitPower( disabled, fixed, unit );
				txPowerGroup->setEnabled( value >= 0 );
				if( value >= 0 )
				{
					txPowerGroup->setTxPower( disabled, fixed, value, unit );
				}
			}
			else
			{
				txPowerGroup->clear();
				txPowerGroup->setEnabled( false );
			}

			if( card->has.ap_addr )
			{
				card->getAPMac(buffer);
				macAPLabel->setValue(buffer);
			}
			else
			{
				macAPLabel->setValue( tr("unavailable") );
			}
			
			if( card->has.bitrate )
			{
				int bitrate = card->getBitRate();
				if( bitrate >= 0 )
				{
					map< int, int >::iterator bitrateIterator = bitrateIndices.find( bitrate );
					if( bitrateIterator != bitrateIndices.end() )
					{
						bitrateSelection->setCurrentItem( bitrateIterator->second );
					}
					else
					{
						int newBitrateIndex = bitrateIndices.size() + 1;
						bitrateSelection->insertItem( bitrate2String( bitrate), newBitrateIndex );
						bitrateIndices[ bitrate ] = newBitrateIndex;
						bitrateValues[ newBitrateIndex ] = bitrate;
						bitrateSelection->setCurrentItem( newBitrateIndex );
					}
				}
			}
			else
			{
				bitrateSelection->setValue( tr("unavailable") );
			}
			
			freq = 0.0;
			numChannel = channel = 0;
			
			if( card->getFreguencyAndChannel( freq, channel, numChannel ) < 0 )
			{
				channelSelection->setAltLabel( i18n( "unavailable" ) );
				channelSelection->showValue2();
			}
			else
			{
				if( channel >= 0 )
				{
					channelSelection->setCurrentItem( channel - 1 );
					channelSelection->showValue();
				}
				else
				{
					channelSelection->setAltLabel( i18n ("unavailable") );
					channelSelection->showValue2();
				}
			}
			
			if( card->has.power )
			{
				bool powerEnabled;
				int flags;
				if( card->getPower( powerEnabled, flags ) >= 0 )
				{
					powerGroup->setPower( powerEnabled, flags );
				}
			}
			
			int sens = 0;
			int sensRange = 0;
			card->getSensitivity( sens, sensRange );
	
			if( card->has.sens )
			{
				if( card->has.range )
				{
					if( sens < 0 )
					{
						sensitivityLabel->setValue( QString::number( sens ) + " dBm" );
					}
					else
					{
						sensitivityLabel->setValue( QString::number( sens ) + "/" + QString::number( sensRange ) );
					}
				}
				else
				{
					sensitivityLabel->setValue( QString::number( sens ) );
				}
			}
			else
			{
				sensitivityLabel->setValue( i18n( "unavailable" ) );
			}
			
			if( card->has.rts && !rtsGroup->hasFocus() && !rtsGroup->hasChanged() )
			{
				iw_param rts;
				card->getRTS( rts );
				rtsGroup->set( !rts.disabled, rts.value );
			}
			
			if( card->has.frag && !fragGroup->hasFocus() && !fragGroup->hasChanged() )
			{
				iw_param frag;
				card->getFrag( frag );
				fragGroup->set( !frag.disabled, frag.value );
			}
			
			iw_discarded discarded;
			int missedBeacon;
			if( card->getStats( discarded, missedBeacon ) )
			{
				trafficTab->setDiscarded( discarded, missedBeacon );
			}
			else
			{
				trafficTab->setDiscardedUnknown();
			}
		}
		else
		{
			if( isWave )
			{
				setWave(false);
			}
		}
		if( !IPEdit->hasFocus() )
		{
			IPEdit->setValue( card->getIP() );
		}
		if( !gatewayEdit->hasFocus() )
		{
			gatewayEdit->setValue( card->getGateway() );
			gateway = gatewayEdit->getText();
		}
	}
}


/*!
	\fn WWidget::secondTimerSlot()
*/
void WWidget::secondTimerSlot()
{
	if( card->getTraffic( sent, recv ) )
	{
		trafficTab->setTraffic( recv, sent );
		trafficTab->setEnabled( true );
	}
	else
	{
		trafficTab->setEnabled( false );
	}

	if( tab->isTabEnabled( "Traffic" ) )
		trafficTab->trafficWidget->paintDiagramm();

	if( scanning && scanThread->finished() )
	{
		scanning = false;
		scanComplete( scanThread->apInfos, scanThread->error, scanThread->errorNo );
		scanWidget->setScanning( false );
	}
	
	if( dhcpThreadRunning && dhcpThread->finished() )
	{
		dhcpThreadRunning = false;
		std::string result = dhcpThread->getResult();
		if( result == NULL )
		{
			KMessageBox::error( this, i18n( "Failed to run dhcpcd" ) );
		}
		else if( result.length() > 0 )
		{
			KMessageBox::detailedError( this, i18n( "Failed to run dhcpcd." ), QString( result ) );
		}
		dhcpEntry->enableButton( true );
	}

	dhcpEntry->setStartText( ! card->isDhcpRunning() );
	adjustSize();
}


/*!
	\fn WWidget::setStaticInfo( )
*/
void WWidget::setStaticInfo( )
{
	bool isWave = card->isWave();

	powerGroup->setEnabled( isWave );
	txPowerGroup->setEnabled( isWave );
	scanWidget->setScanningEnabled( isWave );
	
	int i;
	
	if( isWave )
	{
		bitrateSelection->clear();
		bitrateIndices.clear();
		bitrateValues.clear();
		if( card->has.range && card->has.bitrate )
		{
			for( i = 0; i < card->range->num_bitrates; i++ )
			{
				bitrateSelection->insertItem( bitrate2String( card->range->bitrate[ i ] ), i );
				bitrateIndices[ card->range->bitrate[ i ] ] = i;
 				bitrateValues[ i ] = card->range->bitrate[ i ];
			}
		}
		
		channelSelection->clear();
		if( card->has.range )
		{
			for( i = 0; i < card->range->num_frequency; i++ )
			{
				double freq;
				freq = WCard::frequency2Float( card->range->freq[ i ] );
				channelSelection->insertItem( QString::number( i + 1 ) + " | " + frequency2QString( freq ) );
			}
		}
		
		powerGroup->insertPowers( card );
		txPowerGroup->insertTxPowers( card );

		encGroup->setKeys( card->getKeys() );
	}
}


/*!
	\fn WWidget::deviceSlot(const QString & text)
	when the device was changed
*/
void WWidget::deviceSlot(const QString & text)
{
	card->setDevice( text.data() );
	KGlobal::config()->setGroup( "KWaveControl" );
	KGlobal::config()->writeEntry( "device", text );
	if( card->getTraffic( sent, recv ) )
	{
		trafficTab->clear( recv, sent );
	}
	setStaticInfo();
}


/*!
	\fn WWidget::IPChangedSlot( const QString & text )
*/
void WWidget::IPChangedSlot( const QString & text )
{
	if( text.isEmpty() )
		return;
	char* ip = card->setIP( text.data() );
	if( ip == NULL )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set ip." ), strerror( errno ) );
	}
}


/*!
	\fn WWidget::gatewaySlot( const QString & text )
 */
void WWidget::gatewaySlot( const QString & text )
{
	if( text.isEmpty() )
		return;
	if( gateway.length() > 0 )
	{
		if( card->deleteGateway( gateway.data() ) < 0 )
		{
			KMessageBox::detailedError( this, i18n( "Failed to remove gateway." ), strerror( errno ) );
			return;
		}
	}

	if( text.length() > 0 )
	{
		if( card->setGateway( text.data() ) < 0 )
		{
			KMessageBox::detailedError( this, i18n( "Failed to set gateway." ), strerror( errno ) );
		}
	}
}



/*!
	\fn WWidget::nickSlot( const QString & text )
	when the nickname was changed
*/
void WWidget::nickSlot( const QString & text )
{
	if( card->setNickName( text.ascii() ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set nickname." ), strerror( errno ) );
	}
	// has.nickname actualize
	card->setHas();
	card->getNickName( buffer );
	nickEdit->setValue(buffer);
}


/*!
	\fn WWidget::essidSlot( const QString & text )
	when the essid was changed
*/
void WWidget::essidSlot( const QString & text )
{
	if( card->setESSID( text.ascii() ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set ESSID." ), strerror( errno ) );
	}
	card->getESSID( buffer );
	essidEdit->setValue(buffer);
}


/*!
	\fn WWidget::bitrateSlot( int index )
	when the bitrate was changed
*/
void WWidget::bitrateSlot( int index )
{
	if( card->setBitRate( bitrateValues[ index ] ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set bitrate." ), strerror( errno ) );
	}
}


/*!
	\fn WWidget::modeSlot( int index )
	when the mode was changed
*/
void WWidget::modeSlot( int index )
{
	int newMode = card->setMode( index );
	if( newMode < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set mode." ), strerror( errno ) );
	}
	else
	{
		modeSelection->setCurrentItem( newMode );
	}

}


/*!
	\fn WWidget::sensitivitySlot(int index)
	when the sensitivity was changed
*/
void WWidget::sensitivitySlot(int index)
{
	if( card->setSensitivity( index+1 ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set sensitivity." ), strerror( errno ) );
	}
}


/*!
	\fn WWidget::channelSlot( int index )
*/
void WWidget::channelSlot( int index )
{
	if( card->setChannel( index + 1, false ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set channel." ), strerror( errno ) );
	}
}


/*!
	\fn WWidget::txPowerSlot( bool disabled, bool fixed, int value, int unit )
*/
void WWidget::txPowerSlot( bool disabled, bool fixed, int value, int unit )
{
	if( card->setTransmitPower( disabled, fixed, value, unit ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set txPower." ), strerror( errno ) );
	}
}


/*!
	\fn WWidget::powerSlot( bool enabled, int mode )
*/
void WWidget::powerSlot( bool enabled, int mode )
{
	if( card->setPower( enabled, mode ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set power management." ), strerror( errno ) );
	}
}


/*!
	\fn WWidget::setWave(bool is)
	set isWave true, if the card is a wavelan device
*/
void WWidget::setWave( bool is )
{
	modeSelection->showValue(is);
	linkBar->showValue(is);
	signalBar->showValue(is);
	noiseBar->showValue(is);
	encodingLabel->showValue(is);
	protocolLabel->showValue(is);
	bitrateSelection->showValue(is);
	channelSelection->showValue(is);
	sensitivityLabel->showValue(is);
	essidEdit->showValue(is);
	macAPLabel->showValue(is);
	nickEdit->showValue(is);
	
	txPowerGroup->setEnabled( is );
	powerGroup->setEnabled( is );
	fragGroup->setEnabled( is );
	rtsGroup->setEnabled( is );
	encGroup->setEnabled( is );

	isWave = is;
	if( is )
	{
		card->setHas();
		setStaticInfo();
	}
	else
	{
		trafficTab->setDiscardedUnknown();
	}
}


/*!
    \fn WWidget::getWirelessDevices( QString preset )
 */
void WWidget::getWirelessDevices( QString preset )
{
	ifstream procFile( "/proc/net/wireless" );
	
	if( procFile.good() )
	{
		char* buffer = new char[512];
		char* device = new char[32];
	
		procFile.getline( buffer, 510 );
			if( ! procFile.good() )
				return;
		procFile.getline( buffer, 510 );
			if( ! procFile.good() )
				return;
		procFile.get( );
		int index = 0;
		
		while( procFile.good() )
		{
			procFile >> device;
			if( ! procFile.good() )
				return;
			procFile.getline( buffer, 510 );
			if( ! procFile.good() )
				return;
			procFile.get( );
			if( strlen(device) > 0 )
			{
				device[ strlen(device) -1 ] = '\0';
				if( ! deviceSelection->contains( device ) )
				{
					deviceSelection->insertItem( device, index );
					if( QString(device) == preset )
						deviceSelection->setCurrentItem( index );
					index++;
				}
			}
			
		}
		delete [] device;
		delete [] buffer;
		procFile.close();
	}
	if( ! preset.isEmpty() )
		deviceSelection->setValue( preset );
}


/*!
    \fn WWidget::scanSlot()
 */
void WWidget::scanSlot()
{
	if( ! scanning )
	{
		scanThread->start();
		scanning = true;
	}
}


/*!
    \fn WWidget::scanComplete( APInfos apInfos, int error, int errorNo )
 */
void WWidget::scanComplete( APInfos apInfos, int error, int errorNo )
{
	switch( error )
	{
		case 0:
			scanWidget->update( card, apInfos );
			break;
		case 1:
			KMessageBox::detailedError( this, i18n( "Scanning failed" ), i18n( "Interface doesn't support scanning" ) );
			break;
		case 2:
			KMessageBox::detailedError( this, i18n( "Interface doesn't support scanning" ), strerror( errorNo ) );
			break;
		case 3:
			KMessageBox::detailedError( this, i18n( "Scanning failed" ), strerror( errorNo ) );
			break;
		case 4:
			KMessageBox::detailedError( this, i18n( "Scanning failed" ), i18n( "Unhandled signal" ) );
			break;
		case 5:
			KMessageBox::detailedError( this, i18n( "Scanning failed" ), i18n( "Allocation failed" ) );
			break;
		case 6:
			KMessageBox::detailedError( this, i18n( "Failed to read scan data" ), strerror( errorNo ) );
			break;
		default:
			KMessageBox::detailedError( this, i18n( "Scanning failed" ), i18n( "Unknown error" ) );
			break;
	}
}


/*!
    \fn WWidget::bitrate2String( int bitrate )
 */
QString WWidget::bitrate2String( int bitrate )
{
	QString s;
	if( bitrate >= WCard::GIGA_I )
	{
		if( bitrate % WCard::GIGA_I != 0 )
			s = QString::number( (float)bitrate / WCard::GIGA_I, 'f', 1 ) + " Gbit";
		else
			s = QString::number( (int)(bitrate / WCard::GIGA_I) ) + " Gbit";
	}
	else if( bitrate >= WCard::MEGA_I )
	{
		if( bitrate % WCard::MEGA_I != 0 )
			s = QString::number( (float)bitrate / WCard::MEGA_I, 'f', 1 ) + " Mbit";
		else
			s = QString::number( (int)(bitrate / WCard::MEGA_I) ) + " Mbit";
	}
	else if( bitrate >= WCard::KILO_I )
	{
		if( bitrate % WCard::KILO_I != 0 )
			s = QString::number( (float)bitrate / WCard::KILO_I, 'f', 1 ) + " kbit";
		else
			s = QString::number( (int)(bitrate / WCard::KILO_I) ) + " kbit";
	}
	else
	{
		s = QString::number( bitrate ) + " bit";
	}
	return s;
}


/*!
	\fn WWidget::rootSlot()
	restarts the program as root (using kdesu)
 */
void WWidget::rootSlot()
{
	if( execlp( "kdesu", "kdesu", cmd.ascii() , NULL ) == -1 )
	{
		KMessageBox::detailedError( this, i18n( "Using kdesu to restart as root failed." ), strerror( errno ) );
	}
}


/*!
	\fn WWidget::connectToAp( QString essid, double frequency, QString mode, bool enc, bool dhcp )
 */
void WWidget::connectToAp( QString essid, double frequency, QString mode, bool enc, bool dhcp )
{
	if( ! essid.isEmpty() )
	{
		card->setESSID( essid.ascii() );
	}
	if( frequency > 0 )
	{
		card->setChannel( (int)frequency, true );
	}
	if( ! mode.isEmpty() )
	{
		int i;
		for( i = 0; i < NUM_OPERATION_MODES; i++ )
		{
			if( strcmp( iw_operation_mode[ i ], mode.ascii() ) == 0 )
			{
				card->setMode( i );
				break;
			}
		}
	}
	if( dhcp )
	{
		dhcpTimer->start( 1000, true );
	}
}


/*!
	\fn WWidget::getProfile()
 */
Profile WWidget::getProfile()
{
	Profile profile;
	profile.essid = essidEdit->getText();
	profile.mode = modeSelection->getCurrentItem();
	int channel, numChannel;
	card->getFreguencyAndChannel( profile.frequency, channel, numChannel );
	if( channel >= 0 )
		profile.frequency = channel;
	profile.ip = IPEdit->getText();

	profile.security = encGroup->getMode();
	if( profile.security != IW_ENCODE_DISABLED )
	{
		profile.keyIndex = encGroup->getCurrentIndex() + 1;
		profile.key = encGroup->getCurrentKey();
		
	}
	else
	{
		profile.keyIndex = -1;
		profile.key = "";
	}

	QString device = card->getDevice();
	QString file = QString( "/var/run/dhcpcd-" ) + device + QString( ".pid" );
	std::fstream fin;
	fin.open( file.ascii(), std::ios::in );
	profile.dhcp = fin.is_open();
	fin.close();
	if( profile.dhcp )
	{
		QString dhcpMessage = "The device " + device + " seems to be configured by dhcpcd.\n";
		dhcpMessage += "Do you want to save the use of dhcpcd in the profile?\n";
		dhcpMessage += "(So the next time you load this profile dhcpcd will be run automatically, otherwise the settings will be restored statically.)";
		profile.dhcp = ( KMessageBox::questionYesNo( this, dhcpMessage, "save use of dhcp" ) == KMessageBox::Yes );
	}
	
	return profile;
}


/*!
    \fn WWidget::frequency2QString( double frequency )
 */
QString WWidget::frequency2QString( double frequency )
{
	if( frequency > WCard::GIGA )
	{
		return QString::number( frequency / WCard::GIGA,'f', 3 ) + " GHz";
	}
	else if( frequency > WCard::MEGA )
	{
		return QString::number( frequency / WCard::MEGA,'f', 3 ) + " MHz";
	}
	else if( frequency > WCard::KILO )
	{
		return QString::number( frequency / WCard::KILO,'f', 3 ) + " kHz";
	}
	else
	{
		return QString::number( frequency,'f', 3 ) + " Hz";
	}
}


/*!
    \fn WWidget::setProfile( Profile profile )
 */
void WWidget::setProfile( Profile profile )
{
	essidSlot( profile.essid );
	if( card->setChannel( profile.frequency, false ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set channel." ), strerror( errno ) );
	}

	if( ! profile.dhcp )
	{
		IPChangedSlot( profile.ip );
		gatewaySlot( profile.gateway );
	}
	else
	{
		// let the device settle down a few seconds
		dhcpTimer->start( 1000, true );
	}

	modeSlot( profile.mode );

	if( profile.security != IW_ENCODE_DISABLED )
	{
		int length;
		unsigned char *key = encGroup->hexToKey( profile.key, length );
		encryptionSlot( profile.security, profile.keyIndex, key, length );
	}
	else
	{
		encryptionModeSlot( profile.security );
	}
}


/*!
    \fn WWidget::whatsThisSlot()
 */
void WWidget::whatsThisSlot()
{
	QWhatsThis::enterWhatsThisMode();
}


/*!
    \fn WWidget::setWhatsThis()
 */
void WWidget::setWhatsThis()
{
	essidEdit->setWhatsThis( QWhatsThisStrings::essid() );
	nickEdit->setWhatsThis( QWhatsThisStrings::nickname() );
	channelSelection->setWhatsThis( QWhatsThisStrings::channel() );
	sensitivityLabel->setWhatsThis( QWhatsThisStrings::sensitivity() );
	modeSelection->setWhatsThis( QWhatsThisStrings::mode() );
	bitrateSelection->setWhatsThis( QWhatsThisStrings::bitrate() );
	QWhatsThis::add( powerGroup, QWhatsThisStrings::power() );
	QWhatsThis::add( txPowerGroup, QWhatsThisStrings::txPower() );
	macAPLabel->setWhatsThis( QWhatsThisStrings::ap() );
	linkBar->setWhatsThis( QWhatsThisStrings::link() );
	signalBar->setWhatsThis( QWhatsThisStrings::signal() );
	noiseBar->setWhatsThis( QWhatsThisStrings::noise() );
	deviceSelection->setWhatsThis( QWhatsThisStrings::device() );
	IPEdit->setWhatsThis( QWhatsThisStrings::ip() );
	gatewayEdit->setWhatsThis( QWhatsThisStrings::gateway() );
	pollingSpin->setWhatsThis( QWhatsThisStrings::polling() );
	QWhatsThis::add( actionButton( KDialogBase::User2 ), QWhatsThisStrings::root() );
	QWhatsThis::add( rtsGroup, QWhatsThisStrings::rts() );
	QWhatsThis::add( fragGroup, QWhatsThisStrings::frag() );
	QWhatsThis::add( encGroup, QWhatsThisStrings::encryption() );
}


/*!
	\fn WWidget::slotPolling( double value )
 */
void WWidget::slotPolling( double value )
{
	pollingChanged( value );
}


/*!
    \fn WWidget::rtsSlot( bool enabled, int value )
 */
void WWidget::rtsSlot( bool enabled, int value )
{
	iw_param rts;
	rts.disabled = ! enabled;
	rts.value = value;
	rts.fixed = 1;
	if( card->setRTS( rts ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set RTS threshold." ), strerror( errno ) );
	}
	rtsGroup->clearFocus();
}


/*!
    \fn WWidget::fragSlot( bool enabled, int value )
 */
void WWidget::fragSlot( bool enabled, int value )
{
	iw_param frag;
	frag.disabled = ! enabled;
	frag.value = value;
	frag.fixed = 1;
	if( card->setFrag( frag ) < 0 )
	{
		KMessageBox::detailedError( this, "Failed to set fragmentation threshold.", strerror( errno ) );
	}
	fragGroup->clearFocus();
}


/*!
    \fn WWidget::tabSelectedSlot( int )
 */
void WWidget::tabSelectedSlot( int )
{
	rtsGroup->clearFocus();
	rtsGroup->clearChanged();
	fragGroup->clearFocus();
	fragGroup->clearChanged();
	if( card->isWave() )
	{
		encGroup->setKeys( card->getKeys() );
	}
}


/*!
    \fn WWidget::encryptionSlot( int mode, int index, unsigned char* key, int length )
 */
void WWidget::encryptionSlot( int mode, int index, unsigned char* key, int length )
{
	if( card->setEncoding( mode, index, key, length ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set encryption." ), strerror( errno ) );
	}
	encGroup->setKeys( card->getKeys() );
}



/*!
    \fn WWidget::encryptionSlot( int index, unsigned char* key, int length )
 */
void WWidget::encryptionSlot( int index, unsigned char* key, int length )
{
	encryptionSlot( encGroup->getMode(), index, key, length );
}


/*!
    \fn WWidget::encryptionSlot( int index )
 */
void WWidget::encryptionSlot( int index )
{
	Encoding encodingBefore = card->getEncoding();
	if( card->setEncoding( index ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set encryption." ), strerror( errno ) );
	}
	Encoding encodingAfter = card->getEncoding();
	if( ( encodingBefore.mode != IW_ENCODE_DISABLED ) && ( encodingAfter.mode == IW_ENCODE_DISABLED ) )
	{
		unsigned char c = 0;
		card->setEncoding( encodingBefore.mode, index, &c, 1 );
		encGroup->setKeys( card->getKeys() );
	}
}


/*!
    \fn WWidget::encryptionModeSlot( int mode )
 */
void WWidget::encryptionModeSlot( int mode )
{
	if( card->setEncodingMode( mode ) < 0 )
	{
		KMessageBox::detailedError( this, i18n( "Failed to set encryption." ), strerror( errno ) );
	}
	Encoding encoding = card->getEncoding();
	if( ( mode != IW_ENCODE_DISABLED ) && ( encoding.mode == IW_ENCODE_DISABLED ) )
	{
		unsigned char c = 0;
		card->setEncoding( mode, encoding.key, &c, 1 );
	}

	encGroup->setKeys( card->getKeys() );
}


/*!
    \fn WWidget::getScanWidget()
 */
KScanWidget* WWidget::getScanWidget()
{
	return scanWidget;
}


/*!
    \fn WWidget::dhcpSlot()
 */
void WWidget::dhcpSlot()
{
	if( !card->isDhcpRunning() )
	{
		if( !dhcpThreadRunning && !dhcpThread->running() )
		{
			dhcpThreadRunning = true;
			dhcpThread->start();
			dhcpEntry->enableButton( false );
		}
	}
	else
	{
		if( card->stopDhcpcd() != 0 )
		{
			KMessageBox::detailedError( this, i18n( "Failed to stop dhcpcd." ), strerror( errno ) );
		}
	}
}
