/*
 * Decompiled with CFR 0.152.
 */
package org.limewire.nio;

import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import org.limewire.nio.ByteBufferCache;
import org.limewire.service.ErrorService;

public class CircularByteBuffer {
    private static final DevNull DEV_NULL = new DevNull();
    private final ByteBufferCache cache;
    private final int capacity;
    private ByteBuffer in;
    private ByteBuffer out;
    private ByteOrder order = ByteOrder.BIG_ENDIAN;
    private boolean lastOut = true;

    public CircularByteBuffer(int capacity, ByteBufferCache cache) {
        this.cache = cache;
        this.capacity = capacity;
    }

    private void initBuffers() {
        if (this.in == null) {
            assert (this.out == null);
            this.in = this.cache.getHeap(this.capacity);
            this.out = this.in.asReadOnlyBuffer();
        } else assert (this.out != null);
    }

    public final int remainingIn() {
        int o;
        if (this.in == null) {
            return this.capacity;
        }
        int i = this.in.position();
        if (i > (o = this.out.position())) {
            return this.in.capacity() - i + o;
        }
        if (i < o) {
            return o - i;
        }
        return this.lastOut ? this.in.capacity() : 0;
    }

    public final int remainingOut() {
        return this.capacity() - this.remainingIn();
    }

    public void put(ByteBuffer src) {
        if (src.remaining() > this.remainingIn()) {
            throw new BufferOverflowException();
        }
        if (!src.hasRemaining()) {
            return;
        }
        this.lastOut = false;
        this.initBuffers();
        if (src.remaining() > this.in.remaining()) {
            int oldLimit = src.limit();
            src.limit(src.position() + this.in.remaining());
            this.in.put(src);
            this.in.rewind();
            src.limit(oldLimit);
        }
        this.in.put(src);
    }

    public void put(CircularByteBuffer src) {
        if (src.remainingOut() > this.remainingIn()) {
            throw new BufferOverflowException();
        }
        this.initBuffers();
        if (this.in.remaining() < src.remainingOut()) {
            src.out.limit(this.in.remaining());
            this.in.put(src.out);
            this.in.rewind();
            src.out.limit(src.out.capacity());
        }
        this.in.put(src.out);
        this.lastOut = false;
    }

    public byte get() {
        if (this.remainingOut() < 1) {
            throw new BufferUnderflowException();
        }
        if (!this.out.hasRemaining()) {
            this.out.rewind();
        }
        byte ret = this.out.get();
        this.releaseIfEmpty();
        this.lastOut = true;
        return ret;
    }

    public void get(byte[] dest) {
        this.get(dest, 0, dest.length);
    }

    public void get(byte[] dest, int offset, int length) {
        if (this.remainingOut() < length) {
            throw new BufferUnderflowException();
        }
        if (length <= 0) {
            return;
        }
        this.lastOut = true;
        if (this.out.remaining() < length) {
            int remaining = this.out.remaining();
            this.out.get(dest, offset, remaining);
            offset += remaining;
            length -= remaining;
            this.out.rewind();
        }
        this.out.get(dest, offset, length);
        this.releaseIfEmpty();
    }

    public void get(ByteBuffer dest) {
        if (this.remainingOut() < dest.remaining()) {
            throw new BufferUnderflowException();
        }
        if (dest.remaining() <= 0) {
            return;
        }
        this.lastOut = true;
        if (this.out.remaining() < dest.remaining()) {
            dest.put(this.out);
            this.out.rewind();
        }
        this.out.limit(this.out.position() + dest.remaining());
        dest.put(this.out);
        this.out.limit(this.out.capacity());
        this.releaseIfEmpty();
    }

    private void releaseIfEmpty() {
        if (this.in != null && this.out != null && this.out.position() == this.in.position()) {
            this.cache.release(this.in);
            this.in = null;
            this.out = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int write(WritableByteChannel sink, int len) throws IOException {
        int written;
        int thisTime = 0;
        for (written = 0; this.remainingOut() > 0 && written < len; written += thisTime) {
            if (!this.out.hasRemaining()) {
                this.out.rewind();
            }
            if (this.in.position() > this.out.position()) {
                if (len == Integer.MAX_VALUE) {
                    this.out.limit(this.in.position());
                } else {
                    this.out.limit(Math.min(this.in.position(), len - written + this.out.position()));
                }
            }
            try {
                thisTime = sink.write(this.out);
            }
            finally {
                if (thisTime > 0) {
                    this.lastOut = true;
                }
            }
            this.out.limit(this.out.capacity());
            if (thisTime == 0) break;
        }
        this.releaseIfEmpty();
        return written;
    }

    public int write(WritableByteChannel sink) throws IOException {
        return this.write(sink, Integer.MAX_VALUE);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int read(ReadableByteChannel source) throws IOException {
        int read = 0;
        int thisTime = 0;
        while (this.remainingIn() > 0) {
            this.initBuffers();
            if (!this.in.hasRemaining()) {
                this.in.rewind();
            }
            if (this.out.position() > this.in.position()) {
                this.in.limit(this.out.position());
            }
            try {
                thisTime = source.read(this.in);
            }
            finally {
                if (thisTime > 0) {
                    this.lastOut = false;
                }
            }
            this.in.limit(this.in.capacity());
            if (thisTime == 0) break;
            if (thisTime == -1) {
                if (read == 0) {
                    throw new IOException();
                }
                return read;
            }
            read += thisTime;
        }
        return read;
    }

    public int size() {
        return this.remainingOut();
    }

    public int capacity() {
        return this.capacity;
    }

    public String toString() {
        return "circular buffer in:" + this.in + " out:" + this.out;
    }

    public void order(ByteOrder order) {
        this.order = order;
    }

    public int getInt() throws BufferUnderflowException {
        if (this.remainingOut() < 4) {
            throw new BufferUnderflowException();
        }
        if (this.order == ByteOrder.BIG_ENDIAN) {
            return this.getU() << 24 | this.getU() << 16 | this.getU() << 8 | this.getU();
        }
        return this.getU() | this.getU() << 8 | this.getU() << 16 | this.getU() << 24;
    }

    private int getU() {
        return this.get() & 0xFF;
    }

    public void discard(int num) {
        if (this.remainingOut() < num) {
            throw new BufferUnderflowException();
        }
        try {
            this.write(DEV_NULL, num);
        }
        catch (IOException impossible) {
            ErrorService.error(impossible);
        }
    }

    private static class DevNull
    implements WritableByteChannel {
        private DevNull() {
        }

        public int write(ByteBuffer src) throws IOException {
            int ret = src.remaining();
            src.position(src.limit());
            return ret;
        }

        public void close() throws IOException {
        }

        public boolean isOpen() {
            return true;
        }
    }
}

