#
# (C) Tenable Network Security, Inc.
#


  desc = "
Synopsis :

The remote web server contains an ASP.NET script that is susceptible
to an information disclosure vulnerability. 

Description :

The remote host is running the Spam Quarantine Management web
component of MailMarshal SMTP, a mail server for Windows. 

The version of the Spam Quarantine Management web component installed
on the remote host fails to sanitize input to the 'emailTextBox'
parameter of the 'Register.aspx' script before using it in database
queries.  By appending a long string of blanks and his own email
address, an unauthenticated attacker may be able to leverage this
issue to reset and retrieve the password to any user account provided
he knows the email address associated with it. 

See also :

http://www.sec-1labs.co.uk/advisories/BTA_Full.pdf
http://archives.neohapsis.com/archives/fulldisclosure/2007-07/0323.html
http://www.nessus.org/u?1835b90f

Solution :

Upgrade to MailMarshal SMTP 6.2.1 or later. 

Risk factor : 

High / CVSS Base Score : 7.6
(CVSS2#AV:N/AC:H/Au:N/C:C/I:C/A:C)";


if (description)
{
  script_id(25711);
  script_version("$Revision: 1.3 $");

  script_cve_id("CVE-2007-3796");
  script_bugtraq_id(24936);
  script_xref(name:"OSVDB", value:"38182");

  script_name(english:"MailMarshal Spam Quarantine Password Retrieval Vulnerability");
  script_summary(english:"Checks version in SMTP banner");

  script_description(english:desc);

  script_category(ACT_GATHER_INFO);
  script_family(english:"CGI abuses");

  script_copyright(english:"This script is Copyright (C) 2007-2008 Tenable Network Security, Inc.");

  script_dependencies("find_service_3digits.nasl", "doublecheck_std_services.nasl", "http_version.nasl");
  script_exclude_keys("Settings/disable_cgi_scanning");
  script_require_ports("Services/smtp", 25, "Services/www", 80);

  exit(0);
}


include("global_settings.inc");
include("http_func.inc");
include("http_keepalive.inc");
include("smtp_func.inc");


# Grab the version from the SMTP banner.
port = get_kb_item("Services/smtp");
if (!port) port = 25;
if (!get_port_state(port)) exit(0);

ver = NULL;
banner = get_smtp_banner(port:port);
if (banner && " ESMTP MailMarshal " >< banner)
{
  pat = " ESMTP MailMarshal \(v([0-9][0-9.]+)\)";
  matches = egrep(pattern:pat, string:banner);
  if (matches)
  {
    foreach match (split(matches))
    {
      match = chomp(match);
      item = eregmatch(pattern:pat, string:match);
      if (!isnull(item))
      {
        ver = item[1];
        break;
      }
    }
  }
}


# If it's a vulnerable version...
if (
  ver &&
  ver =~ "^([0-5]\.|6\.([01]\.|2\.0[^0-9]?))"
)
{
  report = NULL;

  # If we're being paranoid, just flag it as vulnerable.
  if (report_paranoia > 1)
    report = string(
      desc,
      "\n\n",
      "Plugin output :\n",
      "\n",
      "According to its SMTP banner, version ", ver, " of MailMarshal is\n",
      "installed on the remote host, but Nessus did not check whether the\n",
      "optional Spam Quarantine component is available because of the Report\n",
      "Paranoia setting in effect when this scan was run.\n"
    );
  # Otherwise, make sure the affected component is installed.
  else 
  {
    port = get_http_port(default:80);
    if (!get_port_state(port)) exit(0);
    if (!can_host_asp(port:port)) exit(0);

    # Loop through directories.
    if (thorough_tests) dirs = make_list("/SpamConsole", cgi_dirs());
    else dirs = make_list(cgi_dirs());

    foreach dir (dirs)
    {
      url = string(dir, "/Register.aspx");
      req = http_get(item:url, port:port);
      res = http_keepalive_send_recv(port:port, data:req, bodyonly:TRUE);
      if (res == NULL) exit(0);

      # If it is...
      if (
        "MailMarshal" >< res && 
        "Spam Quarantine Management" >< res &&
        '<form name="Form1" method="post" action="Register.aspx"' >< res
      )
      {
        report = string(
          desc,
          "\n\n",
          "Plugin output :\n",
          "\n",
          "According to its SMTP banner, version ", ver, " of MailMarshal is\n",
          "installed on the remote host and the affected component is accessible\n",
          "under the directory '", dir, "'."
        );
        break;
      }
    }
  }

  if (report) security_hole(port:port, data:report);
}
