/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller;

import net.sourceforge.rssowl.model.Category;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.document.DocumentGenerator;
import net.sourceforge.rssowl.util.i18n.ITranslatable;
import net.sourceforge.rssowl.util.shop.HotkeyShop;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.URLShop;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MenuAdapter;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;

/**
 * This is the Menu that is displayed on top of the application RSSOwl
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class RSSOwlMenu implements ITranslatable {

  /** Toplevel menu structure used to set hotkeys */
  public static final String[] menuStructure = new String[] { "MENU_FILE", "MENU_EDIT", "MENU_WINDOW", "MENU_GOTO", "TABLE_HEADER_FEED", "HEADER_RSS_FAVORITES", "MENU_TOOLS" };

  /** Sublevel menu structure that may be assigned with hotkeys */
  public static String[][] subMenuStructure = new String[][] { { "MENU_NEW_FAVORITE", "BUTTON_OPEN", "MENU_CLOSE", "MENU_CLOSE_ALL", "MENU_SAVE", "MENU_GENERATE_PDF", "MENU_GENERATE_RTF", "MENU_GENERATE_HTML", "TOOLTIP_PRINT", "MENU_WORK_OFFLINE", "MENU_EXIT" }, { "MENU_EDIT_CUT", "MENU_EDIT_COPY", "MENU_EDIT_PASTE", "MENU_EDIT_SELECT_ALL", "POP_PROPERTIES" }, { "MENU_BROWSER_FOR_NEWSTEXT", "MENU_TOOLBAR", "MENU_QUICKVIEW", "HEADER_RSS_FAVORITES", "MENU_DISPLAY_FEED_PDF", "MENU_DISPLAY_FEED_RTF", "MENU_DISPLAY_FEED_HTML", "MENU_MINIMIZE_RSSOWL" }, { "MENU_PREVIOUS_TAB", "MENU_NEXT_TAB", "MENU_NEXT_NEWS", "MENU_NEXT_UNREAD_NEWS" }, { "POP_MARK_UNREAD", "POP_MARK_ALL_READ", "MENU_RELOAD", "BUTTON_SEARCH", "POP_MAIL_LINK", "POP_COPY_NEWS_URL", "POP_BLOG_NEWS" },
      { "BUTTON_ADDTO_FAVORITS", "BUTTON_AGGREGATE_ALL", "BUTTON_RELOAD_ALL", "BUTTON_SEARCH_ALL", "BUTTON_MARK_ALL_READ" }, { "MENU_FEEDSEARCH", "MENU_FEED_DISCOVERY", "MENU_VALIDATE", "MENU_PREFERENCES" } };

  private MenuItem about;
  private MenuItem addToFavorites;
  private MenuItem aggregateAll;
  private MenuItem blogNews;
  private MenuItem cascadeItemCreateOPML;
  private MenuItem cascadeItemImportBlogroll;
  private MenuItem cascadeItemImportOPML;
  private MenuItem closeAllTabs;
  private MenuItem closeCurrentTab;
  private MenuItem copy;
  private MenuItem copyUrlItem;
  private MenuItem customizeToolBar;
  private MenuItem cut;
  private MenuItem delete;
  private MenuItem donate;
  private MenuItem edit;
  private MenuItem exitItem;
  private MenuItem export;
  private MenuItem exportHtml;
  private MenuItem exportPdf;
  private MenuItem exportRtf;
  private MenuItem faq;
  private MenuItem favorites;
  private MenuItem feedDiscovery;
  private MenuItem file;
  private MenuItem find;
  private MenuItem gotoNews;
  private Menu gotoNewsMenu;
  private MenuItem gotoNextNews;
  private MenuItem gotoNextTab;
  private MenuItem gotoNextUnreadNews;
  private MenuItem gotoPreviousTab;
  private MenuItem homepage;
  private MenuItem importMenu;
  private MenuItem info;
  private Menu infomenu;
  private boolean isMac;
  private MenuItem license;
  private MenuItem mailingList;
  private MenuItem mailLinkToFriend;
  private MenuItem markAllRead;
  private MenuItem markAllReadItem;
  private MenuItem markUnreadItem;
  private Menu menu;
  private MenuItem minimizeWindow;
  private MenuItem newCategory;
  private MenuItem newFavorite;
  private MenuItem newItem;
  private MenuItem newsheader;
  private Menu newsheaderMenu;
  private MenuItem newSubCategory;
  private MenuItem open;
  private MenuItem paste;
  private MenuItem preferences;
  private MenuItem printNews;
  private MenuItem properties;
  private MenuItem readNewsInHTML;
  private MenuItem readNewsInPDF;
  private MenuItem readNewsInRTF;
  private MenuItem reload;
  private MenuItem reloadAll;
  private MenuItem rssSearch;
  private MenuItem saveNewsFeed;
  private MenuItem searchAll;
  private MenuItem selectAll;
  Shell shell;
  private MenuItem tellMyFriends;
  private MenuItem tools;
  private Menu toolsMenu;
  private MenuItem tutorial;
  private MenuItem update;
  private MenuItem validateFeed;
  private MenuItem welcome;
  private MenuItem window;
  private Menu windowmenu;
  EventManager eventManager;
  MenuItem favoritesTreeWindow;
  Menu filemenu;
  MenuItem quickviewWindow;
  GUI rssOwlGui;
  MenuItem toolBarWindow;
  MenuItem useBrowserForNewsText;
  MenuItem workOffline;

  /**
   * Instantiate a new RSSOwl Menu.
   * 
   * @param rssOwlGui Access some methods of the main controller
   * @param shell The shell
   * @param eventManager The manager handling user interaction
   */
  public RSSOwlMenu(GUI rssOwlGui, Shell shell, EventManager eventManager) {
    this.rssOwlGui = rssOwlGui;
    this.shell = shell;
    this.eventManager = eventManager;
    isMac = GlobalSettings.isMac();
    initComponents();
  }

  /** Init the mnemonics */
  public void initMnemonics() {
    MenuManager.initMnemonics(menu);
  }

  /**
   * Set the Favorites Tree Menuitem to checked or not
   * 
   * @param enabled TRUE sets item unchecked
   */
  public void setFavoritesTreeMenuEnabled(boolean enabled) {
    if (WidgetShop.isset(favoritesTreeWindow))
      favoritesTreeWindow.setSelection(enabled);
  }

  /**
   * Set the Quickview Menuitem to checked or not
   * 
   * @param enabled TRUE sets item unchecked
   */
  public void setQuickviewMenuEnabled(boolean enabled) {
    if (WidgetShop.isset(quickviewWindow))
      quickviewWindow.setSelection(enabled);
  }

  /**
   * Set the ToolBar Menuitem to checked or not
   * 
   * @param enabled TRUE sets item unchecked
   */
  public void setToolBarMenuEnabled(boolean enabled) {
    if (WidgetShop.isset(toolBarWindow))
      toolBarWindow.setSelection(enabled);
  }

  /**
   * Set the option "useBrowserForNewsText" selected or not
   * 
   * @param enabled TRUE if selected
   */
  public void setUseBrowserForNewsText(boolean enabled) {
    useBrowserForNewsText.setSelection(enabled);
  }

  /**
   * Update the accelerators on the selected menuitem
   * 
   * @param menuItem Selected menuitem
   * @param type Translation key of the selected menuitem
   * @param points TRUE if "..." should be added to the item
   * @param removePrintable If TRUE, remove all accelerators that are printable.
   * This is required in case the user has set focus into an editable field.
   */
  public void updateAccelerator(MenuItem menuItem, String type, boolean points, boolean removePrintable) {
    updateAccelerator(menuItem, type, type, points, removePrintable);
  }

  /** Update the accelerators on the menuitems */
  public void updateAccelerators() {
    updateAccelerators(false);
  }

  /**
   * Update the accelerators on the menuitems
   * 
   * @param removePrintable If TRUE, remove all accelerators that are printable.
   * This is required in case the user has set focus into an editable field.
   */
  public void updateAccelerators(boolean removePrintable) {

    /** Menu: File */
    updateAccelerator(newFavorite, "LABEL_FAVORITE", "MENU_NEW_FAVORITE", true, removePrintable);
    updateAccelerator(open, "BUTTON_OPEN", true, removePrintable);
    updateAccelerator(closeCurrentTab, "MENU_CLOSE", false, removePrintable);
    updateAccelerator(closeAllTabs, "MENU_CLOSE_ALL", false, removePrintable);
    updateAccelerator(saveNewsFeed, "MENU_SAVE", true, removePrintable);

    if (GlobalSettings.useIText()) {
      updateAccelerator(exportPdf, "MENU_GENERATE_PDF", true, removePrintable);
      updateAccelerator(exportRtf, "MENU_GENERATE_RTF", true, removePrintable);
    }
    updateAccelerator(exportHtml, "MENU_GENERATE_HTML", true, removePrintable);

    if (GlobalSettings.usePrinting())
      updateAccelerator(printNews, "TOOLTIP_PRINT", true, removePrintable);

    updateAccelerator(workOffline, "MENU_WORK_OFFLINE", false, removePrintable);

    /** Exit item on Mac placed in Application Menu and not File Menu */
    if (!GlobalSettings.isMac())
      updateAccelerator(exitItem, "MENU_EXIT", false, removePrintable);

    /** Menu: Edit */
    updateAccelerator(cut, "MENU_EDIT_CUT", false, removePrintable);
    updateAccelerator(paste, "MENU_EDIT_PASTE", false, removePrintable);
    updateAccelerator(selectAll, "MENU_EDIT_SELECT_ALL", false, removePrintable);
    updateAccelerator(copy, "MENU_EDIT_COPY", false, removePrintable);
    updateAccelerator(properties, "POP_PROPERTIES", false, removePrintable);

    /** Menu: Newsheader */
    updateAccelerator(find, "BUTTON_SEARCH", true, removePrintable);
    updateAccelerator(reload, "MENU_RELOAD", false, removePrintable);
    updateAccelerator(addToFavorites, "BUTTON_ADDTO_FAVORITS", true, removePrintable);
    updateAccelerator(mailLinkToFriend, "POP_MAIL_LINK", false, removePrintable);
    updateAccelerator(blogNews, "POP_BLOG_NEWS", false, removePrintable);
    updateAccelerator(copyUrlItem, "POP_COPY_NEWS_URL", false, removePrintable);
    updateAccelerator(markUnreadItem, "POP_MARK_UNREAD", false, removePrintable);
    updateAccelerator(markAllReadItem, "POP_MARK_ALL_READ", false, removePrintable);

    /** Menu: Favorites */
    updateAccelerator(markAllRead, "BUTTON_MARK_ALL_READ", false, removePrintable);
    updateAccelerator(reloadAll, "BUTTON_RELOAD_ALL", false, removePrintable);
    updateAccelerator(aggregateAll, "BUTTON_AGGREGATE_ALL", false, removePrintable);
    updateAccelerator(searchAll, "BUTTON_SEARCH_ALL", false, removePrintable);

    /** Menu: View */
    if (GlobalSettings.useInternalBrowser())
      updateAccelerator(useBrowserForNewsText, "MENU_BROWSER_FOR_NEWSTEXT", false, removePrintable);
    updateAccelerator(minimizeWindow, "MENU_MINIMIZE_RSSOWL", false, removePrintable);
    updateAccelerator(toolBarWindow, "MENU_TOOLBAR", false, removePrintable);
    updateAccelerator(quickviewWindow, "MENU_QUICKVIEW", false, removePrintable);
    updateAccelerator(favoritesTreeWindow, "HEADER_RSS_FAVORITES", false, removePrintable);

    if (GlobalSettings.useIText()) {
      updateAccelerator(readNewsInPDF, "MENU_DISPLAY_FEED_PDF", false, removePrintable);
      updateAccelerator(readNewsInRTF, "MENU_DISPLAY_FEED_RTF", false, removePrintable);
    }
    updateAccelerator(readNewsInHTML, "MENU_DISPLAY_FEED_HTML", false, removePrintable);

    /** Menu: Goto */
    updateAccelerator(gotoPreviousTab, "MENU_PREVIOUS_TAB", false, removePrintable);
    updateAccelerator(gotoNextTab, "MENU_NEXT_TAB", false, removePrintable);
    updateAccelerator(gotoNextNews, "MENU_NEXT_NEWS", false, removePrintable);
    updateAccelerator(gotoNextUnreadNews, "MENU_NEXT_UNREAD_NEWS", false, removePrintable);

    /** Menu: Tools */
    updateAccelerator(rssSearch, "MENU_FEEDSEARCH", true, removePrintable);
    updateAccelerator(feedDiscovery, "MENU_FEED_DISCOVERY", true, removePrintable);
    updateAccelerator(validateFeed, "MENU_VALIDATE", true, removePrintable);
    updateAccelerator(preferences, "MENU_PREFERENCES", true, removePrintable);
  }

  /** Update all controlls text with i18n */
  public void updateI18N() {
    file.setText(GUI.i18n.getTranslation("MENU_FILE"));
    newItem.setText(GUI.i18n.getTranslation("POP_NEW"));
    newCategory.setText(GUI.i18n.getTranslation("LABEL_CATEGORY") + "...");
    newSubCategory.setText(GUI.i18n.getTranslation("POP_SUB_CATEGORY") + "...");
    importMenu.setText(GUI.i18n.getTranslation("POP_IMPORT"));
    cascadeItemImportOPML.setText(GUI.i18n.getTranslation("POP_FROM_OPML") + "...");
    cascadeItemImportBlogroll.setText(GUI.i18n.getTranslation("POP_IMPORT_BLOGROLL") + "...");
    cascadeItemCreateOPML.setText(GUI.i18n.getTranslation("POP_EXPORT_OPML") + "...");
    export.setText(GUI.i18n.getTranslation("BUTTON_EXPORT"));
    edit.setText(GUI.i18n.getTranslation("MENU_EDIT"));
    delete.setText(GUI.i18n.getTranslation("MENU_EDIT_DELETE") + "\t" + GUI.i18n.getTranslation("LABEL_KEY_DEL"));
    window.setText(GUI.i18n.getTranslation("MENU_WINDOW"));
    customizeToolBar.setText(GUI.i18n.getTranslation("POP_CUSTOMIZE_TOOLBAR") + "...");
    preferences.setText(GUI.i18n.getTranslation("MENU_PREFERENCES") + "...");
    mailingList.setText(GUI.i18n.getTranslation("MENU_MAILING_LIST"));
    welcome.setText(GUI.i18n.getTranslation("MENU_WELCOME"));
    tutorial.setText(GUI.i18n.getTranslation("MENU_TUTORIAL"));
    license.setText(GUI.i18n.getTranslation("MENU_LICENSE"));
    update.setText(GUI.i18n.getTranslation("MENU_UPDATE"));
    info.setText(GUI.i18n.getTranslation("MENU_INFO"));
    about.setText(GUI.i18n.getTranslation("MENU_ABOUT"));
    donate.setText(GUI.i18n.getTranslation("MENU_DONATE"));
    tools.setText(GUI.i18n.getTranslation("MENU_TOOLS"));
    gotoNews.setText(GUI.i18n.getTranslation("MENU_GOTO"));
    newsheader.setText(GUI.i18n.getTranslation("TABLE_HEADER_FEED"));
    homepage.setText(GUI.i18n.getTranslation("CHANNEL_INFO_HOMEPAGE"));
    tellMyFriends.setText(GUI.i18n.getTranslation("MENU_TELL_FRIENDS"));
    favorites.setText(GUI.i18n.getTranslation("HEADER_RSS_FAVORITES"));

    /** Update accelerators */
    updateAccelerators();

    /** Update the mnemonics */
    MenuManager.initMnemonics(menu.getItems());
  }

  /**
   * Update the state of the "Work Offline" menuitem, either checked or not.
   */
  public void updateOfflineModeItem() {
    if (GlobalSettings.isMac())
      workOffline.setSelection(GlobalSettings.workOffline);
    else
      workOffline.setImage(GlobalSettings.workOffline ? PaintShop.iconChecked : null);

    /** Update Shell's Title */
    shell.setText(WidgetShop.getShellTitle());
  }

  /** Init all components */
  private void initComponents() {

    /** Main Menu of RSSOwl */
    menu = new Menu(shell, SWT.BAR);
    shell.setMenuBar(menu);

    /**
     * File - Menu
     */
    file = new MenuItem(menu, SWT.CASCADE);
    file.setText(GUI.i18n.getTranslation("MENU_FILE"));
    filemenu = new Menu(shell, SWT.DROP_DOWN);
    file.setMenu(filemenu);

    /** Drop Down: New */
    newItem = new MenuItem(filemenu, SWT.CASCADE);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      newItem.setImage(PaintShop.iconNew);

    /** Menu containing: New Favorite, New Category and New Sub Category */
    Menu newMenu = new Menu(shell, SWT.DROP_DOWN);
    newItem.setMenu(newMenu);
    newItem.setText(GUI.i18n.getTranslation("POP_NEW"));

    /** Create a new Favorite */
    newFavorite = new MenuItem(newMenu, SWT.PUSH);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      newFavorite.setImage(PaintShop.iconAddToFavorites);

    /** Create a new favorite on selection */
    newFavorite.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionNewFavorite();
      }
    });

    /** Separator */
    new MenuItem(newMenu, SWT.SEPARATOR);

    /** Create new Category */
    newCategory = new MenuItem(newMenu, SWT.PUSH);
    newCategory.setText(GUI.i18n.getTranslation("LABEL_CATEGORY") + "...");

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      newCategory.setImage(PaintShop.iconAddCategory);

    /** Create a new category on selection */
    newCategory.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionNewCategory();
      }
    });

    /** Create new Sub-Category */
    newSubCategory = new MenuItem(newMenu, SWT.PUSH);
    newSubCategory.setText(GUI.i18n.getTranslation("POP_SUB_CATEGORY") + "...");

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      newSubCategory.setImage(PaintShop.iconSubCategory);

    /** Create Sub-Category on selection */
    newSubCategory.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionNewCategory(true);
      }
    });

    /** Register New-Items to MenuManager */
    MenuManager.registerNewItems(newCategory, newSubCategory);

    /** Open a newsfeed from a local file */
    open = new MenuItem(filemenu, SWT.PUSH);
    open.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionOpenFeed();
      }
    });

    /** Separator */
    new MenuItem(filemenu, SWT.SEPARATOR);

    /** Close the current displayed tab */
    closeCurrentTab = new MenuItem(filemenu, SWT.PUSH);
    closeCurrentTab.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionCloseCurrent();
      }
    });

    /** Close all displayed tabs */
    closeAllTabs = new MenuItem(filemenu, SWT.PUSH);
    closeAllTabs.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionCloseAll();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerCloseTab(closeCurrentTab, closeAllTabs);

    /** Separator */
    new MenuItem(filemenu, SWT.SEPARATOR);

    /** Save feed to local file */
    saveNewsFeed = new MenuItem(filemenu, SWT.PUSH);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac()) {
      saveNewsFeed.setImage(PaintShop.loadImage("/img/icons/save_as.gif"));
      saveNewsFeed.addDisposeListener(DisposeListenerImpl.getInstance());
    }

    saveNewsFeed.addListener(SWT.Selection, new Listener() {
      public void handleEvent(Event e) {
        eventManager.actionSaveFeed();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerSaveRSS(saveNewsFeed);

    /** In case iText is allowed */
    if (GlobalSettings.useIText()) {

      /** Separator */
      new MenuItem(filemenu, SWT.SEPARATOR);

      /** Export displayed newsfeed to PDF */
      exportPdf = new MenuItem(filemenu, SWT.PUSH);
      exportPdf.setText(GUI.i18n.getTranslation("MENU_GENERATE_PDF"));

      /** On Mac, icons in the menu are not displayed */
      if (!GlobalSettings.isMac())
        exportPdf.setImage(PaintShop.iconPDF);

      exportPdf.addSelectionListener(new SelectionAdapter() {
        public void widgetSelected(SelectionEvent e) {
          eventManager.actionExportFeed(DocumentGenerator.PDF_FORMAT);
        }
      });

      /** Export displayed newsfeed to RTF */
      exportRtf = new MenuItem(filemenu, SWT.PUSH);
      exportRtf.setText(GUI.i18n.getTranslation("MENU_GENERATE_RTF"));

      /** On Mac, icons in the menu are not displayed */
      if (!GlobalSettings.isMac())
        exportRtf.setImage(PaintShop.iconRTF);

      exportRtf.addSelectionListener(new SelectionAdapter() {
        public void widgetSelected(SelectionEvent e) {
          eventManager.actionExportFeed(DocumentGenerator.RTF_FORMAT);
        }
      });
    }

    /** Export displayed newsfeed to HTML */
    exportHtml = new MenuItem(filemenu, SWT.PUSH);
    exportHtml.setText(GUI.i18n.getTranslation("MENU_GENERATE_HTML"));

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      exportHtml.setImage(PaintShop.iconHTML);

    exportHtml.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionExportFeed(DocumentGenerator.HTML_FORMAT);
      }
    });

    /** Register to MenuManager */
    MenuManager.registerExport(exportPdf, exportRtf, exportHtml);

    /** Separator */
    new MenuItem(filemenu, SWT.SEPARATOR);

    /** Printing is only supported on Windows and Mac */
    if (GlobalSettings.usePrinting()) {
      printNews = new MenuItem(filemenu, SWT.PUSH);

      /** On Mac, icons in the menu are not displayed */
      if (!GlobalSettings.isMac())
        printNews.setImage(PaintShop.iconPrint);

      /** Print newstext */
      printNews.addListener(SWT.Selection, new Listener() {
        public void handleEvent(Event e) {
          eventManager.actionPrintNews();
        }
      });

      /** Register to MenuManager */
      MenuManager.regiserPrint(printNews);

      /** Separator */
      new MenuItem(filemenu, SWT.SEPARATOR);
    }

    /** Import OPML files */
    importMenu = new MenuItem(filemenu, SWT.CASCADE);
    importMenu.setText(GUI.i18n.getTranslation("POP_IMPORT"));
    if (!GlobalSettings.isMac())
      importMenu.setImage(PaintShop.iconImport);

    Menu selectImport = new Menu(shell, SWT.DROP_DOWN);
    importMenu.setMenu(selectImport);

    /** Import OPML */
    cascadeItemImportOPML = new MenuItem(selectImport, SWT.PUSH);
    cascadeItemImportOPML.setText(GUI.i18n.getTranslation("POP_FROM_OPML") + "...");
    if (!GlobalSettings.isMac())
      cascadeItemImportOPML.setImage(PaintShop.iconImportOpml);
    cascadeItemImportOPML.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionImportOPML();
      }
    });

    /** Import Synchronized Blogroll */
    cascadeItemImportBlogroll = new MenuItem(selectImport, SWT.PUSH);
    cascadeItemImportBlogroll.setText(GUI.i18n.getTranslation("POP_IMPORT_BLOGROLL") + "...");
    if (!GlobalSettings.isMac())
      cascadeItemImportBlogroll.setImage(PaintShop.iconAddBlogroll);
    cascadeItemImportBlogroll.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionImportBlogroll();
      }
    });

    /** Export to OPML */
    export = new MenuItem(filemenu, SWT.CASCADE);
    export.setText(GUI.i18n.getTranslation("BUTTON_EXPORT"));
    if (!GlobalSettings.isMac())
      export.setImage(PaintShop.iconExport);

    /** Register to MenuManager */
    MenuManager.registerExportMenuItem(export);

    Menu selectexport = new Menu(shell, SWT.DROP_DOWN);
    export.setMenu(selectexport);

    /** Create an OPML file from all favorites */
    cascadeItemCreateOPML = new MenuItem(selectexport, SWT.PUSH);
    cascadeItemCreateOPML.setText(GUI.i18n.getTranslation("POP_EXPORT_OPML") + "...");
    if (!GlobalSettings.isMac())
      cascadeItemCreateOPML.setImage(PaintShop.iconExportOpml);
    cascadeItemCreateOPML.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionCategoryToOPML(Category.getRootCategory(), "rssowl.opml");
      }
    });

    /** Separator */
    new MenuItem(filemenu, SWT.SEPARATOR);

    /** Offline Mode */
    workOffline = new MenuItem(filemenu, GlobalSettings.isMac() ? SWT.CHECK : SWT.PUSH);
    updateOfflineModeItem();
    workOffline.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        GlobalSettings.workOffline = !GlobalSettings.workOffline;

        /** Update the Icon if not on Mac */
        if (!GlobalSettings.isMac())
          workOffline.setImage(GlobalSettings.workOffline ? PaintShop.iconChecked : null);

        /** Update Shell's Title */
        shell.setText(WidgetShop.getShellTitle());
      }
    });

    /** Exit item on Mac placed in Application Menu and not File Menu */
    if (!GlobalSettings.isMac()) {

      /** Separator */
      new MenuItem(filemenu, SWT.SEPARATOR);

      /** Exit RSSOwl */
      exitItem = new MenuItem(filemenu, SWT.PUSH);
      exitItem.addListener(SWT.Selection, new Listener() {
        public void handleEvent(Event e) {
          eventManager.actionExit();
        }
      });
    }

    /**
     * Edit Menu
     */
    edit = new MenuItem(menu, SWT.CASCADE);
    edit.setText(GUI.i18n.getTranslation("MENU_EDIT"));
    Menu editMenu = new Menu(shell, SWT.DROP_DOWN);
    edit.setMenu(editMenu);
    editMenu.addMenuListener(new MenuAdapter() {
      public void menuShown(MenuEvent e) {
        MenuManager.handleEditMenuState();
      }
    });

    /** Cut */
    cut = new MenuItem(editMenu, SWT.PUSH);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac()) {
      cut.setImage(PaintShop.loadImage("/img/icons/cut.gif"));
      cut.addDisposeListener(DisposeListenerImpl.getInstance());
    }

    cut.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.handleEditAction(EventManager.EDIT_ACTION_CUT);
      }
    });

    /** Copy */
    copy = new MenuItem(editMenu, SWT.PUSH);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      copy.setImage(PaintShop.iconCopy);

    copy.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.handleEditAction(EventManager.EDIT_ACTION_COPY);
      }
    });

    /** Paste */
    paste = new MenuItem(editMenu, SWT.PUSH);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac()) {
      paste.setImage(PaintShop.loadImage("/img/icons/paste.gif"));
      paste.addDisposeListener(DisposeListenerImpl.getInstance());
    }

    paste.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.handleEditAction(EventManager.EDIT_ACTION_PASTE);
      }
    });

    /** Separator */
    new MenuItem(editMenu, SWT.SEPARATOR);

    /** Delete */
    delete = new MenuItem(editMenu, SWT.PUSH);
    delete.setText(GUI.i18n.getTranslation("MENU_EDIT_DELETE") + "\t" + GUI.i18n.getTranslation("LABEL_KEY_DEL"));

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      delete.setImage(PaintShop.iconDelete);

    delete.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.handleEditAction(EventManager.EDIT_ACTION_DELETE);
      }
    });

    /** Select All */
    selectAll = new MenuItem(editMenu, SWT.PUSH);
    selectAll.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.handleEditAction(EventManager.EDIT_ACTION_SELECTALL);
      }
    });

    /** Separator */
    new MenuItem(editMenu, SWT.SEPARATOR);

    /** Properties */
    properties = new MenuItem(editMenu, SWT.PUSH);
    properties.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.handleEditAction(EventManager.EDIT_ACTION_PROPERTIES);
      }
    });

    /** Register to MenuManager */
    MenuManager.registerEditMenu(copy, paste, cut, selectAll, delete, properties);

    /**
     * View - Menu
     */
    window = new MenuItem(menu, SWT.CASCADE);
    window.setText(GUI.i18n.getTranslation("MENU_WINDOW"));
    windowmenu = new Menu(shell, SWT.DROP_DOWN);
    window.setMenu(windowmenu);

    /** Display newstext in internal browser if possible */
    if (GlobalSettings.useInternalBrowser()) {
      useBrowserForNewsText = new MenuItem(windowmenu, SWT.CHECK);
      useBrowserForNewsText.addSelectionListener(new SelectionAdapter() {
        public void widgetSelected(SelectionEvent e) {
          eventManager.actionSetBrowserView(useBrowserForNewsText.getSelection());
        }
      });

      /** Separator */
      new MenuItem(windowmenu, SWT.SEPARATOR);
    }

    /** Show / Hide Toolbar control */
    toolBarWindow = new MenuItem(windowmenu, SWT.CHECK);
    toolBarWindow.addListener(SWT.Selection, new Listener() {
      public void handleEvent(Event e) {
        eventManager.actionShowToolBar(toolBarWindow.getSelection());
      }
    });

    /** Show / Hide Quickview control */
    quickviewWindow = new MenuItem(windowmenu, SWT.CHECK);
    quickviewWindow.addListener(SWT.Selection, new Listener() {
      public void handleEvent(Event e) {
        eventManager.actionShowQuickview(quickviewWindow.getSelection());
      }
    });

    /** Show / Hide Favorites Tree panel */
    favoritesTreeWindow = new MenuItem(windowmenu, SWT.CHECK);
    favoritesTreeWindow.addListener(SWT.Selection, new Listener() {
      public void handleEvent(Event event) {
        rssOwlGui.setFavoritesMinimized(!favoritesTreeWindow.getSelection(), true);
      }
    });

    /** Separator */
    new MenuItem(windowmenu, SWT.SEPARATOR);

    /** Customize ToolBar */
    customizeToolBar = new MenuItem(windowmenu, SWT.NONE);
    customizeToolBar.setText(GUI.i18n.getTranslation("POP_CUSTOMIZE_TOOLBAR") + "...");
    customizeToolBar.addListener(SWT.Selection, new Listener() {
      public void handleEvent(Event event) {
        eventManager.actionCustomizeToolBar();
      }
    });

    /** Separator */
    new MenuItem(windowmenu, SWT.SEPARATOR);

    /** In case iText is used */
    if (GlobalSettings.useIText()) {

      /** View displayed feed as PDF */
      readNewsInPDF = new MenuItem(windowmenu, SWT.NONE);
      readNewsInPDF.addListener(SWT.Selection, new Listener() {
        public void handleEvent(Event event) {
          eventManager.actionViewFeedInDocument(DocumentGenerator.PDF_FORMAT);
        }
      });

      /** View displayed feed as RTF */
      readNewsInRTF = new MenuItem(windowmenu, SWT.NONE);
      readNewsInRTF.addListener(SWT.Selection, new Listener() {
        public void handleEvent(Event event) {
          eventManager.actionViewFeedInDocument(DocumentGenerator.RTF_FORMAT);
        }
      });
    }

    /** View displayed feed as HTML */
    readNewsInHTML = new MenuItem(windowmenu, SWT.NONE);
    readNewsInHTML.addListener(SWT.Selection, new Listener() {
      public void handleEvent(Event event) {
        eventManager.actionViewFeedInDocument(DocumentGenerator.HTML_FORMAT);
      }
    });

    /** Register to Menu Manager */
    MenuManager.registerReadNewsInDocument(readNewsInPDF, readNewsInRTF, readNewsInHTML);

    /** Separator */
    new MenuItem(windowmenu, SWT.SEPARATOR);

    /** Minimize application window */
    minimizeWindow = new MenuItem(windowmenu, SWT.NONE);
    minimizeWindow.addListener(SWT.Selection, new Listener() {
      public void handleEvent(Event e) {
        eventManager.actionMinimizeWindow();
      }
    });

    /**
     * Goto - Menu
     */
    gotoNews = new MenuItem(menu, SWT.CASCADE);
    gotoNews.setText(GUI.i18n.getTranslation("MENU_GOTO"));
    gotoNewsMenu = new Menu(shell, SWT.DROP_DOWN);
    gotoNews.setMenu(gotoNewsMenu);

    /** Goto the previous tab if possible */
    gotoPreviousTab = new MenuItem(gotoNewsMenu, SWT.NONE);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      gotoPreviousTab.setImage(PaintShop.iconBackward);

    gotoPreviousTab.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionGotoPreviousTab();
      }
    });

    /** Goto the next tab if possible */
    gotoNextTab = new MenuItem(gotoNewsMenu, SWT.NONE);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      gotoNextTab.setImage(PaintShop.iconForward);

    gotoNextTab.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionGotoNextTab();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerGotoTab(gotoPreviousTab, gotoNextTab);

    /** Separator */
    new MenuItem(gotoNewsMenu, SWT.SEPARATOR);

    /** Goto the next news if possible */
    gotoNextNews = new MenuItem(gotoNewsMenu, SWT.NONE);
    gotoNextNews.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionGotoNextNews();
      }
    });

    /** Goto the next unread news if possible */
    gotoNextUnreadNews = new MenuItem(gotoNewsMenu, SWT.NONE);
    gotoNextUnreadNews.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionGotoNextUnreadNews();
      }
    });

    /**
     * Newsheader - Menu
     */
    newsheader = new MenuItem(menu, SWT.CASCADE);
    newsheader.setText(GUI.i18n.getTranslation("TABLE_HEADER_FEED"));
    newsheaderMenu = new Menu(shell, SWT.DROP_DOWN);
    newsheader.setMenu(newsheaderMenu);

    /** Mark selected news unread */
    markUnreadItem = new MenuItem(newsheaderMenu, SWT.NONE);
    markUnreadItem.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionMarkNewsUnread();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerMarkUnreadItem(markUnreadItem);

    /** Mark all news read */
    markAllReadItem = new MenuItem(newsheaderMenu, SWT.NONE);
    markAllReadItem.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionMarkAllNewsRead();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerMarkAllReadItem(markAllReadItem);

    /** Separator */
    new MenuItem(newsheaderMenu, SWT.SEPARATOR);

    /** Reload displayed newstab */
    reload = new MenuItem(newsheaderMenu, SWT.PUSH);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      reload.setImage(PaintShop.iconReload);

    reload.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionReload();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerReload(reload);

    /** Separator */
    new MenuItem(newsheaderMenu, SWT.SEPARATOR);

    /** Search in newsfeed */
    find = new MenuItem(newsheaderMenu, SWT.PUSH);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      find.setImage(PaintShop.iconFind);

    find.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionSearch();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerSearch(find);

    /** Separator */
    new MenuItem(newsheaderMenu, SWT.SEPARATOR);

    /** Mail Link of selected news to friend */
    mailLinkToFriend = new MenuItem(newsheaderMenu, SWT.NONE);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      mailLinkToFriend.setImage(PaintShop.iconMail);

    mailLinkToFriend.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionMailNewsTip();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerNewsTipMail(mailLinkToFriend);

    /** Copy selected news URL */
    copyUrlItem = new MenuItem(newsheaderMenu, SWT.NONE);
    copyUrlItem.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionCopyNewsUrl();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerCopyLink(copyUrlItem);

    /** Separator */
    new MenuItem(newsheaderMenu, SWT.SEPARATOR);

    blogNews = new MenuItem(newsheaderMenu, SWT.NONE);
    blogNews.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionBlogNews();
      }
    });

    /** Register to MenuManager */
    MenuManager.regiserBlogNews(blogNews);

    /**
     * Favorites - Menu
     */
    favorites = new MenuItem(menu, SWT.CASCADE);
    favorites.setText(GUI.i18n.getTranslation("HEADER_RSS_FAVORITES"));
    Menu favoritesmenu = new Menu(shell, SWT.DROP_DOWN);
    favorites.setMenu(favoritesmenu);

    /** Add newsfeed to favorites */
    addToFavorites = new MenuItem(favoritesmenu, SWT.PUSH);
    addToFavorites.setText(GUI.i18n.getTranslation("BUTTON_ADDTO_FAVORITS"));

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      addToFavorites.setImage(PaintShop.iconAddToFavorites);

    addToFavorites.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionAddToFavorites();
      }
    });

    /** Register to MenuManager */
    MenuManager.registerAddFavoriteMenu(addToFavorites);

    /** Separator */
    new MenuItem(favoritesmenu, SWT.SEPARATOR);

    /** Aggregate all categories */
    aggregateAll = new MenuItem(favoritesmenu, SWT.NONE);
    aggregateAll.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionAggregateAllCategories();
      }
    });

    /** Reload all categories */
    reloadAll = new MenuItem(favoritesmenu, SWT.NONE);
    reloadAll.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionReloadAllCategories();
      }
    });

    /** Search in all categories */
    searchAll = new MenuItem(favoritesmenu, SWT.NONE);
    searchAll.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionSearchInAllCategories();
      }
    });

    /** Separator */
    new MenuItem(favoritesmenu, SWT.SEPARATOR);

    /** Mark all categories read */
    markAllRead = new MenuItem(favoritesmenu, SWT.NONE);
    markAllRead.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionMarkAllCategoriesRead();
      }
    });

    /**
     * Tools - Menu
     */
    tools = new MenuItem(menu, SWT.CASCADE);
    tools.setText(GUI.i18n.getTranslation("MENU_TOOLS"));
    toolsMenu = new Menu(shell, SWT.DROP_DOWN);
    tools.setMenu(toolsMenu);

    /** Search for RSS / RDF / Atom newsfeeds */
    rssSearch = new MenuItem(toolsMenu, SWT.NONE);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      rssSearch.setImage(PaintShop.iconSearch);

    rssSearch.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionSearchFeeds();
      }
    });

    /** Feed Discovery on a website */
    feedDiscovery = new MenuItem(toolsMenu, SWT.NONE);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      feedDiscovery.setImage(PaintShop.iconFeedDiscovery);

    feedDiscovery.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionDiscoverFeeds();
      }
    });

    /** Separator */
    new MenuItem(toolsMenu, SWT.SEPARATOR);

    /** Validate newsfeeds */
    validateFeed = new MenuItem(toolsMenu, SWT.NONE);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      validateFeed.setImage(PaintShop.iconValidate);

    validateFeed.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionValidateFeeds(false);
      }
    });

    /** Separator */
    new MenuItem(toolsMenu, SWT.SEPARATOR);

    /** RSSOwl's preferences */
    preferences = new MenuItem(toolsMenu, SWT.NONE);

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac()) {
      preferences.setImage(PaintShop.loadImage("/img/icons/preferences.gif"));
      preferences.addDisposeListener(DisposeListenerImpl.getInstance());
    }

    preferences.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionOpenPreferences();
      }
    });

    /**
     * Help - Menu
     */
    info = new MenuItem(menu, SWT.CASCADE);
    info.setText(GUI.i18n.getTranslation("MENU_INFO"));
    infomenu = new Menu(shell, SWT.DROP_DOWN);
    info.setMenu(infomenu);

    /** Open RSSOwl's tutorial */
    tutorial = new MenuItem(infomenu, SWT.PUSH);
    tutorial.setText(GUI.i18n.getTranslation("MENU_TUTORIAL"));

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac()) {
      tutorial.setImage(PaintShop.loadImage("/img/icons/tutorial.gif"));
      tutorial.addDisposeListener(DisposeListenerImpl.getInstance());
    }

    tutorial.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionOpenTutorial();
      }
    });

    /** Open RSSOwl's FAQ */
    faq = new MenuItem(infomenu, SWT.PUSH);
    faq.setText("FAQ");

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac()) {
      faq.setImage(PaintShop.loadImage("/img/icons/faq.gif"));
      faq.addDisposeListener(DisposeListenerImpl.getInstance());
    }

    faq.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionOpenFAQ();
      }
    });

    /** Separator */
    new MenuItem(infomenu, SWT.SEPARATOR);

    /** Display welcome tab */
    welcome = new MenuItem(infomenu, SWT.PUSH);
    welcome.setText(GUI.i18n.getTranslation("MENU_WELCOME"));
    welcome.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionShowWelcome();
      }
    });

    /** Show license in a tab */
    license = new MenuItem(infomenu, SWT.PUSH);
    license.setText(GUI.i18n.getTranslation("MENU_LICENSE"));
    license.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionShowLicense();
      }
    });

    /** Separator */
    new MenuItem(infomenu, SWT.SEPARATOR);

    /** Open RSSOwl's homepage */
    homepage = new MenuItem(infomenu, SWT.PUSH);
    homepage.setText(GUI.i18n.getTranslation("CHANNEL_INFO_HOMEPAGE"));
    homepage.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionOpenURL(URLShop.RSSOWL_WEBPAGE);
      }
    });

    /** Open mailinglist */
    mailingList = new MenuItem(infomenu, SWT.PUSH);
    mailingList.setText(GUI.i18n.getTranslation("MENU_MAILING_LIST"));
    mailingList.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionOpenURL(URLShop.RSSOWL_LIST);
      }
    });

    /** Separator */
    new MenuItem(infomenu, SWT.SEPARATOR);

    /** Check for an updated RSSOwl version */
    update = new MenuItem(infomenu, SWT.PUSH);
    update.setText(GUI.i18n.getTranslation("MENU_UPDATE"));

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac()) {
      update.setImage(PaintShop.loadImage("/img/icons/update.gif"));
      update.addDisposeListener(DisposeListenerImpl.getInstance());
    }

    update.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionCheckUpdate();
      }
    });

    /** Separator */
    new MenuItem(infomenu, SWT.SEPARATOR);

    /** Open link to make a donation */
    donate = new MenuItem(infomenu, SWT.PUSH);
    donate.setText(GUI.i18n.getTranslation("MENU_DONATE"));
    donate.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionOpenURL(URLShop.RSSOWL_DONATE);
      }
    });

    /** Separator */
    new MenuItem(infomenu, SWT.SEPARATOR);

    /** Advertise RSSOwl in a mail */
    tellMyFriends = new MenuItem(infomenu, SWT.PUSH);
    tellMyFriends.setText(GUI.i18n.getTranslation("MENU_TELL_FRIENDS"));
    tellMyFriends.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionTellFriends();
      }
    });

    /** Separator */
    new MenuItem(infomenu, SWT.SEPARATOR);

    /** About info for RSSOwl */
    about = new MenuItem(infomenu, SWT.PUSH);
    about.setText(GUI.i18n.getTranslation("MENU_ABOUT"));

    /** On Mac, icons in the menu are not displayed */
    if (!GlobalSettings.isMac())
      about.setImage(PaintShop.iconInfo);

    about.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionOpenAbout();
      }
    });
  }

  /**
   * Update the accelerators on the selected menuitem
   * 
   * @param menuItem Selected menuitem
   * @param text Translation key of the MenuItem's text
   * @param type Translation key of the selected menuitem
   * @param points TRUE if "..." should be added to the item
   * @param removePrintable If TRUE, remove all accelerators that are printable.
   * This is required in case the user has set focus into an editable field.
   */
  private void updateAccelerator(MenuItem menuItem, String text, String type, boolean points, boolean removePrintable) {
    String pointsStr = (points == true) ? "..." : "";

    /** Get Hotkey value */
    int hotKeyValue = HotkeyShop.getHotkeyValue(type);

    /** Check if this hotkey is to be disabled */
    boolean disable = removePrintable && HotkeyShop.isPrintableAccelerator(hotKeyValue);

    /** Assign Hotkey */
    if (hotKeyValue > 0 && !disable) {
      menuItem.setAccelerator(hotKeyValue);

      /**
       * The Mac OS Menu-Manager will add the accelerator as text to the
       * menuitem
       */
      if (isMac)
        menuItem.setText(GUI.i18n.getTranslation(text) + pointsStr);

      /** Explicitly set the accelerator as text to the menuitem */
      else
        menuItem.setText(GUI.i18n.getTranslation(text) + pointsStr + "\t" + HotkeyShop.getHotkeyName(type));
    }

    /** Disable Hotkey */
    else if (disable) {
      menuItem.setAccelerator(HotkeyShop.NULL_ACCELERATOR_VALUE);
    }

    /** Remove the hotkey */
    else {
      menuItem.setAccelerator(HotkeyShop.NULL_ACCELERATOR_VALUE);
      menuItem.setText(GUI.i18n.getTranslation(text) + pointsStr);
    }
  }

  /** Static initializer */
  static {

    /** Sub-Menu Structure for Linux */
    if (GlobalSettings.isLinux())
      subMenuStructure = new String[][] { { "MENU_NEW_FAVORITE", "BUTTON_OPEN", "MENU_CLOSE", "MENU_CLOSE_ALL", "MENU_SAVE", "MENU_GENERATE_PDF", "MENU_GENERATE_RTF", "MENU_GENERATE_HTML", "MENU_WORK_OFFLINE", "MENU_EXIT" }, { "MENU_EDIT_CUT", "MENU_EDIT_COPY", "MENU_EDIT_PASTE", "MENU_EDIT_SELECT_ALL", "POP_PROPERTIES" }, { "MENU_BROWSER_FOR_NEWSTEXT", "MENU_TOOLBAR", "MENU_QUICKVIEW", "HEADER_RSS_FAVORITES", "MENU_DISPLAY_FEED_PDF", "MENU_DISPLAY_FEED_RTF", "MENU_DISPLAY_FEED_HTML", "MENU_MINIMIZE_RSSOWL" }, { "MENU_PREVIOUS_TAB", "MENU_NEXT_TAB", "MENU_NEXT_NEWS", "MENU_NEXT_UNREAD_NEWS" }, { "POP_MARK_UNREAD", "POP_MARK_ALL_READ", "MENU_RELOAD", "BUTTON_SEARCH", "POP_MAIL_LINK", "POP_COPY_NEWS_URL", "POP_BLOG_NEWS" },
          { "BUTTON_ADDTO_FAVORITS", "BUTTON_AGGREGATE_ALL", "BUTTON_RELOAD_ALL", "BUTTON_SEARCH_ALL", "BUTTON_MARK_ALL_READ" }, { "MENU_FEEDSEARCH", "MENU_FEED_DISCOVERY", "MENU_VALIDATE", "MENU_PREFERENCES" } };

    /** Sub-Menu Structure for Mac */
    else if (GlobalSettings.isMac())
      subMenuStructure = new String[][] { { "MENU_NEW_FAVORITE", "BUTTON_OPEN", "MENU_CLOSE", "MENU_CLOSE_ALL", "MENU_SAVE", "MENU_GENERATE_HTML", "TOOLTIP_PRINT", "MENU_WORK_OFFLINE" }, { "MENU_EDIT_CUT", "MENU_EDIT_COPY", "MENU_EDIT_PASTE", "MENU_EDIT_SELECT_ALL", "POP_PROPERTIES" }, { "MENU_BROWSER_FOR_NEWSTEXT", "MENU_TOOLBAR", "MENU_QUICKVIEW", "HEADER_RSS_FAVORITES", "MENU_DISPLAY_FEED_HTML", "MENU_MINIMIZE_RSSOWL" }, { "MENU_PREVIOUS_TAB", "MENU_NEXT_TAB", "MENU_NEXT_NEWS", "MENU_NEXT_UNREAD_NEWS" }, { "POP_MARK_UNREAD", "POP_MARK_ALL_READ", "MENU_RELOAD", "BUTTON_SEARCH", "POP_MAIL_LINK", "POP_COPY_NEWS_URL", "POP_BLOG_NEWS" }, { "BUTTON_ADDTO_FAVORITS", "BUTTON_AGGREGATE_ALL", "BUTTON_RELOAD_ALL", "BUTTON_SEARCH_ALL", "BUTTON_MARK_ALL_READ" },
          { "MENU_FEEDSEARCH", "MENU_FEED_DISCOVERY", "MENU_VALIDATE", "MENU_PREFERENCES" } };

    /** Sub-Menu Structure for Solaris */
    else if (GlobalSettings.isSolaris())
      subMenuStructure = new String[][] { { "MENU_NEW_FAVORITE", "BUTTON_OPEN", "MENU_CLOSE", "MENU_CLOSE_ALL", "MENU_SAVE", "MENU_GENERATE_PDF", "MENU_GENERATE_RTF", "MENU_GENERATE_HTML", "TOOLTIP_PRINT", "MENU_WORK_OFFLINE", "MENU_EXIT" }, { "MENU_EDIT_CUT", "MENU_EDIT_COPY", "MENU_EDIT_PASTE", "MENU_EDIT_SELECT_ALL", "POP_PROPERTIES" }, { "MENU_TOOLBAR", "MENU_QUICKVIEW", "HEADER_RSS_FAVORITES", "MENU_DISPLAY_FEED_PDF", "MENU_DISPLAY_FEED_RTF", "MENU_DISPLAY_FEED_HTML", "MENU_MINIMIZE_RSSOWL" }, { "MENU_PREVIOUS_TAB", "MENU_NEXT_TAB", "MENU_NEXT_NEWS", "MENU_NEXT_UNREAD_NEWS" }, { "POP_MARK_UNREAD", "POP_MARK_ALL_READ", "MENU_RELOAD", "BUTTON_SEARCH", "POP_MAIL_LINK", "POP_COPY_NEWS_URL", "POP_BLOG_NEWS" },
          { "BUTTON_ADDTO_FAVORITS", "BUTTON_AGGREGATE_ALL", "BUTTON_RELOAD_ALL", "BUTTON_SEARCH_ALL", "BUTTON_MARK_ALL_READ" }, { "MENU_FEEDSEARCH", "MENU_FEED_DISCOVERY", "MENU_VALIDATE", "MENU_PREFERENCES" } };
  }
}