/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller;

import net.sourceforge.rssowl.model.TreeItemData;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

/**
 * Helper class for Tree Traversal wrapping a TreeItem.
 * 
 * @author bpasero
 * @version 1.2.3
 */
public class TreeNode {
  private boolean filterRead;
  private boolean isUnread;
  private TreeItem item;

  /**
   * Create a new TreeNode wrapping a TreeItem.
   * 
   * @param item The Item to wrap inside this helper.
   * @param filterRead If TRUE, filter read Items.
   */
  public TreeNode(TreeItem item, boolean filterRead) {
    this.item = item;
    this.filterRead = filterRead;

    if (WidgetShop.isset(item))
      isUnread = ((TreeItemData) item.getData()).isStatusUnread();
  }

  /**
   * Get the first child of this Node or NULL if no childs at all.
   * 
   * @return TreeNode The first child of this Node or NULL if no childs at all.
   */
  public TreeNode getFirstChild() {
    if (WidgetShop.isset(item) && !exclude() && item.getItemCount() > 0)
      return new TreeNode(item.getItems()[0], filterRead);

    return null;
  }

  /**
   * Get the wrapped TreeItem.
   * 
   * @return TreeItem The wrapped TreeItem.
   */
  public TreeItem getItem() {
    return item;
  }

  /**
   * Get the next Sibling of this Node or NULL if no next Sibling at all.
   * 
   * @return TreeNode The next Sibling of this Mode or NULL if no next Sibling
   * at all.
   */
  public TreeNode getNextSibling() {
    if (!WidgetShop.isset(item))
      return null;

    TreeItem parent = item.getParentItem();

    /** Item is not Root-Leveld */
    if (WidgetShop.isset(parent)) {
      int index = parent.indexOf(item);

      if (parent.getItemCount() > index + 1)
        return new TreeNode(parent.getItem(index + 1), filterRead);

      return null;
    }

    /** Item is Root-Leveld */
    Tree tree = item.getParent();
    int index = tree.indexOf(item);

    if (tree.getItemCount() > index + 1)
      return new TreeNode(tree.getItem(index + 1), filterRead);

    return null;
  }

  /**
   * Get the parent Node of this Node or NULL if no parent at all.
   * 
   * @return TreeNode The parent Node of this Node or NULL if no parent at all.
   */
  public TreeNode getParent() {
    if (!WidgetShop.isset(item))
      return null;

    return new TreeNode(item.getParentItem(), filterRead);
  }

  /**
   * Check wether this Node contains any child Nodes.
   * 
   * @return boolean TRUE if this Node contains any child Nodes, FALSE
   * otherwise.
   */
  public boolean hasChildNodes() {
    return WidgetShop.isset(item) && !exclude() && item.getItemCount() > 0;
  }

  /**
   * Check wether this Node is to be exlucded because of applied filter rules.
   * 
   * @return boolean TRUE if this Node is to be exlucded because of applied
   * filter rules. FALSE otherwise.
   */
  private boolean exclude() {
    return filterRead && !isUnread;
  }
}