/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller;

import net.sourceforge.rssowl.controller.panel.BrowserPanel;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.BrowserShop;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.URLShop;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.swt.SWTError;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import java.io.File;

/**
 * This object is used to display RSSOwl's tutorial in a new shell that holds
 * the internal browser widget. If the browser widget is not supported on the
 * used OS, the tutorial is loaded in the external browser.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class TutorialBrowser {
  private Display display;
  private String path;
  private GUI rssOwlGui;
  private Shell shell;

  /**
   * Instantiate a new TutorialBrowser
   * 
   * @param rssOwlGui The maincontroller
   * @param display The display
   * @param shell The shell
   */
  public TutorialBrowser(Display display, Shell shell, GUI rssOwlGui) {
    this.display = display;
    this.shell = shell;
    this.rssOwlGui = rssOwlGui;

    /** Get location of the tutorial HTML */
    path = new File(".").getAbsolutePath() + GlobalSettings.PATH_SEPARATOR + "doc" + GlobalSettings.PATH_SEPARATOR + "tutorial" + GlobalSettings.PATH_SEPARATOR + "en" + GlobalSettings.PATH_SEPARATOR + "index.html";

    /** In case local tutorial is not available, choose online tutorial */
    if (!new File(path).exists())
      path = URLShop.RSSOWL_TUTORIAL;
  }

  /**
   * Show the tutorial in the internal browser on a new Shell.
   */
  private void openTutorialShell() {

    /** Shell holding the browser */
    Shell browserShell = new Shell(display);

    try {

      /** Browser widget */
      BrowserPanel browser = new BrowserPanel(rssOwlGui, browserShell, true);
      browser.openUrl(path);

      /** Open the shell and display the browser */
      browserShell.setText("RSSOwl " + GUI.i18n.getTranslation("MENU_TUTORIAL"));
      browserShell.setLayout(new FillLayout());

      /** On Mac do not set Shell Image since it will change the Dock Image */
      if (!GlobalSettings.isMac())
        browserShell.setImages(PaintShop.iconOwl);

      /** Focus on browser */
      browser.getBrowserPanel().setFocus();

      /** Maximize browser shell */
      browserShell.setMaximized(true);

      /** Open the browser shell */
      browserShell.open();
    }

    /**
     * The internal browser is not yet supported on all operating systems. If it
     * fails, try the def. browser.
     */
    catch (SWTError e) {
      GUI.logger.log("openTutorialShell", e);

      /** Dispose browser shell if opened */
      if (WidgetShop.isset(browserShell))
        browserShell.dispose();

      GlobalSettings.openBrowserExtern = true;

      /** Inform user about error */
      MessageBoxFactory.showError(shell, BrowserShop.createBrowserError(e));

      /** Open in external Browser */
      BrowserShop.openLink(path);
    }
  }

  /**
   * Show the tutorial in the browser
   */
  void show() {

    /** Display tutorial in the internal browser */
    if (GlobalSettings.useInternalBrowser() && !GlobalSettings.openBrowserExtern)
      openTutorialShell();

    /** Display the tutorial in the external browser */
    else
      BrowserShop.openLink(path);
  }
}