/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.dialog;

import net.sourceforge.rssowl.controller.DisposeListenerImpl;
import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.BrowserShop;
import net.sourceforge.rssowl.util.shop.FontShop;
import net.sourceforge.rssowl.util.shop.LayoutDataShop;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.URLShop;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

/**
 * Class displays a Dialog with info about RSSOwl
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class AboutDialog extends Dialog {

  /** Size of the used logo */
  private static final int LOGO_SIZE = 460;

  private Label aboutbanner;
  private String title;

  /**
   * Creates an about dialog with an OK button. Note that the dialog will have
   * no visual representation (no widgets) until it is told to open.
   * <p>
   * Note that the <code>open</code> method blocks for input dialogs.
   * </p>
   * 
   * @param parentShell the parent shell
   * @param dialogTitle the dialog title, or <code>null</code> if none
   */
  public AboutDialog(Shell parentShell, String dialogTitle) {
    super(parentShell);
    this.title = dialogTitle;
  }

  /**
   * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
   */
  protected void configureShell(Shell shell) {
    super.configureShell(shell);

    /** On Mac do not set Shell Image since it will change the Dock Image */
    if (!GlobalSettings.isMac())
      shell.setImages(PaintShop.iconOwl);

    shell.setText(title);
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
   */
  protected void createButtonsForButtonBar(Composite parent) {

    /** Override parent, DLU dependant margins */
    ((GridLayout) parent.getLayout()).marginHeight = 10;
    ((GridLayout) parent.getLayout()).marginWidth = 10;

    /** Create Buttons */
    createButton(parent, IDialogConstants.OK_ID, GUI.i18n.getTranslation("BUTTON_OK"), true).setFont(FontShop.dialogFont);
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
   */
  protected Control createDialogArea(Composite parent) {

    /** Composite to hold all components */
    Composite composite = (Composite) super.createDialogArea(parent);
    composite.setLayout(LayoutShop.createGridLayout(1, 0, 0));
    composite.setBackground(GUI.display.getSystemColor(SWT.COLOR_WHITE));

    /** Group to hold the about-banner */
    Composite bannerHolder = new Composite(composite, SWT.NONE);
    bannerHolder.setLayout(LayoutShop.createFillLayout(0, 0));
    bannerHolder.setBackground(GUI.display.getSystemColor(SWT.COLOR_WHITE));

    /** Banner holder */
    aboutbanner = new Label(bannerHolder, SWT.NONE);
    aboutbanner.setImage(PaintShop.loadImage("/img/aboutbanner.png"));
    aboutbanner.setCursor(GUI.display.getSystemCursor(SWT.CURSOR_HAND));
    aboutbanner.setToolTipText(URLShop.RSSOWL_WEBPAGE);
    aboutbanner.addDisposeListener(DisposeListenerImpl.getInstance());
    aboutbanner.setBackground(GUI.display.getSystemColor(SWT.COLOR_WHITE));

    /** Open project homepage on mouse click */
    aboutbanner.addMouseListener(new MouseAdapter() {
      public void mouseDown(MouseEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_WEBPAGE);
      }
    });

    /** Composite to hold the about text */
    Composite aboutTextHolder = new Composite(composite, SWT.NONE);
    aboutTextHolder.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 1, LOGO_SIZE));
    aboutTextHolder.setLayout(LayoutShop.createFillLayout(10, 10));
    aboutTextHolder.setBackground(GUI.display.getSystemColor(SWT.COLOR_WHITE));

    /** Copyright Label - Selectable and therefor using StyledText */
    StyledText aboutTextLabel = new StyledText(aboutTextHolder, SWT.MULTI | SWT.WRAP | SWT.READ_ONLY);
    aboutTextLabel.setText((char) 169 + " 2003-2006 RSSOwl Development Team. All rights reserved.\n\n" + BrowserShop.getOwlAgent());
    aboutTextLabel.setFont(FontShop.dialogFont);
    aboutTextLabel.setCaret(null);
    aboutTextLabel.setBackground(GUI.display.getSystemColor(SWT.COLOR_WHITE));

    /** Holder for the separator to the OK and Cancel buttons */
    Composite sepHolder = new Composite(parent, SWT.NONE);
    sepHolder.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    sepHolder.setLayout(LayoutShop.createGridLayout(1, 0, 0));
    sepHolder.setBackground(GUI.display.getSystemColor(SWT.COLOR_WHITE));

    /** Separator */
    Label separator = new Label(sepHolder, SWT.SEPARATOR | SWT.HORIZONTAL);
    separator.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

    return composite;
  }

  /**
   * @see org.eclipse.jface.window.Window#getShellStyle()
   */
  protected int getShellStyle() {
    int style = SWT.TITLE | SWT.BORDER | SWT.APPLICATION_MODAL | getDefaultOrientation();

    /** Follow Apple's Human Interface Guidelines for Application Modal Dialogs */
    if (!GlobalSettings.isMac())
      style |= SWT.CLOSE;

    return style;
  }

  /**
   * Set the layout data of the button to a GridData with appropriate widths
   * This method was slightly modified so that it is not setting a heightHint.
   * 
   * @param button The button to layout
   */
  protected void setButtonLayoutData(Button button) {
    GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
    int widthHint = convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH);
    data.widthHint = Math.max(widthHint, button.computeSize(SWT.DEFAULT, SWT.DEFAULT, true).x);
    button.setLayoutData(data);
  }
}