/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.dialog;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.model.Category;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.FontShop;
import net.sourceforge.rssowl.util.shop.LayoutDataShop;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * Class displays a Dialog prompting for a category title.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class CategoryDialog extends TitleAreaDialog {

  /** Min. width of the dialog in DLUs */
  private static final int dialogMinWidth = 320;

  private String catName;
  private Text catTitle;
  private String dialogMessage;
  private boolean isEdit;
  private Button okButton;
  private Category parent;
  private String title;

  /**
   * Creates an input dialog with OK and Cancel buttons. Prompts for a category
   * title. Note that the dialog will have no visual representation (no widgets)
   * until it is told to open.
   * <p>
   * Note that the <code>open</code> method blocks for input dialogs.
   * </p>
   * 
   * @param parentShell the parent shell
   * @param dialogTitle the dialog title, or <code>null</code> if none
   * @param dialogMessage the dialog dialogMessage, or <code>null</code> if
   * none
   * @param parent The parent Category for validation
   */
  public CategoryDialog(Shell parentShell, String dialogTitle, String dialogMessage, Category parent) {
    this(parentShell, dialogTitle, dialogMessage, "", parent);
  }

  /**
   * Creates an input dialog with OK and Cancel buttons. Prompts for a category
   * title. Note that the dialog will have no visual representation (no widgets)
   * until it is told to open.
   * <p>
   * Note that the <code>open</code> method blocks for input dialogs.
   * </p>
   * 
   * @param parentShell the parent shell
   * @param dialogTitle the dialog title, or <code>null</code> if none
   * @param dialogMessage the dialog dialogMessage, or <code>null</code> if
   * none
   * @param catName Preset the Title field
   * @param parent The parent Category for validation
   */
  public CategoryDialog(Shell parentShell, String dialogTitle, String dialogMessage, String catName, Category parent) {
    super(parentShell);
    this.title = dialogTitle;
    this.catName = catName;
    this.dialogMessage = dialogMessage;
    this.parent = parent;

    /** If a catName is given the user is editing the category */
    isEdit = !catName.equals("");
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#close()
   */
  public boolean close() {

    /** Dispose title image */
    getTitleImageLabel().getImage().dispose();

    return super.close();
  }

  /**
   * Returns the string typed into this input dialog.
   * 
   * @return the input string
   */
  public String getCatName() {
    return catName;
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#buttonPressed(int)
   */
  protected void buttonPressed(int buttonId) {
    if (buttonId == IDialogConstants.OK_ID)
      catName = catTitle.getText();
    else
      catName = null;

    super.buttonPressed(buttonId);
  }

  /**
   * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
   */
  protected void configureShell(Shell shell) {
    super.configureShell(shell);

    /** On Mac do not set Shell Image since it will change the Dock Image */
    if (!GlobalSettings.isMac())
      shell.setImages(PaintShop.iconOwl);

    shell.setText(title);
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
   */
  protected void createButtonsForButtonBar(Composite parent) {

    /** Different Button order on Mac */
    if (GUI.display.getDismissalAlignment() == SWT.RIGHT) {

      /** Create Cancel Button */
      createButton(parent, IDialogConstants.CANCEL_ID, GUI.i18n.getTranslation("BUTTON_CANCLE"), false).setFont(FontShop.dialogFont);

      /** Create OK Button */
      okButton = createButton(parent, IDialogConstants.OK_ID, GUI.i18n.getTranslation("BUTTON_OK"), true);
    } else {

      /** Create OK Button */
      okButton = createButton(parent, IDialogConstants.OK_ID, GUI.i18n.getTranslation("BUTTON_OK"), true);

      /** Create Cancel Button */
      createButton(parent, IDialogConstants.CANCEL_ID, GUI.i18n.getTranslation("BUTTON_CANCLE"), false).setFont(FontShop.dialogFont);
    }

    /** Apply Font to Ok Button */
    okButton.setFont(FontShop.dialogFont);

    /** Give Focus to input text field */
    catTitle.setFocus();

    /** Preset cat title */
    if (catName != null) {
      catTitle.setText(catName);
      catTitle.selectAll();
    }

    /** Since not category title is preset, disable OK Button */
    else {
      okButton.setEnabled(false);
    }
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
   */
  protected Control createDialogArea(Composite parent) {

    /** Composite to hold all components */
    Composite composite = new Composite((Composite) super.createDialogArea(parent), SWT.NONE);
    composite.setLayout(LayoutShop.createGridLayout(2, 5, 20));
    composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

    /** Title Image */
    setTitleImage(PaintShop.loadImage("/img/icons/newcat_big.gif"));

    /** Title Message */
    setMessage(dialogMessage, IMessageProvider.INFORMATION);

    /** Title Label */
    Label titleLabel = new Label(composite, SWT.NONE);
    titleLabel.setText(GUI.i18n.getTranslation("LABEL_TITLE") + ": ");
    titleLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
    titleLabel.setFont(FontShop.dialogFont);

    /** Title input field */
    catTitle = new Text(composite, SWT.SINGLE | SWT.BORDER);
    catTitle.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    catTitle.setFont(FontShop.dialogFont);
    catTitle.addModifyListener(new ModifyListener() {
      public void modifyText(ModifyEvent e) {
        validateInput();
      }
    });

    /** Tweak Text Widget */
    WidgetShop.tweakTextWidget(catTitle);

    /** Holder for the separator to the OK and Cancel buttons */
    Composite sepHolder = new Composite(parent, SWT.NONE);
    sepHolder.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    sepHolder.setLayout(LayoutShop.createGridLayout(1, 0, 0));

    /** Separator */
    Label separator = new Label(sepHolder, SWT.SEPARATOR | SWT.HORIZONTAL);
    separator.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

    return composite;
  }

  /**
   * @see org.eclipse.jface.window.Window#getShellStyle()
   */
  protected int getShellStyle() {
    int style = SWT.TITLE | SWT.BORDER | SWT.APPLICATION_MODAL | getDefaultOrientation();

    /** Follow Apple's Human Interface Guidelines for Application Modal Dialogs */
    if (!GlobalSettings.isMac())
      style |= SWT.CLOSE;

    return style;
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#initializeBounds()
   */
  protected void initializeBounds() {
    super.initializeBounds();
    Point bestSize = getShell().computeSize(convertHorizontalDLUsToPixels(dialogMinWidth), SWT.DEFAULT);
    Point location = getInitialLocation(bestSize);
    getShell().setBounds(location.x, location.y, bestSize.x, bestSize.y);
  }

  /**
   * Set the layout data of the button to a GridData with appropriate widths
   * This method was slightly modified so that it is not setting a heightHint.
   * 
   * @param button The button to layout
   */
  protected void setButtonLayoutData(Button button) {
    GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
    int widthHint = convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH);
    data.widthHint = Math.max(widthHint, button.computeSize(SWT.DEFAULT, SWT.DEFAULT, true).x);
    button.setLayoutData(data);
  }

  /**
   * Validates the input.
   * <p>
   * The default implementation of this framework method delegates the request
   * to the supplied input validator object; if it finds the input invalid, the
   * warning dialogMessage is displayed in the dialog's dialogMessage line. This
   * hook method is called whenever the text changes in the input field.
   * </p>
   */
  protected void validateInput() {
    String errorMessage = null;

    /** Check if there was an error */
    if ((isEdit && !catName.equals(catTitle.getText())) || !isEdit) {
      if (parent.getSubCategories().containsKey(catTitle.getText()))
        errorMessage = GUI.i18n.getTranslation("ERROR_CAT_EXISTS");
    }

    /** Show an error message */
    if (errorMessage != null)
      setMessage(errorMessage, IMessageProvider.ERROR);

    /** Display info message */
    else
      setMessage(dialogMessage, IMessageProvider.INFORMATION);

    /** Set state of OK button */
    okButton.setEnabled(errorMessage == null && !catTitle.getText().equals(""));
  }
}