/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.panel;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.controller.ViewForm;
import net.sourceforge.rssowl.controller.forms.Hyperlink;
import net.sourceforge.rssowl.controller.forms.HyperlinkAdapter;
import net.sourceforge.rssowl.controller.forms.HyperlinkEvent;
import net.sourceforge.rssowl.util.shop.BrowserShop;
import net.sourceforge.rssowl.util.shop.FontShop;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.URLShop;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;

/**
 * The update panel is placed inside a TabItem and shown whenever a new version
 * of RSSOwl was released and the user wants to be informed about it. The panel
 * shows the Changelog and offers a link to download the new version.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class UpdatePanel {
  private Color bannerColor;
  private Image bannerImage;
  private Display display;
  private String message;
  private ViewForm messageViewForm;
  private Composite parent;
  private String title;

  /**
   * Instantiate a new UpdatePanel
   * 
   * @param display The display
   * @param parent The parent composite
   * @param title Title of the update panel's TabItem
   * @param message Message containing Changelog of the new version
   */
  public UpdatePanel(Display display, Composite parent, String title, String message) {
    this.parent = parent;
    this.display = display;
    this.title = title;
    this.message = message;
    bannerColor = new Color(display, 216, 216, 216);
    bannerImage = PaintShop.loadImage("/img/aboutbanner.png");
    initComponents();
  }

  /**
   * Dispose Images and Colors
   */
  public void dispose() {
    bannerColor.dispose();
    bannerImage.dispose();
  }

  /**
   * Get the Update Panel
   * 
   * @return Composite The update panel
   */
  public Composite getPanel() {
    return messageViewForm;
  }

  /**
   * Init all Components
   */
  private void initComponents() {

    /** ViewForm containing Banner and Hyperlinks */
    messageViewForm = new ViewForm(parent, SWT.BORDER | SWT.FLAT);
    messageViewForm.setBackground(bannerColor);
    messageViewForm.addDisposeListener(new DisposeListener() {
      public void widgetDisposed(DisposeEvent e) {
        UpdatePanel.this.dispose();
      }
    });

    /** Composite for the Banner */
    Composite bannerHolder = new Composite(messageViewForm, SWT.NONE);
    bannerHolder.setLayout(LayoutShop.createGridLayout(1, 0));
    bannerHolder.setBackground(bannerColor);

    /** Apply Banner Composite to Top Left corner */
    messageViewForm.setTopLeft(bannerHolder, false);

    /** Label to show the Banner */
    Label banner = new Label(bannerHolder, SWT.NONE);
    banner.setImage(bannerImage);
    banner.setLayoutData(new GridData(SWT.CENTER, SWT.BEGINNING, true, false));
    banner.setCursor(display.getSystemCursor(SWT.CURSOR_HAND));
    banner.setToolTipText(URLShop.RSSOWL_WEBPAGE);
    banner.setBackground(bannerColor);

    /** Open project homepage on mouse click */
    banner.addMouseListener(new MouseAdapter() {
      public void mouseUp(MouseEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_WEBPAGE);
      }
    });

    /** Welcome Title */
    Label titleLabel = new Label(bannerHolder, SWT.CENTER);
    titleLabel.setText(title);
    titleLabel.setLayoutData(new GridData(SWT.CENTER, SWT.BEGINNING, true, false));
    titleLabel.setBackground(bannerColor);
    titleLabel.setFont(FontShop.textBoldFont);

    /** Composite holding the bottom message area */
    Composite messageHolder = new Composite(messageViewForm, SWT.NONE);
    messageHolder.setLayout(LayoutShop.createGridLayout(1, 10, 5));
    messageHolder.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, true, true));
    messageHolder.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Composite holding the download controls */
    Composite downloadHolder = new Composite(messageHolder, SWT.NONE);
    downloadHolder.setLayout(LayoutShop.createGridLayout(2, 0, 10));
    downloadHolder.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, true, false));
    downloadHolder.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** "Download" Label */
    CLabel downloadLabel = new CLabel(downloadHolder, SWT.NONE);
    downloadLabel.setImage(PaintShop.iconMail);
    downloadLabel.setText(GUI.i18n.getTranslation("LABEL_DOWNLOAD") + ": ");
    downloadLabel.setFont(FontShop.textBoldFont);
    downloadLabel.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false));
    downloadLabel.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Hyperlink for Download */
    Hyperlink downloadLink = new Hyperlink(downloadHolder, SWT.NONE);
    downloadLink.setUnderlined(true);
    downloadLink.setText(URLShop.RSSOWL_DOWNLOAD);
    downloadLink.setFont(FontShop.textFont);
    downloadLink.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false));
    downloadLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    downloadLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    downloadLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_DOWNLOAD);
      }
    });

    /** Text control displaying the message */
    StyledText messageText = new StyledText(messageHolder, SWT.MULTI | SWT.READ_ONLY | SWT.WRAP | SWT.V_SCROLL);
    messageText.setText(message);
    messageText.setCaret(null);
    messageText.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
    messageText.setFont(FontShop.textFont);
    messageText.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Apply messageHolder as Content to ViewForm */
    messageViewForm.setContent(messageHolder, true);
  }
}