/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.panel;

import net.sourceforge.rssowl.controller.EventManager;
import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.controller.ViewForm;
import net.sourceforge.rssowl.controller.forms.Hyperlink;
import net.sourceforge.rssowl.controller.forms.HyperlinkAdapter;
import net.sourceforge.rssowl.controller.forms.HyperlinkEvent;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.BrowserShop;
import net.sourceforge.rssowl.util.shop.FontShop;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.URLShop;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseTrackAdapter;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;

/**
 * The welcome panel is the first Tab that is displayed to the user after first
 * start of RSSOwl. It contains some Hyperlinks to open the FAQ or Tutorial for
 * example.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class WelcomePanel {
  private Color bannerColor;
  private Image bannerImage;
  private Display display;
  private ViewForm messageViewForm;
  private Composite parent;
  Image arrowHotImage;
  Image arrowImage;
  EventManager eventManager;

  /**
   * Instantiate a new WelcomePanel
   * 
   * @param display The display
   * @param eventManager The eventManager
   * @param parent The parent composite
   */
  public WelcomePanel(Display display, EventManager eventManager, Composite parent) {
    this.parent = parent;
    this.display = display;
    this.eventManager = eventManager;
    bannerImage = PaintShop.loadImage("/img/aboutbanner.png");
    arrowImage = PaintShop.loadImage("/img/arrow.gif");
    arrowHotImage = PaintShop.loadImage("/img/arrow_hot.gif");
    bannerColor = new Color(display, 216, 216, 216);
    initComponents();
  }

  /**
   * Dispose Images and Colors
   */
  public void dispose() {
    bannerColor.dispose();
    bannerImage.dispose();
    arrowImage.dispose();
    arrowHotImage.dispose();
  }

  /**
   * Get the Welcome Panel
   * 
   * @return Composite The welcome panel
   */
  public Composite getPanel() {
    return messageViewForm;
  }

  /**
   * Init all Components
   */
  private void initComponents() {

    /** ViewForm containing Banner and Hyperlinks */
    messageViewForm = new ViewForm(parent, SWT.BORDER | SWT.FLAT);
    messageViewForm.setBackground(bannerColor);
    messageViewForm.addDisposeListener(new DisposeListener() {
      public void widgetDisposed(DisposeEvent e) {
        WelcomePanel.this.dispose();
      }
    });

    /** Composite for the Banner */
    Composite bannerHolder = new Composite(messageViewForm, SWT.NONE);
    bannerHolder.setLayout(LayoutShop.createGridLayout(1, 0));
    bannerHolder.setBackground(bannerColor);

    /** Apply Banner Composite to Top Left corner */
    messageViewForm.setTopLeft(bannerHolder, false);

    /** Label to show the Banner */
    Label banner = new Label(bannerHolder, SWT.NONE);
    banner.setImage(bannerImage);
    banner.setLayoutData(new GridData(SWT.CENTER, SWT.BEGINNING, true, false));
    banner.setCursor(display.getSystemCursor(SWT.CURSOR_HAND));
    banner.setToolTipText(URLShop.RSSOWL_WEBPAGE);
    banner.setBackground(bannerColor);

    /** Open project homepage on mouse click */
    banner.addMouseListener(new MouseAdapter() {
      public void mouseUp(MouseEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_WEBPAGE);
      }
    });

    /** Welcome Title */
    Label titleLabel = new Label(bannerHolder, SWT.CENTER);
    titleLabel.setText(GUI.i18n.getTranslation("LABEL_WELCOME_TITLE"));
    titleLabel.setLayoutData(new GridData(SWT.CENTER, SWT.BEGINNING, true, false));
    titleLabel.setBackground(bannerColor);
    titleLabel.setFont(FontShop.textBoldFont);

    /** Align the message area centered */
    Composite messageHolderAlignment = new Composite(messageViewForm, SWT.NONE);
    messageHolderAlignment.setLayout(new GridLayout(1, false));
    messageHolderAlignment.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Composite holding the bottom message area */
    Composite messageHolder = new Composite(messageHolderAlignment, SWT.NONE);
    messageHolder.setLayout(LayoutShop.createGridLayout(3, 10, 10, SWT.DEFAULT, SWT.DEFAULT, true));
    messageHolder.setLayoutData(new GridData(SWT.CENTER, SWT.BEGINNING, true, true));
    messageHolder.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Apply message area as content to ViewForm */
    messageViewForm.setContent(messageHolderAlignment, true);

    /** First Column Container */
    Composite firstColumn = new Composite(messageHolder, SWT.NONE);
    firstColumn.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.HORIZONTAL_ALIGN_CENTER));
    firstColumn.setLayout(LayoutShop.createGridLayout(1, 0, 0, 1, 2, false));
    firstColumn.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Container for "First Steps" */
    Composite firstSteps = new Composite(firstColumn, SWT.NONE);
    firstSteps.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
    firstSteps.setLayout(LayoutShop.createGridLayout(1, 0, 10, 1));
    firstSteps.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** "First Steps" Label */
    CLabel firstStepsLabel = new CLabel(firstSteps, SWT.NONE);
    firstStepsLabel.setImage(PaintShop.iconMail);
    firstStepsLabel.setText(GUI.i18n.getTranslation("LABEL_FIRST_STEPS"));
    firstStepsLabel.setFont(FontShop.textBoldFont);
    firstStepsLabel.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, true, false));
    firstStepsLabel.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Container for "First Steps" Hyperlinks */
    Composite firstStepsHolder = new Composite(firstSteps, SWT.NONE);
    firstStepsHolder.setLayout(LayoutShop.createGridLayout(1, 24, 0));
    firstStepsHolder.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Hyperlink for Tutorial */
    Hyperlink tutorialLink = new Hyperlink(firstStepsHolder, SWT.NONE);
    tutorialLink.setUnderlined(true);
    tutorialLink.setText(GUI.i18n.getTranslation("MENU_TUTORIAL"));
    tutorialLink.setFont(FontShop.textFont);
    tutorialLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    tutorialLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    tutorialLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {
        eventManager.actionOpenTutorial();
      }
    });

    /** Hyperlink for the FAQ */
    Hyperlink faqLink = new Hyperlink(firstStepsHolder, SWT.NONE);
    faqLink.setUnderlined(true);
    faqLink.setText("FAQ");
    faqLink.setFont(FontShop.textFont);
    faqLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    faqLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    faqLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {

        /** Restore View if necessary */
        restoreView();

        /** Load Feed */
        eventManager.actionOpenFAQ();
      }
    });

    /** Container for "News" */
    Composite news = new Composite(firstColumn, SWT.NONE);
    news.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
    news.setLayout(LayoutShop.createGridLayout(1, 0, 10, 1));
    news.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** "News" Label */
    CLabel newsLabel = new CLabel(news, SWT.NONE);
    newsLabel.setImage(PaintShop.iconMail);
    newsLabel.setText(GUI.i18n.getTranslation("LABEL_NEWS"));
    newsLabel.setFont(FontShop.textBoldFont);
    newsLabel.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, true, false));
    newsLabel.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Container for the Hyperlinks */
    Composite newsHolder = new Composite(news, SWT.NONE);
    newsHolder.setLayout(LayoutShop.createGridLayout(1, 24, 0));
    newsHolder.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Hyperlink for mail contact */
    Hyperlink newsFeedLink = new Hyperlink(newsHolder, SWT.NONE);
    newsFeedLink.setUnderlined(true);
    newsFeedLink.setText(GUI.i18n.getTranslation("LABEL_RSSOWL_NEWSFEED"));
    newsFeedLink.setFont(FontShop.textFont);
    newsFeedLink.setToolTipText(URLShop.RSSOWL_FEED);
    newsFeedLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    newsFeedLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    newsFeedLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {

        /** Restore View if necessary */
        restoreView();

        /** Load Feed */
        eventManager.actionOpenFeed(URLShop.RSSOWL_FEED);
      }
    });

    /** Second Column Container */
    Composite secondColumn = new Composite(messageHolder, SWT.NONE);
    secondColumn.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.HORIZONTAL_ALIGN_CENTER));
    secondColumn.setLayout(LayoutShop.createGridLayout(1, 0, 0, 1, 2, false));
    secondColumn.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Container for "Support" */
    Composite support = new Composite(secondColumn, SWT.NONE);
    support.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
    support.setLayout(LayoutShop.createGridLayout(1, 0, 10, 1));
    support.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** "Support" Label */
    CLabel supportLabel = new CLabel(support, SWT.NONE);
    supportLabel.setImage(PaintShop.iconMail);
    supportLabel.setText(GUI.i18n.getTranslation("LABEL_SUPPORT"));
    supportLabel.setFont(FontShop.textBoldFont);
    supportLabel.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, true, false));
    supportLabel.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Container for the Hyperlinks */
    Composite supportHolder = new Composite(support, SWT.NONE);
    supportHolder.setLayout(LayoutShop.createGridLayout(1, 24, 0));
    supportHolder.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Hyperlink for Discussion Forum */
    Hyperlink forumLink = new Hyperlink(supportHolder, SWT.NONE);
    forumLink.setUnderlined(true);
    forumLink.setText(GUI.i18n.getTranslation("LABEL_DISCUSSION_FORUM"));
    forumLink.setFont(FontShop.textFont);
    forumLink.setToolTipText(URLShop.RSSOWL_FORUM);
    forumLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    forumLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    forumLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_FORUM);
      }
    });

    /** Hyperlink for Mailinglist */
    Hyperlink mailingListLink = new Hyperlink(supportHolder, SWT.NONE);
    mailingListLink.setUnderlined(true);
    mailingListLink.setText(GUI.i18n.getTranslation("MENU_MAILING_LIST"));
    mailingListLink.setFont(FontShop.textFont);
    mailingListLink.setToolTipText(URLShop.RSSOWL_LIST);
    mailingListLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    mailingListLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    mailingListLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_LIST);
      }
    });

    /** Container for "Misc." */
    Composite misc = new Composite(secondColumn, SWT.NONE);
    misc.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
    misc.setLayout(LayoutShop.createGridLayout(1, 0, 10, 1));
    misc.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** "Misc." Label */
    CLabel miscLabel = new CLabel(misc, SWT.NONE);
    miscLabel.setImage(PaintShop.iconMail);
    miscLabel.setText(GUI.i18n.getTranslation("MENU_MISC"));
    miscLabel.setFont(FontShop.textBoldFont);
    miscLabel.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, true, false));
    miscLabel.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Container for the Hyperlinks */
    Composite miscHolder = new Composite(misc, SWT.NONE);
    miscHolder.setLayout(LayoutShop.createGridLayout(1, 24, 0));
    miscHolder.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Hyperlink for Donation */
    Hyperlink donateLink = new Hyperlink(miscHolder, SWT.NONE);
    donateLink.setUnderlined(true);
    donateLink.setText(GUI.i18n.getTranslation("MENU_DONATE"));
    donateLink.setFont(FontShop.textFont);
    donateLink.setToolTipText(URLShop.RSSOWL_DONATE);
    donateLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    donateLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    donateLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_DONATE);
      }
    });

    /** Hyperlink for Promotion */
    Hyperlink promoteLink = new Hyperlink(miscHolder, SWT.NONE);
    promoteLink.setUnderlined(true);
    promoteLink.setText(GUI.i18n.getTranslation("LABEL_PROMOTION"));
    promoteLink.setFont(FontShop.textFont);
    promoteLink.setToolTipText(URLShop.RSSOWL_PROMOTION);
    promoteLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    promoteLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    promoteLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_PROMOTION);
      }
    });

    /** Third Column Container */
    Composite thirdColumn = new Composite(messageHolder, SWT.NONE);
    thirdColumn.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.HORIZONTAL_ALIGN_CENTER));
    thirdColumn.setLayout(LayoutShop.createGridLayout(1, 0, 0, 1, 2, false));
    thirdColumn.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Container for "Contact" */
    Composite contact = new Composite(thirdColumn, SWT.NONE);
    contact.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
    contact.setLayout(LayoutShop.createGridLayout(1, 0, 10, 1));
    contact.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** "Contact" Label */
    CLabel contactLabel = new CLabel(contact, SWT.NONE);
    contactLabel.setImage(PaintShop.iconMail);
    contactLabel.setText(GUI.i18n.getTranslation("LABEL_CONTACT"));
    contactLabel.setFont(FontShop.textBoldFont);
    contactLabel.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, true, false));
    contactLabel.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Container for the Hyperlinks */
    Composite contactHolder = new Composite(contact, SWT.NONE);
    contactHolder.setLayout(LayoutShop.createGridLayout(1, 24, 0));
    contactHolder.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Hyperlink for mail contact */
    Hyperlink contactLink = new Hyperlink(contactHolder, SWT.NONE);
    contactLink.setUnderlined(true);
    contactLink.setText("RSSOwl Team");
    contactLink.setToolTipText("contact@rssowl.org");
    contactLink.setFont(FontShop.textFont);
    contactLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    contactLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    contactLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_CONTACT);
      }
    });

    /** Hyperlink for mail contact */
    Hyperlink newsLink = new Hyperlink(newsHolder, SWT.NONE);
    newsLink.setUnderlined(true);
    newsLink.setText("RSSOwl.org");
    newsLink.setToolTipText(URLShop.RSSOWL_WEBPAGE);
    newsLink.setFont(FontShop.textFont);
    newsLink.setForeground(display.getSystemColor(SWT.COLOR_BLUE));
    newsLink.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
    newsLink.addHyperlinkListener(new HyperlinkAdapter() {
      public void linkActivated(HyperlinkEvent e) {
        BrowserShop.openLink(URLShop.RSSOWL_WEBPAGE);
      }
    });

    /** Container for Arrow Image to close Welcome tab */
    Composite startRSSOwl = new Composite(messageHolder, SWT.NONE);
    startRSSOwl.setLayoutData(new GridData(SWT.CENTER, SWT.BEGINNING, true, false, 3, 1));
    startRSSOwl.setLayout(LayoutShop.createGridLayout(1, 0, 50));
    startRSSOwl.setBackground(display.getSystemColor(SWT.COLOR_WHITE));

    /** Show a green, clickable Arrow */
    final Label startRSSOwlLabel = new Label(startRSSOwl, SWT.NONE);
    startRSSOwlLabel.setImage(arrowImage);
    startRSSOwlLabel.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.HORIZONTAL_ALIGN_CENTER));
    startRSSOwlLabel.setToolTipText(GUI.i18n.getTranslation("LABEL_START"));
    startRSSOwlLabel.setCursor(display.getSystemCursor(SWT.CURSOR_HAND));

    /** Simulate a "Hot" Image on MouseOver */
    startRSSOwlLabel.addMouseTrackListener(new MouseTrackAdapter() {

      /** Mouse enters Arrow */
      public void mouseEnter(MouseEvent e) {
        if (WidgetShop.isset(startRSSOwlLabel) && PaintShop.isset(arrowHotImage))
          startRSSOwlLabel.setImage(arrowHotImage);
      }

      /** Mouse exits Arrow */
      public void mouseExit(MouseEvent e) {
        if (WidgetShop.isset(startRSSOwlLabel) && PaintShop.isset(arrowImage))
          startRSSOwlLabel.setImage(arrowImage);
      }
    });

    /** Close welcome tab on mouse click */
    startRSSOwlLabel.addMouseListener(new MouseAdapter() {
      public void mouseUp(MouseEvent e) {
        eventManager.actionCloseCurrent();
      }
    });
  }

  /**
   * Restore the normal View in order to display a Newsfeed for example.
   */
  void restoreView() {
    GUI.rssOwlGui.getRSSOwlNewsTabFolder().getNewsHeaderTabFolder().setMaximized(false);
    GUI.rssOwlGui.setFavoritesMinimized(!GlobalSettings.isFavoritesTreeShown, false);
    GUI.rssOwlGui.getRSSOwlQuickview().setShowQuickview(GlobalSettings.isQuickviewShown, false);
    GUI.rssOwlGui.getRSSOwlQuickview().setShowToolBar(GlobalSettings.isToolBarShown, false);
  }
}