/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.popup;

import net.sourceforge.rssowl.controller.EventManager;
import net.sourceforge.rssowl.controller.FavoritesTree;
import net.sourceforge.rssowl.controller.GUI;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TreeItem;

/**
 * The popupmenu appears after clicking in the favoritesTree and supports some
 * feed management functions like add new favorit / category or remove them.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class FavoritesTreePopup {

  /** Constants for the different menuStructure types */
  public static final int BLOGROLL_CAT_MENU = 3;

  /** Constants for the different menuStructure types */
  public static final int BLOGROLL_FAV_MENU = 4;

  /** Constants for the different menuStructure types */
  public static final int CAT_MENU = 1;

  /** Constants for the different menuStructure types */
  public static final int EMPTY_TREE_MENU = 5;

  /** Constants for the different menuStructure types */
  public static final int FAV_MENU = 2;

  private int currentMenu;
  private EventManager eventManager;
  private boolean requiresUpdate;
  private Menu treePopUpMenu;
  FavoritesTree rssOwlFavoritesTree;
  GUI rssOwlGui;
  Shell shell;

  /**
   * Instantiate a new FavoritesTreePopup
   * 
   * @param shell The shell
   * @param rssOwlFavoritesTree The RSSOwl URL tree
   * @param rssOwlGui The maincontroller
   * @param eventManager The event manager
   */
  public FavoritesTreePopup(Shell shell, FavoritesTree rssOwlFavoritesTree, GUI rssOwlGui, EventManager eventManager) {
    this.shell = shell;
    this.rssOwlFavoritesTree = rssOwlFavoritesTree;
    this.rssOwlGui = rssOwlGui;
    this.eventManager = eventManager;
    currentMenu = -1;
    requiresUpdate = false;
    initComponents();
  }

  /**
   * Get the Popup menu that is set on the favorites tree.
   * 
   * @return Menu Return the tree popup menu
   */
  public Menu getTreePopUpMenu() {
    return treePopUpMenu;
  }

  /**
   * Marks the popup as required of an update.
   */
  public void setRequiresUpdate() {
    requiresUpdate = true;
  }

  /**
   * Update the context Menu on the Favorites Tree.
   * 
   * @param style One of the supported styles
   * @param selectedTreeItem The treeitem that will get the popup
   * @return boolean TRUE if the Menu was updated, FALSE otherwise.
   */
  public boolean updateMenu(int style, TreeItem selectedTreeItem) {

    /** Return in these cases */
    if (style == currentMenu && style != BLOGROLL_CAT_MENU && style != CAT_MENU && !requiresUpdate)
      return false;

    /** First reset the popup */
    if (style != -1)
      resetPopup();

    /** Remember Menu */
    currentMenu = style;
    requiresUpdate = false;

    switch (style) {

      /** Menu for category */
      case CAT_MENU:
        new CategoryPopup(shell, rssOwlGui, treePopUpMenu, eventManager, selectedTreeItem);
        break;

      /** Menu for a favorit */
      case FAV_MENU:
        new FavoritePopup(shell, treePopUpMenu, eventManager);
        break;

      /** Menu for a blogroll category */
      case BLOGROLL_CAT_MENU:
        new BlogrollCategoryPopup(shell, rssOwlGui, treePopUpMenu, eventManager, selectedTreeItem);
        break;

      /** Menu for a blogroll favorite */
      case BLOGROLL_FAV_MENU:
        new BlogrollFavoritePopup(shell, treePopUpMenu, eventManager);
        break;

      /** Menu when the tree is empty */
      case EMPTY_TREE_MENU:
        new EmptyTreePopup(shell, treePopUpMenu, eventManager);
        break;
    }

    /** Set menu to URL tree */
    rssOwlFavoritesTree.getFavoritesTree().setMenu(treePopUpMenu);

    /** Return: Menu was updated */
    return true;
  }

  /** Init Popup menu */
  private void initComponents() {
    treePopUpMenu = new Menu(shell, SWT.POP_UP);
  }

  /** Reset the menue */
  private void resetPopup() {
    MenuItem items[] = treePopUpMenu.getItems();
    for (int i = 0; i < items.length; i++) {
      MenuItem item = items[i];
      item.dispose();
    }
  }
}