/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.properties;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.FileShop;
import net.sourceforge.rssowl.util.shop.FontShop;
import net.sourceforge.rssowl.util.shop.LayoutDataShop;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.StringShop;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import java.util.ArrayList;

/**
 * Class displays a PropertyPage prompting for a path to a browser executable
 * and some other settings concerning the use of the browser in RSSOwl
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class BrowserProperties extends PropertyPage {
  private Button blockPopupsCheck;
  private Button openNewBrowserWindowCheck;
  Text browserArguments;
  Text browserPath;
  Button openBrowserExternCheck;

  /**
   * Instantiate a new BrowserProperties PropertyPage
   * 
   * @param parent Composite's parent
   */
  public BrowserProperties(Composite parent) {
    super(parent);
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#applyButtonPressed()
   */
  public void applyButtonPressed() {
    GlobalSettings.customBrowser = browserPath.getText();
    GlobalSettings.customBrowserArguments = browserArguments.getText();

    /** Internal browser settings */
    if (GlobalSettings.useInternalBrowser()) {
      GlobalSettings.openNewBrowserWindow = openNewBrowserWindowCheck.getSelection();
      GlobalSettings.openBrowserExtern = openBrowserExternCheck.getSelection();

      if (!GlobalSettings.isLinux())
        GlobalSettings.blockPopups = blockPopupsCheck.getSelection();
    }

    updatePropertiesChangeManager();
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#updatePropertiesChangeManager()
   */
  public void updatePropertiesChangeManager() {

    /** Save custom browser path */
    propertyChangeManager.setCustomBrowserPath(browserPath.getText());

    /** Save custom browser arguments */
    propertyChangeManager.setCustomBrowserArguments(browserArguments.getText());

    /** If the OS supports internal browser */
    if (GlobalSettings.useInternalBrowser()) {
      propertyChangeManager.setOpenBrowserInNewTab(openNewBrowserWindowCheck.getSelection());
      propertyChangeManager.setOpenBrowserExtern(openBrowserExternCheck.getSelection());

      if (!GlobalSettings.isLinux())
        propertyChangeManager.setBlockPopups(blockPopupsCheck.getSelection());
    }
  }

  /** Init all components */
  protected void initComponents() {

    /** Remember buttons that require Mnemonics */
    ArrayList buttons = new ArrayList();

    /** Group for the custom browser path components */
    Group externalBrowserGroup = new Group(composite, SWT.NONE);
    externalBrowserGroup.setLayoutData(LayoutDataShop.createGridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL, 2));
    externalBrowserGroup.setText(GUI.i18n.getTranslation("MENU_SELECT_EXTERNAL_BROWSER"));
    externalBrowserGroup.setLayout(new GridLayout(2, false));
    externalBrowserGroup.setFont(FontShop.dialogFont);

    /** Message Label */
    Label messageLabel = new Label(externalBrowserGroup, SWT.WRAP);
    messageLabel.setText(GUI.i18n.getTranslation("DIALOG_SELECT_EXTERNAL_BROWSER") + ": ");
    messageLabel.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    messageLabel.setFont(dialogFont);

    /** Input for the custom browser path */
    browserPath = new Text(externalBrowserGroup, SWT.SINGLE | SWT.BORDER);
    browserPath.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    browserPath.setFont(dialogFont);
    browserPath.setText(propertyChangeManager.getCustomBrowserPath());
    browserPath.addModifyListener(new ModifyListener() {

      /** Disable Browser arguments if no Browser is given */
      public void modifyText(ModifyEvent e) {
        browserArguments.setEnabled(!browserPath.getText().equals(""));

        /** Check if the given File exists */
        boolean fileExists = FileShop.exists(browserPath.getText());

        /** Enable the "Use external Browser" check if File exists */
        if (fileExists && GlobalSettings.useInternalBrowser())
          openBrowserExternCheck.setSelection(true);

        /** Show error in case path is not valid */
        if (StringShop.isset(browserPath.getText()))
          setErrorMessage(fileExists ? null : GUI.i18n.getTranslation("ERROR_FILE_NOT_FOUND"));

        /** Reset Error Message */
        else
          setErrorMessage(null);
      }
    });

    /** Tweak Text Widget */
    WidgetShop.tweakTextWidget(browserPath);

    /** Button to search in Filesystem */
    Button searchBrowser = new Button(externalBrowserGroup, SWT.PUSH);
    searchBrowser.setText(GUI.i18n.getTranslation("BUTTON_SEARCH") + "...");
    searchBrowser.setFont(dialogFont);
    searchBrowser.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        String file = FileShop.getFilePath(composite.getShell(), null, null, SWT.OPEN, browserPath.getText(), GUI.i18n.getTranslation("MENU_BROWSER"));

        if (file != null)
          browserPath.setText(file);
      }
    });

    /** Apply layoutdata to button */
    setButtonLayoutData(searchBrowser);

    /** Add Mnemonic to Button later */
    buttons.add(searchBrowser);

    /** Error Label in case the path is not valid */
    errorMessageLabel = new CLabel(externalBrowserGroup, SWT.NONE);
    errorMessageLabel.setFont(dialogFont);
    errorMessageLabel.setLayoutData(new GridData(SWT.BEGINNING, SWT.CENTER, false, false));

    /** Show error label in case a path is not valid */
    if (StringShop.isset(browserPath.getText()))
      setErrorMessage(FileShop.exists(browserPath.getText()) ? null : GUI.i18n.getTranslation("ERROR_FILE_NOT_FOUND"));

    /** Group for the arguments components */
    Group argumentsGroup = new Group(composite, SWT.NONE);
    argumentsGroup.setLayoutData(LayoutDataShop.createGridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL, 2));
    argumentsGroup.setText(GUI.i18n.getTranslation("GROUP_ARGUMENTS"));
    argumentsGroup.setLayout(new GridLayout(2, false));
    argumentsGroup.setFont(FontShop.dialogFont);

    /** Browser arguments */
    browserArguments = new Text(argumentsGroup, SWT.SINGLE | SWT.BORDER);
    browserArguments.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    browserArguments.setFont(dialogFont);
    browserArguments.setText(propertyChangeManager.getCustomBrowserArguments());

    /** Tweak Text Widget */
    WidgetShop.tweakTextWidget(browserArguments);

    /** Disable Browser arguments if no Browser is given */
    browserArguments.setEnabled(!browserPath.getText().equals(""));

    /** Menu to insert the wildcards */
    WidgetShop.createWildCardMenu(browserArguments, new String[] { "[URL]" });

    /** Message Label */
    Label argumentsLabel = new Label(argumentsGroup, SWT.WRAP);
    argumentsLabel.setText(GUI.i18n.getTranslation("LABEL_BROWSER_USAGE"));
    argumentsLabel.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    argumentsLabel.setFont(dialogFont);

    /** Only display if OS supports internal browser */
    if (GlobalSettings.useInternalBrowser()) {

      /** Group holding the misc. settings */
      Group settingsGroup = new Group(composite, SWT.NONE);
      settingsGroup.setLayoutData(LayoutDataShop.createGridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL, 2));
      settingsGroup.setText(GUI.i18n.getTranslation("MENU_MISC"));
      settingsGroup.setLayout(new GridLayout(2, false));
      settingsGroup.setFont(FontShop.dialogFont);

      /** Block Popups - On Linux, Mozilla is taking care of Popups for itself */
      if (!GlobalSettings.isLinux()) {
        blockPopupsCheck = new Button(settingsGroup, SWT.CHECK);
        blockPopupsCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
        blockPopupsCheck.setSelection(propertyChangeManager.isBlockPopups());
        blockPopupsCheck.setText(GUI.i18n.getTranslation("BUTTON_BLOCK_POPUPS"));
        blockPopupsCheck.setFont(dialogFont);

        /** Add Mnemonic to Button later */
        buttons.add(blockPopupsCheck);
      }

      /** Open Browser in new Tab */
      openNewBrowserWindowCheck = new Button(settingsGroup, SWT.CHECK);
      openNewBrowserWindowCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
      openNewBrowserWindowCheck.setSelection(propertyChangeManager.isOpenBrowserInNewTab());
      openNewBrowserWindowCheck.setText(GUI.i18n.getTranslation("MENU_OPENNEW_BROWSER"));
      openNewBrowserWindowCheck.setFont(dialogFont);

      /** Add Mnemonic to Button later */
      buttons.add(openNewBrowserWindowCheck);

      /** Open browser extern */
      openBrowserExternCheck = new Button(settingsGroup, SWT.CHECK);
      openBrowserExternCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
      openBrowserExternCheck.setSelection(propertyChangeManager.isOpenBrowserExtern());
      openBrowserExternCheck.setText(GUI.i18n.getTranslation("MENU_BROWSER_EXTERN"));
      openBrowserExternCheck.setFont(dialogFont);

      /** Add Mnemonic to Button later */
      buttons.add(openBrowserExternCheck);
    }

    /** Set Mnemonics to Buttons */
    WidgetShop.initMnemonics((Button[]) buttons.toArray(new Button[buttons.size()]));

    /** Fill with some spacer */
    LayoutShop.setDialogSpacer(composite, 2, GlobalSettings.useInternalBrowser() ? 1 : 3);
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#restoreButtonPressed()
   */
  protected void restoreButtonPressed() {

    /** Internal Browser settings */
    if (GlobalSettings.useInternalBrowser()) {
      openBrowserExternCheck.setSelection(false);
      openNewBrowserWindowCheck.setSelection(true);

      if (!GlobalSettings.isLinux())
        blockPopupsCheck.setSelection(true);
      setErrorMessage(null);
    }

    browserPath.setText("");
    browserArguments.setText("");
    updatePropertiesChangeManager();
  }

}