/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.properties;

import net.sourceforge.rssowl.controller.DisposeListenerImpl;
import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.controller.dialog.PreferencesDialog;
import net.sourceforge.rssowl.util.i18n.Dictionary;
import net.sourceforge.rssowl.util.i18n.LanguageDetector;
import net.sourceforge.rssowl.util.i18n.RSSOwlI18n;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.PaintShop;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Locale;

/**
 * Class displays a PropertyPage to select the language used in RSSOwl
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class LanguageProperties extends PropertyPage {
  private Hashtable languages;
  private PreferencesDialog rssOwlPropertiesDialog;
  String selectedLanguage;

  /**
   * Instatiate a new LanguageProperties PropertyPage
   * 
   * @param parent Composite's parent
   * @param rssOwlGui Access some methods from the maincontroller
   * @param rssOwlPropertiesDialog Access some methods from the opened prop.
   * dialog
   */
  public LanguageProperties(Composite parent, GUI rssOwlGui, PreferencesDialog rssOwlPropertiesDialog) {
    super(parent, rssOwlGui);
    this.rssOwlPropertiesDialog = rssOwlPropertiesDialog;
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#applyButtonPressed()
   */
  public void applyButtonPressed() {
    if (selectedLanguage != null && !selectedLanguage.equals(Dictionary.selectedLanguage)) {
      GUI.i18n = new RSSOwlI18n(new Locale(selectedLanguage));
      rssOwlGui.updateI18N();

      /** Update propertys */
      rssOwlPropertiesDialog.updateI18N();
    }
    updatePropertiesChangeManager();
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#updatePropertiesChangeManager()
   */
  public void updatePropertiesChangeManager() {
    propertyChangeManager.setSelectedLanguage(selectedLanguage);
  }

  /**
   * Select one of the buttons in dependance of the selected language. English
   * is default.
   * 
   * @param language Selected language
   */
  private void setButtonState(String language) {

    /** Select the button that matches the language */
    Enumeration keys = languages.keys();
    while (keys.hasMoreElements()) {
      String key = (String) keys.nextElement();

      if (key.equals(language))
        ((Button) languages.get(key)).setSelection(true);
      else
        ((Button) languages.get(key)).setSelection(false);
    }
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#initComponents()
   */
  protected void initComponents() {
    languages = new Hashtable();

    /** Get selected language from PropertyChangeManager */
    selectedLanguage = propertyChangeManager.getSelectedLanguage();

    /** Composite for the labels and buttons */
    Composite langHolder = new Composite(composite, SWT.NONE);
    langHolder.setLayout(LayoutShop.createGridLayout(4, 10, 0, 15, 5, false));
    langHolder.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, true, false, 2, 1));

    /**
     * Add a button and clabel to the composite for each language that is
     * supported
     */
    for (int a = 0; a < Dictionary.languages.size(); a++) {

      /** Locale of the language */
      final Locale locale = (Locale) Dictionary.locales.get(Dictionary.languages.get(a));

      /** Radio button to select the language */
      final Button selectLanguageButton = new Button(langHolder, SWT.RADIO);
      selectLanguageButton.setFont(dialogFont);
      selectLanguageButton.addDisposeListener(DisposeListenerImpl.getInstance());
      selectLanguageButton.setImage(PaintShop.getLanguageFlag(locale.getCountry().toLowerCase(), getClass()));
      selectLanguageButton.addSelectionListener(new SelectionAdapter() {
        public void widgetSelected(SelectionEvent e) {
          if (selectLanguageButton.getSelection())
            selectedLanguage = locale.getLanguage();
        }
      });

      /** CLabel with flag of the language */
      Label selectLanguageLabel = new Label(langHolder, SWT.NONE);
      selectLanguageLabel.setText(GUI.i18n.getTranslation(locale.getLanguage()));
      selectLanguageLabel.setFont(dialogFont);

      /** Add radio button to list of selectable languages */
      languages.put(locale.getLanguage(), selectLanguageButton);
    }

    /** Set button state */
    setButtonState(selectedLanguage);
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#restoreButtonPressed()
   */
  protected void restoreButtonPressed() {
    setButtonState(LanguageDetector.parseLanguage(Locale.getDefault()));
    selectedLanguage = LanguageDetector.parseLanguage(Locale.getDefault());
    updatePropertiesChangeManager();
  }
}