/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.properties;

import net.sourceforge.rssowl.controller.DisposeListenerImpl;
import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.FontShop;
import net.sourceforge.rssowl.util.shop.LayoutDataShop;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;

/**
 * Class displays a PropertyPage to set view-releated properties in RSSOwl. This
 * is for example the window-layout: Either two columns or three.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class ViewProperties extends PropertyPage {
  private Button curvedTabRadio;
  private Button focusNewTabsCheck;
  private Button showChannelInfoCheck;
  private Button showErrorsCheck;
  private Button simpleTabRadio;
  private Button threeColumnGUIRadio;
  private Button twoColumnGUIRadio;
  private Button useBrowserForNewstextCheck;
  Button displayTabsCheck;
  Button showCloseButtonCheck;

  /**
   * Instatiate a new ViewProperties PropertyPage
   * 
   * @param parent Composite's parent
   * @param rssOwlGui The MainController
   */
  public ViewProperties(Composite parent, GUI rssOwlGui) {
    super(parent, rssOwlGui);
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#applyButtonPressed()
   */
  public void applyButtonPressed() {

    /** State of 2 / 3 column window layout */
    if (GlobalSettings.isThreeColumnGUI != threeColumnGUIRadio.getSelection()) {
      GlobalSettings.isThreeColumnGUI = threeColumnGUIRadio.getSelection();
      rssOwlGui.changeWindowLayout();
    }

    /** Tab Style */
    if (GlobalSettings.simpleTabs != simpleTabRadio.getSelection()) {
      GlobalSettings.simpleTabs = simpleTabRadio.getSelection();
      rssOwlGui.getRSSOwlNewsTabFolder().getNewsHeaderTabFolder().setSimple(GlobalSettings.simpleTabs);
    }

    /** A change in using Tabs at all */
    if (GlobalSettings.displaySingleTab == displayTabsCheck.getSelection()) {
      GlobalSettings.displaySingleTab = !displayTabsCheck.getSelection();
      rssOwlGui.getRSSOwlNewsTabFolder().getNewsHeaderTabFolder().setSelectionBackground((GlobalSettings.displaySingleTab) ? PaintShop.grayViewFormColor : GUI.display.getSystemColor(SWT.COLOR_WHITE));
      rssOwlGui.getRSSOwlNewsTabFolder().getNewsHeaderTabFolder().setTabHeight(GlobalSettings.displaySingleTab ? 0 : -1);

      /** Single Tab View looks better if single tab is bottom aligned */
      if (GlobalSettings.displaySingleTab)
        rssOwlGui.getRSSOwlNewsTabFolder().getNewsHeaderTabFolder().setTabPosition(SWT.BOTTOM);
      else
        rssOwlGui.getRSSOwlNewsTabFolder().getNewsHeaderTabFolder().setTabPosition(GlobalSettings.tabPositionIsTop ? SWT.TOP : SWT.BOTTOM);

      /** Layout CTabFolder */
      rssOwlGui.getRSSOwlNewsTabFolder().getNewsHeaderTabFolder().layout();
    }

    GlobalSettings.showTabCloseButton = showCloseButtonCheck.getSelection();

    /** Use browser for Newstext */
    if (GlobalSettings.useInternalBrowser() && GlobalSettings.useBrowserForNewsText != useBrowserForNewstextCheck.getSelection()) {
      GlobalSettings.useBrowserForNewsText = useBrowserForNewstextCheck.getSelection();
      rssOwlGui.getEventManager().actionSetBrowserView(GlobalSettings.useBrowserForNewsText);
    }

    /** Show channel info */
    GlobalSettings.showChannelInfo = showChannelInfoCheck.getSelection();

    /** Show errors */
    GlobalSettings.showErrors = propertyChangeManager.isShowErrors();

    /** Focus on new tabs */
    GlobalSettings.focusNewTabs = focusNewTabsCheck.getSelection();

    updatePropertiesChangeManager();
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#updatePropertiesChangeManager()
   */
  public void updatePropertiesChangeManager() {
    propertyChangeManager.setThreeColumnGUI(threeColumnGUIRadio.getSelection());
    propertyChangeManager.setShowChannelInfo(showChannelInfoCheck.getSelection());
    propertyChangeManager.setSimpleTabs(simpleTabRadio.getSelection());
    propertyChangeManager.setDisplaySingleTab(!displayTabsCheck.getSelection());
    propertyChangeManager.setShowTabCloseButton(showCloseButtonCheck.getSelection());
    propertyChangeManager.setFocusNewTabs(focusNewTabsCheck.getSelection());
    propertyChangeManager.setShowErrors(showErrorsCheck.getSelection());

    if (GlobalSettings.useInternalBrowser())
      propertyChangeManager.setUseBrowserForNewsText(useBrowserForNewstextCheck.getSelection());
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#initComponents()
   */
  protected void initComponents() {

    /** Group for the 2 / 3 column GUI components */
    Group columnGUIGroup = new Group(composite, SWT.NONE);
    columnGUIGroup.setLayoutData(LayoutDataShop.createGridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL, 2));
    columnGUIGroup.setText(GUI.i18n.getTranslation("GROUP_WINDOW_LAYOUT"));
    columnGUIGroup.setLayout(LayoutShop.createGridLayout(2, 5, 5, 5, 15, false));
    columnGUIGroup.setFont(FontShop.dialogFont);

    Label columnGUILabel = new Label(columnGUIGroup, SWT.WRAP);
    columnGUILabel.setFont(dialogFont);
    columnGUILabel.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    columnGUILabel.setText(GUI.i18n.getTranslation("LABEL_SELECT_WINDOW_LAYOUT") + ": ");

    /** Two column GUI */
    twoColumnGUIRadio = new Button(columnGUIGroup, SWT.RADIO);
    twoColumnGUIRadio.setFont(dialogFont);
    twoColumnGUIRadio.setSelection(!propertyChangeManager.isThreeColumnGUI());
    twoColumnGUIRadio.setImage(PaintShop.loadImage("/img/icons/twocolumn.gif"));
    twoColumnGUIRadio.addDisposeListener(DisposeListenerImpl.getInstance());

    /** Three column GUI */
    threeColumnGUIRadio = new Button(columnGUIGroup, SWT.RADIO);
    threeColumnGUIRadio.setFont(dialogFont);
    threeColumnGUIRadio.setSelection(propertyChangeManager.isThreeColumnGUI());
    threeColumnGUIRadio.setImage(PaintShop.loadImage("/img/icons/threecolumn.gif"));
    threeColumnGUIRadio.addDisposeListener(DisposeListenerImpl.getInstance());

    /** Tab Style */
    Group tabStyleGroup = new Group(composite, SWT.NONE);
    tabStyleGroup.setText(GUI.i18n.getTranslation("GROUP_TAB_LAYOUT"));
    tabStyleGroup.setLayout(new GridLayout(2, false));
    tabStyleGroup.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    tabStyleGroup.setFont(FontShop.dialogFont);

    /** Traditional style */
    simpleTabRadio = new Button(tabStyleGroup, SWT.RADIO);
    simpleTabRadio.setText(GUI.i18n.getTranslation("LABEL_TRADITIONAL_TABS"));
    simpleTabRadio.setSelection(propertyChangeManager.isSimpleTabs());
    simpleTabRadio.setFont(dialogFont);

    /** Curved style */
    curvedTabRadio = new Button(tabStyleGroup, SWT.RADIO);
    curvedTabRadio.setText(GUI.i18n.getTranslation("LABEL_CURVED_TABS"));
    curvedTabRadio.setSelection(!propertyChangeManager.isSimpleTabs());
    curvedTabRadio.setFont(dialogFont);

    /** Display newsfeeds in tabs */
    displayTabsCheck = new Button(tabStyleGroup, SWT.CHECK);
    displayTabsCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    displayTabsCheck.setSelection(!propertyChangeManager.isDisplaySingleTab());
    displayTabsCheck.setText(GUI.i18n.getTranslation("BUTTON_DISPLAY_TABS"));
    displayTabsCheck.setFont(dialogFont);
    displayTabsCheck.addSelectionListener(new SelectionAdapter() {

      /** Disable tab controls if tabs are disabled */
      public void widgetSelected(SelectionEvent e) {
        setTabControlsEnabled(displayTabsCheck.getSelection());

        /** Show warning label in case a restart of RSSOwl is required */
        setWarningMessage((GlobalSettings.displaySingleTab == displayTabsCheck.getSelection()) ? GUI.i18n.getTranslation("LABEL_RESTART") : null);
      }
    });

    /** Focus new opened tabs */
    focusNewTabsCheck = new Button(tabStyleGroup, SWT.CHECK);
    focusNewTabsCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    focusNewTabsCheck.setSelection(propertyChangeManager.isFocusNewTabs());
    focusNewTabsCheck.setText(GUI.i18n.getTranslation("BUTTON_FOCUS_TABS"));
    focusNewTabsCheck.setFont(dialogFont);

    /** Show the Close Button */
    showCloseButtonCheck = new Button(tabStyleGroup, SWT.CHECK);
    showCloseButtonCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    showCloseButtonCheck.setSelection(propertyChangeManager.isShowTabCloseButton());
    showCloseButtonCheck.setText(GUI.i18n.getTranslation("BUTTON_SHOW_TAB_CLOSE"));
    showCloseButtonCheck.setFont(dialogFont);
    showCloseButtonCheck.addSelectionListener(new SelectionAdapter() {

      /** Disable tab controls if tabs are disabled */
      public void widgetSelected(SelectionEvent e) {

        /** Show warning label in case a restart of RSSOwl is required */
        setWarningMessage((GlobalSettings.showTabCloseButton != showCloseButtonCheck.getSelection()) ? GUI.i18n.getTranslation("LABEL_RESTART") : null);
      }
    });

    /** Misc. View related settings */
    Group settingsGroup = new Group(composite, SWT.NONE);
    settingsGroup.setLayoutData(LayoutDataShop.createGridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL, 2));
    settingsGroup.setText(GUI.i18n.getTranslation("MENU_MISC"));
    settingsGroup.setLayout(new GridLayout(2, false));
    settingsGroup.setFont(FontShop.dialogFont);

    /** Use browser for Newstext */
    if (GlobalSettings.useInternalBrowser()) {
      useBrowserForNewstextCheck = new Button(settingsGroup, SWT.CHECK);
      useBrowserForNewstextCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
      useBrowserForNewstextCheck.setSelection(propertyChangeManager.isUseBrowserForNewsText());
      useBrowserForNewstextCheck.setText(GUI.i18n.getTranslation("MENU_BROWSER_FOR_NEWSTEXT"));
      useBrowserForNewstextCheck.setFont(dialogFont);
    }

    /** Show channel info */
    showChannelInfoCheck = new Button(settingsGroup, SWT.CHECK);
    showChannelInfoCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    showChannelInfoCheck.setSelection(propertyChangeManager.isShowChannelInfo());
    showChannelInfoCheck.setText(GUI.i18n.getTranslation("MENU_CHANNELINFO"));
    showChannelInfoCheck.setFont(dialogFont);

    /** Display errors */
    showErrorsCheck = new Button(settingsGroup, SWT.CHECK);
    showErrorsCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    showErrorsCheck.setSelection(propertyChangeManager.isShowErrors());
    showErrorsCheck.setText(GUI.i18n.getTranslation("BUTTON_SHOW_ERRORS"));
    showErrorsCheck.setFont(dialogFont);

    /** Set state of tab controls */
    setTabControlsEnabled(displayTabsCheck.getSelection());

    /** Set Mnemonics to Buttons */
    if (GlobalSettings.useInternalBrowser())
      WidgetShop.initMnemonics(new Button[] { simpleTabRadio, curvedTabRadio, displayTabsCheck, focusNewTabsCheck, showCloseButtonCheck, useBrowserForNewstextCheck, showChannelInfoCheck, showErrorsCheck });
    else
      WidgetShop.initMnemonics(new Button[] { simpleTabRadio, curvedTabRadio, displayTabsCheck, focusNewTabsCheck, showCloseButtonCheck, showChannelInfoCheck, showErrorsCheck });

    /** Warning Label in case a restart is needed */
    errorMessageLabel = new CLabel(composite, SWT.NONE);
    errorMessageLabel.setFont(dialogFont);
    errorMessageLabel.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, true, false, 2, 1));
    ((GridData) errorMessageLabel.getLayoutData()).heightHint = 32;

    /** Show warning label in case a restart of RSSOwl is required */
    if (GlobalSettings.displaySingleTab == displayTabsCheck.getSelection())
      setWarningMessage(GUI.i18n.getTranslation("LABEL_RESTART"));
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#restoreButtonPressed()
   */
  protected void restoreButtonPressed() {
    twoColumnGUIRadio.setSelection(true);
    threeColumnGUIRadio.setSelection(false);
    showChannelInfoCheck.setSelection(true);
    simpleTabRadio.setSelection(true);
    displayTabsCheck.setSelection(true);
    showCloseButtonCheck.setSelection(true);
    curvedTabRadio.setSelection(false);
    focusNewTabsCheck.setSelection(true);
    showErrorsCheck.setSelection(true);

    if (GlobalSettings.useInternalBrowser())
      useBrowserForNewstextCheck.setSelection(GlobalSettings.isWindows());

    /** Set state of tab controls */
    setTabControlsEnabled(displayTabsCheck.getSelection());

    /** Show warning label in case a restart of RSSOwl is required */
    setWarningMessage((GlobalSettings.displaySingleTab == displayTabsCheck.getSelection()) ? GUI.i18n.getTranslation("LABEL_RESTART") : null);
  }

  /**
   * Enable / Disable tab controls
   * 
   * @param enabled TRUE if enabled
   */
  void setTabControlsEnabled(boolean enabled) {
    focusNewTabsCheck.setEnabled(enabled);
    simpleTabRadio.setEnabled(enabled);
    curvedTabRadio.setEnabled(enabled);
    showCloseButtonCheck.setEnabled(enabled);
  }
}