/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.statusline;

import net.sourceforge.rssowl.controller.thread.ExtendedThread;

/**
 * Objects from this class represent a Job in RSSOwl, which is currently
 * performed. This may be the loading, reloading or searching of a newsfeed. A
 * job may either be created from an aggregation, or not.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class LoadJob {
  private ExtendedThread aggregationThread;
  private boolean isFromAggregation;
  private ExtendedThread loadingThread;
  private int style;
  private String title;

  /**
   * Instantiate a new LoadJob
   * 
   * @param identifier The unique identifier (e.g. the URL of the feed)
   * @param title The title of the job (e.g. the Title of the feed)
   * @param isFromAggregation If TRUE the job is created from an aggregation
   * @param style The style of the job (reload, load or search)
   * @param loadingThread The thread that is running the job
   */
  public LoadJob(String identifier, String title, boolean isFromAggregation, int style, ExtendedThread loadingThread) {
    this.isFromAggregation = isFromAggregation;
    this.style = style;
    this.loadingThread = loadingThread;
    this.title = (title != null) ? title : identifier;
  }

  /**
   * Cancel this job. This is done by stopping or interrupting the Thread(s)
   * that is performing the job.
   * 
   * @param forceAggregationStop If TRUE, forces the Thread that is running the
   * aggregation to stop. This only works in case this job is part of an
   * aggregation.
   */
  public synchronized void cancelJob(boolean forceAggregationStop) {
    loadingThread.stopThread();

    /** If wanted, cancel entire aggregation */
    if (aggregationThread != null && forceAggregationStop)
      aggregationThread.stopThread();
  }

  /**
   * Get the style of this Job. That is either loading, reloading or searching
   * of a feed.
   * 
   * @return Returns the style.
   */
  public int getStyle() {
    return style;
  }

  /**
   * Get the title of the Job.
   * 
   * @return Returns the title.
   */
  public String getTitle() {
    return title;
  }

  /**
   * If this job is from an aggregation, this Method returns TRUE
   * 
   * @return Returns the isFromAggregation.
   */
  public boolean isFromAggregation() {
    return isFromAggregation;
  }

  /**
   * Set the aggregation Thread to this Job, if it is from an aggregation.
   * 
   * @param aggregationThread The aggregationThread to set.
   */
  public void setAggregationThread(ExtendedThread aggregationThread) {
    this.aggregationThread = aggregationThread;
  }
}