/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.dao;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.model.Category;
import net.sourceforge.rssowl.util.shop.StringShop;

/**
 * Custom exception that occurs if there was an error while parsing a RSS XML
 * and creating the Channel
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.0
 */
public class NewsfeedFactoryException extends Exception {

  /** Status Codes: 401 Unauthorized */
  public static final int ERROR_AUTH_REQUIRED = 1;

  /** Status Codes: 4xx except for 401 */
  public static final int ERROR_FILE_NOT_FOUND = 0;

  /** File is not a valid XML document */
  public static final int ERROR_INVALID_NEWSFEED = 3;

  /** File is not a valid Newsfeed */
  public static final int ERROR_INVALID_XML = 2;

  /** RSSOwl is in offline mode */
  public static final int ERROR_WORKING_OFFLINE = 4;

  /** Unique serialVersionUID for serialization */
  static final long serialVersionUID = 3256440292004541489L;

  private String httpStatus;
  private String msg;
  private String reason;
  private String title;
  private int type;
  private String url;

  /**
   * Instantiate a new NewsfeedFactoryException
   * 
   * @param url URL of the loading Newsfeed
   * @param reason Additional info about the Exception
   * @param httpStatus The HTTP status of the connection
   * @param type The type of the error
   */
  public NewsfeedFactoryException(String url, String reason, String httpStatus, int type) {
    this.url = url;
    this.reason = reason;
    this.httpStatus = httpStatus;
    this.type = type;
    msg = GUI.i18n.getTranslation("ERROR_NEWSFEED_GENERAL");

    if (url != null)
      title = Category.getTitleForLink(url);
  }

  /**
   * Get the HTTP Status
   * 
   * @return String The HTTP Status
   */
  public String getHttpStatus() {
    return httpStatus;
  }

  /**
   * @see java.lang.Throwable#getMessage()
   */
  public String getMessage() {
    StringBuffer message = new StringBuffer();
    message.append(msg);

    if (getReason() != null) {
      message.append("\n\n");
      message.append(GUI.i18n.getTranslation("ERROR_REASON"));
      message.append(": ");
      message.append(getReason());
    }

    if (getHttpStatus() != null) {
      message.append("\n");
      message.append(GUI.i18n.getTranslation("ERROR_HTTP_STATUS"));
      message.append(": ");
      message.append(getHttpStatus());
    }

    return message.toString();
  }

  /**
   * Get the error message
   * 
   * @return String The error message
   */
  public String getMsg() {
    return msg;
  }

  /**
   * Get the error reason in dependance of the given error type
   * 
   * @return String The error reason
   */
  public String getReason() {
    String reason = null;

    switch (type) {
      case ERROR_FILE_NOT_FOUND:
        reason = GUI.i18n.getTranslation("ERROR_FILE_NOT_FOUND");
        break;

      case ERROR_AUTH_REQUIRED:
        reason = GUI.i18n.getTranslation("ERROR_AUTH_REQUIRED");
        break;

      case ERROR_INVALID_XML:
        reason = GUI.i18n.getTranslation("ERROR_NOT_A_XML");
        break;

      case ERROR_INVALID_NEWSFEED:
        reason = GUI.i18n.getTranslation("ERROR_NOT_A_RSS");
        break;

      case ERROR_WORKING_OFFLINE:
        reason = GUI.i18n.getTranslation("ERROR_WORKING_OFFLINE");
        break;
    }

    /** A reason was already supplied */
    if (StringShop.isset(this.reason) && this.reason.indexOf(" ") < 0)
      reason += " (" + this.reason + ")";
    else if (StringShop.isset(this.reason) && this.reason.indexOf(" ") >= 0)
      reason += ". " + this.reason;

    return reason;
  }

  /**
   * Get the Title of the loading Newsfeed
   * 
   * @return String The title.
   */
  public String getTitle() {
    return title;
  }

  /**
   * Get the URL of the loading Newsfeed
   * 
   * @return String The URL.
   */
  public String getUrl() {
    return url;
  }
}