/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.dao.feedparser;

import net.sourceforge.rssowl.dao.NewsfeedFactoryException;
import net.sourceforge.rssowl.model.Channel;
import net.sourceforge.rssowl.model.Enclosure;
import net.sourceforge.rssowl.model.NewsItem;
import net.sourceforge.rssowl.util.shop.StringShop;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.Text;

import java.util.Iterator;
import java.util.List;

/**
 * Parser for the RSS 2.0 Format <br />
 * Specification: http://blogs.law.harvard.edu/tech/rss
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class RSS_2_0_Parser extends BaseRSSParser {

  /**
   * Instantiate a new Parser for this format.
   * 
   * @param document The document containing the data to parse
   * @param rssChannel The Channel to fill with data from the document
   * @param url The URL of the Newsfeed that is parsed
   * @param nameSpaces Possible Namespaces of the XML document
   * @throws NewsfeedFactoryException If any error occurs
   */
  public RSS_2_0_Parser(Document document, Channel rssChannel, String url, Namespace nameSpaces[]) throws NewsfeedFactoryException {
    super(document, rssChannel, url, nameSpaces);
  }

  /**
   * @see net.sourceforge.rssowl.dao.feedparser.AbstractFeedParser#parse()
   */
  public void parse() throws NewsfeedFactoryException {

    /** Parse common attributes */
    super.parse();

    /** Temp String */
    String str;

    /** Get Channel Element */
    Element channel = getChildElement(root, "channel");

    /** TTL of channel */
    str = getChildValue("ttl", channel);
    if (StringShop.isset(str))
      rssChannel.setTtl(str);

    /** Category of channel */
    str = getChildValue("category", channel);
    if (StringShop.isset(str))
      rssChannel.setCategory(str);

    /** Generator of channel */
    str = getChildValue("generator", channel);
    if (StringShop.isset(str))
      rssChannel.setGenerator(str);

    /** Get the Channel Newsitems */
    List items = getChildren(channel, "item");

    /** Try validating for items in root element */
    if (items.size() == 0)
      items = getChildren(root, "item");

    Iterator itemsIt = items.iterator();

    /** For each newsitem */
    while (itemsIt.hasNext()) {
      Element item = (Element) itemsIt.next();
      NewsItem newsItem = new NewsItem();

      /** Title of NewsItem */
      str = getChildValue("title", item);
      if (StringShop.isset(str))
        newsItem.setTitle(str);

      /** Link of NewsItem */
      str = getChildValue("link", item);
      if (StringShop.isset(str)) {
        newsItem.setLink(str);
      }

      /** Description of NewsItem */
      str = getChildValue("description", item);
      if (StringShop.isset(str))
        newsItem.setDescription(str);

      /** Set part of description as title if title is not available */
      if (!StringShop.isset(newsItem.getTitle()) && newsItem.getDescription() != null)
        newsItem.setTitle(newsItem.getDescription(), true);

      /** pubDate of NewsItem */
      str = getChildValue("pubDate", item);
      if (StringShop.isset(str)) {
        newsItem.setPubDate(str, true);
        rssChannel.addAvailableNewsItemInfo("TABLE_HEADER_PUBDATE");
      }

      /** Author of NewsItem */
      str = getChildValue("author", item);
      if (StringShop.isset(str)) {
        newsItem.setAuthor(str);
        rssChannel.addAvailableNewsItemInfo("TABLE_HEADER_AUTHOR");
      }

      /** Category of NewsItem */
      str = getChildValue("category", item);
      if (StringShop.isset(str)) {
        newsItem.setCategory(str);
        rssChannel.addAvailableNewsItemInfo("TABLE_HEADER_CATEGORY");
      }

      /** GUID of NewsItem */
      Element guid = getChildElement(item, "guid");

      if (guid != null) {
        newsItem.setGuid(guid.getTextTrim());

        /** If Title is NULL, set guid as title */
        if (!StringShop.isset(newsItem.getTitle()))
          newsItem.setTitle(Text.normalizeString(newsItem.getGuid()));
      }

      /** Source of NewsItem */
      Element source = getChildElement(item, "source");

      if (source != null) {
        if (getAttributeValue(source, "url") != null) {
          newsItem.setSource(getAttributeValue(source, "url").trim());
        } else if (StringShop.isset(source.getText())) {
          newsItem.setSource(source.getTextTrim());
        }
      }

      /** Enclosures of NewsItem */
      List enclosures = getChildren(item, "enclosure");

      if (enclosures != null) {
        Iterator enclosuresIt = enclosures.iterator();

        while (enclosuresIt.hasNext()) {
          Element enclosure = (Element) enclosuresIt.next();
          Enclosure rssEnclosure = new Enclosure();

          /** URL of enclosure */
          if (getAttributeValue(enclosure, "url") != null)
            rssEnclosure.setUrl(getAttributeValue(enclosure, "url"));

          /** Length of enclosure */
          if (getAttributeValue(enclosure, "length") != null)
            rssEnclosure.setLength(getAttributeValue(enclosure, "length"));

          /** Type of enclosure */
          if (getAttributeValue(enclosure, "type") != null)
            rssEnclosure.setType(getAttributeValue(enclosure, "type"));

          newsItem.insertEnclosure(rssEnclosure);
        }
      }

      /** Comments to this NewsItem */
      List comments = getChildren(item, "comments");

      if (comments != null) {
        Iterator commentsIt = comments.iterator();

        /** For each comment */
        while (commentsIt.hasNext()) {
          String comment = ((Element) commentsIt.next()).getText();

          if (StringShop.isset(comment))
            newsItem.addComment(comment);
        }
      }

      /** Possible <origurl>Tag set by AmphetaRate for feed XML url */
      str = getChildValue("origurl", item);
      if (StringShop.isset(str))
        newsItem.setOrigurl(str);

      /** Parse dublin core module */
      parseDCModule(item, newsItem);

      /** Parse content module */
      parseContentModule(item, newsItem);

      /** Add NewsItem to Channel */
      rssChannel.insertItem(newsItem);
    }
  }
}