/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.archive;

import java.io.File;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * An archive item is a Model Object that stores archive entries. Each archive
 * entry consists of a newstitle, newslink and newsdate. These are used later,
 * to determine which news of the item were read and which are unread. The
 * archive item has a unique identifier: Its feedurl. The file attribute is set
 * to a unique file to save the item later.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class ArchiveItem {
  private boolean dirty;
  private Hashtable entries;
  private String feedurl;
  private File file;

  /**
   * Instantiate a new ArchiveItem
   * 
   * @param feedurl The URL of the newsfeed
   */
  public ArchiveItem(String feedurl) {
    this.feedurl = feedurl;
    entries = new Hashtable();

    /** Mark as in need of being saved on flush */
    dirty = true;
  }

  /**
   * Add a new archive entry to this archive item
   * 
   * @param rssOwlArchiveEntry The archive entry consists of a newstitle and
   * newslink
   */
  public void addEntry(ArchiveEntry rssOwlArchiveEntry) {
    entries.put(rssOwlArchiveEntry.toID(), rssOwlArchiveEntry);

    /** Mark as in need of being saved on flush */
    dirty = true;
  }

  /**
   * @return Returns the entries.
   */
  public Hashtable getEntries() {
    return entries;
  }

  /**
   * @return Returns the feedurl.
   */
  public String getFeedurl() {
    return feedurl;
  }

  /**
   * @return Returns the file.
   */
  public File getFile() {
    return file;
  }

  /**
   * Check wether this Item has changed since it has been loaded from the File.
   * 
   * @return boolean TRUE if this Item has changed since it has been loaded and
   * requires to be saved to the File again, FALSE otherwise.
   */
  public boolean isDirty() {
    return dirty;
  }

  /**
   * Check if the given newsitem is read or not. This is simply done by
   * checking, if the newsitem exists as entry in the connected archive item.
   * 
   * @param newstitle Title of the news
   * @param newslink Link of the news
   * @param newsdate The publish date of the news
   * @return TRUE if the news was read
   */
  public boolean isNewsRead(String newstitle, String newslink, String newsdate) {
    return entries.containsKey(ArchiveEntry.toID(newstitle, newslink, newsdate));
  }

  /**
   * Remove an archive entry from this archive item.
   * 
   * @param entry The archive entry
   */
  public void removeEntry(ArchiveEntry entry) {
    removeEntry(entry.getNewsTitle(), entry.getNewsLink(), entry.getNewsdate());
  }

  /**
   * Remove an archive entry from this archive item.
   * 
   * @param newstitle Title of the news
   * @param newslink Link of the news
   * @param newsdate The publish date of the news
   */
  public void removeEntry(String newstitle, String newslink, String newsdate) {
    String entryID = ArchiveEntry.toID(newstitle, newslink, newsdate);

    /** Remove the entry if given */
    entries.remove(entryID);

    /** Mark as in need of being saved on flush */
    dirty = true;
  }

  /**
   * Set wether this Item has changed since it has been loaded from the File and
   * needs to be saved to the File again on shutdown.
   * 
   * @param dirty Set to TRUE if this Item has changed since it has been loaded
   * and requires to be saved to the File again, FALSE otherwise.
   */
  public void setDirty(boolean dirty) {
    this.dirty = dirty;
  }

  /**
   * @param file The file to set.
   */
  public void setFile(File file) {
    this.file = file;
  }

  /**
   * @see java.lang.Object#toString()
   */
  public String toString() {
    StringBuffer strBuf = new StringBuffer();
    String fileName = (file != null) ? file.getName() : "NULL";
    strBuf.append("FeedURL: " + feedurl + "   FileName: " + fileName + "\n");

    /** For each entry */
    Enumeration elements = entries.elements();
    while (elements.hasMoreElements())
      strBuf.append(elements.nextElement());

    return new String(strBuf);
  }
}