/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <urlglib/url_info.h>
#include <urlglib/ug_i18n.h>
#include "batch_dialog.h"

#include <gdk/gdkkeysyms.h>

static void batch_dialog_decide_sensitive(BatchDialog* bd)
{
	gboolean  responsive = FALSE;
	UrlInfo*  uinfo;

	const gchar* str = gtk_entry_get_text (GTK_ENTRY(bd->entry));

	uinfo = url_info_new (str);

	if (uinfo->protocol && uinfo->address && uinfo->filename) {
		for (; *str ; ++str) {
			if ( *str == '*' ) {
				responsive=TRUE;
				break;
			}
		}
	}
	gtk_dialog_set_response_sensitive (GTK_DIALOG (bd->self),
	                                   GTK_RESPONSE_OK,
	                                   responsive);
	url_info_free (uinfo);
}

// signal handler --------------------------------------------------------------

static void radio1_clicked(GtkWidget* widget, BatchDialog* bd)
{
	gtk_widget_set_sensitive (bd->spin_from, TRUE);
	gtk_widget_set_sensitive (bd->spin_to, TRUE);
	gtk_widget_set_sensitive (bd->spin_width, TRUE);
	gtk_widget_set_sensitive (bd->entry_a, FALSE);
	gtk_widget_set_sensitive (bd->entry_z, FALSE);
}

static void radio2_clicked(GtkWidget* widget, BatchDialog* bd)
{
	gtk_widget_set_sensitive (bd->spin_from, FALSE);
	gtk_widget_set_sensitive (bd->spin_to, FALSE);
	gtk_widget_set_sensitive (bd->spin_width, FALSE);
	gtk_widget_set_sensitive (bd->entry_a, TRUE);
	gtk_widget_set_sensitive (bd->entry_z, TRUE);
}

BatchDialog* batch_dialog_new (GtkWindow* parent)
{
	BatchDialog* batdlg = g_malloc (sizeof (BatchDialog));

	batch_dialog_init (batdlg, parent);
	return batdlg;
}

void batch_dialog_destroy (BatchDialog* bd)
{
	batch_dialog_finalize (bd);
	g_free (bd);
}

void batch_dialog_init (BatchDialog* bd, GtkWindow* parent)
{
	GtkTable*  table;
	GtkWidget* vbox;
	GtkWidget* label;
	GtkAdjustment* spinner_adj;
	GtkEditable* editable;

	// initial dialog
	bd->self = gtk_dialog_new_with_buttons (_("Batch mission"),
	                                        parent,
	                                        GTK_DIALOG_DESTROY_WITH_PARENT,
	                                        GTK_STOCK_CANCEL,
	                                        GTK_RESPONSE_CANCEL,
	                                        GTK_STOCK_OK,
	                                        GTK_RESPONSE_OK,
	                                        NULL);
	gtk_window_set_resizable (GTK_WINDOW (bd->self), FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(bd->self), GTK_RESPONSE_OK, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER(bd->self), 5);

	// attach & pack widgets
	table = (GtkTable*)gtk_table_new (5, 6, FALSE);
	vbox = GTK_DIALOG (bd->self)->vbox;
	gtk_box_set_spacing (GTK_BOX(vbox), 5);
	gtk_box_pack_start (GTK_BOX(vbox), GTK_WIDGET (table), FALSE, FALSE, 0);

	// URL entry
	label = gtk_label_new_with_mnemonic (_("_URL:"));
	bd->entry = gtk_entry_new ();
	editable = GTK_EDITABLE (bd->entry);
	g_signal_connect_swapped (editable, "changed", G_CALLBACK(batch_dialog_decide_sensitive), bd);

	gtk_table_attach (table, label, 0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 3, 3);
	gtk_table_attach (table, bd->entry, 1, 6, 0, 1,
	                  GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);

	label = gtk_label_new (_("Example 1:"));
	gtk_table_attach (table, label, 0, 1, 1, 2, GTK_SHRINK, GTK_SHRINK, 3, 3);
	label = gtk_label_new ("http://example1.format1.com/path_name/pre*.jpg");
	gtk_table_attach (table, label, 1, 6, 1, 2, GTK_SHRINK, GTK_SHRINK, 3, 3);

	label = gtk_label_new (_("Example 2:"));
	gtk_table_attach (table, label, 0, 1, 2, 3, GTK_SHRINK, GTK_SHRINK, 3, 3);
	label = gtk_label_new ("http://example2.format2.com/path_name/fn(*).jpg");
	gtk_table_attach (table, label, 1, 6, 2, 3, GTK_SHRINK, GTK_SHRINK, 3, 3);

	// spin "From"
	bd->radio_button = gtk_radio_button_new_with_mnemonic (NULL, _("_From:"));
	g_signal_connect (bd->radio_button, "clicked",
	                  G_CALLBACK(radio1_clicked), bd);
	gtk_table_attach (table, bd->radio_button, 0, 1, 3, 4,
	                  GTK_SHRINK, GTK_SHRINK, 3, 3);
	spinner_adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0,
	                                                    99999.0, 1.0,
	                                                    5.0, 5.0);
	bd->spin_from = gtk_spin_button_new (spinner_adj, 1.0, 0);
	gtk_table_attach (table, bd->spin_from, 1, 2, 3, 4,
	                  GTK_SHRINK, GTK_SHRINK, 3, 3);

	// spin "To"
	label = gtk_label_new_with_mnemonic (_("_To:"));
	gtk_table_attach (table, label, 2, 3, 3, 4, GTK_SHRINK, GTK_SHRINK, 3, 3);
	spinner_adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0,
	                                                    99999.0, 1.0,
	                                                    5.0, 5.0);
	bd->spin_to = gtk_spin_button_new (spinner_adj, 1.0, 0);
	gtk_label_set_mnemonic_widget (GTK_LABEL(label), bd->spin_to);
	gtk_table_attach (table, bd->spin_to, 3, 4, 3, 4,
	                  GTK_SHRINK, GTK_SHRINK, 3, 3);

	// spin "Width"
	label = gtk_label_new_with_mnemonic (_("_Wildcard bytes:"));
	gtk_table_attach (table, label, 4, 5, 3, 4, GTK_SHRINK, GTK_SHRINK, 3, 3);
	spinner_adj = (GtkAdjustment *) gtk_adjustment_new (2.0, 1.0,
	                                                    20.0, 1.0,
	                                                    5.0, 5.0);
	bd->spin_width = gtk_spin_button_new (spinner_adj, 1.0, 0);
	gtk_label_set_mnemonic_widget (GTK_LABEL(label), bd->spin_width);

	gtk_table_attach (table, bd->spin_width, 5, 6, 3, 4,
	                  GTK_SHRINK, GTK_SHRINK, 3, 3);

	// a - z
	bd->radio_button = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(bd->radio_button),
	                                                                   _("F_rom:"));
	g_signal_connect (bd->radio_button, "clicked",
	                  G_CALLBACK(radio2_clicked), bd);
	gtk_table_attach (table, bd->radio_button, 0, 1, 4, 5,
	                  GTK_SHRINK, GTK_SHRINK, 3, 3);
	bd->entry_a = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY(bd->entry_a), "a");
	gtk_entry_set_max_length (GTK_ENTRY(bd->entry_a), 1);
	gtk_entry_set_width_chars (GTK_ENTRY(bd->entry_a), 2);
	gtk_widget_set_sensitive (bd->entry_a, FALSE);
	gtk_table_attach (table, bd->entry_a, 1, 2, 4, 5,
	                  GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);

	bd->entry_z = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY(bd->entry_z), "z");
	gtk_entry_set_max_length (GTK_ENTRY(bd->entry_z), 1);
	gtk_entry_set_width_chars (GTK_ENTRY(bd->entry_z), 2);
	gtk_widget_set_sensitive (bd->entry_z, FALSE);
	gtk_table_attach (table, bd->entry_z, 3, 4, 4, 5,
	                  GTK_FILL | GTK_EXPAND, GTK_SHRINK, 3, 3);

	label = gtk_label_new (_("Warning: case sensitive"));
	gtk_table_attach (table, label, 4, 6, 4, 5,
	                  GTK_FILL, GTK_SHRINK, 3, 3);
	gtk_widget_show_all (vbox);
}

void batch_dialog_finalize (BatchDialog* bd)
{
	gtk_widget_destroy (bd->self);
}

gint batch_dialog_get_mode (BatchDialog* bd)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (bd->radio_button)) )
		return BATCH_MODE_CHAR;
	return BATCH_MODE_NUM;
}

gint batch_dialog_run (BatchDialog* bd)
{
	gint response;

	response = gtk_dialog_run (GTK_DIALOG (bd->self));

	if ( response==GTK_RESPONSE_CANCEL || response==GTK_RESPONSE_DELETE_EVENT)
		return BATCH_MODE_NONE;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (bd->radio_button)) )
		return BATCH_MODE_CHAR;

	return BATCH_MODE_NUM;
}

const gchar* batch_dialog_get_pattern (BatchDialog* bd)
{
	return gtk_entry_get_text (GTK_ENTRY(bd->entry));
}

gchar batch_dialog_get_char_min (BatchDialog* bd)
{
	return *gtk_entry_get_text (GTK_ENTRY(bd->entry_a));
}

gchar batch_dialog_get_char_max (BatchDialog* bd)
{
	return *gtk_entry_get_text (GTK_ENTRY(bd->entry_z));
}

gint batch_dialog_get_int_min (BatchDialog* bd)
{
	return gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(bd->spin_from));
}

gint batch_dialog_get_int_max (BatchDialog* bd)
{
	return gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(bd->spin_to));
}

gint batch_dialog_get_int_width (BatchDialog* bd)
{
	return gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(bd->spin_width));
}
